/*

 Copyright 2001-2003  The Apache Software Foundation 

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 */
package jp.cssj.sakae.opentype;

import java.io.IOException;
import java.io.RandomAccessFile;

import jp.cssj.sakae.opentype.cff.CffGlyphList;
import jp.cssj.sakae.opentype.cff.CffTable;
import jp.cssj.sakae.opentype.table.CmapTable;
import jp.cssj.sakae.opentype.table.GlyfTable;
import jp.cssj.sakae.opentype.table.HeadTable;
import jp.cssj.sakae.opentype.table.HheaTable;
import jp.cssj.sakae.opentype.table.HmtxTable;
import jp.cssj.sakae.opentype.table.LocaTable;
import jp.cssj.sakae.opentype.table.MaxpTable;
import jp.cssj.sakae.opentype.table.NameTable;
import jp.cssj.sakae.opentype.table.Os2Table;
import jp.cssj.sakae.opentype.table.PostTable;
import jp.cssj.sakae.opentype.table.Table;
import jp.cssj.sakae.opentype.table.TableDirectory;
import jp.cssj.sakae.opentype.table.TableFactory;
import jp.cssj.sakae.opentype.table.VheaTable;
import jp.cssj.sakae.opentype.table.VmtxTable;
import jp.cssj.sakae.opentype.truetype.TrueTypeGlyphList;

/**
 * The TrueType font.
 * 
 * @version $Id: Font.java,v 1.2 2007-05-06 04:27:02 miyabe Exp $
 * @author <a href="mailto:david@steadystate.co.uk">David Schweinsberg</a>
 */
public class OpenTypeFont {
	private TableDirectory tableDirectory = null;

	private Table[] tables;

	private Os2Table os2;

	private CmapTable cmap;

	private HeadTable head;

	private HheaTable hhea;

	private HmtxTable hmtx;

	private VheaTable vhea;

	private VmtxTable vmtx;

	private MaxpTable maxp;

	private NameTable name;

	private PostTable post;

	private GlyphList glyphList;

	public OpenTypeFont(RandomAccessFile raf) throws IOException {
		this.tableDirectory = new TableDirectory(raf);
		this.tables = new Table[this.tableDirectory.getNumTables()];

		// Load each of the tables
		for (int i = 0; i < this.tableDirectory.getNumTables(); i++) {
			this.tables[i] = TableFactory.create(this.tableDirectory.getEntry(i), raf);
		}

		// Get references to commonly used tables
		this.os2 = (Os2Table) getTable(Table.OS_2);
		this.cmap = (CmapTable) getTable(Table.cmap);
		this.head = (HeadTable) getTable(Table.head);
		this.hhea = (HheaTable) getTable(Table.hhea);
		this.hmtx = (HmtxTable) getTable(Table.hmtx);
		this.vhea = (VheaTable) getTable(Table.vhea);
		this.vmtx = (VmtxTable) getTable(Table.vmtx);
		this.maxp = (MaxpTable) getTable(Table.maxp);
		this.name = (NameTable) getTable(Table.name);
		this.post = (PostTable) getTable(Table.post);

		// Initialize the tables that require it
		this.hmtx.init(this.hhea.getNumberOfHMetrics(), this.maxp.getNumGlyphs()
				- this.hhea.getNumberOfHMetrics());
		if (this.vmtx != null) {
			this.vmtx.init(this.vhea.getNumberOfHMetrics(), this.maxp.getNumGlyphs()
					- this.vhea.getNumberOfHMetrics());
		}

		GlyfTable glyf = (GlyfTable) getTable(Table.glyf);
		if (glyf != null) {
			// TrueType
			LocaTable loca = (LocaTable) getTable(Table.loca);
			glyf.init(loca);
			loca.init(this.maxp.getNumGlyphs(), this.head.getIndexToLocFormat() == 0);
			this.glyphList = new TrueTypeGlyphList(glyf, this.head, this.hmtx,
					this.maxp);
		} else {
			CffTable cff = (CffTable) getTable(Table.CFF);
			if (cff != null) {
				// CFF
				cff.init(this.hmtx);
				this.glyphList = new CffGlyphList(cff, this.head,
						this.maxp);
			} else {
				throw new IOException("Unsupported font file format.");
			}
		}

	}

	public Table getTable(int tableType) {
		for (int i = 0; i < this.tables.length; i++) {
			if ((this.tables[i] != null)
					&& (this.tables[i].getType() == tableType)) {
				return this.tables[i];
			}
		}
		return null;
	}

	public Os2Table getOS2Table() {
		return this.os2;
	}

	public CmapTable getCmapTable() {
		return this.cmap;
	}

	public HeadTable getHeadTable() {
		return this.head;
	}

	public HheaTable getHheaTable() {
		return this.hhea;
	}

	public HmtxTable getHmtxTable() {
		return this.hmtx;
	}

	public VheaTable getVheaTable() {
		return this.vhea;
	}

	public VmtxTable getVmtxTable() {
		return this.vmtx;
	}

	public MaxpTable getMaxpTable() {
		return this.maxp;
	}

	public NameTable getNameTable() {
		return this.name;
	}

	public PostTable getPostTable() {
		return this.post;
	}

	public int getAscent() {
		return this.hhea.getAscender();
	}

	public int getDescent() {
		return this.hhea.getDescender();
	}

	public int getNumGlyphs() {
		return this.maxp.getNumGlyphs();
	}

	public Glyph getGlyph(int i) {
		return this.glyphList.getGlyph(i);
	}

	public TableDirectory getTableDirectory() {
		return this.tableDirectory;
	}
}
