///////////////////////////////////////////////////////////////
// Processingによる二分法の対話的プログラム開発用フレームワーク
//    Crowbar（クロウバー）
//                                  白井　達也（Tatsuya Shirai)
//                       shirai＠mech.suzuka-ct.ac.jp, @tatsuva
//                                       2012/04/20    Ver. 0.9
//                                       2012/05/14    Ver. 1.0
//                                       2012/05/15    Ver. 1.1
///////////////////////////////////////////////////////////////
//
// [A] Main()中で使用したいパラメータをsetup()内で宣言する
// setup()内でパラメータを定義： 
//  パラメータの値は基本的に文字列として扱う．
//  それをgetFloat()やgetInt()などの形で数値に変換して取得する．勿論，getStr()で文字列のまま受け取ることも可能．
//  ただし実際には数値をパラメータとして受け取ることが大半だと思われるので，基本形は数値入力を前提としたキー入力
//  制限規則とし，文字列全てを受け入れるにはdefineStr()を用いることとする．
//  0) 基本形
//     crowbar.define(String)：数値入力を前提としたキー入力制限規則．Stringは画面に表示されるパラメータの説明文
//  1) パラメータの入力可能文字制約（数字入力のみ/全ての文字列を受け入れる）
//     基本形は数値入力に必要なキー以外は無視する（0-9 + - .のみ）：フォーマットを規制する訳ではない
//     文字列入力全般を許可する場合はdefineStr(String)を用いる．
//     例）crowbar.defineStr(String)
//  2) 初期値を与える：ENTERキーのみの入力で選択される（trim()は行わないので空白スペースが混入したらＮＧ）
//     例）crowbar.define(String1, String2): String2が初期値．文字列で与えること
//  3) 選択肢（0-9のキーで選択の十択のみ）
//     例）crowbar.defineSel(String, String):    半角スペースで区切った文字列で与えること
//     例）crowbar.defineSel(String, String, 0); キーに対応する初期値を与えることができます．
//  パラメータは一つも与えなくてもＯＫ
//  パラメータの読み出しは，crowbar.getFloat()のように自動読み出しや
//                          crowbar.getFloat(0)のようにIndexを与えて読み出す．
//  インデックスを与えた場合，自動読み出しのインデックスはその次のIndexになる点に注意
//
// [B] Main()内にユーザプログラムを記述
// Main()内でパラメータをユーザプログラムの変数に渡す
//   crowbar.getFloat(int), crowbar.getInt(int), crowbar.getStr(int)
//   crowbar.getFloat/Int/Str()に引数を明示的に与えなかった場合は最後に読み
//   出されたindexの次の引数配列のデータが自動的に読みだされる
//  Main()関数は一度だけ実行して停止する．無限ループに入ったら'Q'キーを押しても止まらない．
//
// 【グラフィックス画面について】
// グラフィックス画面のサイズは横の文字数，縦の行数，フォントサイズにより自動計算されます．
// setup()内の   crowbar = new crowbarClass(60, 40); の引数で変更されます．
// crowbar.clrscr()で実行画面を消去する．ただし，この場合はテキストバッファもクリアされる．
// テキストバッファは消去したくない場合は，crowbar._clrscr()を実行すれば良い．
//
// crowbar.windowSize(pixel, pixel)でグラフィック用のウィンドウのサイズを自由に変えられる
// （それに合わせて最大の文字数／行数の値を再設定しない）
//
// crowbar.enableOverlap()で，ウィンドウからはみ出る文字列を折り返して次行に表示する．
// crowbar.disableOverlap()で，はみ出した文字列は表示しない．
//
// crowbar.disableDisplay()で，crowbar.write()/writeln()の出力をグラフィックス
// 　領域には出力しない．
// crowbar.enableDisplay()でグラフィックス出力を再開
// crowbar.switchDisplay(boolean)で変数により変更可能
//
// 逆に，PDEのテキストエリアに文字列を表示するのを抑制したい場合もある．
// その時には適時，以下のコマンドを実行する．
// crowbar.disableTextArea()で抑制
// crowbar.enableTextArea()で出力
//
// 全パラメータを入力した後に確認するために入力されたデータを表示する．
// もしこれを省きたいならば，setup()中などに，crowbar.noShowParameters()を記述する．
//
// crowbar.nonStop()指定時はMain()を実行終了後，draw()の終端に記述されたユーザ
// プログラムを繰り返し（Processing本来の動作として）実行する．
// 安全に終了するには，crowbar.stop()を実行する．
// crowbar.nonStop()時には[q]/[Q]キーによる強制終了は機能しない．
// crowbar.nonStop()はできるだけsetup()内に記述して下さい．
//
// Main()とuserDraw()実行中のcrowbar.write()/writeln()の出力は以下のコマンドを
// setup()内に記述することでテキストファイルに出力されます．
// ファイルが作成されるのはスケッチのフォルダ内です．（Ctrl+K）
// crowbar.enableLogOutput() : ファイル名は default.txt
// crowbar.enableLogOutput(ファイル名) : 指定したファイル名で記録
//
// プログラムの説明をパラメータ入力前に一度だけ表示することが可能です．
// setup()内に，crowbar.programComment(文字列) を記述して下さい．
// これは省略可能ですし，複数個の宣言も可能です．
//
// プログラムを再び始めるかどうかの確認中にテキスト画面のバッファを上下にスクロール可能です．
// キーボードの[-]または[←]で上に，[+}または[→]で下に半ページ下がります．
// キーボードの[↑]と[↓]で一行単位で上と下に画面がスクロールします．
// [y]を押して再びプログラムのパラメータ入力からやり直す際にはプログラム実行後の画面まで戻ります．
//
// テキスト出力をログファイルに記録するには，crowbar.enableLogOutput(ファイル名)をOptions()で宣言する．
// ファイル名が指定されない場合はdefault.txtというファイル名で保存される．
// 以上はVer.1.00.00の仕様．これを1.1.00以降では以下のように改める．
// テキスト出力を全てログファイルに記録するには，crowbar.startLogging(ファイル名)をOptions()で宣言する．
// ファイル名が指定されない場合はdefault.txtというファイル名で保存される．
// テキスト出力を一時停止するにはcrowbar.pauseLogging(), 再開するにはcrowbar.restartLogging()を記述する．
// テキスト出力を終了するにはcrowbar.stopLogging()を実行する．
// 改めて別のファイル名でログファイルに記録開始するにはcrowbar.startLogging()で異なるファイル名を指定する．
// 標準ではProcessingはテキストファイルの追記ができない（削除して新規作成される）ようなので，とりあえずは諦める．
// 後のバージョンでは追記を可能とする拡張を行うことも考えるが，一旦ファイルを全て読み込んでから改めて全て出力し直す必要があると思われる．
// 非常に効率が悪いので十分な調査を行ってから実装を考える．(2012/5/21)

// メインのインスタンスの宣言（絶対に削除してはいけません）
crowbarClass crowbar;

///////////////////////////
// テキストバッファのクラス
// （スクロール対応用）
///////////////////////////
// メモリの確保をリスト構造にするか配列にするか
// 効率を考えて配列によるページ単位の確保を当初は選択したが，応用性が悪い．
// たとえば１ページの行数を途中で自由に変えることができない．
// なお，あとから行が挿入されることは考えていない．
// しかしオーバーラップによる行数の再計算のことを考えるとリスト構造でも大差ない可能性が高い．
// 再描画も下から上に遡ることは無いので単向のリスト構造で良いと考えた
// オーバーラップかつスクロールに対応するための戦略：
// 　オーバーラップして折り返す分は次のラインデータに入れる．後日，実行ウィンドウのサイズ変更に
// 対応する際には，一旦，改行されたラインデータを繋げ直して，それからまたオーバーラップし直す．
// その際に行数が減るかも知れない．不要になったラインデータはリスト構造から切り離し，GCに開放させる．
// 一行分のクラス（最小単位）
class lineClass {
  protected int     no;    // 行番号．０から始まる
  protected String  line;
  protected boolean flag;  // 入力済みか？（空行を考慮．lineの文字数では判断できないため．さらにnewはできてもメモリは解放できない？ため）
  protected boolean overlap;  // 前の行の折り返しか（後日，実行ウィンドウのサイズ変更に対応するために）
  lineClass         next;  // 次の行の位置
  // コンストラクタ
  lineClass() {
    clear();
  }
  // クリア
  void clear() {
    no         = 0;
    line       = "";
    flag       = false;
    overlap    = false;
    next       = null;
  }
  // 行単位のデータのセット
  void set(String str, boolean over) {
    line    = str;
    flag    = true;
    overlap = over;
    if (next == null) next = new lineClass();  // リスト構造を再利用する場合は確保しない
    next.no = no + 1;
  }
}

///////////////////////////////
// テキストバッファ本体のクラス
class textbufferClass {
  private int   cwidth;              // 実行ウィンドウのテキストの文字数
  private int   fontSize;            // 実行ウィンドウのテキストのフォントサイズ
  lineClass     currentWriteLine;    // 現在の書き込み可能な行データ
  lineClass     topLine;             // 一番最初の行データ
  protected int totalLines;          // 出力済みのテキストバッファの行数を単純に返す（オーバーラップは考慮しない）
  protected int readingLine;         // 現在の読み出しされている行番号
  private String buff;               // 現在の未改行のバッファ

// コンストラクタ
  textbufferClass(int max_x, int fsize) {
    cwidth           = max_x;
    fontSize         = fsize;
    topLine          = new lineClass();
    totalLines       = 0;
    currentWriteLine = topLine;
    readingLine      = 0;
    buff = "";
  }
  // 総行数を正確に取得する際に用いる（いまのところ使用箇所は無い）
  int refreshTotalLines() {
    lineClass temp = topLine;
    totalLines = 0;
    while (temp.next != null) {
      if (temp.flag != true) break;  // GCを期待しないで，古いリスト構造を再利用する思想で行く
      temp = temp.next;
      totalLines++;
    }
    return totalLines;
  }
  ///////////////////////
  // オーバーラーップ関係
  // コンストラクタで指定された一行の文字数やフォントサイズを変更する場合に（未対応）
  void changeConditions(int max_x, int fsize) {
    cwidth   = max_x;
    fontSize = fsize;
    // TODO: オーバーラップされている行を再度オーバーラップし直す処理を以下に記述
  }
  // 与えられた文字列の長さを返す
  float getStringLength(String str) {
    return textWidth(str) / fontSize; // これでOKだった
  }
  // オーバーラップ処理
  
  ///////////////////
  // その他
  ///////////////////
  // 全データのクリア
  // GCで消えるのを期待するのではなく古いリスト構造を再利用する
  void clear() {
    lineClass  temp, next;
    temp = topLine;
    while (true) {
      if (temp.next == null)  break;
      if (temp.flag == false) break;
      next = temp.next;
      temp.clear();
      temp = next;
    }
    buff = "";
    totalLines  = 0;
    currentWriteLine = topLine;
    readingLine = 0;
  }
  // 現在の行のデータを追加する（未改行）
  void set(String str) {
    buff += str;
  }
  // データ書き込みの本体
  void _setLine(String str, boolean overlap) {
    currentWriteLine.set(str, overlap);
    currentWriteLine = currentWriteLine.next;
    totalLines++;
  }
  // 自動的に一行分のデータをセットする （実際にはstrは""しか来ないが…）
  // オーバーラップ有効時に折り返し分を新たに行データを作成してそちらに続ける．
  void setLine(String str, boolean overlap) {
    str = buff + str;
    if (!overlap) {
      _setLine(str, false);
    } else {
      // オーバーラップ有効時の分割処理
      // lineClassのoverlapはオーバーラップの”発生した”行のみtrueとする
      String buffer;  // 分割する際の頭側の文字列
      int    df;      // strからbufferに移動する文字列の長さ
      while (true) {
        // オーバーラップの発生していない行（あるいは残りの部分）
        if (getStringLength(str) < cwidth) {
          _setLine(str, false);
          break;
        }
        // オーバーラップしている分を１行分，切り取る．
        // 繰り返しループになっているのは，キャラクタ数と文字幅が一致しないため．
        // 半角文字は0.5文字幅とカウントするので，とりあえず全角と仮定して移動する．
        // そして足りていない分がゼロになるまで漸近的に移動して行く．
        buffer  = "";
        while ((df = cwidth - int(getStringLength(buffer))) > 0) {
          // 足りていないと思われる文字数分をstrからbufferに”移動”する．
          buffer += str.substring(0, df);
          str = str.substring(df);
        }
        // これはオーバーラップした分なので第2引数はtrue
        _setLine(buffer, true);
      }
    }
    buff = "";
  }
  // バッファをフラッシュ（書き出す）
  void flushBuffer(boolean overlap) {
    if (buff.length() > 0) setLine("", overlap);
  }
  // 指定された行番号の行データを返す（０行から始める）
  lineClass getTargetLine(int line) {
    lineClass temp = topLine;
    int  num;
    for (num = 0; num < line; num++) {
      if (temp.next == null) break;
      if (temp.flag != true) break;
      temp = temp.next;
    }
    return temp;
  }
  // 表示開始行をダイレクトに指定する
  void setReadingLine(int num) {
    if (num < 0) num = 0;
    if (num >= totalLines - 1) num = totalLines - 1;
    readingLine = num;
  }
  // 表示開始行数を指定行数分，増加する
  void incrementReadingLine(int num) {
    setReadingLine(readingLine + num);
  }
  // デバッグ用：テキストバッファを全て表示
  void displayAllTextBuffer() {
    int  i;
    lineClass temp;
    temp = topLine;
    i = 0;
    while (true) {
      if (temp == null) break;
      print(str(i) + ":" + str(temp.no) + ":" + str(temp.flag) +":" + str(temp.overlap) + ":" + str(temp.line.length()));
      println("[" + temp.line + "]");
      if (temp.next == null) break;
      temp = temp.next;
      i++;
    }
  }
}

// キーボード入力規則のクラス
class keyInputConstraintClass {
  private boolean point;   // [.]キーが押されたか
  private boolean sign;    // [+],[-]キーが押されたか
  private boolean inputted; // 数字が既に入力済みか
  // コンストラクタ
  keyInputConstraintClass() { reset(); }
  // 初期化
  void reset() {
    point = sign = inputted = false;
  }
  // 浮動小数点時の入力規則チェック
  boolean checkFloat(char c) {
    boolean flag = true;
    switch (c) {
      case '+' :
      case '-' :
        if ((sign) || (inputted)) flag = false;
          else sign = true;
        break;
      case '.' :
        if ((point) || (!inputted)) flag = false;
          else point = true;
        break;
      default :
        if (isNumberChar(c)) inputted = true;
          else               flag = false;
        break;
    }
    return flag;    
  }
  // 整数入力時の入力規則チェック（現状は使われていない）
  boolean checkInt(char c) {
    if (c == '.') return false;
    return checkFloat(c);
  }
}

// ログファイル記録用クラス
class loggingClass {
  private String       _defaultFileName = "default.txt";
  protected String     logFileName;    // ファイル名（スケッチフォルダに保存）
  private boolean      loggingNow;        // trueの間はログを記録する
  private PrintWriter  output;         // ログ出力用ファイルポインタ
  // コンストラクタ
  loggingClass() {
//  logFileName   = _defaultFileName;
    logFileName   = "";
    loggingNow    = false;
    output        = null;
  }
  // ログ関係
  // 状態確認とフラグセット（ユーザは使用しないで下さい）
  void setFileName(String filename) {
    if (filename.length() > 0) logFileName = filename;
      else                     logFileName = _defaultFileName;
  }
  // ログ記録開始
  void startLogging() {
    if (output != null) stopLogging();
    output     = createWriter(logFileName);
    loggingNow = true;
    println2Log("--- Logging start : " + nowDateTime() + "---");
  }
  // ログ記録終了
  void stopLogging() {
    if (output != null) {
      println2Log("--- Logging finished : " + nowDateTime() + "---");
      // ファイルを閉じる
      output.flush();
      output.close();
    }
    // 設定を初期化する
    loggingNow  = false;
//  logFileName = "";
    output      = null;
  }
  // ログファイルにのみprint()
  void print2Log(String str)   { if (loggingNow) output.print(str);   }
  // ログファイルにのみprintln()
  void println2Log(String str) { if (loggingNow) output.println(str); }
  // ログファイルにのみnewline()
  void newline2Log() { println2Log(""); }
  // ログファイルの保存を一時的に停止する
  void pauseLogging()   { loggingNow = false; }
  // ログファイルの保存を再開する
  void restartLogging() {
    if (output != null) loggingNow = true;
  }
}

/////////////////////////////////////////
// キーボード入力とコンソール出力のクラス
/////////////////////////////////////////
class consoleClass {
  // 実行ウィンドウ関係の設定
  private float cx = 0.0, cy = 0.0;        // キャラクタの表示位置（カレントポジション），半角文字は0.5文字
  private int   cwidth, cheight;           // キャラクタの横文字数，行数
  private int   gwx, gwy;                  // スクリーンサイズ
  private int   colwidth, rowheight;       // 一文字の文字送り，行間（pixel）
  private int   frameWidth;                // 額縁の幅
  private int   fontSize;                  // フォントサイズ
  // 色の初期値
  int backgroundColor = #ffffee;  // 背景色
  int textColor       = #000000;  // 文字の色
  // 機能選択のフラグ
  protected boolean nodisplay   = false;   // グラフィックス表示の抑制
  protected boolean noTextArea  = false;   // PDEのテキストエリアへの文字列の表示の抑制
  protected boolean noOverlap   = true;    // ウィンドウからはみ出す文字列を折り返して次行に表示しない
  // テキスト画面のスクロール表示関係
  textbufferClass      textbuffer;         // バッファ関係のクラス
  private boolean      recordTextBuffer;   // trueの間はテキストバッファに記録する
  // キーボード入力規則のチェック
  keyInputConstraintClass  keyCheck;
  // ログファイル関係
  loggingClass         logging;
  private boolean      enableLogging;  // テキストログを記録する（setup()でのみ有効）

  // コンストラクタ（引数は横方向の最大文字数と縦方向の最大行数）
  consoleClass(int max_x, int max_y) {
    // グラフィック関係
    cwidth     = max_x;
    cheight    = max_y;
    fontSize   = 14;
    colwidth   = int(fontSize * 1.2);
    rowheight  = int(fontSize * 1.4);
    frameWidth = 10;
    gwx        = cwidth  * colwidth  + frameWidth * 2;
    gwy        = cheight * rowheight + frameWidth * 2;
    // スクリーンの設定
    size(gwx, gwy);
    _clrscr();  
    // フォント関係
    PFont font = createFont("MS Gothic",fontSize,true);  // フォントを変換
    textFont(font);         // フォントを設定
    textSize(fontSize);     // フォントサイズを設定（不要？）
    textAlign(LEFT, TOP);   // 配置
    // スクロール表示用のバッファ関連
    textbuffer       = new textbufferClass(cwidth, fontSize);
    recordTextBuffer = true;
    // キーボード入力規則チェック
    keyCheck = new keyInputConstraintClass();
    // ログファイル関係
    logging = new loggingClass();
    enableLogging = false;
  }

  // ----------- 画面出力関係-----------------
  // 文字位置をグラフィックス座標に変換
  int px(float x) { return int(x * colwidth  + frameWidth); }
  int py(float y) { return int(y * rowheight + frameWidth); }

  // 出力位置指定（廃止予定：使っても良いがテキストバッファは非対応）
  void locate(int x, int y) {
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    cx = x;
    cy = y;
  }
  // 行数が最大値に達したか
  boolean rowCheck() {
    // 達したならば画面消去
    if (cy >= cheight) {
      _clrscr();
      textbuffer.readingLine += cheight;  // ここだけは強制的にインクリメントする必要がある（まだ出力していない）
      return true;
    }
    return false;
  }
  // BSキーが押された場合の処理
  void backSpace() {
    if (keyBufferLength() > 0) {
      keyBuffer = keyBuffer.substring(0, keyBuffer.length() - 1);
      writeBS();
    }
  }
  // テキスト文字0.5文字分を削除（パラメータ入力時のみ有効なため）
  void writeBS() {
    if (cx <= 0) return;
    if (!nodisplay) {
      cx -= 0.5;
      noStroke();
      fill(backgroundColor);    // 背景色
      rect(px(cx), py(cy), colwidth/2, rowheight);
      fill(textColor);          // 塗りつぶし色
      stroke(textColor);
    }
//  if (!noTextArea) print("\b");  // backspaceのエスケープシーケンスは無効の模様．
  }
  // 与えられた文字の大きさを返す．全角ならば１，半角ならば0.5
  float charLength(char c) {
    return textWidth(c) / fontSize;
  }
  // 与えられた文字列の長さを返す
  float getStringLength(String str) {
    return textWidth(str) / fontSize; // これでOKだった
  }
  // 一文字表示（文字列は全てこの関数を介してtext()で表示する
  void putchar(char c) {
    text(c, px(cx), py(cy));
    cx += charLength(c);
  }

  ///////////////////////
  // テキストバッファ関連
  ///////////////////////
  // テキストバッファへの記録を一時的に停止する
  void stopRecordTextBuffer()  { recordTextBuffer = false; }
  // テキストバッファへの記録を再開する
  void startRecordTextBuffer() { recordTextBuffer = true;  }
  // テキストバッファの指定された行から１ページ分を再描画する
  void redrawPage() {
    redrawPage(textbuffer.readingLine);
  }
  void redrawPage(int line) {
    int       lineCount, dcy;
    lineClass thisline;

    // 表示開始行数のチェック
    textbuffer.incrementReadingLine(0);
    // 初期化
    _clrscr();
    thisline = textbuffer.getTargetLine(line);
    // 再描画
    // スクロール表示時は一時的にオーバーラップを強制的にオフにする
    boolean over_flag = noOverlap;
    disableOverlap();
    for (lineCount = 0; lineCount < cheight; lineCount += dcy) {
      if (!thisline.flag) break; // これ以降，テキストが出力されていないので抜ける
      dcy = writeln(thisline.line);
      thisline = thisline.next;
    }
    // 元の状態に戻す
    noOverlap = over_flag;
  }
  // ページ単位のスクロール
  // pageが正で下，負で上にスクロール
  void pageScroll(float page) {
    textbuffer.incrementReadingLine(int(cheight * page));
    redrawPage();
  }
  // 指定した行数だけ上/下にスクロールする
  // num が正で下，負で上にスクロール
  void pageLineScroll(int num) {
    textbuffer.incrementReadingLine(num);
    redrawPage();
  }
  // 最新のページを再描画（カーソル位置が復元されないのは仕方が無い
  void redrawNewestPage() {
    int  linenumber;
    if (textbuffer.totalLines < cheight) {
      textbuffer.setReadingLine(0);
    } else {
      textbuffer.setReadingLine(textbuffer.totalLines - int(cheight / 2.0));
    }
    redrawPage();
  }

  // フォントサイズの初期設定（setup()以外で変更した場合は変なことが起こるでしょう）
  void setFontSize(int size) {
    fontSize   = size;
    colwidth   = int(fontSize * 1.2);
    rowheight  = int(fontSize * 1.4);
    gwx        = cwidth  * colwidth  + frameWidth * 2;
    gwy        = cheight * rowheight + frameWidth * 2;
    // スクリーンの設定
    size(gwx, gwy);
    _clrscr();  
    // フォント関係
    PFont font = createFont("MS Gothic",fontSize,true);  // フォントを変換
    textFont(font);         // フォントを設定
    textSize(fontSize);     // フォントサイズを設定（不要？）
  }
  // 文字色と背景色の初期設定（setup()以外で変更した場合は変なことが起こるでしょう）
  void setcolor(int textcolor, int backgroundcolor) {
    textColor = textcolor;
    backgroundColor = backgroundcolor;
//  _clrscr();
  }
  // 画面消去（画面のみクリア）
  void _clrscr() {
    background(backgroundColor);    // 背景色
    fill(textColor);                // 塗りつぶし色
    cx = cy = 0;
  }
  // 画面消去（テキストバッファもクリア）
  void clrscr() {
    _clrscr();
    textbuffer.clear();
  }
  // テキスト出力（単純に描画するだけ）: userDraw()中で利用するのに適している
  int _write(String message) {
    int  i, num = 1;
    for (i = 0; i < message.length(); i++) {
      putchar(message.charAt(i));
      // 折り返し表示に関する処理
      if (noOverlap != true) {
        if (cx >= cwidth) {
          cx = 0;
          cy++;
          num++;
        }
      }
    }
    return num;
  }
  void _write(int x, int y, String message) {
    locate(x, y);
    _write(message);
  }
  // テキスト出力（テキストバッファ，ログ記録，バックスペースによる消去に対応）
  // 戻り値は実行ウィンドウに出力した実際の行数（オーバーラップを考慮）
  int write(String message) {
    int i;
    int num = 0;
    if (!nodisplay) {
      rowCheck();
      num = _write(message);
      if (recordTextBuffer) textbuffer.set(message);
    }
    // コンソールに出力
    if (!noTextArea) {
      if (recordTextBuffer) print(message);
    }
    // ログファイルに記録
    if (enableLogging) logging.print2Log(message);
    return num;
  }
  // テキスト出力（改行）
  int writeln(String message) {
    int num;

    num = write(message);
    // 以下改行のみ出力
    if (!nodisplay) {
      if (recordTextBuffer) textbuffer.setLine("", !noOverlap);
      cy++;
      cx = 0;
    }
    // コンソールに出力
    if (!noTextArea) {
      if (recordTextBuffer) println();
    }
    // ログファイルに記録
    if (enableLogging) logging.println2Log("");
    return num;
  }
  // 改行
  void newline() { writeln(""); }
  // ウィンドウサイズの変更
  void windowSize(int x, int y) {
    gwx = x;
    gwy = y;
    size(gwx, gwy);
  }
  // ディスプレイ表示のチェンジ
  void displaySwitch(boolean flag) { nodisplay = flag; }
  // ディスプレイ表示の抑制と許可
  void disableDisplay() { nodisplay = true;  }
  void enableDisplay()  { nodisplay = false; }
  // テキストエリア表示のチェンジ
  void textAreaSwitch(boolean flag) { noTextArea = flag; }
  // テキストエリア表示の抑制と許可
  void disableTextArea() { noTextArea = true;  }
  void enableTextArea()  { noTextArea = false; }
  // オーバーラップ表示（折り返し）の抑制と許可
  void disableOverlap()  { noOverlap = true;  }
  void enableOverlap()   { noOverlap = false; }
  
  // 【テキスト出力のログファイルへの出力】
  // Main() と userDraw()実行中のcrowbar.write()/writeln()をログに記録
  // ログファイルの記録開始：Options()の中で宣言するのが無難だが，
  // それMain()やuserDraw()の中で宣言しても構わない．
  // ファイルへの追記は仕様上不可能なようなので，現状では未対応
  // 既に記録開始していてクローズされていない場合は，現在記録中のログファイルを閉じて新しく記録を開始する
  // 1-1) ログファイルのオープン（引数無し：デフォルトのファイル名で保存）
  void startLogging() { startLogging(""); }
  // 1-2) ログファイルのオープン（ファイル名指定あり）
  void startLogging(String fname) {
    logging.setFileName(fname);
    println("ログファイルを作成します：" + logging.logFileName);
    logging.startLogging();
  }
  // 2) ログファイルのクローズ（明示しないでもcrowbar終了時に閉じてくれる）
  void closeLogging() {
    logging.stopLogging();
  }
  // 3-1) ログ記録の一時中断
  void pauseLogging()   { logging.pauseLogging();   }
  // 3-2) ログ記録の再開
  void restartLogging() { logging.restartLogging(); }
  // 以下はユーザが用いてはいけない（システム用）
  void enableLogOutput()  { enableLogging = true;  }    // ログファイル出力を許可
  void disableLogOutput() { enableLogging = false; }    // ログファイルの出力を不許可
  boolean isEnableLogging() { return enableLogging; }    // 現在，ログファイルへの出力は許可されているか？

  // --------------- キーボード入力関連 -------------------
  // 以下の関数や変数をユーザは利用しないことが望ましい
  //
  private String keyBuffer = "";
  boolean inputNow = false;    // エコーバックあり
  boolean inputFixed = false;  // ENTERが入力されたらtrue
  // バッファクリア  
  void flushKeyBuffer() {
    keyBuffer = "";
    inputFixed = false;
  }
  // 文字列長を返す
  int keyBufferLength() {
    return keyBuffer.length();
  }
  // コンソールからの入力を開始
  void startInput() {
    inputFixed = false;
    inputNow   = true;
  }
  // キーボード入力完了
  void inputFinish() {
    inputFixed = true;
    inputNow   = false;
  }
  // 文字列の追加
  void strcat(String str) {
    keyBuffer += str;
  }
  // キーバッファーを読み出す
  String getKeyBuffer() {
    String buf;
    buf = keyBuffer;  // インスタンスのクローンが作成されるのなら良いけれど．
    keyBuffer = "";
    inputNow   = false;
    inputFixed = false;
    return buf;  // 文字列データのみを返さなくて良いのか？
  }
}

//////////////////////////////
// パラメータの最小単位のクラス
//////////////////////////////
class arguments {
  boolean ready;
  String  message;
  String  value;
  String  initialValue;   // ""の場合は初期値の指定なしと判断
  String [] selectValues; // nullの場合は選択肢タイプではないと判断（設定時は空白で区切られた文字列）
  char    parameterType;  // 'A': Normal, 'N': 数値のみ, 'S': 選択肢
  // コンストラクタ
  arguments() {
    ready   = false;
    message = value = initialValue = "";
    selectValues  = null;
    parameterType = 'A';
  }
  // 値を設定（標準）
  void set(String str) {
    value = str;
    ready = true;
  }
}

//////////////////////////////////////////////////////////////////////////////
// プロセスコントロールのクラス（キーボード入力とコンソール出力のクラスを継承）
//////////////////////////////////////////////////////////////////////////////
class crowbarClass extends consoleClass {
  int          status = 0;          // プロセスの状態を管理
  String []    comment;             // プログラムのコメント．パラメータ入力前に表示される．
  arguments [] argv;                // パラメータ群（ユーザにより対話的に入力される）
  int          argc;                // パラメータの数
  private int obtainArgIndex = 0;   // パラメータのコメント宣言時の自動登録用カウンタ
  private int readArgIndex = 0;     // パラメータ自動読み出し用カウンタ
  boolean noShowParams  = false;    // 入力済みのパラメータを確認のために表示するか？
  boolean _noShowParams = false;    // setup()中でcrowbar.noShowParameters()が指定された場合の対策．再実行時にクリアされてしまうため．
  boolean running       = false;    // Main()あるいはcrowbar.nonStop()指定時にユーザプログラム実行中はtrue
  boolean afterRunning  = false;    // Main()が終わった後，テキストバッファを再描画可能な間のみtrue;（userDraw()が動いていてもそれは気にしない．今のところ）
  boolean nonstop  = false;         // Main()終了後はdraw()の終端に記述したコードを繰り返し実行する．nonStop()でtrue/stop()でfalse
  boolean _nonstop = false;         // setup()中でcrowbar.nonStop()が指定された場合の対策．再実行時にクリアされてしまうため．

  // コンストラクタ
  crowbarClass(int x, int y) {
    // consoleClassのコンストラクタを呼ぶ
    super(x, y);
    // パラメータの初期化
    argc = 0;
    argv = (arguments [])new arguments[1]; // 常に一つ余分に作成
    argv[0] = new arguments();
    // コメントの初期化
    comment = null;
  }
 // パラメータの値のセット
  void setValue(String str) {
    argv[obtainArgIndex].set(str);
  }
  // キー入力ルール（数値のみ）
  boolean is_numbersOnly() {
    if (argv[obtainArgIndex].parameterType == 'N') return true;
    return false;
  }
  // キー入力ルール（選択肢）
  // 選択肢型ならば選択肢数（最大１０）を返す
  // 選択肢型ではないならば０を返す
  int is_selectOnly() {
    arguments arg;
    arg = argv[obtainArgIndex];
    if (arg.parameterType == 'S') {
      // チェック１
      if (arg.selectValues == null) {
        println("SystemError: 選択肢型なのに選択肢がセットされていません．");
        exit();
      }
      // チェック２
      if (arg.selectValues.length > 10) {
        println("SystemError: 選択肢型の選択肢の数が１０個より多い．");
        exit();
      }
      return arg.selectValues.length;
    }
    return 0;
  }
  // 選択肢型の選択されたパラメータ値を返す
  String get_selectValue(int index) {
    return argv[obtainArgIndex].selectValues[index];
  }
  // 初期値を取得
  String get_initialValue() {
    return argv[obtainArgIndex].initialValue;
  }
  // 実行可能か？
  boolean allReady() {
    int i;
    // パラメータ入力が指定されていない
    if (argv == null) return true;
    // パラメータ入力が指定されている
    for (i = 0; i < argc; i++) {
      if (argv[i].ready != true) return false;
    }
    return true;
  }
  // 全ての入力完了フラグをリセット
  void allResetArguments() {
    int i;
    for (i = 0; i < argv.length; i++) argv[i].ready = false;
    readArgIndex   = 0;
    obtainArgIndex = 0;
  }
  // パラメータが一つも指定されていない場合にtrue
  boolean isNoParameter() {
    if (argc == 0) return true;
    return false;
  }
  // 入力済みの全パラメータを表示する
  void showAllParameters() {
    int i, _readArgIndex;
    // インデックスのバックアップ
    _readArgIndex = readArgIndex;
    if (noShowParams) return; // 念のため
    for (i = 0; i < argc; i++) {
      if (!argv[i].ready) break;  // 入力済みの分のみ表示するので．
      write(get_paramMessage(i) + "：[");
      write(getStr(i));
      writeln("]");
    }
    // インデックスのリストア
    readArgIndex = _readArgIndex;
  }
  // 入力済みの全パラメータをログファイルにのみ出力
  void showAllParameters2Log() {
    boolean _display, _textarea;
    // フラグのバックアップ
    _display  = nodisplay;
    _textarea = noTextArea;
    // テキスト出力を一時的に禁止
    disableDisplay();
    disableTextArea();
    // 禁止した上で出力すればログにのみ出力される
    showAllParameters();
    // フラグを戻す
    if (!_display)  enableDisplay();
    if (!_textarea) enableTextArea();
  }
  // ------------------------------
  // パラメータ表示の確認表示の抑制
  void noShowParameters() { noShowParams = true;  }
  void showParameters()   { noShowParams = false; } // 必要性は微妙
  // パラメータ説明の取得
  String get_paramMessage(int index) {
    String    message;
    arguments arg;

    // インデックスの範囲のチェック
    if ((index >= argc) || (index < 0)) {
      println();
      print("[警告] パラメータ読み出しのIndexが範囲を超えています！(Index=");
      println(str(index)+")");
      return "";
    }
    // 説明の取得
    arg     = argv[index];
    // 全てのタイプに共通
    message = arg.message;
    // 以下，オプションによってメッセージを追加
    // 初期値が与えられている場合
    if (arg.initialValue.length() > 0) message += "（初期値：" + arg.initialValue + "）";
    // パラメータの種類によって
    if (arg.parameterType == 'N') {
      // 数字のみ
      message += "／数字のみ";
    } else if (arg.parameterType == 'S') {
      // 選択肢型
      int i;
      message += "／（";
      for (i = 0; i < arg.selectValues.length; i++) {
        if (i > 0) message += ",";
        message += str(i) + ":" + arg.selectValues[i];
      }
      message += "）";
    }
    return message;
  }
  // --------------------------------------------
  // パラメータの宣言（画面に表示するメッセージ）
  // --------------------------------------------
  // パラメータの宣言（標準：数字のみ）
  void define(String str) {
    defineStr(str);
    argv[argc - 1].parameterType = 'N';
  }
  // パラメータの宣言（標準：数字のみ：初期値あり）
  void define(String str1, String str2) {
    defineStr(str1, str2);
    argv[argc - 1].parameterType = 'N';
  }
  // パラメータの宣言（数字のみ：初期値あり，float渡し）
  void define(String str, float f) { define(str, str(f)); }
  // パラメータの宣言（数字のみ：初期値あり，int渡し）
  void define(String str, int i)   { define(str, str(i)); }

  // パラメータの宣言（入力制限規則なし）
  void defineStr(String str) {
    argc = argv.length;
    argv = (arguments [])expand(argv, argc + 1);
    argv[argc] = new arguments();
    // 現在の値をセット
    argv[argc - 1].message = str;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり）
  void defineStr(String str1, String str2) {
    defineStr(str1);
    argv[argc - 1].initialValue = str2;
  }
  // パラメータの宣言（入力制限規則なし：初期値あり，float渡し）
  void defineStr(String str, float f) { define(str, str(f)); }
  // パラメータの宣言（入力制限規則なし：初期値あり，int渡し）
  void defineStr(String str, int i)   { define(str, str(i)); }

  // パラメータの宣言（選択肢型：入力制限規則は0-9のみ）
  void defineSel(String str1, String str2) {
    String list[];
    if (str2.length() <= 0) {
      println("Error: 選択肢型のパラメータ宣言の第２パラメータが空文字列です．");
      exit();
    }
    list = split(str2, ' ');
    if (list.length > 10) {
      println("Error: 選択肢型のパラメータ宣言の選択肢の数は１０個以下です．");
      exit();
    }
    defineStr(str1);
    argv[argc - 1].parameterType = 'S';
    argv[argc - 1].selectValues = list;
  }
  // パラメータの宣言（選択型：初期値あり）
  void defineSel(String str1, String str2, int num) {
    defineSel(str1, str2);
    if (num < 0) {
      println("Error: 選択肢型の初期値は0から9の整数です．");
      exit();
    }
    if (num >= argv[argc - 1].selectValues.length) {
      println("Error: 選択肢型の初期値は0から選択肢の数-1までです．");
      exit();
    }
    argv[argc - 1].initialValue = str(num);
  } 

  //-------------------------
  // パラメータの値を受け取る
  //-------------------------
  // 1) indexを明示的に指定する
  // 1-a) パラメータ名を文字列で返す
  String getStr(int index) {
    if ((index >= argc) || (index < 0)) {
      println();
      print("[警告] パラメータ読み出しのIndexが範囲を超えています！(Index=");
      println(str(index)+")");
      return "";
    }
    // インデックスの自動更新
    readArgIndex = index + 1;
    return argv[index].value;
  }
  // 1-b) パラメータをfloatで返す
  float getFloat(int index) { return float(getStr(index)); }
  // 1-c) パラメータをintで返す
  int getInt(int index) { return int(getStr(index)); }

  // 2) indexを明示的に指定しないで順番に自動的に読み出す
  // 2-a) パラメータを文字列で渡す（自動）
  String getStr() { return getStr(readArgIndex); }
  // 2-b) パラメータをfloatで返す（自動）
  float getFloat() { return getFloat(readArgIndex); }
  // 2-c) パラメータをintで返す（自動）
  int getInt() { return getInt(readArgIndex); }

  // ----------------------------------------------------------------------------
  // Main()終了後にdraw()終端に記述したコードを繰り返し実行するためのコントロール
  // ----------------------------------------------------------------------------
  void nonStop() { nonstop = true; }   // userDraw()関数が有効
  void stop()    { nonstop = false; }  // userDraw()関数が終了したらプログラム終了
  
  // ----------------------------------------------
  // プログラムの説明（パラメータ入力前に表示される
  // ----------------------------------------------
  // プログラムコメントを表示する
  void outputProgramComments() {
    int i, commNum;
    commNum = comment.length;
    for (i = 0; i < commNum; i++) writeln(comment[i]);
  }
  // プログラムコメントを宣言（複数宣言可能）．setup()内でユーザが宣言する．
  void programComment(String str) {
    String addStr[];
    addStr    = new String[1];
    addStr[0] = str;
    if (comment == null) {
      // 一個目のコメントの場合
      comment = (String [])new String[1];
      comment = addStr;
    } else {
      // 二個目以降のコメントの場合
      comment = concat(comment, addStr);
    }
  }
}

///////////////
// キー入力判定
///////////////
// 数字入力用キーか？
boolean isKeyNumbers(char c) {
  if (isNumberChar(c)) return true;
  switch (c) {
    case '.' :
//  case ',' :  // お節介過ぎる
    case '+' :
    case '-' :
      return true;
  }
  return false;
}

void keyPressed() {
  if (crowbar.running == false) {
    // ユーザプログラムが動作していない時（パラメータ入力中など）
    if ((key == 'Q') || (key == 'q')) exit();
    if (crowbar.inputNow) {
      // パラメータ入力中
      if (key == ENTER) {
        crowbar.inputFinish();
        crowbar.newline();
      } else if (key == BACKSPACE) {
        if (crowbar.keyBufferLength() > 0) {
          crowbar.backSpace();
        }
      } else {
        // パラメータの入力
        boolean flag = true;
        int     num;
        // キー入力ルール
        // 数値のみの場合
        if (crowbar.is_numbersOnly()) {
          if (!isKeyNumbers(key)) flag = false;
            else                  flag = crowbar.keyCheck.checkFloat(key);
        }
        // 選択肢の場合
        if ((num = crowbar.is_selectOnly()) > 0) {
          // 既に１文字入力済み（半角空白も含む）
          if (crowbar.keyBufferLength() >= 1) flag = false;
          // 数字キー（0-9）ではない
          if (!isNumberChar(key))             flag = false;
          // 選択肢の数よりも大きな数字
          if (int(str(key)) >= num)           flag = false;
        }
        if (flag) {
          crowbar.strcat(str(key));
          crowbar.write(str(key));
        }
      }
    }
    // スクロール可能な状態（プログラムが一旦終了してキー入力を待っている間のみ）
    if (crowbar.afterRunning) {
      if (key == '-') {
        crowbar.pageScroll(-0.5);
      } else if (key == '+') {
        crowbar.pageScroll(0.5);
      } else if (key == CODED) {
        if (keyCode == UP) {
          crowbar.pageLineScroll(-1);
        } else if (keyCode == DOWN) {
          crowbar.pageLineScroll(1);
        } else if (keyCode == LEFT) {
          crowbar.pageScroll(-0.5);
        } else if (keyCode == RIGHT) {
          crowbar.pageScroll(0.5);
        }
      }
    }
    redraw();
  } else {
    // crowbar.nonStop()指定時のユーザプログラム実行中のキー入力判定
    // コードはuserKeyPressed()に記述すると良い．
    userKeyPressed();
  }
}

///////////////
// プロセス管理
///////////////
void draw() {
  switch (crowbar.status) {
    case 0: // 初期状態
      // setup()中でcrowbar.nonStop()/noShowParams()が実行された場合の対策
      if (crowbar.nonstop)      crowbar._nonstop      = true;
      if (crowbar.noShowParams) crowbar._noShowParams = true;
      // プログラムコメントの表示
      if (crowbar.comment != null) {
        crowbar.writeln("【プログラム説明】");
        crowbar.outputProgramComments();
      }
      crowbar.status++;
      break;
    case 1: // パラメータ入力開始
      crowbar.allResetArguments();
      // setup()中でcrowbar.nonStop()/noShowParams()が実行された場合の対策
      if (crowbar._nonstop)      crowbar.nonstop = true;
      if (crowbar._noShowParams) crowbar.noShowParams = true;
      if (!crowbar.isNoParameter()) {
        crowbar.newline();
        crowbar.writeln("【パラメータの入力】");
      }
      crowbar.status++;
      break;
    case 2: // パラメータ入力中
      if (!crowbar.isNoParameter()) {
        crowbar.keyCheck.reset();  // キー入力規則のリセット
        crowbar.write(crowbar.get_paramMessage(crowbar.obtainArgIndex) + ":");
        crowbar.flushKeyBuffer();
        crowbar.startInput();
        // ループ停止
        noLoop();
      }
      crowbar.status++;
      break;
    case 3: // 入力確定
      if (!crowbar.isNoParameter()) {
        String inputs;
        if (crowbar.inputFixed) {
          boolean flag = true;
          inputs = crowbar.getKeyBuffer();
          // 空かつ初期値が指定されている場合
          if (inputs.length() == 0) {
            inputs = crowbar.get_initialValue();
          }
          // 選択肢型の場合
          if (crowbar.is_selectOnly() > 0) {
            int  num;
            // 念のためにチェック
            if (((num = int(inputs)) < 0) || (num > 9)) {
              flag = false;            
            } else if (num > crowbar.is_selectOnly()) {
              flag = false;
            }
            if (!flag) {
              // あり得ないはず（入力時にチェックしているので）
              println("SystemError: 選択肢型の入力値が異常です（" + str(num) + "）．");
              exit();
            }
            inputs = crowbar.get_selectValue(num);
          }
          crowbar.setValue(inputs);
          crowbar.obtainArgIndex++;
          // ループ再開
          loop();
          crowbar.status++;
        }
      } else crowbar.status++;
      break;
    case 4: // パラメータ入力完了かの判定
      if (!crowbar.isNoParameter()) {
        if (crowbar.allReady()) {
          crowbar.status++;
        } else {
          crowbar.status -= 2;
        }
      } else crowbar.status++;
      break;
    case 5: // 実行か？ 再入力か？
      if (!crowbar.isNoParameter()) {
        crowbar.newline();
        crowbar.writeln("【全てのパラメータの入力が完了】");
        if (!crowbar.noShowParams) crowbar.showAllParameters();
        crowbar.write("プログラムを実行する(Yes) /パラメータを入力し直す(Retry) : ");
      }
      crowbar.status++;
      break;
    case 6: // キー判定結果
      if (!crowbar.isNoParameter()) {
        if (keyPressed) {
          if ((key == 'y') || (key == 'Y')) {
            crowbar.writeln(str(key));
            crowbar.newline();
            crowbar.status++;
          } else if ((key == 'r') || (key == 'R')) {
            crowbar.writeln(str(key));
            crowbar.newline();
            crowbar.status -= 5;
          }
        }
      } else crowbar.status++;
      break;
    case 7: // メインプログラム実行前処理（あるならば）
      preMain(); // optionCodeタブ参照
      crowbar.running = true;
      crowbar.status++;
      break;
    case 8: // ログファイルへの出力を許可する:これはもしかしたらstatus==0から許可するかも知れない．余計な文字情報は記録しないという仕様はお節介だったかもという反省．
      crowbar.enableLogOutput();
      if (crowbar.isEnableLogging()) {
//      crowbar.logging.startLogging();
        // 全パラメータを出力する
        crowbar.logging.newline2Log();
        crowbar.showAllParameters2Log();
        crowbar.logging.newline2Log();
      }
      crowbar.status++;
      break;
    case 9: // メインプログラム実行
      Main();
      // ループ再開
      loop();
      crowbar.status++;
      break;
    case 10: // 繰り返し実行コード（オプション）
      if (crowbar.nonstop) break;
      crowbar.status++;
      break;
    case 11: // ログファイルを閉じる
      crowbar.running = false;
      if (crowbar.isEnableLogging()) {
        // ログファイルを開いたままなら閉じる
        crowbar.logging.newline2Log();
        crowbar.logging.stopLogging();
        crowbar.disableLogOutput();
      }
      crowbar.status++;
      break;
    case 12: // メインプログラム実行後処理（あるならば）
      postMain(); // optionCodeタブ参照
      crowbar.status++;
      break;
    case 13: // リトライ確認メッセージ
      crowbar.newline();
      crowbar.writeln("【プログラム終了】");
      crowbar.write("もう一度，実行しますか？(Yes/No)：");
      // テキストバッファ関連
      crowbar.afterRunning = true;
      crowbar.textbuffer.flushBuffer(!crowbar.noOverlap);
      crowbar.stopRecordTextBuffer();
      crowbar.status++;
      break;
    case 14: // キー判定結果
      if (keyPressed) {
        if ((key == 'y') || (key == 'Y')) {
          crowbar.writeln(str(key));
          crowbar.newline();
          // テキストバッファ―関連
          crowbar.redrawNewestPage();
          crowbar.afterRunning = false;
          crowbar.startRecordTextBuffer();
          crowbar.status = 1;
        } else if ((key == 'n') || (key == 'N')) {
          crowbar.writeln(str(key));
          // テキストバッファ―関連
          crowbar.redrawNewestPage();
          crowbar.afterRunning = false;
          crowbar.startRecordTextBuffer();
          crowbar.status++;
        }
      }
      break;
    case 15: // 終了
      exit();
      break;
  }
  if (crowbar.running) {
    // crowbar.nonStop()指定時の繰り返し処理はuserDraw()に記述
    userDraw();
  }
}

void mouseDragged() {
  if (crowbar.running) userMouseDragged();
}
void mouseMoved() {
  if (crowbar.running) userMouseMoved();
}
void mousePressed() {
  if (crowbar.running) userMousePressed();
}
void mouseReleased() {
  if (crowbar.running) userMouseReleased();
}
void mouseClicked() {
  if (crowbar.running) userMouseClicked();
}
void keyReleased() {
  if (crowbar.running) userKeyReleased();
}
void keyTyped() {
  if (crowbar.running) userKyeTyped();
}

// システム初期設定
void setup()
{
  initializeMain();
//frameRate(10);
  Setup();
  Options();
}

