// Crowbar組込グラフィックライブラリTomahawk
// by T.Shirai
// 開発開始 2012/6/7

// tomahawkMode : 0, 1, 2 はTomahawkのモードを切り替える非常に重要なパラメータで，実行中に変更するものではない．
// 0 : このCrowbarにはTomahawkが存在しない場合（tomahawkClass.pdeを削除し，crowbarClass.pde冒頭のコメントを外す）
//     Tomahawkは非常に負荷が大きいのでビューポートを利用する必要がない場合はこのモードの使用を勧めます．
// 1 : Tomahawkを利用するがCrowbarのテキスト表示を背景に描画する．ビューポートはその上のレイヤーとして上書きされる．
//     実行ウィンドウの一部を画像描画用のビューポートとして活用する場合などに適している．
//     テキスト専用レイヤーをブレンドしないため比較的負荷が少ない．
// 2 : Tomahawkを利用する．さらにCrowbarのテキスト出力を専用のレイヤーに描画する．
//     レイヤーは背景，ビューポート，テキスト表示専用レイヤーの順でブレンドされるためテキスト表示が隠れることが無い．
//     オーバーヘッドが最も重いモード．
// （テキスト専用レイヤー使用時：2の制約）：Processingの仕様に基づく
//   まず，フォントの見栄えが悪い．テキストの色も何故か指定した値よりも少し暗くなる．
//   レイヤーを重ねるのにblend()を使用している．テクニックとしては不透明度０（つまり透明）かつ黒い背景に不透明度255で
//   グラフィックス描画を行ない，blend(BLEND)すると不透明度255のPixelのみが合成されるという手法を取っています．
//   ところがtext()はbackground(#000000, 0)の背景色#000000を利用してくれるのですが，不透明度255で描画してしまいます．
//   そこで文字をtext()で描画する度にそのエリアをpixel単位でチェックして背景色#000000の場合は強制的にcolor(#00000, 0)
//   に変更します．したがって文字色として#000000を指定すると区別が付きませんので内部で勝手に#000002に文字色を変更して
//   います．この不透明度の変更の処理にも余計なオーバーヘッドが生じています．あまり大きなフォントサイズを指定すると，
//   blend()にも時間が掛かります．

/*
class vectorClass {
  float x, y;
  vectorClass() {
    x = y = 0.0;
  }
  void scale(float a) {
    x *= a;
    y *= a;
  }
}
*/

// 現在の線色や座標などの情報を保持
class currentDrawingClass {
  boolean  isFill;       // 塗り潰す？
  int      fillColor;    // 塗り潰し色
  int      fillTrans;    // 塗り潰しの不透明度
  boolean  isStroke;     // 外形線描く？
  int      strokeColor;  // 外形線色
  int      strokeTrans;  // 外形線の不透明度
  int      textColor;    // テキスト文字の色
  int      vpx, vpy;     // ビューポート内の最後に描画した座標
  float    wpx, wpy;     // ワールド座標における最後に描画した座標
  currentDrawingClass() {
    setDefault();
  }
  // 複製
  void clone(currentDrawingClass src) {
    isFill      = src.isFill;
    fillColor   = src.fillColor;
    fillTrans   = src.fillTrans;
    isStroke    = src.isStroke;
    strokeColor = src.strokeColor;
    strokeTrans = src.strokeTrans;
    textColor   = src.textColor;
    vpx         = src.vpx;
    vpy         = src.vpy;
    wpx         = src.wpx;
    wpy         = src.wpy;
  }
  // 初期値に設定
  void setDefault() {
    isFill       = true;
    fillColor    = #ffffff;
    fillTrans    = 255;
    isStroke     = true;
    strokeColor  = #000000;
    strokeTrans  = 255;
    textColor    = fillColor;
    vpx = vpy    = 0;
    wpx = wpy    = 0.0;
  }
}


class viewportClass {
  protected PGraphics pg;                              // ビューのグラフィックスのハンドル
  protected float     wx_min, wx_max, wy_min, wy_max;  // ビューのワールド座標
  protected int       vx, vy;                          // ビューの実行スクリーンに対する左上の座標
  protected int       vwidth, vheight;                 // ビューの横幅／縦幅
  boolean             monoRatio;                       // ピクセル比は縦も横も同一にするか
  float               pixelRatioX, pixelRatioY;        // ピクセル比（縦と横）ワールド座標→ビューポート内のPixel座標変換用
  protected int       bgColor;                         // ビューの背景色
  protected boolean   transparentView;                 // ビューの背景が透明なビュー
  protected int       transparency;                    // ビューの透明度（枠は除く）： 0～255（デフォルトは255）
  boolean             drawFrame;                       // 枠の表示を行なうか？
  int                 frameColor;                      // 枠の色
  protected String    label;                           // ビューの説明（オプション）
  protected int       no;                              // 表示順（優先度）：数字が大きいほど後（上）に表示される
  float               magnification;                   // 画面の拡大率（ピクセル比にこの値を掛ける）
  float               offsetX, offsetY;                // ワールド座標系におけるオフセット量
  protected boolean   visible;                         // ビューの表示をダイナミックにOn/Offできる
  currentDrawingClass cdSet;                           // 最新の描画関係の情報
  // コンストラクタ
  viewportClass(String name, int x, int y, int width, int height) {
    // 座標関係の初期化
    vx         = x;
    vy         = y;
    vwidth     = width;
    vheight    = height;
    monoRatio  = true;
    world(0.0, 0.0, vwidth, vheight);  // 初期値で設定
    // 背景色，フレーム
    bgColor      = _BGCOLOR;
    transparentView = false;
    drawFrame    = true;
    transparency = 255;
    frameColor   = #000000;
    // 最新の描画関係の情報
    cdSet      = new currentDrawingClass();
    // 拡大・並進パラメータ
    magnification = 1.0;
    offsetX = offsetY = 0.0;
    // デバッグ情報
    label     = name;
    no        = -1;
    // ビューポート用のグラフィックスを確保
    pg        = createGraphics(vwidth, vheight, JAVA2D);    // P3Dで描画すると円に変な線が沢山入るので．
    pg.beginDraw();
    pg.smooth();                      // 一応，念のために
    pg.endDraw();
    visible   = true;
  }
  // クラスの複製
  void clone(viewportClass src) {
    vx              = src.vx;
    vy              = src.vy;
    vwidth          = src.vwidth;
    vheight         = src.vheight;
    monoRatio       = src.monoRatio;
    bgColor         = src.bgColor;
    transparentView = src.transparentView;
    drawFrame       = src.drawFrame;
    transparency    = src.transparency;
    frameColor      = src.frameColor;
    cdSet.clone(src.cdSet);
    magnification   = src.magnification;
    offsetX         = src.offsetX;
    offsetY         = src.offsetY;
    label           = src.label;
    no              = src.no;
    visible         = src.visible;
  }
  
  // 確認用のビュー番号をセット
  void setNo(int n) { no = n; }

  // ピクセル比の計算
  void calcPixelRatio() {
    float px, py;
    px = vwidth  / (wx_max - wx_min);
    py = vheight / (wy_max - wy_min);
    if (monoRatio) {
      // 縦と横のピクセル比を同一に
      if (px > py) {
        // 縦優先
        pixelRatioX = py;
        pixelRatioY = py;
      } else {
        // 横優先
        pixelRatioX = px;
        pixelRatioY = px;
      }
    } else {
      // 別々
      pixelRatioX = px;
      pixelRatioY = py;
    }
  }
  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  float world_minX() { return wx_min; }
  float world_maxX() { return wx_max; }
  float world_minY() { return wy_min; }
  float world_maxY() { return wy_max; }
  // 【変更】
  // (a) 全指定
  void world(float minx, float miny, float maxx, float maxy) {
    if ((minx >= maxx) || (miny >= maxy)) {
      println("ワールド座標系の設定が正しくありません（最小値と最大値が逆です）");
      exit();
    }
    wx_min = minx;
    wy_min = miny;
    wx_max = maxx;
    wy_max = maxy;
    calcPixelRatio();
  }
  // (b) ｘ，ｙそれぞれの最小値／最大値を個別に変更
  float world_minX(float x) {
    float oldvalue = wx_min;
    world(x, wy_min, wx_max, wy_max);
    return oldvalue;
  }
  float world_minY(float y) {
    float oldvalue = wy_min;
    world(wx_min, y, wx_max, wy_max);
    return oldvalue;
  }
  float world_maxX(float x) {
    float oldvalue = wx_max;
    world(wx_min, wy_min, x, wy_max);
    return oldvalue;
  }
  float world_maxY(float y) {
    float oldvalue = wy_max;
    world(wx_min, wy_min, wx_max, y);
    return oldvalue;
  }

  // ---------------------------------------------------
  // ワールド座標系からビューポートのPixel座標系への変換
  // ---------------------------------------------------
  // a) 単にピクセル比を掛けたもの（半径や高さ／幅用）
  int _px(float x) { return int(x * magnification * pixelRatioX); }
  int _py(float y) { return int(y * magnification * pixelRatioY); }
  // b) 座標変換
  int px(float x) {
    float center = (wx_max + wx_min) / 2.0 - offsetX;
    return int((x - center) * magnification * pixelRatioX) + (vwidth / 2);
  }
  int py(float y) {
    float center = (wy_max + wy_min) / 2.0 - offsetY;
    return vheight - (int((y - center) * magnification * pixelRatioY) + (vheight / 2));
  }
  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isVisible() { return visible; }
  // (a) 表示
  void visible()   { visible = true;  }
  // (b) 非表示
  void unVisible() { visible = false; }
  // (c) 引数による指定
  void changeVisible(boolean flag) { if (flag) visible(); else unVisible(); }
  // (d) トグル動作
  boolean toggleVisible() {
    if (isVisible()) {
      unVisible();
      return false;
    } else {
      visible();
      return true;
    }
  }
  // ビューポートの移動
  void moveView(int x, int y)      { vx  = x;  vy  = y;  }
  void moveViewRel(int dx, int dy) { vx += dx; vy += dy; }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // (a) フレームカラーを修得（現在のフレームのみ）
  int  frameColor()     { return frameColor; }
  // (b) フレームカラーの設定（＋変更前の値）
  int frameColor(int c) {
    int oldcolor = frameColor();
    frameColor = c & #ffffff;
    return oldcolor;
  }
  // (c) フレームの表示／非表示
  void drawFrame(boolean flag) { drawFrame = flag; }

  // (a) 不透明度の取得
  int transparency() { return transparency; }
  // (b) 不透明度の設定（＋変更前の値）
  int transparency(int trans) {
    int oldtrans = transparency();
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    transparency = cdSet.fillTrans = cdSet.strokeTrans = trans;  // 塗り潰しや線の不透明度も変更する
    return oldtrans;
  }

  // 背景色の設定
  // (a) 色を取得（不透明度は無視）
  int viewBgColor() { return bgColor; }
  // (b) 色を設定（不透明度は255）
  int viewBgColor(int c) {
    int oldcolor = viewBgColor();
    bgColor = c & #ffffff;
    return oldcolor;
  }
  // 背景を透明なビューポートにする／しない
  void transparentView(boolean flag) { transparentView = flag; }
/*
  // 指定されたビューの背景色を変更する
  void viewBgColor(int v, int bg) {
    viewportClass vp = getView(v);
    if (vp == null) return;
    vp.bgColor(bg);
  }
  // 現在のビューの背景色を変更する
  void viewBgColor(int bg) {
    cv.bgColor(bg);
  }
*/

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す
  float scale() { return magnification; }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float scale(float s) {
    float oldscale;
    if (s <= 0.0) s = 1.0;  // 不正な値
    oldscale = scale();
    magnification = s;
    return oldscale;
  }

  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 現在のビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float offsetX() { return offsetX; }
  // (a-2) ｙ方向
  float offsetY() { return offsetY; }
  // (b) オフセット量を変更する（＋変更前の値を返す）：現在のビューポート
  // (b-1) ｘ方向
  float offsetX(float offset) {
    float oldoffset = offsetX();
    offsetX = offset;
    return oldoffset;
  }
  // (b-2) ｙ方向
  float offsetY(float offset) {
    float oldoffset = offsetY();
    offsetY = offset;
    return oldoffset;
  }
  // (b-3) ｘ方向／ｙ方向同時：常にtrueを返す
  boolean offset(float offx, float offy) {
    offsetX(offx);
    offsetY(offy);
    return true;
  }

  // ------
  // 色関係
  // ------
  // (a) fill()
  // (a-1) 現在の塗り潰し色を取得
  int fill() { return cdSet.fillColor; }
  // (a-2) 塗り潰し色を変更（＋現在の値を返す）
  int fill(int c) { return this.fill(c, cdSet.fillTrans); }
  // (a-3) 塗り潰し色を変更（不透明度対応＋現在の値を返す）
  int fill(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.fill();
    cdSet.fillColor = c & #ffffff;
    cdSet.fillTrans = trans;
    pg.beginDraw();
    pg.fill(cdSet.fillColor, cdSet.fillTrans);
    pg.endDraw();
    cdSet.isFill = false;
    return oldcolor;
  }
  // (a-4) 塗り潰し無し
  void noFill() {
    cdSet.isFill = false;
    pg.beginDraw();
    pg.noFill();
    pg.endDraw();
  }
  // (a-5) 再び塗り潰し
  void reFill() {
    this.fill(cdSet.fillColor, cdSet.fillTrans);
  }
  // (b) stroke()
  // (b-1) 現在の線色を取得
  int stroke() { return cdSet.strokeColor; }
  // (b-2) 線色を変更（＋現在の値を返す）
  int stroke(int c) { return this.stroke(c, cdSet.fillTrans); }
  // (b-3) 線色を変更（不透明度対応＋現在の値を返す）
  int stroke(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.stroke();
    cdSet.strokeColor = c & #ffffff;
    cdSet.strokeTrans = trans;
    pg.stroke(cdSet.strokeColor, cdSet.strokeTrans);
    return oldcolor;
  }
  // (b-4) 線無し
  void noStroke() {
    cdSet.isStroke = false;
    pg.beginDraw();
    pg.noStroke();
    pg.endDraw();
  }
  // (b-5) 再び線有り
  void reStroke() {
    this.stroke(cdSet.strokeColor, cdSet.strokeTrans);
  }
  // (c) 背景色
  int getBgColor() {
    return bgColor;
  }

  // 現在のビューポートを画面消去する
  void clrView() {
    pg.beginDraw();
    if (transparentView) pg.background(bgColor, 0);
      else               pg.background(bgColor, transparency);
    pg.endDraw();
  }

  // --------------
  // 座標を移動する
  // --------------
  void _moveTo(int x, int y) {
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void moveTo(float x, float y) {
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  void _moveToRel(int dx, int dy) {
    cdSet.vpx += dx;
    cdSet.vpy += dy;
  }
  void moveToRel(float dx, float dy) {
    cdSet.wpx += dx;
    cdSet.wpy += dy;
  }
  // ------------
  // 直線描画関係
  // ------------
  // 直線を描く（ピクセル座標）
  void _line(int x1, int y1, int x2, int y2) {
    pg.beginDraw();
    pg.line(x1, y1, x2, y2);
    pg.endDraw();
    cdSet.vpx = x2;
    cdSet.vpy = y2;
  }
  // 直線を描く（ワールド）
  void line(float x1, float y1, float x2, float y2) {
    _line(px(x1), py(y1), px(x2), py(y2));
    cdSet.wpx = x2;
    cdSet.wpy = y2;
  }
  // 始点省略（絶対座標）
  void _lineTo(int x, int y)    { _line(cdSet.vpx, cdSet.vpy, x, y); }
  void lineTo(float x, float y) { this.line(cdSet.wpx, cdSet.wpy, x, y); }
  // 始点省略（相対座標）
  void _lineRel(int dx, int dy)    { _line(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy); }
  void lineRel(float dx, float dy) { this.line(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy); }
  // ----------
  // 点線を描く
  // ----------
  // (a) 絶対座標指定
  // (a-1) ピクセル座標
  void _dashedLine(int x0, int y0, int x1, int y1, int step) {
//  int    step = 10;
    float  px0, py0, px1, py1, normv;
    float  unitx, unity;
    float  dx, dy;
    int    countmax, count;
    boolean stroke = true;

    cdSet.vpx = x1;
    cdSet.vpy = y1;
    countmax = int(norm(x0, y0, x1, y1) / step);
    if (countmax <= 0) {
      _line(x0, y0, x1, y1);
      return;
    }
    px0 = x0;
    py0 = y0;
    for (count = 0; count < countmax; count++) {
      dx = x1 - px0;
      dy = y1 - py0;
      normv = sqrt(dx * dx + dy * dy);
      if (normv < step) {
        _line(int(px0), int(py0), x1, y1);
        return;
      }
      unitx = dx / normv;
      unity = dy / normv;
      px1 = px0 + unitx * step;
      py1 = py0 + unity * step;
      if (stroke) _line(int(px0), int(py0), int(px1), int(py1));
      stroke = !stroke;
      px0 = px1;
      py0 = py1;
    }
  }
  // (a-2) ワールド座標系
  void dashedLine(float x1, float y1, float x2, float y2, float step) {
    _dashedLine(px(x1), py(y1), px(x2), py(y2), int((_px(step) + _py(step)) / 2.0));
    cdSet.wpx = x2;
    cdSet.wpy = y2;
  }
  // (b) 始点省略（絶対座標）
  void _dashedLineTo(int x, int y, int step)      { _dashedLine(cdSet.vpx, cdSet.vpy, x, y, step); }
  void dashedLineTo(float x, float y, float step) { dashedLine(cdSet.wpx, cdSet.wpy, x, y, step);  }
  // (c) 始点省略（相対座標）
  void _dashedLineRel(int dx, int dy, int step)      { _dashedLine(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy, step); }
  void dashedLineRel(float dx, float dy, float step) { dashedLine(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy, step);  }

  // --------
  // 円を描く
  // --------
  // 真円を描く（生：半径指定）
  void _ellipse(int x, int y, int r) {
    _ellipse(x, y, r, r);
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  // 楕円を描く（生：幅と高さを指定）
  void _ellipse(int x, int y, int w, int h) {
    pg.beginDraw();
    pg.ellipse(x, y, w, h);
    pg.endDraw();
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  // 真円を描く（ワールド：半径指定）
  void ellipse(float x, float y, float r) {
    this.ellipse(x, y, r, r);
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // 楕円を描く（ワールド：幅と高さ指定）
  void ellipse(float x, float y, float w, float h) {
    _ellipse(px(x), py(y), _px(w), _py(h));
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // -----------
  // 三角形を描く
  // -----------
  void _triangle(int x1, int y1, int x2, int y2, int x3, int y3) {
    pg.beginDraw();
    pg.triangle(x1, y1, x2, y2, x3, y3);
    pg.endDraw();
    cdSet.vpx = x3;
    cdSet.vpy = y3;
  }
  void triangle(float x1, float y1, float x2, float y2, float x3, float y3) {
    _triangle(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3));
    cdSet.wpx = x3;
    cdSet.wpy = y3;
  }
  // 中心座標を指定（回転対応）
  void _triangleC(int x, int y, int s, float angle) {
    int   dx1, dy1, dx2, dy2, dx3, dy3;
    float da;
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = int(rotate2Dx(dx1, dy1, angle + da));
    dy2 = int(rotate2Dy(dx1, dy1, angle + da));
    dx3 = int(rotate2Dx(dx1, dy1, angle - da));
    dy3 = int(rotate2Dy(dx1, dy1, angle - da));
    dx1 = int(rotate2Dx(dx1, dy1, angle));
    dy1 = int(rotate2Dy(dx1, dy1, angle));
    _triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void _triangleC(int x, int y, int s) {
    _triangleC(x, y, s, 0.0);
  }
  void triangleC(float x, float y, float s, float angle) {
    float dx1, dy1, dx2, dy2, dx3, dy3, da;
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = rotate2Dx(dx1, dy1, angle + da);
    dy2 = rotate2Dy(dx1, dy1, angle + da);
    dx3 = rotate2Dx(dx1, dy1, angle - da);
    dy3 = rotate2Dy(dx1, dy1, angle - da);
    dx1 = rotate2Dx(dx1, dy1, angle);
    dy1 = rotate2Dy(dx1, dy1, angle);
    this.triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  void triangleC(float x, float y, float s) {
    triangleC(x, y, s, 0.0);
  }
  // ------------
  // 四辺形を描く
  // ------------
  void _quad(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
    pg.beginDraw();
    pg.quad(x1, y1, x2, y2, x3, y3, x4, y4);
    pg.endDraw();
    cdSet.vpx = x4;
    cdSet.vpy = y4;
  }
  void quad(float x1, float y1, float x2, float y2, float x3, float y3, float x4, float y4) {
    _quad(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3), px(x4), py(y4));
    cdSet.wpx = x4;
    cdSet.wpy = y4;
  }
  // ------------
  // 長方形を描く
  // ------------
  // (a) 左上始点
  void _rect(int x, int y, int w, int h) {
    pg.beginDraw();
    pg.rect(x, y, w, h);
    pg.endDraw();
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void rect(float x, float y, float w, float h) {
    _rect(px(x), py(y), _px(w), _py(h));
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // (b) 中央指定
  void _rectC(int x, int y, int w, int h) {
    _rect(x - w / 2, y + h / 2, w, h);
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void rectC(float x, float y, float w, float h) {
    this.rect(x - w / 2.0, y + h / 2.0, w, h);
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // (b) 左上，右下指定
  void _rectLR(int x1, int y1, int x2, int y2) {
    _rect(x1, y1, x2 - x1, y2 - y1);
    cdSet.vpx = x1;
    cdSet.vpy = y1;
  }
  void rectLR(float x1, float y1, float x2, float y2) {
    this.rect(x1, y1, x2 - x1, y2 - y1);
    cdSet.wpx = x1;
    cdSet.wpy = y1;
  }
  // ---------
  // 円弧を描く
  // ---------
  void _arc(int x, int y, int w, int h, float start, float stop) {
    pg.beginDraw();
    pg.arc(x, y, w, h, start, stop);
    pg.endDraw();
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void arc(float x, float y, float w, float h, float start, float stop) {
    _arc(px(x), py(y), _px(w), _py(h), start, stop);
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // ------------
  // 点を描画する
  // ------------
  void _point(int x, int y) {
    pg.beginDraw();
    pg.point(x, y);
    pg.endDraw();
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void point(float x, float y) {
    _point(px(x), py(y));
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  // ----------------
  // テキスト表示関係
  // ----------------
  void prepareFont(int fontsize) {
    prepareFont(_FONTNAME, fontsize);
  }
  void prepareFont(String fontname, int fontsize) {
    if (crowbar.status != 0) {
      println("エラー: フォントの宣言はOptions()かSetup()で行って下さい，");
      return;
    }
    PFont font = createFont(fontname, fontsize, false);  // フォントを変換
    pg.beginDraw();
    pg.textFont(font);         // フォントを設定
    pg.textMode(SCREEN);
    pg.textSize(fontsize);     // フォントサイズを設定（不要？）
    pg.textAlign(LEFT, TOP);   // 配置
    pg.endDraw();
  }
  void textSize(int size) {
    pg.beginDraw();
    pg.textSize(size);
    pg.endDraw();
  }
  void textAlign(int align) {
    pg.beginDraw();
    pg.textAlign(align);
    pg.endDraw();
  }
  void textAlign(int align, int yalign) {
    pg.beginDraw();
    pg.textAlign(align, yalign);
    pg.endDraw();
  }
  // 実際に描画する
  void _text(String str, int x, int y) {
    pg.beginDraw();
    pg.fill(cdSet.textColor);
    pg.text(str, x, y);
    pg.fill(cdSet.fillColor);
    cdSet.vpx = x + int(pg.textWidth(str));
    cdSet.vpy = y;
    pg.endDraw();
  }
  void _text(String str) {
    _text(str, cdSet.vpx, cdSet.vpy);
  }
  void text(String str, float x, float y) {
    _text(str, px(x), py(y));
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  void text(String str) {
    _text(str);
    // ワールド座標を動かすのは面倒なのでパス（不可能ではないが）．
  }
  // 文字色関係
  // (a) 文字色の取得
  int textColor() {
    return cdSet.textColor;
  }
  // (b) 文字色の設定（＋変更前の値を返す）
  int textColor(int c) {
    int oldcolor = this.textColor();
    cdSet.textColor = c & #ffffff;
    return oldcolor;
  }
}

// Tomahawk本体クラス（Crowbarに継承されます）
class tomahawkClass {
  protected int               viewMax;       // 宣言済みのビューポートの数
  protected int               cvNumber;      // 現在注目しているビューポートの番号
  protected viewportClass     cv;            // 現在注目しているビューポートクラス
  protected viewportClass []  viewStack;     // 宣言済みビューポート
  protected viewportClass []  cloneStack;    // Options(), Setup()で宣言した状態をバックアップ＆リストアするための領域
  protected int               tomahawkMode;  // mode == 0: noTomahawk, 1: Basic, 2: Advanced
  tomahawkClass(int mode) {
//  background(_BGCOLOR);  // Crowbarで指定する背景色で初期化
    viewStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    viewStack[0] = null;
    cloneStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    cloneStack[0] = null;
    cvNumber     = 0;
    cv           = null;
    viewMax      = 0;
    tomahawkMode = mode;
  }
  // 現在のビューポートの状態をバックアップ
  void backupViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) cloneStack[i].clone(viewStack[i]);
  }
  // バックアップされたビューポートの状態をリストアする
  void restoreViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) viewStack[i].clone(cloneStack[i]);
  }
  // デバッグ用（全ビューの情報を表示する）
  void displayAllViewData() {
    int  i;
    viewportClass vp;
    for (i = 0; i < viewMax; i++) {
      vp = viewStack[i];
      print(str(i) + ":[" + str(vp.no) + "]" + vp.label + "/(" + str(vp.vx) + "," + str(vp.vy) + "," + str(vp.vwidth) + "," + str(vp.vheight) + ")");
      print("-(" + str(vp.wx_min) + "," + str(vp.wy_min) + "," + str(vp.wx_max) + "," + str(vp.wy_max) + ")");
      print(" Color=#" + hex(vp.viewBgColor()) + ":Ratio =[" + str(vp.pixelRatioX) + "," + str(vp.pixelRatioY) + "]:Scale=" + str(vp.scale()));
      print(" Offset=" + str(vp.offsetX()) + "," + str(vp.offsetY()) + "Visible:" + str(vp.visible));
      println();
    }
  }
  
  // ----------------
  // ビューポート関連
  // ----------------
  // ビューポートの新規作成（成功した場合はビュー番号を返す．失敗した場合は -1 を返す）
  int createView(String label, int x, int y, int width, int height) {
    if (crowbar.status != 0) {
      println("エラー：ビューポートの作成はOptions()かSetup()内で行なって下さい．");
      exit();
    }
    if (viewMax > 0) {
      viewStack  = (viewportClass [])expand(viewStack,  viewMax + 1);  // 最初だけは配列数１で確保されているため
      cloneStack = (viewportClass [])expand(cloneStack, viewMax + 1);  // 最初だけは配列数１で確保されているため
    }
    if (viewStack.length != viewMax + 1) return -1;  // 正しく配列の拡大ができなかった場合
    viewMax++;
    cvNumber = viewMax - 1;
    if ((cv = viewStack[cvNumber] = new viewportClass(label, x, y, width, height)) == null) return -1;  // 実際に確保する
    cloneStack[cvNumber] = new viewportClass(label, x, y, width, height);                               // バックアップ領域を確保
    viewStack[cvNumber].setNo(cvNumber);  // デバッグ用にビュー番号をセット
    // 色情報などのデフォルト設定
    cv.fill(#ffffff,   cv.transparency); // 透明度に対応．初期色
    cv.stroke(#000000, cv.transparency); // 透明度に対応．初期色
    cv.world(0, 0, width, height);
    return cvNumber;
  }
  // 全画面指定のビューポートを作成する
  int createView(String label) {
    return createView(label, 0, 0, width, height);
  }
  // ビューポートを分割して新しく確保したビューポート番号を返す．失敗した場合は-1を返す．
  // (a) 水平方向
  int splitViewH(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合（第二引数）は 0 < rate < 1.0の範囲でして下さい．");
      return -1;
    }
    if (isValidViewNumber(v) == false) { // 不正なビュー番号
      println("エラー：指定されたビューポート番号（第一引数）は不正です／" + str(v));
      return -1;
    }
    // 分割
    viewportClass vp = getView(v);
    int xL, yL, wL, hL;
    int xR, yR, wR, hR;
    // 先に左のビューポートを縮小する
    xL = vp.vx;
    yL = vp.vy;
    wL = vp.vwidth;
    hL = vp.vheight;

    vp.vwidth = int(wL * rate);  // 左のビューポートの変更後の幅
    cv.world(0, 0, vp.vwidth, vp.vheight);
    // 終わったら新規作成
    xR = xL + vp.vwidth;            // 右のビューポートの座標
    yR = yL;
    wR = wL - vp.vwidth;
    hR = hL;
    return createView(label, xR, yR, wR, hR);
  }
  // (b) 垂直方向
  int splitViewH(String label, float rate) {
    return splitViewH(cvNumber, label, rate);
  }
  int splitViewV(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合（第二引数）は 0 < rate < 1.0の範囲でして下さい．");
      return -1;
    }
    if (isValidViewNumber(v) == false) { // 不正なビュー番号
      println("エラー：指定されたビューポート番号（第一引数）は不正です／" + str(v));
      return -1;
    }
    // 分割
    viewportClass vp = getView(v);
    int xU, yU, wU, hU;
    int xL, yL, wL, hL;
    // 先に上のビューポートを縮小する
    xU = vp.vx;
    yU = vp.vy;
    wU = vp.vwidth;
    hU = vp.vheight;

    vp.vheight = int(hU * rate);  // 上のビューポートの変更後の高さ
    cv.world(0, 0, vp.vwidth, vp.vheight);
    // 終わったら新規作成
    xL = xU;
    yL = yU + vp.vheight;
    wL = wU;
    hL = hU - vp.vheight;
    return createView(label, xL, yL, wL, hL);
  }
  int splitViewV(String label, float rate) {
    return splitViewV(cvNumber, label, rate);
  }

  // 指定されたビュー番は正しいか？
  boolean isValidViewNumber(int n) {
    if (n >= viewMax)         return false;
    if (n < 0)                return false;
    if (n > viewStack.length) return false;
    return true;
  }
  // 現在のビューポート番号を返す
  int view() { return cvNumber; }
  // ビューポートを選択（成功した場合はtrue）
  boolean view(int n) {
    if (isValidViewNumber(n) == false) return false;
    cvNumber = n;
    cv = viewStack[cvNumber];
    return true;
  }
  // 指定された番号のビューポートクラスを返す
  viewportClass getView(int no) {
    if (isValidViewNumber(no) == false) return null;
    return viewStack[no];
  }
  // ------------------------
  // 現在のビューポートの移動
  // ------------------------
  // 一つ次のビューポートへ
  int nextView() {
    if (++cvNumber >= viewMax) cvNumber = viewMax - 1;
    view(cvNumber);
    return cvNumber;
  }
  // 一つ前のビューポートへ
  int prevView() {
    if (--cvNumber < 0) cvNumber = 0;
    view(cvNumber);
    return cvNumber;
  }
  // 最初に確保されたビューポートへ
  void firstView() { view(0);           }
  // 最後に確保されたビューポートへ
  void lastView()  { view(viewMax - 1); }
  
  // ----------------------------
  // ビューポートの上下関係の移動
  // ----------------------------
  // ビューポートの表示順を表わす行列を返す
  int [] getViewOrderList() {
    int [] order = new int[viewMax];
    int i;
    for (i = 0; i < viewMax; i++) order[viewStack[i].no] = i;
    return order;
  }
  // ビューポートの表示順リストを実際のビューポートのデータに書き戻す
  void updateViewOrder(int [] order) {
    int i;
    for (i = 0; i < viewMax; i++) viewStack[order[i]].no = i;
  }
  // 指定したビューポートの表示順を一番上に移す
  void moveToTop(int v) {
    if (viewMax == 1) return;  // 不要なので
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].no; i < viewMax - 1; i++) order[i] = order[i + 1];
    order[i] = v;
    updateViewOrder(order);
  }
  // 指定したビューポートの表示順を一番下に移す
  void moveToBottom(int v) {
    if (viewMax == 1) return;  // 不要なので
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].no; i > 0; i--) order[i] = order[i - 1];
    order[0] = v;
    updateViewOrder(order);
  }
  // 現在のビューポートの表示順を一番上に移す
  void moveToTop()    { moveToTop(cvNumber); }
  // 現在のビューポートの表示順を一番下に移す
  void moveToBottom() { moveToBottom(cvNumber); }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // (a) フレームカラーを修得
  int frameColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).frameColor;
  }
  // (b) フレームカラーの設定（＋変更前の値）
  // (b-2) 指定したビューポート
  int frameColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    int oldcolor = frameColor(v);
    getView(v).frameColor(c);
    return oldcolor;
  }
  // (c) フレームの表示／非表示
  void viewDrawFrame(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).drawFrame(flag);
  }
  // (a) 不透明度の取得
  int transparency(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency();
  }
  // (b) 不透明度の設定（＋変更前の値）
  int transparency(int v, int trans) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency(trans);
  }
  // 指定されたビューの背景色を変更する
  // (a) 背景色を取得
  int viewBgColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor();
  }
  // (b) 背景色を指定
  int viewBgColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor(c);
  }
  // ビューポートの背景を透明にするか？（trueで透明）
  void transparentView(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).transparentView(flag);    
  }

  // ------------------------
  // ビューポートの位置の変更
  // ------------------------
  // (a) 指定した位置に移動
  // (a-1) 現在のビューポート
  // (a-2) 指定したビューポート
  void moveView(int v, int x, int y) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).moveView(x, y);
  }
  void moveViewRel(int v, int dx, int dy) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).moveViewRel(dx, dy);
  }

  // 座標変換（現在のビュー）
  // 座標変換（ワールド→ビュー：大きさのみ）
  private int _px(float x) { return cv._px(x); }
  private int _py(float y) { return cv._py(y); }
  // 座標変換（ワールド→ビュー）
  private int px(float x) { return cv.px(x); }
  private int py(float y) { return cv.py(y); }

  // -----------------------------------
  // 描画コマンド関係（ユーザによる利用）
  // -----------------------------------
  // 全てのビューポートを画面消去する（現在のビューは変更しない）
  void clrAllView() {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) clrView(i);
    view(cv);
  }
  // 指定されたビューポートを画面消去する（現在のビューポートをそのビューポートに移す）
  void clrView(int i) {
    if (view(i) != true) return;
    view(i);
    cv.clrView();
  }

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す：不正なビュー番号の場合は1.0を返す
  float scale(int v) {
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    return getView(v).scale();
  }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float scale(int v, float s) {
    float oldscale;    
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    oldscale = getView(v).scale();
    getView(v).scale(s);
    return oldscale;
  }
  // (c) 全てのビューポートの倍率を指定した倍率に変更する
  void allScale(float s) {
    int i;
    for (i = 0; i < viewMax; i++) scale(i, s);
  }

  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 指定したビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float offsetX(int v) {
      viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    return vp.offsetX;
  }
  // (a-2) ｙ方向
  float offsetY(int v) {
      viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    return vp.offsetY;
  }
  // (b) オフセット量を変更する（＋変更前の値を返す）：指定したビューポート
  // (b-1) ｘ方向
  float offsetX(int v, float offset) {
    viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    float oldoffset = vp.offsetX;
    vp.offsetX = offset;
    return oldoffset;
  }
  // (b-2) ｙ方向
  float offsetY(int v, float offset) {
    viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    float oldoffset = vp.offsetY;
    vp.offsetY = offset;
    return oldoffset;
  }
  // (b-3) ｘ方向／ｙ方向同時：成功した場合はtrueを返す
  boolean offset(int v, float offx, float offy) {
    float oldoffset;
    viewportClass vp = getView(v);
    if (vp == null) return false;  // 不正なビュー
    vp.offsetX = offx;
    vp.offsetY = offy;
    return true;
  }

  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  // (a) ワールド座標系の設定値の参照：指定したビューポート：不正なビューポート番号が指定された場合は0.0を返す
  float world_minX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_min;
  }
  float world_maxX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_max;
  }
  float world_minY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_min;
  }
  float world_maxY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_max;
  }
  // 【変更】
  // (a) ワールド座標系の変更：指定したビューポート：成功した場合はtrueを返す
  boolean world(int v, float x1, float y1, float x2, float y2) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).world(x1, y1, x2, y2);
    return true;
  }
  // (b) ｘ，ｙそれぞれの最小値／最大値を個別に変更
  // 指定したビューポートの値を変更：成功した場合はtrueを返す
  boolean world_minX(int v, float x) { return world(v, x, world_maxX(v), world_minY(v), world_maxY(v)); }
  boolean world_maxX(int v, float x) { return world(v, world_minX(v), x, world_minY(v), world_maxY(v)); }
  boolean world_minY(int v, float y) { return world(v, world_minX(v), world_maxX(v), y, world_maxY(v)); }
  boolean world_maxY(int v, float y) { return world(v, world_minX(v), world_maxX(v), world_minY(v), y); }

  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).isVisible();
  }
  // (a) 表示
  void visible(int v) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).visible();
  }
  // (b) 非表示
  void unVisible(int v) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).unVisible();
  }
  // (c) 引数による指定
  void changeVisible(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    if (flag) visible(v); else unVisible(v);
  }
  // (d) トグル動作
  boolean toggleVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).toggleVisible();
  }

  // 与えられた座標（マウスクリックなど）はどのビューポートを指しているのか？
  // （どのビューポートも含まれないならば-1）
  // trans == true の場合は透明なビューポートを無視する
  int detectActiveView(int x, int y) { return detectActiveView(x, y, true); }
  int detectActiveView(int x, int y, boolean trans) {
    int [] order = getViewOrderList();
    int i, x1, y1, x2, y2;
    viewportClass vp;
    for (i = viewMax - 1; i >= 0; i--) {
      vp = viewStack[order[i]];
      if (vp.isVisible() == false) continue;  // 非表示は無視
      if ((trans == true) && (vp.transparentView == true)) continue;
      if (vp.transparency == 0) {           // ビューポートが透明背景
        if (trans == true) continue;
        int col;
        vp.pg.beginDraw();
        col = vp.pg.get(x - vp.vx, y - vp.vy);
        vp.pg.endDraw();
        if ((col & 0x0ff000000) == 0) continue;  // もし描画されていないならば無視
      }
      x1 = vp.vx;
      y1 = vp.vy;
      x2 = x1 + vp.vwidth;
      y2 = y1 + vp.vheight;
      if ((x1 <= x) && (x <= x2) && (y1 <= y) && (y <= y2)) return order[i];
    }
    return -1;
  }

// =========================================================================================================

  // -----------------------------------------------
  // ビューの合成（crowbarClass.pdeのdraw()にて使用）
  // -----------------------------------------------
  void refreshView() {
    if (tomahawkMode == 0) return;
//    if (tomahawkMode == 2) background(_BGCOLOR);
    if (crowbar.status == 10) {
      // ビューの合成
      int i;
      int [] order = getViewOrderList();
      // 表示順のリスト作成（きちんと欠番，重複なく並んでいることは上下関係の操作時に確認済みという前提）
      viewportClass vp;
      for (i = 0; i < viewMax; i++) {
        vp = viewStack[order[i]];
        if (vp.isVisible() == false) continue;
        if (vp.drawFrame) {
          vp.pg.beginDraw();
          vp.pg.noFill();
          vp.pg.stroke(vp.frameColor, 255);
          vp.pg.rect(0, 0, vp.vwidth - 1, vp.vheight - 1);
          vp.pg.endDraw();
        }
        blend(vp.pg, 0, 0, vp.vwidth, vp.vheight, vp.vx, vp.vy, vp.vwidth, vp.vheight, BLEND);
      }
    }
    if (tomahawkMode == 2) blend(crowbar.screenSetting.pgText, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, BLEND);
  }
}

