package Cuitter::Plugin::Core;

use Mouse::Role;
use Encode;
use POSIX qw( strftime );
use DateTime::Format::DateParse;

requires qw( set_help input output );

sub _init_Core {
    my $self = shift;
    $self->set_help( 'exit', 'exit from cuitter' );
    $self->set_help( 'ft', 'show friends time-line.' );
}

sub prompt {
    my $self = shift;
    $self->output->print( '@'.$self->username.": " );
    my $line = $self->input->getline;
    chomp $line;
    $line = '/ft' unless $line;
    if ( $line =~ /^\// ) {
        $line =~ s/^\///;
        my ( $command, @arg ) = split / /, $line;
        my $exec = "com_$command";
        if ( $self->can( "$exec" ) ) {
            return $self->$exec( @arg );
        }
        else {
            $self->output->print( "$command ? I don't know it.\n" );
            return 1;
        }
    }
    else {
        $self->tweet( $line );
        $self->com_ft;
        return 1;
    }
}

sub loop {
    my $self = shift;
    while ( $self->prompt ) {};
}

sub tweet {
    my ( $self, $line ) = @_;
    $self->client->update( $line );
}

sub com_exit {
    my $self = shift;
    $self->output->print( "Sayonara.\n" );
    return;
}

sub com_ft {
    my ( $self, $rows ) = @_;
    $self->rows( $rows ) if $rows;
    $rows ||= $self->rows;
    $self->update_status( $rows );
    for my $tweet ( @{ $self->status } ) {
        $self->output->print( $tweet->{ display } );
    }
    $self->output->print( '--- Last Update: '. strftime( '%Y-%m-%d %H:%M:%S', localtime() )." ---\n" );
    1;
}

sub update_status {
    my ( $self, $rows ) = @_;
    my $id = 0;
    $self->clear_status;
    $self->status( [] );
    for my $tweet ( @{ $self->client->friends_timeline( { rows => $rows } ) } ) {
        $tweet->{ created_at } = DateTime::Format::DateParse->parse_datetime( $tweet->{ created_at } );
        my $delaysec = time() - $tweet->{ created_at }->epoch;
        $tweet->{ display } = $self->_build_display( $id, $delaysec, $tweet );
        push @{ $self->status }, $tweet;
        $id++;
        last if $id >= $rows;
    }
}

sub _build_display {
    my ( $self, $id, $delaysec, $tweet ) = @_;
    my $delaymin = int( $delaysec / 60 );
    my $delayhour = int( $delaymin / 60 );
    my $delayday = int( $delayhour / 24 );
    $delayhour -= $delayday * 24 if $delayday;
    $delaymin -= $delayhour * 60 if $delayhour;
    $delaysec -= $delaymin * 60 if $delaymin;
    my $delay = $delayday."Day ".sprintf("%02d",$delayhour).":".sprintf("%02d",$delaymin).":".sprintf("%02d",$delaysec);
    "---\n[$id] ". $delay. " ago\n". '@'.$tweet->{user}{screen_name}. ': '. encode( "utf8", $tweet->{text} ). "\n";
}

no Mouse::Role;

1;
__END__

=head1 NAME

Cuitter::Plugin::Core - Cuitter Core Plugin

=head1 DESCRIPTION

This plugin provides core utility methods and commands to Cuitter.

=head1 COMMANDS IT PROVIDES

=head2 /exit

exit from cuitter

=head2 /ft [ maxrows ]

show friends timeline

=head1 LICENSE

Artistic

=head1 AUTHOR

Satoshi Azuma E<lt>ytnobody@gmail.comE<gt>

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
