/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package org.apache.jetspeed.util.template;

// Java imports
import java.util.Stack;

// Cactus and Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;
import junit.awtui.TestRunner;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.util.template.JetspeedLink;

// Turbine imports
import org.apache.turbine.services.pool.PoolService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

/**
 * TestTurbineCache
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestJetspeedLinkFactory.java,v 1.1 2004/04/07 22:02:42 jford Exp $
 */
public class TestJetspeedLinkFactory extends JetspeedTestCase
{
    /**
     * Configuration object to run Turbine outside a servlet container
     * ( uses turbine.properties )
     */
    private static TurbineConfig config = null;
    
    /**
     * Sets up TurbineConfig using the system property:
     * <pre>turbine.properties</pre>
     */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp",
            "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }
    
    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestJetspeedLinkFactory(String name)
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestJetspeedLinkFactory.class.getName() } );
    }
    
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestJetspeedLinkFactory.class );
    }
    
    /**
     * Simple test that get a JetspeedLink object
     *
     * @throws Exception
     */
    public void testSimpleGet() throws Exception
    {
        JetspeedLink jsLink = JetspeedLinkFactory.getInstance();
        System.out.println("Class return by JetspeedLinkFactory: " + jsLink.getClass().getName());
        assertNotNull( "Got JetspeedLink", jsLink);
        assertEquals( "Created class defined by tools.request.jslink"
        , JetspeedResources.getString("tool.request.jslink"
        , "org.apache.jetspeed.util.template.BaseJetspeedLink"), jsLink.getClass().getName());
        assertTrue( "Class instance of JetspeedLink", (jsLink instanceof JetspeedLink));
    }
    
    /**
     * Simple test that gets and put a JetspeedLink object
     *
     * @throws Exception
     */
    public void testGetandPut() throws Exception
    {
        PoolService poolService = (PoolService) TurbineServices.
        getInstance().getService(PoolService.SERVICE_NAME);
        JetspeedLink jsLink = null;
        int beforeSize;
        
        for (int counter = 0; counter < 10; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
        }
        String linkClassName = jsLink.getClass().getName();
        jsLink = null;
        
        for (int counter = 0; counter < 10; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get/put loop - Got JetspeedLink", jsLink);
            assertTrue( "Get/put loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
            beforeSize = poolService.getSize( linkClassName);
            JetspeedLinkFactory.putInstance(jsLink);
            assertTrue( "Class saved in pool", (beforeSize < poolService.getSize( linkClassName)));
            jsLink = null;
        }
    }
    public void testFillPool() throws Exception
    {
        Stack jsLinkStack = new Stack();
        JetspeedLink jsLink = null;

        PoolService poolService = (PoolService) TurbineServices.
        getInstance().getService(PoolService.SERVICE_NAME);
        int poolCapacity;

        jsLink = JetspeedLinkFactory.getInstance();
        String linkClassName = jsLink.getClass().getName();
        poolCapacity = poolService.getCapacity( linkClassName);

        System.out.println("Class Name  " + linkClassName);
        System.out.println("Pool Capacity " + poolCapacity);

        // Fill stack with objects
        for (int counter = 0; counter < poolCapacity; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
            jsLinkStack.push(jsLink);
        }
        
        // Fill up the pool
        while (jsLinkStack.empty() == false)
            JetspeedLinkFactory.putInstance( (JetspeedLink) jsLinkStack.pop());
        assertEquals( "Pool is full", poolCapacity, poolService.getSize(linkClassName));
        
        // Empty pool
        for (int counter = 0; counter < poolCapacity; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
        }
        assertEquals( "Pool is empty", 0, poolService.getSize(linkClassName));
    }
}
