/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.modules.parameters;

//ecs stuff
import org.apache.ecs.html.Select;
import org.apache.ecs.html.Option;

// java stuff
import java.util.Map;
import java.util.Arrays;
import java.util.StringTokenizer;

//turbine support
import org.apache.turbine.util.RunData;

/**
 * Returns simple list box control.
 * <p>Options:
 * <UL>
 * <LI><code>items</code> - comma-separated list of list box items</LI>
 * <LI><code>layout</code> [<strong>$combo</strong>|$list] - combo box vs list box</LI> 
 * <LI><code>size</code> - size of the list box for $list style</LI> 
 * <LI><code>sort</code> [<strong>false</strong>|true] - return sorted list of items</LI>  
 * <LI><code>multiplechoice</code> [<strong>false</strong>|true] - allow multiple selections</LI>  
 * <LI><code>null-if-empty</code> [<strong>false</strong>|true] - do not return a select control if item list is empty</LI>   
 * </UL> 
 * @author <a href="mark_orciuch@ngsltd.com">Mark Orciuch</a>
 * @author <a href="mailto:solutioncenter@infointegrators.com">Dave Trapp</a>
 * @version $Id: ListBox.java,v 1.5 2004/02/23 03:01:20 jford Exp $  
 */
public class ListBox extends ParameterPresentationStyle
{

    public static final String SORT = "sort";
    public static final String ITEMS = "items";
    public static final String LAYOUT = "layout";
    public static final String LAYOUT_COMBO = "$combo";
    public static final String LAYOUT_LIST = "$list";
    public static final String LIST_SIZE = "listsize";
    public static final String MULTIPLE_CHOICE = "multiplechoice";
    public static final String NULL_IF_EMPTY = "null-if-empty";

    protected String layout = null;
    protected String items[] = null; 
    protected String size = null;
    protected boolean multiple = false;

    /**
     * Returns presentation control
     * 
     * @param data
     * @param name
     * @param value
     * @param parms
     * @return string
     */
    public String getContent(RunData data, String name, String value, Map parms)
    {

        init(data);

        Select select = null;
        if ( layout.equalsIgnoreCase(LAYOUT_LIST) )
        {
            select = new Select(name, new Integer(size).intValue());
        } else
        {
            select = new Select(name);
        }

        if ( multiple )
        {
            select.setMultiple(multiple);
        }

        if ( items != null )
        {

            boolean sort = new Boolean((String)this.getParm(SORT, "false")).booleanValue();
            if ( sort )
            {
                Arrays.sort(items);
            }

            for ( int i=0; i < items.length; i++ )
            {
                Option option = new Option(items[i]).addElement(items[i]);
                if (multiple)
                {
                    option.setSelected(value.indexOf(items[i]) >= 0);
                }
                else
                {
                    option.setSelected(items[i].equalsIgnoreCase(value));
                }
                select.addElement(option);
            }
        }

        // If no items to display, do not display empty control
        boolean nullIfEmpty = new Boolean((String)this.getParm(NULL_IF_EMPTY, "false")).booleanValue();
        if ( this.items == null || (nullIfEmpty && items.length == 0) )
        {
            return null;
        }

        return select.toString();

    }

    /**
     * Initialize options
     * 
     * @param data
     */
    protected void init(RunData data)
    {

        this.layout = (String)this.getParm(LAYOUT, LAYOUT_COMBO);
        this.items = this.getItems(data); 
        this.size = (String)this.getParm(LIST_SIZE, "1");
        this.multiple = new Boolean((String)this.getParm(MULTIPLE_CHOICE, "false")).booleanValue();

    }

    /**
     * Parse items into an arrary of strings
     * 
     * @param data
     * @return string array
     */
    protected String[] getItems(RunData data)
    {

        String[] result = null;
        String list = (String)this.getParm(ITEMS, ""); 

        StringTokenizer it = new StringTokenizer(list, ",");
        int size = it.countTokens();

        if ( size > 0 )
        {
            result = new String[size];

            int i = 0;
            while ( it.hasMoreTokens() )
            {
                String item = (String)it.nextToken();
                result[i] = item;
                i++;
            }
        }

        return result; 

    }

}
