/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2009 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.circulation;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTCirculationCategory;
import com.aimluck.eip.cayenne.om.portlet.AvzTCirculationRequest;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTCirculationCategory;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTCirculationRequest;
import com.aimluck.eip.circulation.util.CirculationUtils;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 回覧板カテゴリのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class CirculationCategoryFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(CirculationCategoryFormData.class.getName());

  /** カテゴリ名 */
  private ALStringField category_name;

  /** メモ */
  private ALStringField note;
  
  /** テンプレート */
  private ALStringField ordertemplate;

  /** 変更・削除対象のカテゴリID */
  private Integer category_id;

  private DataContext dataContext;

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    String categoryid = ALEipUtils.getTemp(rundata, context,
        ALEipConstants.ENTITY_ID);
    if (categoryid != null && Integer.valueOf(categoryid) != null) {
      category_id = Integer.valueOf(categoryid);
    }

    dataContext = DatabaseOrmService.getInstance().getDataContext();
  }

  /**
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // カテゴリ名
    category_name = new ALStringField();
    category_name.setFieldName("カテゴリ名");
    category_name.setTrim(true);
    // メモ
    note = new ALStringField();
    note.setFieldName("メモ");
    note.setTrim(true);
    // テンプレート
    ordertemplate = new ALStringField();
    ordertemplate.setFieldName("テンプレート");
    ordertemplate.setTrim(true);
  }

  /**
   * カテゴリの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // カテゴリ名必須項目
    category_name.setNotNull(true);
    // カテゴリ名文字数制限
    category_name.limitMaxLength(50);
    // テンプレート文字数制限
    ordertemplate.limitMaxLength(1000);
    // メモ文字数制限
    note.limitMaxLength(1000);
  }

  /**
   * 回覧板カテゴリのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(ArrayList msgList) {
    try {
      SelectQuery query = new SelectQuery(AvzTCirculationCategory.class);
      Expression exp1 = ExpressionFactory
          .matchExp(_AvzTCirculationCategory.CATEGORY_NAME_PROPERTY, category_name
              .getValue());
      query.setQualifier(exp1);
      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        Expression exp2 = ExpressionFactory.noMatchDbExp(
            _AvzTCirculationCategory.CATEGORY_ID_PK_COLUMN, category_id);
        query.andQualifier(exp2);
      }
      if (dataContext.performQuery(query).size() != 0) {
        msgList.add("カテゴリ『 <span class='em'>" + category_name.toString()
            + "</span> 』は既に登録されています。");
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    // カテゴリ名
    category_name.validate(msgList);
    // メモ
    note.validate(msgList);
    // テンプレート
    ordertemplate.validate(msgList);
    return (msgList.size() == 0);
  }

  /**
   * 回覧板カテゴリをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCirculationCategory category = CirculationUtils.getAvzTCirculationCategory(
          rundata, context);
      if (category == null)
        return false;
      // カテゴリ名
      category_name.setValue(category.getCategoryName());
      // テンプレート
      ordertemplate.setValue(category.getTemplate());
      // メモ
      note.setValue(category.getNote());
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 回覧板カテゴリをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      AvzTCirculationCategory category = (AvzTCirculationCategory) dataContext
          .createAndRegisterNewObject(AvzTCirculationCategory.class);
      category.setCategoryName(category_name.getValue());
      category.setNote(note.getValue());
      category.setUserId(Integer.valueOf(ALEipUtils.getUserId(rundata)));
      category.setCreateDate(Calendar.getInstance().getTime());
      category.setUpdateDate(Calendar.getInstance().getTime());
      category.setTemplate(ordertemplate.getValue());
      dataContext.commitChanges();
    } catch (Exception ex) {
      logger.error("Exception", ex);
      dataContext.rollbackChanges();
      return false;
    }
    return true;
  }

  /**
   * データベースに格納されている回覧板カテゴリを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCirculationCategory category = CirculationUtils.getAvzTCirculationCategory(
          rundata, context);
      if (category == null)
        return false;
      // カテゴリ名
      category.setCategoryName(category_name.getValue());
      // テンプレート
      category.setTemplate(ordertemplate.getValue());
      // メモ
      category.setNote(note.getValue());
      // ユーザーID
      category.setUserId(Integer.valueOf(ALEipUtils.getUserId(rundata)));
      // 更新日
      category.setUpdateDate(Calendar.getInstance().getTime());

      // カテゴリを更新
      dataContext.commitChanges();

    } catch (Exception ex) {
      logger.error("Exception", ex);
      dataContext.rollbackChanges();
      return false;
    }
    return true;
  }

  /**
   * 回覧板カテゴリを削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCirculationCategory category = 
        CirculationUtils.getAvzTCirculationCategory(
          rundata, context);
      if (category == null)
        return false;

      if (category.getCategoryId().intValue() == 1) {
        // カテゴリ「未分類」は削除不可
        msgList.add("カテゴリ『 <span class='em'>未分類</span> 』は削除できません。");
        return false;
      }
      
      // このカテゴリに含まれる依頼をカテゴリ「未分類」に移す。
      SelectQuery query = new SelectQuery(AvzTCirculationRequest.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          _AvzTCirculationRequest.TO_AVZ_TCIRCULATION_CATEGORY_PROPERTY + "."
              + _AvzTCirculationCategory.CATEGORY_ID_PK_COLUMN, category
              .getCategoryId());
      query.setQualifier(exp1);
      List requests = dataContext.performQuery(query);
      if (requests != null && requests.size() > 0) {
        AvzTCirculationRequest request = null;
        AvzTCirculationCategory defaultCategory = CirculationUtils
            .getAvzTCirculationCategory(dataContext, Long.valueOf(1));
        int size = requests.size();
        for (int i = 0; i < size; i++) {
          request = (AvzTCirculationRequest) requests.get(i);
          request.setToAvzTCirculationCategory(defaultCategory);
        }
      }

      // 回覧板カテゴリを削除
      dataContext.deleteObject(category);

      dataContext.commitChanges();
      
      CirculationUtils.clearFilter(rundata, context);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      dataContext.rollbackChanges();
      return false;
    }
    return true;
  }

  /**
   * カテゴリ名を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getCategoryName() {
    return category_name;
  }

  /**
   * テンプレートを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getTemplate() {
    return ordertemplate;
  }

  /**
   * メモを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

}
