/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.extcabinet;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTCabinetFile;
import com.aimluck.eip.cayenne.om.portlet.AvzTCabinetFolder;
import com.aimluck.eip.cayenne.om.portlet.AvzTFolderAcl;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTCabinetFolder;
import com.aimluck.eip.cayenne.om.portlet.auto._AvzTFolderAcl;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipPost;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.extcabinet.util.ExtCabinetUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 個人情報のフォルダフォームデータを管理するクラス <BR>
 * 
 */
public class ExtCabinetFolderFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(ExtCabinetFolderFormData.class
      .getName());

  /** フォルダ名 */
  private ALStringField folder_name;

  /** 親フォルダ ID */
  private ALNumberField parent_id;

  /** 元の親フォルダ ID */
  private ALNumberField org_parent_id;

  /** メモ */
  private ALStringField note;

  /** フォルダ情報一覧 */
  private List<ExtFolderInfo> folder_hierarchy_list;

  /** 選択されたフォルダ情報 */
  private ExtFolderInfo selected_folderinfo = null;

  /** 親フォルダ情報 */
  private ExtFolderInfo parent_folderinfo = null;

  private RunData rundata;

  private String folderid = null;

  private String org_id;

  /** アクセス権限リスト by motegi */
  private List<Map<String, String>> aclRequestList;

  /** アクセス権限リスト（更新時表示用） by motegi */
  private List<ExtCabinetAclResultData> aclList;

  private boolean is_owner;

  /** ログインユーザーID */
  private long login_user_id;

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      // ENTITY ID
      if (rundata.getParameters().containsKey(
        ExtCabinetUtils.REQUEST_KEY_FOLDER_ID)) {
        ALEipUtils.setTemp(
          rundata,
          context,
          ExtCabinetUtils.SESSION_KEY_FOLDER_ID,
          rundata.getParameters().getString(
            ExtCabinetUtils.REQUEST_KEY_FOLDER_ID));
      }
    }

    org_id = DatabaseOrmService.getInstance().getOrgId(rundata);

    String tmpfid =
      ALEipUtils.getTemp(
        rundata,
        context,
        ExtCabinetUtils.SESSION_KEY_FOLDER_ID);
    int fid = ExtCabinetUtils.ROOT_FODLER_ID;
    if (tmpfid != null && !"".equals(tmpfid)) {
      try {
        fid = Integer.parseInt(tmpfid);
      } catch (Exception e) {
        fid = ExtCabinetUtils.ROOT_FODLER_ID;
      }
    } else {
      fid = ExtCabinetUtils.ROOT_FODLER_ID;
    }
    ExtFolderInfo rootExtFolderInfo = null;
    folder_hierarchy_list =
      ExtCabinetUtils.getFolderList(ExtCabinetUtils.ROOT_PARENT_FODLER_ID);
    if (folder_hierarchy_list != null && folder_hierarchy_list.size() > 0) {
      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        ExtFolderInfo info = folder_hierarchy_list.get(i);
        info.setVisible(true);
        if (info.getFolderId() == ExtCabinetUtils.ROOT_FODLER_ID) {
          rootExtFolderInfo = info;
        }
        if (info.getFolderId() == fid) {
          selected_folderinfo = info;
        }
      }
    }
    ExtCabinetUtils.setFolderVisibleForForm(folder_hierarchy_list, rundata);

    if (selected_folderinfo == null) {
      selected_folderinfo = rootExtFolderInfo;
    }

    this.rundata = rundata;
    login_user_id = ALEipUtils.getALEipUser(rundata).getUserId().getValue();

  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      if (!rundata.getParameters().containsKey("parent_id")) {
        parent_id.setValue(selected_folderinfo.getFolderId());
      }

      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        folderid =
          ALEipUtils.getTemp(
            rundata,
            context,
            ExtCabinetUtils.SESSION_KEY_FOLDER_ID);
      }

      if (ALEipConstants.MODE_NEW_FORM.equals(getMode())) {

        aclList =
          ExtCabinetUtils.loadFolderAclList(selected_folderinfo.getFolderId());

        // 親フォルダから引き継いだアクセス権限にログインユーザーのレコードが
        // あれば除外します。これはルートフォルダ直下にフォルダやファイルを作成
        // する時が対象の処理です。
        aclList = ExtCabinetUtils.excludeLoginUser(aclList, login_user_id);
      }

      // motegi start アクセス権限情報をリクエストから取得します。
      if (ALEipConstants.MODE_INSERT.equals(getMode())
        || ALEipConstants.MODE_UPDATE.equals(getMode())) {
        aclRequestList = new ArrayList<Map<String, String>>();
        String[] target_ids =
          rundata.getParameters().getStrings("acl[target_id][]");
        String[] target_types =
          rundata.getParameters().getStrings("acl[target_type][]");
        String[] f =
          rundata.getParameters().getStrings("acl[file_creatable][]");
        String[] c =
          rundata.getParameters().getStrings("acl[folder_creatable][]");
        String[] r = rundata.getParameters().getStrings("acl[readable][]");
        String[] w = rundata.getParameters().getStrings("acl[writable][]");
        String[] d = rundata.getParameters().getStrings("acl[deletable][]");
        if (target_ids != null && target_ids.length > 0) {
          int acl_num = target_ids.length;
          for (int i = 0; i < acl_num; i++) {
            Map<String, String> map = new HashMap<String, String>();
            map.put(ExtCabinetUtils.AUTHORITY_TARGET_ID, target_ids[i]);
            map.put(ExtCabinetUtils.AUTHORITY_TARGET_TYPE, target_types[i]);
            map.put(ExtCabinetUtils.AUTHORITY_F, f[i]);
            map.put(ExtCabinetUtils.AUTHORITY_C, c[i]);
            map.put(ExtCabinetUtils.AUTHORITY_R, r[i]);
            map.put(ExtCabinetUtils.AUTHORITY_U, w[i]);
            map.put(ExtCabinetUtils.AUTHORITY_D, d[i]);
            aclRequestList.add(map);
          }
        }
      }
      // motegi end

    }
    return res;
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // フォルダ名
    folder_name = new ALStringField();
    folder_name.setFieldName("フォルダ名");
    folder_name.setTrim(true);
    // 親フォルダ
    parent_id = new ALNumberField();
    parent_id.setFieldName("追加位置");
    parent_id.setValue(0);
    // メモ
    note = new ALStringField();
    note.setFieldName("メモ");
    note.setTrim(true);
    // 編集前の元の親フォルダ
    org_parent_id = new ALNumberField();
    org_parent_id.setFieldName("編集前の元の親フォルダ");
    org_parent_id.setValue(0);
  }

  /**
   * フォルダの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // フォルダ名必須項目
    folder_name.setNotNull(true);
    // フォルダ名の文字数制限
    folder_name.limitMaxLength(128);
    // 追加位置
    parent_id.limitMinValue(0);
    // メモの文字数制限
    note.limitMaxLength(1000);
  }

  /**
   * フォルダのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // フォルダ名
    folder_name.validate(msgList);
    // 追加位置
    parent_id.validate(msgList);
    // メモ
    note.validate(msgList);

    if (existsFolderName()) {
      msgList.add("このフォルダ名と同じフォルダがすでに存在するため、登録できません。フォルダ名を変更してください。");
    }

    /** 編集アクセス制限 */
    if (ALEipConstants.MODE_INSERT.equals(getMode())) {
      // フォルダ追加の場合には、親フォルダに編集権限があるかチェック
      if (!ExtCabinetUtils.isFolderCreatableFolder(
        (int) parent_id.getValue(),
        rundata)) {
        msgList.add("このフォルダに追加する権限がありません。");
      }
    }

    if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
      // フォルダ追加の場合には、親フォルダに編集権限があるかチェック
      if (parent_id.getValue() != org_parent_id.getValue()) {
        // 更新時に元親フォルダと新親フォルダが異なるということは、フォルダの移動が
        // 指定されたということ。
        // 移動する権限（元親フォルダに削除権限、新親フォルダに追加権限）があるかチェック
        if (!ExtCabinetUtils.isFolderCreatableFolder(
          (int) parent_id.getValue(),
          rundata)) {
          msgList.add("移動先のフォルダに追加権限がありません。");
        }
      }
    }

    return (msgList.size() == 0);
  }

  private boolean existsFolderName() {
    String fname = folder_name.getValue();
    if (fname == null || "".equals(fname)) {
      return false;
    }

    try {
      SelectQuery<AvzTCabinetFolder> query =
        Database.query(AvzTCabinetFolder.class);
      if (ALEipConstants.MODE_INSERT.equals(getMode())) {
        Expression exp =
          ExpressionFactory.matchExp(
            _AvzTCabinetFolder.PARENT_ID_PROPERTY,
            Integer.valueOf((int) parent_id.getValue()));
        query.setQualifier(exp);
      } else if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        Expression exp1 =
          ExpressionFactory.matchExp(
            _AvzTCabinetFolder.PARENT_ID_PROPERTY,
            Integer.valueOf((int) parent_id.getValue()));
        query.setQualifier(exp1);
        Expression exp2 =
          ExpressionFactory.noMatchDbExp(
            _AvzTCabinetFolder.FOLDER_ID_PK_COLUMN,
            Integer.valueOf(folderid));
        query.andQualifier(exp2);
      }

      ResultList<AvzTCabinetFolder> list = query.getResultList();
      if (list != null && list.size() > 0) {
        AvzTCabinetFolder tmpfolder = null;
        int size = list.size();
        for (int i = 0; i < size; i++) {
          tmpfolder = list.get(i);
          if (fname.equals(tmpfolder.getFolderName())) {
            return true;
          }
        }
      }
    } catch (Exception e) {
      return true;
    }

    return false;
  }

  /**
   * フォルダをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCabinetFolder folder =
        ExtCabinetUtils.getAvzTCabinetFolder(rundata, context);
      if (folder == null) {
        return false;
      }

      // フォルダ名
      folder_name.setValue(folder.getFolderName());
      // メモ
      note.setValue(folder.getNote());

      // フォルダ階層の表示切替
      int folder_id = selected_folderinfo.getFolderId();
      List<Integer> disable_list = new ArrayList<Integer>();
      disable_list.add(Integer.valueOf(folder_id));
      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        ExtFolderInfo info = folder_hierarchy_list.get(i);
        boolean flg = false;
        int len = disable_list.size();
        for (int j = 0; j < len; j++) {
          Integer id = disable_list.get(j);
          if (info.getFolderId() == id.intValue()
            || info.getParentFolderId() == id.intValue()) {
            flg = true;
            break;
          }
        }
        if (flg) {
          info.setVisible(false);
          disable_list.add(Integer.valueOf(info.getFolderId()));
        } else {
          info.setVisible(true);
        }
      }

      ExtCabinetUtils.setFolderVisibleForForm(folder_hierarchy_list, rundata);

      for (int i = 0; i < size; i++) {
        ExtFolderInfo info = folder_hierarchy_list.get(i);
        if (info.getFolderId() == selected_folderinfo.getParentFolderId()) {
          // 現在編集中のフォルダの親フォルダを選択させる．
          selected_folderinfo = info;
          parent_id.setValue(info.getFolderId());
        }
      }

      // motegi start
      aclList = ExtCabinetUtils.loadFolderAclList(folder.getFolderId());

      is_owner = (folder.getCreateUserId().intValue() == (int) login_user_id);
      // motegi end
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダをデータベースとファイルシステムから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      String folderid =
        ALEipUtils.getTemp(
          rundata,
          context,
          ExtCabinetUtils.SESSION_KEY_FOLDER_ID);
      int delete_id = Integer.parseInt(folderid);
      if (delete_id <= ExtCabinetUtils.ROOT_FODLER_ID) {
        // フォルダ『ルートフォルダ』は削除不可にする．
        return false;
      }

      List<Integer> folderids = new ArrayList<Integer>();

      // 削除対象のフォルダIDを格納します。
      folderids.add(delete_id);

      // 削除対象フォルダの子フォルダおよびファイルに対して自分が削除権限を持っているかをチェックします。
      List<ExtFolderInfo> children =
        ExtCabinetUtils.getFolderList(Integer.parseInt(folderid));
      if (children != null) {
        for (ExtFolderInfo e : children) {
          List<String> aclFolderHandle =
            ExtCabinetUtils.getFolderAclHandle(rundata, e.getFolderId(), e
              .getCreateUserId());
          if (!ExtCabinetUtils.isDeletable(aclFolderHandle)) {
            if (!ExtCabinetUtils.isReadable(aclFolderHandle)) {
              msgList.add("削除権限が無いフォルダが含まれています。" + ":" + "参照権限がありません");
            } else {
              msgList.add("削除権限が無いフォルダが含まれています。" + ":" + e.getFolderName());
            }
          } else {
            List<AvzTCabinetFile> tmp_files =
              ExtCabinetUtils.getAvzTCabinetFileList(e.getFolderId());
            boolean deletable = true;
            if (tmp_files != null) {
              for (AvzTCabinetFile f : tmp_files) {
                List<String> aclHandle =
                  ExtCabinetUtils.getFileAclHandle(rundata, f.getFileId(), f
                    .getCreateUserId());
                if (!ExtCabinetUtils.isDeletable(aclHandle)) {
                  deletable = false;
                  if (!ExtCabinetUtils.isReadable(aclHandle)) {
                    msgList.add("削除権限が無いファイルが含まれています。" + ":" + "参照権限がありません");
                  } else {
                    msgList.add("削除権限が無いファイルが含まれています。" + ":" + f.getFileName());
                  }
                }
              }
            }
            if (deletable) {
              folderids.add(e.getFolderId());
            }
          }
        }
      }
      // Start
      // 削除対象フォルダ内のファイルに対して自分が削除権限を持っているかをチェックします。
      List<AvzTCabinetFile> tmp_files =
        ExtCabinetUtils.getAvzTCabinetFileList(delete_id);
      boolean deletable = true;
      if (tmp_files != null) {
        for (AvzTCabinetFile f : tmp_files) {
          List<String> aclHandle =
            ExtCabinetUtils.getFileAclHandle(rundata, f.getFileId(), f
              .getCreateUserId());
          if (!ExtCabinetUtils.isDeletable(aclHandle)) {
            deletable = false;
            if (!ExtCabinetUtils.isReadable(aclHandle)) {
              msgList.add("削除権限が無いファイルが含まれています。" + ":" + "参照権限がありません");
            } else {
              msgList.add("削除権限が無いファイルが含まれています。" + ":" + f.getFileName());
            }
          }
        }
      }
      // End
      if (msgList.size() > 0) {
        return false;
      }

      AvzTCabinetFolder folder = ExtCabinetUtils.getFolderByPK(delete_id);
      int delete_parent_id = folder.getParentId();

      // データベースのフォルダ情報を削除する．
      SelectQuery<AvzTCabinetFolder> delfolderquery =
        Database.query(AvzTCabinetFolder.class);
      Expression delfolderexp =
        ExpressionFactory.inDbExp(
          _AvzTCabinetFolder.FOLDER_ID_PK_COLUMN,
          folderids);
      delfolderquery.setQualifier(delfolderexp);
      ResultList<AvzTCabinetFolder> delFolderList =
        delfolderquery.getResultList();

      int delFolderListsize = delFolderList.size();

      // フォルダ情報を削除
      for (int i = 0; i < delFolderListsize; i++) {
        AvzTCabinetFolder delfolder = delFolderList.get(i);

        // entityIdを取得
        Integer entityId = delfolder.getFolderId();
        // フォルダ名を取得
        String folderName = delfolder.getFolderName();

        List<AvzTCabinetFile> cabinetfiles =
          ExtCabinetUtils.getAvzTCabinetFileList(delfolder
            .getFolderId()
            .intValue());
        List<String> fpaths = new ArrayList<String>();
        if ((cabinetfiles != null) && (cabinetfiles.size() > 0)) {
          int tsize = cabinetfiles.size();
          for (int k = 0; k < tsize; k++) {
            fpaths.add((cabinetfiles.get(k)).getFilePath());
          }
        }

        // フォルダ情報を削除
        Database.delete(delfolder);
        Database.commit();

        if (fpaths.size() > 0) {
          // ローカルファイルに保存されているファイルを削除する．
          File file = null;
          int fsize = fpaths.size();
          for (int k = 0; k < fsize; k++) {
            file =
              new File(ExtCabinetUtils.getSaveDirPath(org_id) + fpaths.get(k));
            if (file.exists()) {
              file.delete();
            }
          }
        }
      }

      // 論理フォルダ階層をリロードする．
      folder_hierarchy_list =
        ExtCabinetUtils.getFolderList(ExtCabinetUtils.ROOT_PARENT_FODLER_ID);
      ExtCabinetUtils.setFolderVisibleForForm(folder_hierarchy_list, rundata);
      selected_folderinfo =
        ExtCabinetUtils.getSelectedFolderInfo(
          folder_hierarchy_list,
          delete_parent_id);

      // //セッションには削除したフォルダ情報が格納されているので、一旦クリアします。
      // ALEipUtils.removeTemp(rundata, context,
      // ExtCabinetUtils.SESSION_KEY_FOLDER_ID);
      // ALEipUtils.removeTemp(rundata, context,
      // ExtCabinetSelectData.class.getName()
      // + ALEipConstants.LIST_FILTER);
      // ALEipUtils.removeTemp(rundata, context,
      // ExtCabinetSelectData.class.getName()
      // + ALEipConstants.LIST_FILTER_TYPE);

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダをデータベースとファイルシステムに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    boolean res = false;
    try {
      int uid = ALEipUtils.getUserId(rundata);

      // 新規オブジェクトモデル
      AvzTCabinetFolder folder = Database.create(AvzTCabinetFolder.class);
      // 親フォルダ ID
      folder.setParentId(Integer.valueOf((int) parent_id.getValue()));
      // フォルダ名
      folder.setFolderName(folder_name.getValue());
      // メモ
      folder.setNote(note.getValue());
      // 作成者
      folder.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      folder.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      folder.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());
      // アクセス権限設定
      ExtCabinetUtils.insertFolderAcl(folder, aclRequestList);
      // フォルダを登録
      Database.commit();

      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        ExtFolderInfo info = folder_hierarchy_list.get(i);
        if (info.getFolderId() == parent_id.getValue()) {
          parent_folderinfo = info;
          break;
        }
      }

      selected_folderinfo.setFolderId(folder.getFolderId().intValue());
      selected_folderinfo.setFolderName(folder.getFolderName());
      selected_folderinfo.setHierarchyIndex(parent_folderinfo
        .getHierarchyIndex() + 1);
      selected_folderinfo.setParentFolderId(parent_folderinfo.getFolderId());

      res = true;
    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return res;
  }

  /**
   * データベースとファイルシステムに格納されているフォルダを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗O
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTCabinetFolder folder =
        ExtCabinetUtils.getAvzTCabinetFolder(rundata, context);
      if (folder == null) {
        return false;
      }

      int uid = ALEipUtils.getUserId(rundata);

      // フォルダ名
      folder.setFolderName(folder_name.getValue());
      // 親フォルダ ID
      folder.setParentId(Integer.valueOf((int) parent_id.getValue()));
      // メモ
      folder.setNote(note.getValue());
      // 更新者
      folder.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());

      // motegi start
      is_owner = (folder.getCreateUserId().intValue() == (int) login_user_id);
      // if( is_owner ) {
      SelectQuery<AvzTFolderAcl> query = Database.query(AvzTFolderAcl.class);
      Expression exp =
        ExpressionFactory.matchExp(_AvzTFolderAcl.FOLDER_ID_PROPERTY, folder
          .getFolderId());
      query.setQualifier(exp);
      ResultList<AvzTFolderAcl> acls = query.getResultList();
      Database.deleteAll(acls);

      ExtCabinetUtils.insertFolderAcl(folder, aclRequestList);
      // }
      // motegi end

      // フォルダを更新
      Database.commit();

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダ名を取得する． <BR>
   * 
   * @return
   */
  public ALStringField getFolderName() {
    return folder_name;
  }

  public ALNumberField getParentId() {
    return parent_id;
  }

  /**
   * メモを取得する． <BR>
   * 
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  public List<ExtFolderInfo> getFolderHierarchyList() {
    return folder_hierarchy_list;
  }

  public ExtFolderInfo getSelectedFolderInfo() {
    return selected_folderinfo;
  }

  /**
   * 
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return null;
  }

  // motegi
  public Map<Integer, ALEipPost> getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }

  public List<ExtCabinetAclResultData> getAclList() {
    return aclList;
  }

  public boolean isOwner() {
    return is_owner;
  }

  // public boolean isMovable() {
  // return is_movable;
  // }
}
