/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2009 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.whereaboutsboard;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.eip.cayenne.om.portlet.EipTSchedule;
import com.aimluck.eip.cayenne.om.portlet.EipTScheduleMap;
import com.aimluck.eip.cayenne.om.security.TurbineGroup;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipGroup;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipPost;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.whereaboutsboard.util.WhereaboutsboardUtils;

/**
 * 行先板の検索データを管理するためのクラスです。 <br />
 */
public class WhereaboutsboardGroupSelectData extends
    ALAbstractSelectData<TurbineUser, Object> {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(WhereaboutsboardGroupSelectData.class
      .getName());

  /** 現在の日付と時刻をAipoの型にあわせています */
  private ALDateTimeField nowDatestamp = null;

  /** 現在の日付をAipoの型にあわせています */
  private ALDateTimeField nowDate = null;

  /** 現在の日付と時刻をカレンダークラスから取得 */
  private final Calendar calendar = Calendar.getInstance();

  /** ログインユーザーのマイグループ */
  private List<ALEipGroup> myGroupList = null;

  /** 現在の時刻のみ用のフォーマット */
  private static final SimpleDateFormat nowtime =
    new SimpleDateFormat("HHmmss");

  /** 現在の日付のみ用のフォーマット */
  private static final SimpleDateFormat nowdate =
    new SimpleDateFormat("yyyyMMdd");

  /** ログインユーザーID */
  private String userId = null;

  /**
   * 初期化処理を行います。 <BR>
   * 
   * @param action
   * @param rundata
   * @param context
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    // グループの初期値を取得する
    try {
      String filter = ALEipUtils.getTemp(rundata, context, LIST_FILTER_STR);
      if (filter == null || filter.equals("")) {
        VelocityPortlet portlet = ALEipUtils.getPortlet(rundata, context);
        String groupName =
          portlet.getPortletConfig().getInitParameter("p3b-group");
        if (groupName != null) {
          ALEipUtils.setTemp(rundata, context, LIST_FILTER_STR, groupName);
          ALEipUtils.setTemp(rundata, context, LIST_FILTER_TYPE_STR, "group");
        }
      }
    } catch (Exception ex) {
      logger.debug("Exception", ex);
    }

    userId = Integer.toString(ALEipUtils.getUserId(rundata));
  }

  /**
   * @see com.aimluck.eip.common.ALAbstractSelectData#selectList(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  // 検索条件を設定した SelectQuery を元に検索するメソッド
  @Override
  protected ResultList<TurbineUser> selectList(RunData rundata, Context context) {
    try {
      // ログインユーザーのマイグループ取得（表示用）
      List<ALEipGroup> myGroups = ALEipUtils.getMyGroups(rundata);
      myGroupList = new ArrayList<ALEipGroup>();
      int length = myGroups.size();
      for (int i = 0; i < length; i++) {
        myGroupList.add(myGroups.get(i));
      }

      String filter = ALEipUtils.getTemp(rundata, context, LIST_FILTER_STR);
      String filter_type =
        ALEipUtils.getTemp(rundata, context, LIST_FILTER_TYPE_STR);
      if (filter == null || filter_type == null || filter.equals("")) {
        return new ResultList<TurbineUser>();
      } else {
        SelectQuery<TurbineUser> query = getSelectQuery(rundata, context);
        buildSelectQueryForListView(query);
        query.orderAscending(TurbineUser.LAST_NAME_KANA_PROPERTY);
        return query.getResultList();
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  @Override
  protected SelectQuery<TurbineUser> buildSelectQueryForFilter(
      SelectQuery<TurbineUser> query, RunData rundata, Context context) {
    String filter = ALEipUtils.getTemp(rundata, context, LIST_FILTER_STR);
    String filter_type =
      ALEipUtils.getTemp(rundata, context, LIST_FILTER_TYPE_STR);
    String crt_key = null;
    Attributes map = getColumnMap();
    if (filter == null || filter_type == null || filter.equals("")) {
      return query;
    }
    crt_key = map.getValue(filter_type);
    if (crt_key == null) {
      return query;
    }

    Expression exp = ExpressionFactory.matchExp(crt_key, filter);
    query.andQualifier(exp);
    current_filter = filter;
    current_filter_type = filter_type;
    return query;
  }

  /**
   * @see com.aimluck.eip.common.ALAbstractSelectData#getResultData(java.lang.Object)
   */
  // 結果を保存
  @Override
  protected Object getResultData(TurbineUser obj) {
    try {
      TurbineUser user = obj;

      SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd");

      WhereaboutsboardGroupContainer rd = new WhereaboutsboardGroupContainer();
      rd.initField();
      rd.setName(ALEipUtils.getUserFullName(user.getUserId().intValue()));

      StringBuffer buf_place = new StringBuffer();
      List<EipTScheduleMap> maps =
        getEipTScheduleMapsOnNow(user.getUserId().intValue());

      List<WhereaboutsboardGroupResultData> places =
        new ArrayList<WhereaboutsboardGroupResultData>();
      if (maps != null) {

        for (EipTScheduleMap map : maps) {
          EipTSchedule schedule = map.getEipTSchedule();
          String tmp_place = WhereaboutsboardUtils.getPlace(schedule);

          SelectQuery<EipTScheduleMap> mapquery =
            Database.query(EipTScheduleMap.class);
          Expression mapexp1 =
            ExpressionFactory.matchExp(
              EipTScheduleMap.SCHEDULE_ID_PROPERTY,
              map.getScheduleId());
          mapquery.setQualifier(mapexp1);

          Expression mapexp2 =
            ExpressionFactory
              .matchExp(EipTScheduleMap.USER_ID_PROPERTY, userId);
          mapquery.andQualifier(mapexp2);

          Expression mapexp3 =
            ExpressionFactory.noMatchExp(EipTScheduleMap.STATUS_PROPERTY, "R");
          mapquery.andQualifier(mapexp3);

          // 自分がメンバーが判定します。
          ResultList<EipTScheduleMap> schedulemaps = mapquery.getResultList();
          boolean is_member =
            (schedulemaps != null && schedulemaps.size() > 0) ? true : false;

          // 非公開／完全に隠す に対応
          boolean publicable = schedule.getPublicFlag().equals("O");

          if (!publicable && !is_member) {
            tmp_place = "非公開";
          }
          boolean hidden = schedule.getPublicFlag().equals("P");
          if (hidden && !is_member) {
            // 「完全に隠す」でメンバーでない場合⇒不可視
            continue;
          }

          WhereaboutsboardGroupResultData place =
            new WhereaboutsboardGroupResultData();
          place.initField();
          place.setUserId(user.getUserId().toString());
          place.setScheduleId(schedule.getScheduleId().toString());
          place.setPlace(tmp_place);
          place.setPublicable(publicable);
          place.setMember(is_member);
          place.setDate(sdf1.format(schedule.getStartDate()) + "-00-00");
          places.add(place);

        }
      }
      rd.setPlaces(places);
      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }

  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<TurbineUser> getSelectQuery(RunData rundata,
      Context context) {
    SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);

    Expression exp11 =
      ExpressionFactory.noMatchDbExp(TurbineUser.USER_ID_PK_COLUMN, Integer
        .valueOf(1));
    Expression exp12 =
      ExpressionFactory.noMatchDbExp(TurbineUser.USER_ID_PK_COLUMN, Integer
        .valueOf(2));
    Expression exp13 =
      ExpressionFactory.noMatchDbExp(TurbineUser.USER_ID_PK_COLUMN, Integer
        .valueOf(3));
    query.setQualifier(exp11.andExp(exp12).andExp(exp13));

    Expression exp2 =
      ExpressionFactory.matchExp(TurbineUser.DISABLED_PROPERTY, "F");
    query.andQualifier(exp2);
    Expression exp3 =
      ExpressionFactory.noMatchDbExp(TurbineUser.USER_ID_PK_COLUMN, Integer
        .valueOf(userId));
    query.andQualifier(exp3);

    return buildSelectQueryForFilter(query, rundata, context);
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private List<EipTScheduleMap> getEipTScheduleMapsOnNow(int userId) {

    // 参加メンバーが削除したレコードは除外
    Expression exp00 =
      ExpressionFactory.noMatchExp(EipTScheduleMap.TYPE_PROPERTY, "R");

    // 施設情報レコードは除外
    Expression exp01 =
      ExpressionFactory.noMatchExp(EipTScheduleMap.TYPE_PROPERTY, "F");

    Expression exp1 =
      ExpressionFactory.matchExp(EipTScheduleMap.USER_ID_PROPERTY, userId);

    nowDatestamp = new ALDateTimeField("yyyy-MM-dd HH:mm:00");
    nowDate = new ALDateTimeField("yyyy-MM-dd");

    // 現在日時時刻取得
    nowDatestamp.setValue(calendar.getTime());
    nowDate.setValue(calendar.getTime());

    // 日付だけを取得しnnに代入
    String nn = nowDatestamp.getDay();
    // 曜日だけを取得しdayに代入
    String dayOfWeek = nowDatestamp.getDayOfWeek();
    // 曜日フラグ判別用変数
    String dayFlag = null;
    // 曜日ごとにフラグ設定
    if (dayOfWeek.equals("（日）")) {
      dayFlag = "1______";
    } else if (dayOfWeek.equals("（月）")) {
      dayFlag = "_1_____";
    } else if (dayOfWeek.equals("（火）")) {
      dayFlag = "__1____";
    } else if (dayOfWeek.equals("（水）")) {
      dayFlag = "___1___";
    } else if (dayOfWeek.equals("（木）")) {
      dayFlag = "____1__";
    } else if (dayOfWeek.equals("（金）")) {
      dayFlag = "_____1_";
    } else if (dayOfWeek.equals("（土）")) {
      dayFlag = "______1";
    }
    // 開始日付時刻が現在より前か
    Expression exp2 =
      ExpressionFactory.lessOrEqualExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.START_DATE_PROPERTY, nowDatestamp);
    // 終了日付時刻が現在より後か
    Expression exp3 =
      ExpressionFactory.greaterOrEqualExp(
        EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
          + "."
          + EipTSchedule.END_DATE_PROPERTY,
        nowDatestamp);
    // RepeatPatternが"N"の場合
    Expression exp6 =
      ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "N");
    // RepeatPatternが"DL"の場合
    Expression exp7 =
      ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "DL");
    // RepeatPatternが"DN"の場合
    Expression exp8 =
      ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "DN");
    // RepeatPatternが"MnnL"の場合
    Expression exp9 =
      ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "M" + nn + "L");
    // RepeatPatternが"MnnN"の場合
    Expression exp10 =
      ExpressionFactory.matchExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "M" + nn + "N");
    // RepeatPatternが"WL"の場合
    Expression exp11 =
      ExpressionFactory.likeExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "W" + dayFlag + "L");
    // RepeatPatternが"WN"の場合
    Expression exp12 =
      ExpressionFactory.likeExp(EipTScheduleMap.EIP_TSCHEDULE_PROPERTY
        + "."
        + EipTSchedule.REPEAT_PATTERN_PROPERTY, "W" + dayFlag + "N");

    SelectQuery<EipTScheduleMap> query =
      Database.query(EipTScheduleMap.class, exp00.andExp(exp01.andExp(exp1
        .andExp((exp2).andExp(exp3).andExp(exp6))
        .orExp(((exp1).andExp((exp8).orExp(exp10).orExp(exp12))))
        .orExp(((exp1).andExp((exp7).orExp(exp9).orExp(exp11)))))));

    ResultList<EipTScheduleMap> ulist = query.getResultList();
    ulist = judgeRepeatPatternWithTime(ulist, userId);
    if (ulist.size() == 0) {
      return null;
    }
    return ulist;

  }

  /*
   * javaで時刻による絞込みを行うメソッド
   */
  private ResultList<EipTScheduleMap> judgeRepeatPatternWithTime(
      ResultList<EipTScheduleMap> list, int userId) {

    ResultList<EipTScheduleMap> judgeRepeatPatternWithTimeList =
      new ResultList<EipTScheduleMap>();

    nowDatestamp = new ALDateTimeField("yyyy-MM-dd HH:mm:00");

    // 現在時刻取得
    nowDatestamp.setValue(calendar.getTime());
    int length = list.size();
    for (int i = 0; i < length; i++) {

      EipTScheduleMap emap = list.get(i);

      // repeatPattern取得
      String repeatPattern = emap.getEipTSchedule().getRepeatPattern();
      // スケジュールテーブルStartDateの開始時刻のみ
      int timeOfStartDate =
        Integer.parseInt(nowtime.format(emap.getEipTSchedule().getStartDate()));
      // スケジュールテーブルEndDateの終了時刻のみ
      int timeOfEndDate =
        Integer.parseInt(nowtime.format(emap.getEipTSchedule().getEndDate()));
      // スケジュールテーブルStartDateの開始日付のみ
      int dateOfStartDate =
        Integer.parseInt(nowdate.format(emap.getEipTSchedule().getStartDate()));
      // スケジュールテーブルEndDateの終了日付のみ
      int dateOfEndDate =
        Integer.parseInt(nowdate.format(emap.getEipTSchedule().getEndDate()));
      // 現在時刻
      int nowTime = Integer.parseInt(nowtime.format(nowDatestamp.getValue()));
      // 現在日付
      int nowDate = Integer.parseInt(nowdate.format(nowDatestamp.getValue()));

      // DBから取得したオブジェクトをJavaで条件分け
      if (repeatPattern.equals("N")) {// RepeatPatternが"N"の場合
        judgeRepeatPatternWithTimeList.add(emap);
      } else if (repeatPattern.endsWith("N")) { // RepeatPattern末尾が"N"の場合

        if (timeOfStartDate <= nowTime) { // 開始時刻より後の場合
          if (timeOfEndDate >= nowTime) { // 終了時刻より前の場合

            // 本日付のダミースケジュールレコードの有無判定を入れる
            // map→スケジュール取得→親IDと本日日付を条件でダミーレコードが有れば対象外
            // ※繰り返しスケジュールで、１日だけ削除した場合は、目印としてダミースケジュール
            // が生成されます。
            if (!isExcludeRecord(emap, userId)) {
              judgeRepeatPatternWithTimeList.add(emap);
            }
          }
        }
      } else {
        if (dateOfStartDate <= nowDate) {// 開始日付より後の場合
          if (dateOfEndDate >= nowDate) {// 終了日付より前の場合
            if (timeOfStartDate <= nowTime) {// 開始時刻より後の場合
              if (timeOfEndDate >= nowTime) {// 終了時刻より前の場合
                judgeRepeatPatternWithTimeList.add(emap);
              }
            }
          }
        }
      }
    }

    return judgeRepeatPatternWithTimeList;
  }

  /**
   * 繰り返しデータの中で削除されたものかを判定
   * 
   * @param map
   * @param userId
   * @return
   */
  public boolean isExcludeRecord(EipTScheduleMap map, int userId) {
    Integer parentId = map.getScheduleId();
    SelectQuery<EipTSchedule> query = Database.query(EipTSchedule.class);

    // スケジュールID
    Expression exp1 =
      ExpressionFactory.matchExp(EipTSchedule.PARENT_ID_PROPERTY, parentId);
    query.setQualifier(exp1);

    // ユーザーID
    Expression exp2 =
      ExpressionFactory.matchExp(EipTSchedule.OWNER_ID_PROPERTY, userId);
    query.andQualifier(exp2);

    // 当日のダミーか
    Expression exp3 =
      ExpressionFactory.matchExp(EipTSchedule.START_DATE_PROPERTY, nowDate);
    query.andQualifier(exp3);

    ResultList<EipTSchedule> schedules = query.getResultList();

    // 指定したSchedule IDのレコードが見つからない場合
    if (schedules == null || schedules.size() == 0) {
      return false;
    }
    return true;
  }

  /**
   * 部署名を取得
   * 
   * @return
   */
  public Map<Integer, ALEipPost> getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }

  /**
   * マイグループリストを取得
   * 
   * @return
   */
  public List getMyGroupList() {
    return myGroupList;
  }

  /**
   * ログインユーザーＩＤを取得
   * 
   * @return
   */
  public String getUserId() {
    return userId;
  }

  /**
   * @see com.aimluck.eip.common.ALAbstractSelectData#selectDetail(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected Object selectDetail(RunData rundata, Context context) {
    return null;
  }

  /**
   * @see com.aimluck.eip.common.ALAbstractSelectData#getResultDataDetail(java.lang.Object)
   */
  @Override
  protected Object getResultDataDetail(Object obj) {
    return null;
  }

  /**
   * @see com.aimluck.eip.common.ALAbstractSelectData#getColumnMap()
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    map.putValue("group", TurbineUser.TURBINE_USER_GROUP_ROLE_PROPERTY
      + "."
      + TurbineUserGroupRole.TURBINE_GROUP_PROPERTY
      + "."
      + TurbineGroup.GROUP_NAME_PROPERTY);
    return map;
  }

}