/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2010 Aimluck,Inc.
 * http://aipostyle.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.addressbook.util;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.DataRow;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.addressbook.AddressBookGroup;
import com.aimluck.eip.addressbookuser.beans.AddressBookUserGroupLiteBean;
import com.aimluck.eip.cayenne.om.account.EipMPost;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressGroup;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressbook;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressbookCompany;
import com.aimluck.eip.cayenne.om.portlet.EipTAddressbookGroupMap;
import com.aimluck.eip.cayenne.om.security.TurbineGroup;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * Addressbookのユーティリティクラスです。
 * 
 */
public class AddressBookUtils {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AddressBookUtils.class.getName());

  /** アドレスブックファイルを一時保管するディレクトリの指定 */
  public static final String FOLDER_TMP_FOR_ADDRESSBOOK_FILES =
    JetspeedResources.getString("aipo.tmp.addressbook.directory", "");

  public static final String EMPTY_COMPANY_NAME = "未分類";

  public static EipMAddressGroup getEipMAddressGroup(RunData rundata,
      Context context) {
    String groupid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (groupid == null || Integer.valueOf(groupid) == null) {
        // グループIDが空の場合
        logger.debug("[AddressBook] Group ID...");
        return null;
      }
      SelectQuery<EipMAddressGroup> query =
        Database.query(EipMAddressGroup.class);
      Expression exp1 =
        ExpressionFactory.matchDbExp(
          EipMAddressGroup.GROUP_ID_PK_COLUMN,
          Integer.valueOf(groupid));
      query.setQualifier(exp1);
      Expression exp2 =
        ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
          .valueOf(ALEipUtils.getUserId(rundata)));
      query.andQualifier(exp2);

      List<EipMAddressGroup> groups = query.fetchList();
      if (groups == null || groups.size() == 0) {
        // 指定したカテゴリIDのレコードが見つからない場合
        logger.debug("[AddressBook] Not found Group ID...");
        return null;
      }
      return groups.get(0);
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 指定した取引先情報の取得
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static EipMAddressbookCompany getEipMAddressbookCompany(
      RunData rundata, Context context) {
    // セッションから指定された 取引先ID を取得
    String companyid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (companyid == null || Integer.valueOf(companyid) == null) {
        logger.debug("[AddressBook] Company ID...");
        return null;
      }

      SelectQuery<EipMAddressbookCompany> query =
        Database.query(EipMAddressbookCompany.class);
      Expression exp =
        ExpressionFactory.matchDbExp(
          EipMAddressbookCompany.COMPANY_ID_PK_COLUMN,
          Integer.valueOf(companyid));
      query.setQualifier(exp);

      List<EipMAddressbookCompany> companys = query.fetchList();
      if (companys == null || companys.size() == 0) {
        logger.debug("[AddressBook] Not found Company ID...");
        return null;
      }
      return companys.get(0);
    } catch (Exception ex) {
      logger.error("Exception ", ex);
      return null;
    }
  }

  /**
   * @param id
   * @return
   */
  public static List<String> getMyGroupNames(RunData rundata, int addressid,
      int userid) {
    try {
      SelectQuery<EipTAddressbookGroupMap> query =
        Database.query(EipTAddressbookGroupMap.class);
      Expression exp1 =
        ExpressionFactory.matchExp(
          EipTAddressbookGroupMap.ADDRESS_ID_PROPERTY,
          Integer.valueOf(addressid));
      query.setQualifier(exp1);
      Expression exp2 =
        ExpressionFactory.matchExp(
          EipTAddressbookGroupMap.EIP_TADDRESS_GROUP_PROPERTY
            + "."
            + EipMAddressGroup.OWNER_ID_PROPERTY,
          Integer.valueOf(userid));
      query.andQualifier(exp2);

      List<EipTAddressbookGroupMap> groups = query.fetchList();

      List<String> aList = new ArrayList<String>();
      int size = groups.size();
      for (int i = 0; i < size; i++) {
        aList.add(groups.get(i).getEipTAddressGroup().getGroupName());
      }
      return aList;
    } catch (Exception ex) {
      logger.error("Exception ", ex);
    }
    return null;
  }

  public static String getMyGroupNamesAsString(RunData rundata, int addressid,
      int userid) {
    List<String> aList = getMyGroupNames(rundata, addressid, userid);
    int group_size = aList.size();
    StringBuffer groupNames = new StringBuffer();
    for (int i = 0; i < group_size; i++) {
      groupNames.append(aList.get(i)).append("、");
    }
    if (groupNames.length() == 0) {
      return "";
    } else {
      return groupNames.substring(0, groupNames.length() - 1);
    }
  }

  /**
   * セッションで指定されたアドレスIDを持つアドレス情報を取得する。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static EipMAddressbook getEipMAddressbook(RunData rundata,
      Context context) {
    // セッションから指定された アドレスID を取得
    String addressid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (addressid == null || Integer.valueOf(addressid) == null) {
        logger.debug("[AddressBook] Address ID...");
        return null;
      }

      SelectQuery<EipMAddressbook> query =
        Database.query(EipMAddressbook.class);
      Expression exp11 =
        ExpressionFactory.matchDbExp(
          EipMAddressbook.ADDRESS_ID_PK_COLUMN,
          Integer.valueOf(addressid));
      query.setQualifier(exp11);

      Expression exp21 =
        ExpressionFactory.matchExp(EipMAddressbook.PUBLIC_FLAG_PROPERTY, "T");
      Expression exp22 =
        ExpressionFactory.matchExp(
          EipMAddressbook.OWNER_ID_PROPERTY,
          ALEipUtils.getUserId(rundata));
      Expression exp23 =
        ExpressionFactory.matchExp(EipMAddressbook.PUBLIC_FLAG_PROPERTY, "F");
      query.andQualifier(exp21.orExp(exp22.andExp(exp23)));

      List<EipMAddressbook> addresses = query.fetchList();

      if (addresses == null || addresses.size() == 0) {
        logger.debug("[AddressBook] Not found Address ID...");
        return null;
      }
      return addresses.get(0);
    } catch (Exception ex) {
      logger.error("Exception ", ex);
      return null;
    }
  }

  /**
   * 未分類の会社情報を取得する。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static EipMAddressbookCompany getDummyEipMAddressbookCompany(
      RunData rundata, Context context) {
    try {
      SelectQuery<EipMAddressbookCompany> query =
        Database.query(EipMAddressbookCompany.class);
      Expression exp11 =
        ExpressionFactory.matchExp(
          EipMAddressbookCompany.CREATE_USER_ID_PROPERTY,
          Integer.valueOf(1));
      Expression exp12 =
        ExpressionFactory.matchExp(
          EipMAddressbookCompany.COMPANY_NAME_PROPERTY,
          AddressBookUtils.EMPTY_COMPANY_NAME);
      query.setQualifier(exp11.andExp(exp12));

      List<EipMAddressbookCompany> list = query.fetchList();

      if (list == null || list.size() == 0) {
        logger.debug("[AddressBook] Not found AddressbookCompany ID...");
        return null;
      }
      return list.get(0);
    } catch (Exception ex) {
      logger.error("Exception ", ex);
      return null;
    }
  }

  /**
   * 自分がオーナのグループを取得しID/Objectのマップを返却する。
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static Map<Integer, AddressBookGroup> getGroupMap(RunData rundata,
      Context context) {
    try {
      Map<Integer, AddressBookGroup> groupMap =
        new LinkedHashMap<Integer, AddressBookGroup>();

      SelectQuery<EipMAddressGroup> query =
        Database.query(EipMAddressGroup.class);
      Expression exp =
        ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
          .valueOf(ALEipUtils.getUserId(rundata)));
      query.setQualifier(exp);

      List<EipMAddressGroup> list = query.fetchList();
      int size = list.size();
      for (int i = 0; i < size; i++) {
        EipMAddressGroup record = list.get(i);
        AddressBookGroup group = new AddressBookGroup();
        group.initField();
        group.setGroupId(record.getGroupId().intValue());
        group.setGroupName(record.getGroupName());
        groupMap.put(record.getGroupId(), group);
      }
      return groupMap;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  public static List<EipMAddressGroup> getGroups(int uid) {
    logger.debug("AddressbookUtils getGroups in");
    try {
      SelectQuery<EipMAddressGroup> query =
        Database.query(EipMAddressGroup.class);
      Expression exp =
        ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
          .valueOf(uid));
      query.setQualifier(exp);

      List<EipMAddressGroup> groups = query.fetchList();

      if (groups == null || groups.size() == 0) {
        logger.debug("[AddressBook] Not found group ID...");
        return null;
      }
      logger.debug("AddressbookUtils getGroups out");
      return groups;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  public static List<String> getGroupMember(String gid) {
    List<String> resultList = new ArrayList<String>();

    StringBuffer statement = new StringBuffer();
    statement.append("SELECT ");
    statement.append(" B.last_name, B.first_name, ");
    statement.append(" C.company_name ");
    statement.append("FROM EIP_T_ADDRESSBOOK_GROUP_MAP as A ");
    statement.append("LEFT JOIN EIP_M_ADDRESSBOOK as B ");
    statement.append(" on A.ADDRESS_ID = B.ADDRESS_ID ");
    statement.append("LEFT JOIN EIP_M_ADDRESSBOOK_COMPANY AS C ");
    statement.append(" on B.COMPANY_ID = C.COMPANY_ID ");
    statement.append("WHERE A.GROUP_ID = #bind($gid) ");
    statement
      .append("ORDER BY C.company_name_kana, C.company_name, B.last_name_kana");
    String query = statement.toString();

    try {
      List<DataRow> list =
        Database
          .sql(EipMAddressbook.class, query)
          .param("gid", gid)
          .fetchListAsDataRow();

      int recordNum = list.size();
      DataRow dataRow;
      for (int i = 0; i < recordNum; i++) {
        dataRow = list.get(i);
        StringBuffer strBuf = new StringBuffer();
        strBuf
          .append(
            Database.getFromDataRow(dataRow, EipMAddressbook.LAST_NAME_COLUMN))
          .append(
            Database.getFromDataRow(dataRow, EipMAddressbook.FIRST_NAME_COLUMN));
        String companyName =
          (String) Database.getFromDataRow(
            dataRow,
            EipMAddressbookCompany.COMPANY_NAME_COLUMN);
        if (companyName != null && companyName.trim().length() > 0) {
          strBuf.append(" (").append(companyName).append(")");
        }
        resultList.add(strBuf.toString());
      }
    } catch (Exception ex) {
      logger.error("[AddressbookUtils]", ex);
    }
    return resultList;
  }

  /**
   * ユーザーの所属する部署の一覧を取得します。
   * 
   * @param uid
   *          ユーザーID
   * @return 所属する部署リスト
   */
  public static List<AddressBookUserGroupLiteBean> getPostBeanList(int uid) {
    SelectQuery<TurbineUserGroupRole> query =
      Database.query(TurbineUserGroupRole.class);
    Expression exp1 =
      ExpressionFactory.matchExp(
        TurbineUserGroupRole.TURBINE_USER_PROPERTY,
        Integer.valueOf(uid));
    Expression exp2 =
      ExpressionFactory.greaterExp(
        TurbineUserGroupRole.TURBINE_GROUP_PROPERTY,
        Integer.valueOf(3));
    Expression exp3 =
      ExpressionFactory.matchExp(TurbineUserGroupRole.TURBINE_GROUP_PROPERTY
        + "."
        + TurbineGroup.OWNER_ID_PROPERTY, Integer.valueOf(1));
    query.setQualifier(exp1);
    query.andQualifier(exp2);
    query.andQualifier(exp3);

    List<TurbineUserGroupRole> list = query.fetchList();

    if (list == null || list.size() < 0) {
      return null;
    }

    List<AddressBookUserGroupLiteBean> resultList =
      new ArrayList<AddressBookUserGroupLiteBean>();

    TurbineUserGroupRole ugr = null;
    TurbineGroup group = null;
    AddressBookUserGroupLiteBean bean = null;
    int size = list.size();
    for (int i = 0; i < size; i++) {
      ugr = list.get(i);
      group = ugr.getTurbineGroup();
      EipMPost post = group.getEipMPost();
      bean = new AddressBookUserGroupLiteBean();
      bean.initField();
      bean.setGroupId(post.getPostId());
      bean.setName(post.getPostName());
      resultList.add(bean);
    }

    return resultList;
  }

  /**
   * 一時ファイル保存先のユーザのルートフォルダ
   * 
   * @param org_id
   * @param userId
   * @return
   */
  public static File getRootFolder(String org_id, int userId) {
    String rootPath =
      AddressBookUtils.FOLDER_TMP_FOR_ADDRESSBOOK_FILES
        + File.separator
        + org_id
        + File.separator
        + userId
        + File.separator;
    File folder = new File(rootPath);
    if (!folder.exists()) {
      if (!folder.mkdirs()) {
        return null;
      }
    }
    return folder;
  }
}
