/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.coordinationforgooglecal;

import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.filter.Filter;
import net.fortuna.ical4j.filter.PeriodRule;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.DateTime;
import net.fortuna.ical4j.model.Dur;
import net.fortuna.ical4j.model.Period;
import net.fortuna.ical4j.model.component.VEvent;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.ParameterParser;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.upload.FileItem;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateField;
import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.coordinationforgooglecal.util.CoordinationForGoogleCalUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.util.ALEipUtils;

/**
 * iCalendarファイルを読み込み、一時ファイルに保存するクラスです。 <BR>
 * 
 */
public class CoordinationForGoogleCalFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(CoordinationForGoogleCalFormData.class
      .getName());

  /** 現在選択されているタブ */
  private String currentTab;

  /** 受信した添付ファイル */
  private FileItem attachmentItem = null;

  /** 添付ファイル名 */
  private ALStringField attachmentName = null;

  public static final String KEY_ATTACHMENT_ID = "attachment";

  /** 添付ファイル名の最大文字数 */
  private final int FIELD_ATTACHMENT_MAX_LEN = 128;

  private List results = new ArrayList();

  private ALDateField from_date;

  private ALDateField to_date;

  public static final String CLASSFICATION_PRIVATE = "非公開";

  public static final String CLASSFICATION_PUBLIC = "公開";

  /** インポート期間全予定のエラーチェック */
  private List<String> errorStrings = new ArrayList<String>();

  /** 一時フォルダ名(番号のみ) */
  private String temp_folder;

  /** 一時ファイルのフルパス */
  protected String temp_file_path;

  private String time_select_flg;

  private String period_flg;

  /** エラーメッセージ */
  private List<String> error_msg = new ArrayList<String>();

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    String tabParam = rundata.getParameters().getString("tab");
    currentTab = ALEipUtils.getTemp(rundata, context, "tab");
    if (tabParam == null && currentTab == null) {
      ALEipUtils.setTemp(rundata, context, "tab", "import");
      currentTab = "import";
    } else if (tabParam != null) {
      ALEipUtils.setTemp(rundata, context, "tab", tabParam);
      currentTab = tabParam;
    }

    period_flg = rundata.getParameters().getString("period");

    /** 期間条件を取得 */
    time_select_flg = rundata.getParameters().getString("time_change");

    from_date = new ALDateField();
    to_date = new ALDateField();

    if ("month_select".equals(time_select_flg)) {

      Calendar from = Calendar.getInstance();

      // インポート対象範囲の開始日に選択値をセット
      int from_year = rundata.getParameters().getInt("view_date_year_from");
      int from_month = rundata.getParameters().getInt("view_date_month_from");

      from.set(Calendar.YEAR, from_year);
      from.set(Calendar.MONTH, from_month - 1);

      int from_day = from.getActualMinimum(Calendar.DATE);
      int from_hour = from.getActualMinimum(Calendar.HOUR_OF_DAY);
      int from_minute = from.getActualMinimum(Calendar.MINUTE);
      int from_second = from.getActualMinimum(Calendar.SECOND);

      from.set(Calendar.DAY_OF_MONTH, from_day);
      from.set(Calendar.HOUR_OF_DAY, from_hour);
      from.set(Calendar.MINUTE, from_minute);
      from.set(Calendar.SECOND, from_second);

      from_date.setValue(from.getTime());

      Calendar to = Calendar.getInstance();

      // インポート対象範囲の終了日に選択値をセット
      int to_year = rundata.getParameters().getInt("view_date_year_to");
      int to_month = rundata.getParameters().getInt("view_date_month_to");

      to.set(Calendar.YEAR, to_year);
      to.set(Calendar.MONTH, to_month - 1);

      int to_day = to.getActualMaximum(Calendar.DATE);
      int to_hour = to.getActualMaximum(Calendar.HOUR_OF_DAY);
      int to_minute = to.getActualMaximum(Calendar.MINUTE);
      int to_second = to.getActualMaximum(Calendar.SECOND);

      to.set(Calendar.DAY_OF_MONTH, to_day);
      to.set(Calendar.HOUR_OF_DAY, to_hour);
      to.set(Calendar.MINUTE, to_minute);
      to.set(Calendar.SECOND, to_second);

      to_date.setValue(to.getTime());

    } else {
      // 日単位指定
      String from_date_str = rundata.getParameters().getString("from_date");
      String to_date_str = rundata.getParameters().getString("to_date");

      try {
        from_date.setValue(DateFormat.getDateInstance().parse(from_date_str));
        to_date.setValue(DateFormat.getDateInstance().parse(to_date_str));
      } catch (ParseException e) {
        e.printStackTrace();
      }

    }

    error_msg = new ArrayList<String>();

    initTempFileName();

    /**
     * 下記のコメントアウトのようにアクセスコントロールを追加する場合は
     * com.aimluck.eip.services.accessctl.ALAccessControlConstantsに
     * アクセスコントロール用の定数を追加してください。 (例) public static final String
     * POERTLET_FEATURE_COORDINATIONFORGOOGLECAL_COORDINATIONFORGOOGLECAL_OTHER =
     * "coordinationforgooglecal_coordinationforgooglecal_other";
     */

    // aclPortletFeature =
    // ALAccessControlConstants.POERTLET_FEATURE_COORDINATIONFORGOOGLECAL_COORDINATIONFORGOOGLECAL_SELF;
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    attachmentName = new ALStringField();
    attachmentName.setFieldName("ICSファイル名");
    attachmentName.setTrim(true);
  }

  /**
   * データに値をセットする． <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   *            エラーメッセージのリスト
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    try {

      if (res) {

        if (to_date.getValue().getDate().before(from_date.getValue().getDate())) {

          error_msg
            .add("『 <span class='em'>インポート終了日</span> 』は『 <span class='em'>インポート開始日</span> 』以降の日付で指定してください。");

        }

        // Itemの取得
        ParameterParser parser = rundata.getParameters();
        attachmentItem = parser.getFileItem(KEY_ATTACHMENT_ID);

        if (attachmentItem != null) {

          if (attachmentItem.getSize() > 0) {

            File import_file = new File(attachmentItem.getName());
            attachmentName.setValue(import_file.getName());

            // 拡張子チェック .ics以外はエラーメッセージ
            String filename = attachmentName.getValue();
            String[] icsCheck = filename.split("\\.");
            if (!"ics".equals(icsCheck[icsCheck.length - 1])) {
              error_msg.add("『 <span class='em'>"
                + attachmentItem.getName()
                + "</span> 』の内容に不備があります。ファイル内容や拡張子をご確認ください。");

              return false;

            }

            return true;

          } else {
            error_msg.add("サイズが 0KB のファイルを追加することはできません。");
            return false;
          }

        } else {
          error_msg.add("ファイルが見つかりません。");
          return false;
        }

      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      res = false;
    }

    return res;

  }

  /**
   * の各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */

  @Override
  protected void setValidator() {
    attachmentName.setNotNull(true);
    attachmentName.limitMaxLength(FIELD_ATTACHMENT_MAX_LEN);
    attachmentName.setCharacterType(ALStringField.TYPE_ALL);
  }

  /**
   * のフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {

    attachmentName.validate(msgList);

    return (msgList.size() == 0);

  }

  /**
   * をデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {

    return true;

  }

  /**
   * をデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {

    return true;
  }

  /**
   * をデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    return true;
  }

  /**
   * データベースに格納されているを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      String filepath = getTempFilePath();
      File file = new File(filepath);
      if (file.exists()) {
        file.delete();
      }
      file.createNewFile();
      attachmentItem.write(file.getAbsolutePath());

      results = importICSReader(rundata, context);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 一時フォルダ生成 <BR>
   */
  private void initTempFileName() {
    File tmpfolderRootFolder =
      new File(CoordinationForGoogleCalUtils.ICS_TEMP_FOLDER
        + File.separator
        + CoordinationForGoogleCalUtils.ICS_SCHEDULE_TEMP_FOLDER);
    if (!tmpfolderRootFolder.exists()) {
      tmpfolderRootFolder.mkdirs();
    }

    temp_folder =
      CoordinationForGoogleCalUtils
        .getNewAttachmentFolderName(tmpfolderRootFolder);

    String newfolderpath = tmpfolderRootFolder + File.separator + temp_folder;
    File newfolder = new File(newfolderpath);
    if (!newfolder.exists()) {
      newfolder.mkdirs();
    }

    temp_file_path =
      newfolderpath
        + File.separator
        + CoordinationForGoogleCalUtils.FOLDER_TMP_FOR_USERINFO_ICS_FILENAME;
  }

  /**
   * 選択したicsファイルの内容を読み込む<BR>
   */
  protected List importICSReader(RunData rundata, Context context) {
    try {

      CalendarBuilder builder = new CalendarBuilder();
      // CalendarBuilderのbuild(FileInputStream)でUTF-8に文字コード設定
      net.fortuna.ical4j.model.Calendar calendar =
        builder.build(attachmentItem.getInputStream());

      List con = new ArrayList();

      if (period_flg == null || "".equals(period_flg)) {
        Calendar fd = Calendar.getInstance();
        fd.setTime(from_date.getValue().getDate());
        DateTime from_d = new DateTime();
        from_d.setTime(fd.getTime().getTime());

        Calendar td = Calendar.getInstance();
        td.setTime(to_date.getValue().getDate());
        DateTime to_d = new DateTime();
        to_d.setTime(td.getTime().getTime());

        // Period(final DateTime start, final DateTime end)でインポートする期間を設定する
        Period period = new Period(new DateTime(from_d), new Dur(from_d, to_d));
        Filter filter = new Filter(new PeriodRule(period));

        // VEVENT毎の要素を取得
        con = (List) filter.filter(calendar.getComponents(Component.VEVENT));
      } else {
        con = calendar.getComponents(Component.VEVENT);
      }

      if (con.size() == 0) {

        error_msg.add("指定期間に該当するスケジュールが『 <span class='em'>"
          + attachmentItem.getName()
          + "</span> 』内にありません。");

        return results;
      }
      // icsファイルの読み取った表示内容を格納
      results = new ArrayList<String>();

      Date now = Calendar.getInstance().getTime();

      errorStrings = new ArrayList<String>();

      for (int i = 0; i < con.size(); i++) {

        VEvent ve = (VEvent) con.get(i);

        CoordinationForGoogleCalRecordData r =
          new CoordinationForGoogleCalRecordData();

        // 登録可・不可のフラグ 初期値は"可"
        ALStringField err_flg = new ALStringField();
        err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_POSSIBLE);

        // 開始時間・終了時間のフラグ 初期値は"可"
        ALStringField sta_end_time_err = new ALStringField();
        sta_end_time_err
          .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_POSSIBLE);

        // 作成日のフラ 初期値は"可"
        ALStringField cre_time_err = new ALStringField();
        cre_time_err
          .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_POSSIBLE);

        // 更新日のフラグ 初期値は"可"
        ALStringField las_time_err = new ALStringField();
        las_time_err
          .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_POSSIBLE);

        // タイトルをセット
        ALStringField sum = new ALStringField();
        // SUMMARY要素がなかったら"(タイトルなし)"をセット
        if (ve.getProperties(
          CoordinationForGoogleCalUtils.PROPERTY_NAME_SUMMARY).size() == 0
          || "".equals(ve.getSummary().getValue())) {
          sum.setValue(CoordinationForGoogleCalUtils.SCHEDULE_HAS_NOT_NAME);
        } else {
          sum.setValue(ve.getSummary().getValue());
        }
        r.setSummary(sum);

        // 場所をセット
        ALStringField loc = new ALStringField();
        // LOCATION要素がなかったら空欄をセット
        if (ve.getProperties(
          CoordinationForGoogleCalUtils.PROPERTY_NAME_LOCATION).size() == 0) {
          loc.setValue(CoordinationForGoogleCalUtils.SPACE_DISPLAY);
        } else {
          loc.setValue(ve.getLocation().getValue());
        }
        r.setLocation(loc);

        // 詳細をセット
        String des = null;
        // DESCRIPTION要素がなかったら空欄をセット
        if (ve.getProperties(
          CoordinationForGoogleCalUtils.PROPERTY_NAME_DESCRIPTION).size() == 0) {
          des = CoordinationForGoogleCalUtils.SPACE_DISPLAY;
        } else {
          des = ve.getDescription().getValue();
        }
        r.setDescription(des);

        // 公開状態をセット
        ALStringField cla = new ALStringField();
        // CLASS要素がなかったら"PRIVATE"をセット
        if (ve
          .getProperties(CoordinationForGoogleCalUtils.PROPERTY_NAME_CLASS)
          .size() == 0
          || "".equals(ve.getClassification().getValue())) {
          cla.setValue(CLASSFICATION_PRIVATE);
        } else {
          if (CoordinationForGoogleCalUtils.PROPERTY_NAME_PRIVATE.equals(ve
            .getClassification()
            .getValue())) {
            cla.setValue(CLASSFICATION_PRIVATE);
          } else {
            cla.setValue(CLASSFICATION_PUBLIC);
          }

        }
        r.setClassification(cla);

        // 作成日をセット
        ALDateTimeField cre = new ALDateTimeField();

        // 作成日が現在時刻より後だったらエラーメッセージを追加
        if (now.before(ve.getCreated().getDate())) {
          error_msg
            .add("(※"
              + (i + 1)
              + "件目)　　　『 <span class='em'>作成日</span> 』は『 <span class='em'>現在時刻</span> 』以降の日付で指定してください。");

          errorStrings.add(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);

          err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          cre_time_err
            .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);

          cre.setValue(ve.getCreated().getDate());
        } else {

          // CREATED要素がなかったら現在時刻をセット
          if (ve.getProperties(
            CoordinationForGoogleCalUtils.PROPERTY_NAME_CREATED).size() == 0) {
            cre.setValue(now);
          } else {
            cre.setValue(ve.getCreated().getDate());
          }
        }

        r.setCreated(cre);

        // 更新日をセット
        ALDateTimeField las = new ALDateTimeField();
        // 更新日が現在時刻より後だったらエラーメッセージを追加
        if (now.before(ve.getLastModified().getDate())) {
          error_msg
            .add("(※"
              + (i + 1)
              + "件目)　　　『 <span class='em'>更新日</span> 』は『 <span class='em'>現在時刻</span> 』以降の日付で指定してください。");
          errorStrings.add(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          las_time_err
            .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);

          las.setValue(ve.getLastModified().getDate());

        } else {

          // LAST-MODIFIED要素がなかったら現在時刻をセット
          if (ve.getProperties(
            CoordinationForGoogleCalUtils.PROPERTY_NAME_LAST_MODIFIED).size() == 0) {
            las.setValue(now);
          } else {
            las.setValue(ve.getLastModified().getDate());
          }

        }
        r.setLastModified(las);

        // 繰り返し等ルールをセット
        ALStringField rep = new ALStringField();
        if (ve
          .getProperties(CoordinationForGoogleCalUtils.PROPERTY_NAME_RRULE)
          .size() == 0) {

          // DTSTART要素にVALUE属性がなければ処理なし
          if (ve.getStartDate().getParameters().getParameter(
            CoordinationForGoogleCalUtils.PROPERTY_NAME_VALUE) == null) {
          } else {

            if (ve.getStartDate().getParameters().getParameter(
              CoordinationForGoogleCalUtils.PROPERTY_NAME_VALUE).getValue() == null
              || "".equals(ve.getStartDate().getParameters().getParameter(
                "VALUE").getValue())) {
              rep.setValue(CoordinationForGoogleCalUtils.SPACE_DISPLAY);

              // DTSTARTのVALUE属性がDATEならば終日予定としてセット
            } else if (CoordinationForGoogleCalUtils.PROPERTY_DATETIPE_DATE
              .equals(ve.getStartDate().getParameters().getParameter(
                CoordinationForGoogleCalUtils.PROPERTY_NAME_VALUE).getValue())) {

              rep.setValue(CoordinationForGoogleCalUtils.SCHEDULE_AT_ALL_DAY);

            } else {
              rep.setValue(CoordinationForGoogleCalUtils.SPACE_DISPLAY);
            }
          }

        } else {
          // 繰り返しパターンを取得
          String repeatPatern =
            CoordinationForGoogleCalUtils.getRepeatPatern(ve);
          repeatPatern =
            CoordinationForGoogleCalUtils.getUntil(ve, repeatPatern);
          rep.setValue(repeatPatern);
        }

        r.setRepeatRule(rep);

        // 開始時間が終了時間より後だったらエラーを返す

        // 開始時間をセット
        ALDateTimeField sta = new ALDateTimeField();
        if (ve.getProperties(
          CoordinationForGoogleCalUtils.PROPERTY_NAME_DTSTART).size() == 0) {
          error_msg.add("(※"
            + (i + 1)
            + "件目)　　　『 <span class='em'>開始日時</span> 』は必須項目です。");
          errorStrings.add(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);

        } else {
          sta.setValue(ve.getStartDate().getDate());
          r.setStartDate(sta);
        }

        // 終了時間をセット
        ALDateTimeField end = new ALDateTimeField();
        if (ve
          .getProperties(CoordinationForGoogleCalUtils.PROPERTY_NAME_DTEND)
          .size() == 0) {
          error_msg.add("(※"
            + (i + 1)
            + "件目)　　　『 <span class='em'>終了日時</span> 』は必須項目です。");
          errorStrings.add(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);

        } else {

          if (CoordinationForGoogleCalUtils.SCHEDULE_AT_ALL_DAY.equals(rep
            .getValue())) {

            Calendar mod_end = Calendar.getInstance();
            mod_end.setTime(ve.getEndDate().getDate());
            mod_end.set(Calendar.DAY_OF_MONTH, mod_end
              .get(Calendar.DAY_OF_MONTH) - 1);

            end.setValue(mod_end.getTime());

          } else {
            Date end_d = ve.getEndDate().getDate();
            end.setValue(end_d);

          }
          r.setEndDate(end);
        }

        // 開始時間が終了時間より遅かったらエラーを返す
        if (r.getEndDate().getValue().before(r.getStartDate().getValue())) {
          error_msg
            .add("(※"
              + (i + 1)
              + "件目)　　　『 <span class='em'>終了日時</span> 』は『 <span class='em'>開始日時</span> 』以降の日付で指定してください。");
          errorStrings.add(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          err_flg.setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
          sta_end_time_err
            .setValue(CoordinationForGoogleCalUtils.ERROR_FLAG_IMPOSSIBLE);
        }

        // 最終的な登録可不可をセット
        r.setErrorFlag(err_flg);
        r.setStartEndTimeErr(sta_end_time_err);
        r.setCreatedTimeErr(cre_time_err);
        r.setLastModifiedTimeErr(las_time_err);

        results.add(r);
      }
      return results;

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }

  }

  /**
   * 現在選択されているタブを取得します。 <BR>
   * 
   * @return
   */
  public String getCurrentTab() {
    return currentTab;
  }

  /**
   * 一時ファイルのフルパスを取得します <BR>
   */
  public String getTempFilePath() {
    return temp_file_path;
  }

  /**
   * 一時フォルダ名（番号）を取得します <BR>
   * 
   * @return
   */
  public String getTempFolderIndex() {
    return temp_folder;
  }

  /**
   * インポートしたファイルの結果を返します。 <BR>
   * 
   * @return
   */
  public List getResults() {
    return results;
  }

  public ALDateField getToDate() {
    return to_date;
  }

  public ALDateField getFromDate() {
    return from_date;
  }

  public List<String> getErrorStrings() {
    return errorStrings;
  }

  /**
   * 期間条件の単位 （"unit_day" or "unit_month"）
   */
  public String getTimeSelectFlg() {
    return time_select_flg;
  }

  public String getPeriodFlg() {
    return period_flg;
  }

  public List<String> getErrorMsg() {
    return error_msg;
  }
}
