/* $Id: cxp-handler.c,v 1.15 2005/08/10 16:05:22 yasumichi Exp $ */
/**
 * \if japanese
 * @file
 * @brief cxp-handler.h
 *
 * @author 
 * @date Tue Mar 29 2005
 * $Revision: 1.15 $
 * \endif
 * \if english
 * @file
 * @brief Internal mounting of cxp-handler.h
 *
 * @author Yasumichi Akahoshi
 * @date Tue Mar 29 2005
 * $Revision: 1.15 $
 * \endif
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include <string.h>
#include <gconf/gconf-client.h>
#include "cxp-utils.h"
#include "cxp-handler.h"
#include "cxp-entry-dialog.h"

/*
 * CxpHandlerΥץ饤١Ȥʥ
 * Private member of CxpHandler
 */
typedef struct
{
	GConfClient *client;
	gboolean dispose_has_run;	/**< Is dispose funciton executed */
} CxpHandlerPrivate;

/**
 * \if japanese
 * CxpHandlerΥ󥹥o˴ϢŤ줿ץ饤١Ȥʹ¤Τ뤿ޥǤ
 * \endif
 * \if english
 * It is an internal macro to obtain the private structure related to instance
 * o of the CxpHandler type. 
 * \endif
 */
#define CXP_HANDLER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_HANDLER, CxpHandlerPrivate))

static GObjectClass *parent_class = NULL;

/*
 * ץ饤١ȥåɤΥץȥ
 * Prototypes of privete methods
 */
static void cxp_handler_init (GTypeInstance * instance, gpointer g_class);
static void cxp_handler_class_init (gpointer g_class, gpointer g_class_data);
static void cxp_handler_dispose (GObject * obj);
static void cxp_handler_finalize (GObject * obj);

/**
 * \if japanese
 * 饹CxpHandlerClassޤ
 * @param [in,out] g_class Ԥ¤ GTypeClass
 * @param [in,out] g_class 𤷤󶡤 class_data
 * \endif
 * \if english
 * Class CxpHandlerClass is initialized. 
 * @param [in,out] The GTypeClass structure to initialize.
 * @param [in,out] The class_data member supplied via the GTypeInfo structure.
 * \endif
 */
static void cxp_handler_class_init (gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);

	gobject_class->dispose = cxp_handler_dispose;
	gobject_class->finalize = cxp_handler_finalize;

	g_type_class_add_private (g_class, sizeof (CxpHandlerPrivate));

	parent_class = g_type_class_peek_parent (g_class);
}

/**
 * \if japanese
 * ֥CxpHandlerΥ󥹥󥹤ޤ
 * @param [in,out] instance Ԥ󥹥
 * @param [in,out] g_class instance 뷿Υ饹
 * \endif
 * \if english
 * The instance of object CxpHandler is initialized.
 * @param [in,out] The instance to initialize.
 * @param [in,out] The class of the type the instance is created for.
 * \endif
 */
static void cxp_handler_init (GTypeInstance * instance, gpointer g_class)
{
	CxpHandlerPrivate *priv = CXP_HANDLER_GET_PRIVATE(instance);

	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");

	priv->client = gconf_client_get_default ();
	priv->dispose_has_run = FALSE;
}

/**
 * \if japanese
 * ƤΥե󥹤¾Υ֥Ȥ˥ɥåפĴؿǤ뤬
 * ¾Υ󥹥󥹤ϤΤޤޤˤƤΤǡ饤¦Υ᥽å
 * ƤӽФư(롼פ򻲾Ȥ뤿ˡʣλ֤ʬư)
 * δؿ֤ˡƥ饹Υ᥽å dispose ϢŪ˸ƤӽФɬפ
 * 롣
 * @param [in,out] obj GObject
 * \endif
 * \if english
 * the dispose function is supposed to drop all references to other objects,
 * but keep the instance otherwise intact, so that client method invocations
 * still work. It may be run multiple times (due to reference loops). Before
 * returning, dispose should chain up to the dispose method of the parent
 * class.
 * @param [in,out] obj GObject
 * \endif
 */
static void cxp_handler_dispose (GObject * obj)
{
	CxpHandlerPrivate *priv = CXP_HANDLER_GET_PRIVATE(obj);

	if (priv->dispose_has_run)
	{
		return;
	}

	priv->dispose_has_run = TRUE;
	g_object_unref (priv->client);

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

/**
 * \if japanese
 * 󥹥󥹤νλԤؿǡ᥽å dispose ǳϤ󥹥
 * νλλƥ饹Υ᥽å finalize ϢŪ˸ƤӽФɬפ
 * 
 * @param [in,out] obj GObject
 * \endif
 * \if english
 * instance finalization function, should finish the finalization of the
 * instance begun in dispose and chain up to the finalize method of the parent
 * class.
 * @param [in,out] obj GObject
 * \endif
 */
static void cxp_handler_finalize (GObject * obj)
{
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

/*
 * APIμ
 * Mounting of API
 */

/**
 * \if japanese
 * CxpHandlerΥ֥ȷȥ饹򷿴ƥϿGType롣
 * @return ϿΥˡ (ʣʤ) ̻ҤɽͤǤ
 * \endif
 * \if english
 * The object type and the class of CxpHandler are registered to the type management 
 * system, and the GType type is obtained. 
 * @return A numerical value which represents the unique identifier of a registered type.
 * \endif
 */
GType cxp_handler_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpHandlerClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			cxp_handler_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpHandler),
			0,	/* n_preallocs */
			cxp_handler_init	/* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT, "CxpHandlerType",
					       &info, 0);
	}

	return type;
}

/**
 * \if japanese
 * CxpHandlerޤ
 * @return CxpHandler
 * \endif
 * \if english
 * CxpHandler is generated. 
 * @return New CxpHandler
 * \endif
 */
CxpHandler *cxp_handler_new (void)
{
	CxpHandler *handler;

	handler = CXP_HANDLER (g_object_new (CXP_TYPE_HANDLER, NULL));

	return handler;
}

/**
 * \if japanese
 * ե̾fullpathγĥҤ򸵤˴ϢŤ줿ץꥱưޤ
 * @param [in] handler CxpHandler
 * @param [in] fullpath եХѥ
 * \endif
 * \if english
 * The application related based on the extension of file name fullpath is
 * started.
 * @param [in] handler CxpHandler
 * @param [in] fullpath Absolute path of file
 * \endif
 */
void cxp_handler_launch (CxpHandler * handler, const gchar * fullpath)
{
	CxpHandlerPrivate *priv = CXP_HANDLER_GET_PRIVATE(handler);
	gchar *gconf_dir;
	gchar *mime_type;
	gchar *cmd = NULL;
	gchar *fullcmd;
	gchar *message;
	gint index;
	GtkWidget *dialog;

	g_return_if_fail (fullpath != NULL);

	mime_type = cxp_get_mime_type_for_file (fullpath);
	gconf_dir = g_strdup_printf ("/apps/cxp/mime-types/%s/open", mime_type);
	cmd = gconf_client_get_string (priv->client, gconf_dir, NULL);
	if (cmd == NULL)
	{
		message =
			g_strdup_printf (_("Please input the command that opens the file whose type is %s"),
					mime_type);
		dialog = cxp_entry_dialog_new (_("Undefined extension"),
				message, "");
		if (gtk_dialog_run (GTK_DIALOG (dialog)) ==
				GTK_RESPONSE_OK)
		{
			cmd = cxp_entry_dialog_get_entry_text
				(CXP_ENTRY_DIALOG (dialog));
			if ((cmd != NULL) && (strlen (cmd) > 0))
			{
				gconf_client_set_string (priv->client, gconf_dir, cmd, NULL);
			}
		}
		gtk_widget_destroy (dialog);
		g_free (message);
	}
	g_free (mime_type);

	if ((cmd != NULL) && (strlen (cmd) > 0))
	{
		fullcmd = g_strdup_printf ("%s '%s'", cmd, fullpath);
		g_spawn_command_line_async (fullcmd, NULL);
		g_free (cmd);
		g_free (fullcmd);
	}

}
