/***************************************************************************
 *            cxp-mime-manager.c
 *
 *  Wed 8 31 20:59:53 2005
 *  Copyright  2005  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "cxp-utils.h"
#include "cxp-mime-manager.h"
 
static void cxp_mime_manager_class_init(CxpMimeManagerClass *klass);
static void cxp_mime_manager_init(CxpMimeManager *sp);
static void cxp_mime_manager_finalize(GObject *object);

struct CxpMimeManagerPrivate {
	/* Place Private Members Here */
	gchar *lang;
	gchar *country;
	gchar *last_matched;
	gchar *common;
	gchar **xdgdirs;
	gint parse_flag;
	GHashTable *comments;
	GHashTable *icons;
};

typedef enum {
	/* Place Signal Types Here */
	SIGNAL_TYPE_EXAMPLE,
	LAST_SIGNAL
} CxpMimeManagerSignalType;

typedef struct {
	CxpMimeManager *object;
} CxpMimeManagerSignal;

enum {
	NO_NEED_PARSE,
	NEED_PARSE,
	NEED_GET_COMMON
};

static guint cxp_mime_manager_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType
cxp_mime_manager_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (CxpMimeManagerClass),
			NULL,
			NULL,
			(GClassInitFunc)cxp_mime_manager_class_init,
			NULL,
			NULL,
			sizeof (CxpMimeManager),
			0,
			(GInstanceInitFunc)cxp_mime_manager_init,
		};

		type = g_type_register_static(G_TYPE_OBJECT, 
			"CxpMimeManager", &our_info, 0);
	}

	return type;
}

static void
cxp_mime_manager_class_init(CxpMimeManagerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = cxp_mime_manager_finalize;
	
	/* Create signals here:
	   cxp_mime_manager_signals[SIGNAL_TYPE_EXAMPLE] = g_signal_new(...)
 	*/
}

static void
cxp_mime_manager_init(CxpMimeManager *obj)
{
	gchar **elements;

	obj->priv = g_new0(CxpMimeManagerPrivate, 1);
	/* Initialize private members, etc. */
	obj->priv->lang = g_strdup(g_getenv("LANG"));
	elements = g_strsplit (obj->priv->lang, "_", 2);
	obj->priv->country = g_strdup(elements[0]);
	g_strfreev (elements);
	obj->priv->last_matched = NULL;
	obj->priv->common = NULL;
	obj->priv->parse_flag = NO_NEED_PARSE;
	obj->priv->comments = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	if (g_getenv("XDG_DATA_DIRS") != NULL)
	{
		obj->priv->xdgdirs = g_strsplit (g_getenv("XDG_DATA_DIRS"), ":", 0);
	}
	else
	{
		obj->priv->xdgdirs = g_strsplit (XDG_DATA_DIR, ":", 0);
	}
	obj->priv->icons = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
}

static void
cxp_mime_manager_finalize(GObject *object)
{
	CxpMimeManager *cobj;
	cobj = CXP_MIME_MANAGER(object);
	
	/* Free private members, etc. */
	g_free(cobj->priv->lang);
	g_free(cobj->priv->country);
	g_hash_table_destroy (cobj->priv->comments);
	g_hash_table_destroy (cobj->priv->icons);
	g_strfreev (cobj->priv->xdgdirs);
		
	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

CxpMimeManager *
cxp_mime_manager_new()
{
	CxpMimeManager *obj;
	
	obj = CXP_MIME_MANAGER(g_object_new(CXP_TYPE_MIME_MANAGER, NULL));
	
	return obj;
}

/* タグが開いていたら呼び出されるコールバックです; たとえば <foo bar="baz"> */
static void cxp_mime_manager_on_start_element (GMarkupParseContext * context,
				  const gchar * element_name,
				  const gchar ** attribute_names,
				  const gchar ** attribute_values,
				  gpointer user_data, GError ** error)
{
	CxpMimeManager *manager = CXP_MIME_MANAGER(user_data);
	gint idx;

	if(strcmp(element_name, "comment") == 0)
	{
		if(attribute_names[0] == NULL)
		{
			manager->priv->parse_flag = NEED_GET_COMMON;
		}
		else
		{
			for(idx=0; attribute_names[idx] != NULL; idx++)
			{
				if(strcmp(attribute_names[idx], "xml:lang") == 0)
				{
					if(strcmp(attribute_values[idx], manager->priv->lang) == 0)
					{
						manager->priv->parse_flag = NEED_PARSE;
					}
					else if(strcmp(attribute_values[idx], manager->priv->country) == 0)
					{
						manager->priv->parse_flag = NEED_PARSE;
					}
					else
					{
						manager->priv->parse_flag = NO_NEED_PARSE;
					}
				}
			}
		}
	}
}

/* タグが閉じていたら呼び出されるコールバックです; たとえば </foo> */
static void cxp_mime_manager_on_end_element (GMarkupParseContext * context,
			    const gchar * element_name,
			    gpointer user_data, GError ** error)
{
	CxpMimeManager *manager = CXP_MIME_MANAGER(user_data);

	manager->priv->parse_flag = NO_NEED_PARSE;
}

/* 文字データだったら呼び出されるコールバックです。*/
/* 'text' の終端はヌル文字 '\0' にして下さい。*/
static void cxp_mime_manager_on_text (GMarkupParseContext * context,
			   const gchar * text,
			   gsize text_len, gpointer user_data, GError ** error)
{
	CxpMimeManager *manager = CXP_MIME_MANAGER(user_data);

	switch(manager->priv->parse_flag)
	{
		case NO_NEED_PARSE:
			break;
		case NEED_PARSE:
			manager->priv->last_matched = g_strdup(text);
			break;
		case NEED_GET_COMMON:
			manager->priv->common = g_strdup(text);
			break;
	}
}

gchar *cxp_mime_manager_get_comment_from_mime (CxpMimeManager *manager, const gchar *mimetype)
{
	gchar *result;
	gchar *filename;
	gchar *content;
	gssize len;
	gint idx;
	GMarkupParseContext *context;
	GMarkupParser parser = {
		cxp_mime_manager_on_start_element,
		cxp_mime_manager_on_end_element,
		cxp_mime_manager_on_text,
		NULL,
		NULL
	};

	if((result = g_strdup(g_hash_table_lookup(manager->priv->comments, mimetype))) == NULL)
	{
		manager->priv->parse_flag = NO_NEED_PARSE;

		for(idx=0; manager->priv->xdgdirs[idx] != NULL; idx++)
		{
			filename = g_strdup_printf("%s/mime/%s.xml", manager->priv->xdgdirs[idx], mimetype);
			if (g_file_get_contents (filename, &content, &len, NULL))
			{
				context = g_markup_parse_context_new (&parser, 0, manager, NULL);
				g_markup_parse_context_parse (context, content, len, NULL);
				g_markup_parse_context_free (context);
				if(manager->priv->last_matched != NULL)
				{
					result = g_strdup (manager->priv->last_matched);
					g_hash_table_replace (manager->priv->comments, g_strdup(mimetype), manager->priv->last_matched);
					g_free (manager->priv->common);
				}
				else
				{
					result = g_strdup (manager->priv->common);
					g_hash_table_replace (manager->priv->comments, g_strdup(mimetype), manager->priv->common);
				}
				break;
			}
			g_free (filename);
		}
		if(result == NULL)	
		{
				result = g_strdup ("unknown file");
				g_hash_table_replace (manager->priv->comments, g_strdup(mimetype), g_strdup("unknown file"));
		}
	}

	return result;
}

GdkPixbuf* cxp_mime_manager_lookup_icon (CxpMimeManager *manager, const gchar *mimetype)
{
	GdkPixbuf *icon;
	gchar *filename;

	if((icon = g_hash_table_lookup(manager->priv->icons, mimetype)) == NULL)
	{
		filename = g_strdup_printf(PACKAGE_PIXMAPS_DIR"/%s.png", mimetype);
		if((icon = gdk_pixbuf_new_from_file(filename, NULL)) == NULL)
		{
			icon = gdk_pixbuf_new_from_file(PACKAGE_PIXMAPS_DIR"/not-find.png", NULL);
		}
		g_hash_table_replace (manager->priv->icons, g_strdup(mimetype), icon);
	}

	return	icon;
}
