/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <cxp-utils.h>
#include <gconf/gconf-client.h>
#include <glib/gi18n-lib.h>
#include "cxp-mime-cmd-dialog.h"
 
static void cxp_mime_cmd_dialog_class_init(CxpMimeCmdDialogClass *klass);
static void cxp_mime_cmd_dialog_init(CxpMimeCmdDialog *sp);
static void cxp_mime_cmd_dialog_finalize(GObject *object);
static GObject *cxp_mime_cmd_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam *construct_properties);
static void cxp_mime_cmd_dialog_construct_child (CxpMimeCmdDialog *self);
static void cxp_mime_cmd_dialog_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void cxp_mime_cmd_dialog_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void cxp_mime_cmd_dialog_ok_button_clicked (GtkWidget *button, CxpMimeCmdDialog *self);

enum
{
	CXP_CMD_FILE_OPEN = 0,
	CXP_CMD_FILE_PREVIEW,
	CXP_CMD_COUNT
}; 

struct CxpMimeCmdDialogPrivate {
	/* Place Private Members Here */
	GConfClient *client;
	GtkWidget *entry[CXP_CMD_COUNT];
	gchar *gconf_key[CXP_CMD_COUNT];
	gchar *mime_type;
};

/*
 * Enumeration type to identify properties.
 */
enum
{
	PROPERTY_MIME_TYPE = 1,
};

typedef enum {
	/* Place Signal Types Here */
	SIGNAL_TYPE_EXAMPLE,
	LAST_SIGNAL
} CxpMimeCmdDialogSignalType;

typedef struct {
	CxpMimeCmdDialog *object;
} CxpMimeCmdDialogSignal;

static guint cxp_mime_cmd_dialog_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

const gchar *gconf_key_format[] = {
	"/apps/cxp/mime-types/%s/open",
	"/apps/cxp/mime-types/%s/preview"
};

GType cxp_mime_cmd_dialog_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (CxpMimeCmdDialogClass),
			NULL,
			NULL,
			(GClassInitFunc)cxp_mime_cmd_dialog_class_init,
			NULL,
			NULL,
			sizeof (CxpMimeCmdDialog),
			0,
			(GInstanceInitFunc)cxp_mime_cmd_dialog_init,
		};

		type = g_type_register_static(GTK_TYPE_DIALOG, 
			"CxpMimeCmdDialog", &our_info, 0);
	}

	return type;
}

static void
cxp_mime_cmd_dialog_class_init(CxpMimeCmdDialogClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	GParamSpec *pspec;

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = cxp_mime_cmd_dialog_finalize;
	object_class->constructor = cxp_mime_cmd_dialog_constructor;
	object_class->set_property = cxp_mime_cmd_dialog_set_property;
	object_class->get_property = cxp_mime_cmd_dialog_get_property;
	
	pspec = g_param_spec_string ("mime-type", "MIME type", "Set MIME type", NULL, G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE);
	g_object_class_install_property (object_class, PROPERTY_MIME_TYPE, pspec);

	/* Create signals here:
	   cxp_mime_cmd_dialog_signals[SIGNAL_TYPE_EXAMPLE] = g_signal_new(...)
 	*/
}

static void
cxp_mime_cmd_dialog_init(CxpMimeCmdDialog *obj)
{
	gint index;

	obj->priv = g_new0(CxpMimeCmdDialogPrivate, 1);
	/* Initialize private members, etc. */
	obj->priv->mime_type = NULL;
	for(index=CXP_CMD_FILE_OPEN; index<CXP_CMD_COUNT; index++)
		obj->priv->gconf_key[index] = NULL;
	obj->priv->client = gconf_client_get_default ();
}

static void
cxp_mime_cmd_dialog_finalize(GObject *object)
{
	CxpMimeCmdDialog *cobj;
	gint index;

	cobj = CXP_MIME_CMD_DIALOG(object);
	
	/* Free private members, etc. */
	g_free(cobj->priv->mime_type);
	for(index=CXP_CMD_FILE_OPEN; index<CXP_CMD_COUNT; index++)
		g_free (cobj->priv->gconf_key[index]);
		
	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

static GObject *cxp_mime_cmd_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam *construct_properties)
{
	GObject *obj;
	CxpMimeCmdDialogClass *klass;

	klass = CXP_MIME_CMD_DIALOG_CLASS (g_type_class_peek (CXP_TYPE_MIME_CMD_DIALOG));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);

	cxp_mime_cmd_dialog_construct_child (CXP_MIME_CMD_DIALOG(obj));

	return obj;
}

static void cxp_mime_cmd_dialog_construct_child (CxpMimeCmdDialog *self)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *ok_button;
	GtkWidget *cancel_button;
	gchar *title[] = {
		N_("Open"),
		N_("Preview")
	};
	gchar *message;
	gchar *command;
	gchar *comment;
	gint index;

	comment = cxp_get_mime_comment (self->priv->mime_type);
	message = g_strdup_printf (_("Please set the command that relates to file of '%s(%s)'."), comment, self->priv->mime_type);
	g_free (comment);
	label = gtk_label_new (message);
	g_free (message);
	gtk_misc_set_alignment (GTK_MISC(label), 0, 0);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(self)->vbox), label, TRUE, TRUE, 2);
	gtk_widget_show (label);

	table = gtk_table_new (2, 2, FALSE);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(self)->vbox), table, TRUE, TRUE, 0);
	gtk_widget_show (table);

	for(index=CXP_CMD_FILE_OPEN; index<CXP_CMD_COUNT; index++)
	{
		label = gtk_label_new (title[index]);
		gtk_misc_set_alignment (GTK_MISC(label), 0, 0);
		gtk_table_attach (GTK_TABLE(table), label, 0, 1, index, index+1, GTK_FILL, GTK_EXPAND, 2, 2);
		gtk_widget_show (label);

		self->priv->entry[index] = gtk_entry_new ();
		gtk_table_attach (GTK_TABLE(table), self->priv->entry[index], 1, 2, index, index+1, GTK_FILL, GTK_EXPAND, 2, 2);
		gtk_entry_set_width_chars (GTK_ENTRY(self->priv->entry[index]), 80);
		self->priv->gconf_key[index] = g_strdup_printf(gconf_key_format[index], self->priv->mime_type);
		g_strdelimit (self->priv->gconf_key[index], "+", '@');	/* example: application/xhtml+xml => /apps/cxp/mime-types/application/xhtml@xml/preview */
		command = gconf_client_get_string (self->priv->client, self->priv->gconf_key[index], NULL);
		if(command != NULL)
		{
			gtk_entry_set_text (GTK_ENTRY(self->priv->entry[index]), command);
			g_free (command);
		}
		gtk_widget_show (self->priv->entry[index]);
	}

	ok_button = gtk_button_new_from_stock("gtk-ok");
	gtk_dialog_add_action_widget (GTK_DIALOG (self), ok_button, GTK_RESPONSE_OK);
	gtk_widget_show (ok_button);
	g_signal_connect (ok_button, "clicked", G_CALLBACK(cxp_mime_cmd_dialog_ok_button_clicked), self);

	cancel_button = gtk_button_new_from_stock("gtk-cancel");
	gtk_dialog_add_action_widget (GTK_DIALOG (self), cancel_button, GTK_RESPONSE_CANCEL);
	gtk_widget_show (cancel_button);
}

GtkWidget* cxp_mime_cmd_dialog_new(const gchar *mime_type)
{
	GtkWidget *obj;
	
	obj = GTK_WIDGET(g_object_new(CXP_TYPE_MIME_CMD_DIALOG, "mime-type", mime_type, NULL));
	
	return obj;
}

/*
 * the generic setter for all properties of this type.
 */
static void cxp_mime_cmd_dialog_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	CxpMimeCmdDialog *self = CXP_MIME_CMD_DIALOG (object);
	CxpMimeCmdDialogPrivate *private = self->priv;

	switch (property_id)
	{
	case PROPERTY_MIME_TYPE:
		if (private->mime_type != NULL)
		{
			g_free (private->mime_type);
		}
		private->mime_type = g_value_dup_string (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

/*
 * the generic getter for all properties of this type.
 */
static void cxp_mime_cmd_dialog_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	CxpMimeCmdDialog *self = CXP_MIME_CMD_DIALOG (object);
	CxpMimeCmdDialogPrivate *private = self->priv;

	switch (property_id)
	{
	case PROPERTY_MIME_TYPE:
		g_value_set_string (value, private->mime_type);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void cxp_mime_cmd_dialog_ok_button_clicked (GtkWidget *button, CxpMimeCmdDialog *self)
{
	gint index;

	for(index=CXP_CMD_FILE_OPEN; index<CXP_CMD_COUNT; index++)
		gconf_client_set_string (self->priv->client, self->priv->gconf_key[index], gtk_entry_get_text (GTK_ENTRY(self->priv->entry[index])), NULL);
}
