/* $Id: cxp-gconf-pair-editor.c,v 1.8 2005/06/28 12:39:36 yasumichi Exp $ */
/**
 * \if japanese
 * @file
 * @brief GConfΥǥ쥯ȥˤڥΥԽ륦ɥ
 *
 * GConfΥǥ쥯ȥˤڥΥԽ뤿Υɥ
 * ̾ϡդˤʤͤǷꤵ롣
 * @author Yasumichi Akahoshi <yasumichi@users.sourceforge.jp>
 * @date Thu Mar 19 2005
 * $Revision: 1.8 $
 * \endif
 * \if english
 * @file
 * @brief Editor of GConf pair type.
 *
 *  Window where key to pair type that exists in specific directory of GConf is
 * edited.The key name is decided to become unique internally.
 * @author Yasumichi Akahoshi <yasumichi@users.sourceforge.jp>
 * @date Thu Mar 19 2005
 * $Revision: 1.8 $
 * \endif
 ****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf-client.h>
#include <glib/gi18n.h>
#include <cxp-utils.h>
#include <cxp-gconf-pair-editor.h>

/*
 * \if japanese
 * @brief ꥹȥȥդ˼̤뤿
 * \endif
 * \if english
 * @brief Enumeration type to identify row of list store uniquely
 * \endif
 */
enum
{
	COL_TERMINATOR = -1,
	COL_KEY,	/* GConf key */
	COL_CAR,	/* first member */
	COL_CDR,	/* second member */
	COL_EDITABLE,	/* Is column editable */
	COL_COUNT	/* number of column */
};

/*
 * the private structure 
 */
typedef struct
{
	GConfClient *client;
	GtkWidget *list_view;
	gchar *gconf_dir;
	gchar *c1_title;
	gchar *c2_title;
	gboolean dispose_has_run;
} CxpGConfPairEditorPrivate;

#define CXP_GCONF_PAIR_EDITOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_GCONF_PAIR_EDITOR, CxpGConfPairEditorPrivate))

static GObjectClass *parent_class = NULL;

/*
 * Enumeration type to identify properties.
 */
enum
{
	CXP_GCONF_PAIR_EDITOR_GCONF_DIR = 1,	/* namespace of GConf */
	CXP_GCONF_PAIR_EDITOR_C1_TITLE,		/* Title of first column header. */
	CXP_GCONF_PAIR_EDITOR_C2_TITLE,		/* Title of second column header. */
};

/*
 * prototype of private method.
 */
static void cxp_gconf_pair_editor_class_init (CxpGConfPairEditorClass * klass);
static void cxp_gconf_pair_editor_instance_init (GTypeInstance * instance,
						 gpointer g_class);
static GObject *cxp_gconf_pair_editor_constructor (GType type,
						   guint n_construct_properties,
						   GObjectConstructParam *
						   construct_properties);
static void cxp_gconf_pair_editor_construct_child (CxpGConfPairEditor * self);
static void cxp_gconf_pair_editor_set_property (GObject * object,
						guint property_id,
						const GValue * value,
						GParamSpec * pspec);
static void cxp_gconf_pair_editor_get_property (GObject * object,
						guint property_id,
						GValue * value,
						GParamSpec * pspec);
static void cxp_gconf_pair_editor_dispose (GObject * obj);
static void cxp_gconf_pair_editor_finalize (GObject * obj);
static GtkWidget *cxp_gconf_pair_editor_list_view_new (CxpGConfPairEditor *
						       self);
static void cxp_gconf_pair_editor_get_all_entries (CxpGConfPairEditor * self);
static void cxp_gconf_pair_editor_selection_foreach (GtkTreeModel * model,
						     GtkTreePath * path,
						     GtkTreeIter * iter,
						     gpointer data);
static void cxp_gconf_pair_editor_cell_edited (GtkCellRendererText * cell,
					       gchar * path_string,
					       gchar * new_text,
					       gpointer user_data);
static void cxp_gconf_pair_editor_add_button_clicked (GtkButton * button,
						      gpointer user_data);
static void cxp_gconf_pair_editor_delete_button_clicked (GtkButton * button,
							 gpointer user_data);
/*
 * the class initialization function
 */
static void cxp_gconf_pair_editor_class_init (CxpGConfPairEditorClass * klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GParamSpec *pspec;

	gobject_class->constructor = cxp_gconf_pair_editor_constructor;
	gobject_class->set_property = cxp_gconf_pair_editor_set_property;
	gobject_class->get_property = cxp_gconf_pair_editor_get_property;
	gobject_class->dispose = cxp_gconf_pair_editor_dispose;
	gobject_class->finalize = cxp_gconf_pair_editor_finalize;

	parent_class = g_type_class_peek_parent (klass);

	pspec = g_param_spec_string ("gconf-dir",
				     "gconf directory",
				     "Set gconf directory",
				     NULL,
				     G_PARAM_CONSTRUCT_ONLY |
				     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 CXP_GCONF_PAIR_EDITOR_GCONF_DIR,
					 pspec);

	pspec = g_param_spec_string ("c1-title",
				     "title of 1st column",
				     "Set title of 1st column",
				     NULL,
				     G_PARAM_CONSTRUCT_ONLY |
				     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 CXP_GCONF_PAIR_EDITOR_C1_TITLE, pspec);

	pspec = g_param_spec_string ("c2-title",
				     "title of 2nd column",
				     "Set title of 2nd column",
				     NULL,
				     G_PARAM_CONSTRUCT_ONLY |
				     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 CXP_GCONF_PAIR_EDITOR_C2_TITLE, pspec);

	g_type_class_add_private (klass, sizeof (CxpGConfPairEditorPrivate));
}

/*
 * the instance initialization function
 */
static void cxp_gconf_pair_editor_instance_init (GTypeInstance * instance,
						 gpointer g_class)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (instance);

	private->client = gconf_client_get_default ();
	private->gconf_dir = NULL;
	private->c1_title = NULL;
	private->c2_title = NULL;
	private->dispose_has_run = FALSE;
}

/*
 * the constructor function
 */
static GObject *cxp_gconf_pair_editor_constructor (GType type,
						   guint n_construct_properties,
						   GObjectConstructParam *
						   construct_properties)
{
	GObject *obj;
	CxpGConfPairEditorClass *klass;

	klass = CXP_GCONF_PAIR_EDITOR_CLASS (g_type_class_peek
					     (CXP_TYPE_GCONF_PAIR_EDITOR));
	obj = parent_class->constructor (type, n_construct_properties,
					 construct_properties);

	cxp_gconf_pair_editor_construct_child (CXP_GCONF_PAIR_EDITOR (obj));

	return obj;
}

/*
 * construct widget face.
 */
static void cxp_gconf_pair_editor_construct_child (CxpGConfPairEditor * self)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (self);
	GtkWidget *vbox;
	GtkWidget *scrolled_win;
	GtkWidget *hbutton_box;
	GtkWidget *add_button;
	GtkWidget *del_button;
	GtkWidget *close_button;

	gtk_window_set_destroy_with_parent (GTK_WINDOW (self), TRUE);

	vbox = gtk_vbox_new (FALSE, 2);
	gtk_container_add (GTK_CONTAINER (self), vbox);
	gtk_widget_show (vbox);

	scrolled_win = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled_win, TRUE, TRUE, 0);
	gtk_widget_show (scrolled_win);

	private->list_view = cxp_gconf_pair_editor_list_view_new (self);
	gtk_container_add (GTK_CONTAINER (scrolled_win), private->list_view);
	gtk_widget_show (private->list_view);

	hbutton_box = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbutton_box),
				   GTK_BUTTONBOX_END);
	gtk_box_pack_start (GTK_BOX (vbox), hbutton_box, FALSE, FALSE, 0);
	gtk_widget_show (hbutton_box);

	add_button = gtk_button_new_from_stock ("gtk-add");
	gtk_container_add (GTK_CONTAINER (hbutton_box), add_button);
	gtk_widget_show (add_button);
	g_signal_connect (add_button, "clicked",
			  G_CALLBACK (cxp_gconf_pair_editor_add_button_clicked),
			  self);

	del_button = gtk_button_new_from_stock ("gtk-delete");
	gtk_container_add (GTK_CONTAINER (hbutton_box), del_button);
	gtk_widget_show (del_button);
	g_signal_connect (del_button, "clicked",
			  G_CALLBACK
			  (cxp_gconf_pair_editor_delete_button_clicked), self);

	close_button = gtk_button_new_from_stock ("gtk-close");
	gtk_container_add (GTK_CONTAINER (hbutton_box), close_button);
	gtk_widget_show (close_button);
	g_signal_connect_swapped (close_button, "clicked", G_CALLBACK(gtk_widget_destroy), self);

	cxp_gconf_pair_editor_get_all_entries (self);
}

/*
 * the generic setter for all properties of this type.
 */
static void cxp_gconf_pair_editor_set_property (GObject * object,
						guint property_id,
						const GValue * value,
						GParamSpec * pspec)
{
	CxpGConfPairEditor *self = CXP_GCONF_PAIR_EDITOR (object);
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (self);

	switch (property_id)
	{
	case CXP_GCONF_PAIR_EDITOR_GCONF_DIR:
		if (private->gconf_dir != NULL)
		{
			g_free (private->gconf_dir);
		}
		private->gconf_dir = g_value_dup_string (value);
		break;
	case CXP_GCONF_PAIR_EDITOR_C1_TITLE:
		if (private->c1_title != NULL)
		{
			g_free (private->c1_title);
		}
		private->c1_title = g_value_dup_string (value);
		break;
	case CXP_GCONF_PAIR_EDITOR_C2_TITLE:
		if (private->c2_title != NULL)
		{
			g_free (private->c2_title);
		}
		private->c2_title = g_value_dup_string (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

/*
 * the generic getter for all properties of this type.
 */
static void cxp_gconf_pair_editor_get_property (GObject * object,
						guint property_id,
						GValue * value,
						GParamSpec * pspec)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (object);

	switch (property_id)
	{
	case CXP_GCONF_PAIR_EDITOR_GCONF_DIR:
		g_value_set_string (value, private->gconf_dir);
		break;
	case CXP_GCONF_PAIR_EDITOR_C1_TITLE:
		g_value_set_string (value, private->c1_title);
		break;
	case CXP_GCONF_PAIR_EDITOR_C2_TITLE:
		g_value_set_string (value, private->c2_title);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

/*
 * the dispose function
 */
static void cxp_gconf_pair_editor_dispose (GObject * obj)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (obj);
	GtkTreeModel *model;

	if (private->dispose_has_run)
	{
		/* If dispose did already run, return. */
		return;
	}
	/* Make sure dispose does not run twice. */
	private->dispose_has_run = TRUE;

	/* 
	 * In dispose, you are supposed to free all types referenced from this
	 * object which might themselves hold a reference to self. Generally,
	 * the most simple solution is to unref all members on which you own a 
	 * reference.
	 */
	g_object_unref (private->client);
	g_free (private->gconf_dir);
	g_free (private->c1_title);
	g_free (private->c2_title);

	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

/*
 * instance finalization function
 */
static void cxp_gconf_pair_editor_finalize (GObject * obj)
{
	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

/*
 * making externals of the list view.
 */
static GtkWidget *cxp_gconf_pair_editor_list_view_new (CxpGConfPairEditor *
						       self)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (self);
	GtkWidget *list_view;
	GtkCellRenderer *renderer;
	GtkTreeModel *model;
	GtkTreeViewColumn *column;
	GtkTreeSelection *selection;

	model = GTK_TREE_MODEL (gtk_list_store_new
				(COL_COUNT, G_TYPE_STRING, G_TYPE_STRING,
				 G_TYPE_STRING, G_TYPE_BOOLEAN));
	list_view = gtk_tree_view_new_with_model (model);
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (list_view));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
	gtk_widget_set_size_request (list_view, 400, 300);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (private->c1_title,
							   renderer, "text",
							   COL_CAR, "editable",
							   COL_EDITABLE, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_view), column);
	g_object_set_data (G_OBJECT (renderer), "my_column_num",
			   GUINT_TO_POINTER (COL_CAR));
	g_signal_connect (renderer, "edited",
			  G_CALLBACK (cxp_gconf_pair_editor_cell_edited), self);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (private->c2_title,
							   renderer, "text",
							   COL_CDR, "editable",
							   COL_EDITABLE, NULL);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_view), column);
	g_object_set_data (G_OBJECT (renderer), "my_column_num",
			   GUINT_TO_POINTER (COL_CDR));
	g_signal_connect (renderer, "edited",
			  G_CALLBACK (cxp_gconf_pair_editor_cell_edited), self);


	return list_view;
}

/*
 * The list of the entry that exists in a specific directory is obtained.
 */
static void cxp_gconf_pair_editor_get_all_entries (CxpGConfPairEditor * self)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (self);
	GConfValue *val;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GSList *slist;
	GSList *node;
	gchar *key;
	gchar *car;
	gchar *cdr;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (private->list_view));
	slist = gconf_client_all_entries (private->client, private->gconf_dir,
					  NULL);

	for (node = slist; node != NULL; node = node->next)
	{
		key = g_strdup (gconf_entry_get_key (node->data));
		val = gconf_entry_get_value (node->data);
		if (val && val->type == GCONF_VALUE_PAIR)
		{
			car = g_strdup (gconf_value_get_string
					(gconf_value_get_car (val)));
			cdr = g_strdup (gconf_value_get_string
					(gconf_value_get_cdr (val)));
			gtk_list_store_append (GTK_LIST_STORE (model), &iter);
			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    COL_KEY, key, COL_CAR, car, COL_CDR,
					    cdr, COL_EDITABLE, TRUE,
					    COL_TERMINATOR);
		}
	}

	g_slist_free (slist);
}

/**
 * \if japanese
 * 򤵤줿ԤGtkTreeRowReference롣
 * \endif
 */
static void cxp_gconf_pair_editor_selection_foreach (GtkTreeModel * model,
						     GtkTreePath * path,
						     GtkTreeIter * iter,
						     gpointer data)
{
	GList **list = (GList **) data;
	GtkTreeRowReference *rowref;

	g_assert (list != NULL);

	rowref = gtk_tree_row_reference_new (model, path);
	*list = g_list_append (*list, rowref);
}

/*
 * \if japanese
 * 뤬Խ줿ν
 * \endif
 */
static void cxp_gconf_pair_editor_cell_edited (GtkCellRendererText * cell,
					       gchar * path_string,
					       gchar * new_text,
					       gpointer user_data)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (user_data);
	GtkTreeIter iter;
	GtkTreeModel *model =
		gtk_tree_view_get_model (GTK_TREE_VIEW (private->list_view));
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	guint column_number =
		GPOINTER_TO_UINT (g_object_get_data
				  (G_OBJECT (cell), "my_column_num"));
	gchar *key;
	gchar *car;
	gchar *cdr;

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, column_number,
			    g_strdup (new_text), -1);
	gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, COL_KEY, &key,
			    COL_CAR, &car, COL_CDR, &cdr, -1);

	gconf_client_set_pair (private->client, key, GCONF_VALUE_STRING,
			       GCONF_VALUE_STRING, &car, &cdr, NULL);

	g_free (key);
	g_free (car);
	g_free (cdr);
}

/*
 * \if japanese
 * ɲåܥ󤬥å줿ΥХå
 * \endif
 */
static void cxp_gconf_pair_editor_add_button_clicked (GtkButton * button,
						      gpointer user_data)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (user_data);
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *ukey;
	gchar *key;

	ukey = cxp_utils_make_unique_key ();
	key = g_build_path ("/", private->gconf_dir, ukey, NULL);

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (private->list_view));
	gtk_list_store_append (GTK_LIST_STORE (model), &iter);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, COL_KEY, key,
			    COL_CAR, private->c1_title, COL_CDR,
			    private->c2_title, COL_EDITABLE, TRUE,
			    COL_TERMINATOR);

	gconf_client_set_pair (private->client, key, GCONF_VALUE_STRING,
			       GCONF_VALUE_STRING, &private->c1_title, &private->c2_title, NULL);

	g_free (key);
	g_free (ukey);
}

/*
 * \if japansese
 * ܥ󤬥å줿ΥХå
 * \endif
 */
static void cxp_gconf_pair_editor_delete_button_clicked (GtkButton * button,
							 gpointer user_data)
{
	CxpGConfPairEditorPrivate *private =
		CXP_GCONF_PAIR_EDITOR_GET_PRIVATE (user_data);
	GList *list = NULL;
	GList *node;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeSelection *selection;
	gchar *key;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (private->list_view));
	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW
					     (private->list_view));
	gtk_tree_selection_selected_foreach (selection,
					     cxp_gconf_pair_editor_selection_foreach,
					     &list);

	for (node = list; node != NULL; node = node->next)
	{
		path = gtk_tree_row_reference_get_path ((GtkTreeRowReference
							 *) (node->data));
		if (path)
		{
			if (gtk_tree_model_get_iter (model, &iter, path))
			{
				gtk_tree_model_get (model, &iter, COL_KEY, &key,
						    COL_TERMINATOR);
				if ((key == NULL)
				    || gconf_client_unset (private->client, key,
							   NULL))
				{
					gtk_list_store_remove (GTK_LIST_STORE
							       (model), &iter);
				}
				g_free (key);
			}
			gtk_tree_path_free (path);
		}
	}

	g_list_free (list);
}

/*
 * \if japanese
 * GTypeƥؤϿ
 * \endif
 */
GType cxp_gconf_pair_editor_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpGConfPairEditorClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) cxp_gconf_pair_editor_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpGConfPairEditor),
			0,	/* n_preallocs */
			cxp_gconf_pair_editor_instance_init	/* instance_init */
		};
		type = g_type_register_static (GTK_TYPE_WINDOW,
					       "CxpGConfPairEditorType",
					       &info, 0);
	}
	return type;
}

/**
 * Creates a new dialog box which show detail of file.
 */
GtkWidget *cxp_gconf_pair_editor_new (GtkWindow * parent, const gchar * title,
				      const gchar * gconf_dir,
				      const gchar * c1_title,
				      const gchar * c2_title)
{
	CxpGConfPairEditor *window;

	window = g_object_new (CXP_TYPE_GCONF_PAIR_EDITOR, "gconf-dir",
			       gconf_dir, "c1-title", c1_title, "c2-title",
			       c2_title, NULL);
	gtk_window_set_title (GTK_WINDOW (window), title);
	gtk_window_set_transient_for (GTK_WINDOW (window), parent);

	return GTK_WIDGET (window);
}
