#include  "d2_segment.h"
#include  "d2_triangle.h"
#include  <cstdlib>
#include  <algorithm>


D2_Segment::D2_Segment( const D2_Vector &  p0 ,  const D2_Vector &  p1 )
	: p0( p0 ) , p1( p1 )
{
}

const D2_Vector &  D2_Segment::point_0() const
{
	return( p0 );
}

const D2_Vector &  D2_Segment::point_1() const
{
	return( p1 );
}

FLOAT	D2_Segment::length() const
{
	return( (p1 - p0).r() );
}

bool   D2_Segment::intersect_without_terminal_point
					( const D2_Segment &  seg ) const
{
	return
	  (   D2_Triangle( *this , seg.point_0() ).signed_area_2()
	    * D2_Triangle( *this , seg.point_1() ).signed_area_2() < 0.0
	 &&   D2_Triangle( seg , this -> point_0() ).signed_area_2()
	    * D2_Triangle( seg , this -> point_1() ).signed_area_2() < 0.0 );
}

bool   D2_Segment::intersect( const D2_Segment &  seg ) const
{
	FLOAT	a0 = D2_Triangle( *this , seg.point_0() ).signed_area_2();
	FLOAT	a1 = D2_Triangle( *this , seg.point_1() ).signed_area_2();
	FLOAT	b0 = D2_Triangle( seg , this -> point_0() ).signed_area_2();
	FLOAT	b1 = D2_Triangle( seg , this -> point_1() ).signed_area_2();

	if ( a0 * a1 < 0.0  &&  b0 * b1 < 0.0 )
	{
		return( true );
	}

	// XXX: optimizable
	if ( this -> point_0() == this -> point_1() )
	{
		if ( seg.point_0() == seg.point_1() )
		{
			return( this -> point_0() == seg.point_0() );
		}

		return( b0 == 0.0
		     && seg.check_intersect_on_line( this -> point_0() ) );
	}
	else if ( seg.point_0() == seg.point_1() )
	{
		return( a0 == 0.0
		     && this -> check_intersect_on_line( seg.point_0() ) );
	}


	if ( (a0 == 0.0 && this -> check_intersect_on_line( seg.point_0() ))
	  || (a1 == 0.0 && this -> check_intersect_on_line( seg.point_1() ))
	  || (b0 == 0.0 && seg.check_intersect_on_line( this -> point_0() ))
	  || (b1 == 0.0 && seg.check_intersect_on_line( this -> point_1() )) )
	{
		return( true );
	}

	return( false );
}

bool   D2_Segment::on_segment( const D2_Vector &  p ) const
{
	return( D2_Triangle( *this , p ).signed_area_2() == 0.0
		&& this -> check_intersect_on_line( p ) );
}

bool   D2_Segment::on_segment( const D2_Vector &  p ,
			       FLOAT  max_dist ) const
{
	return( this -> distance( p ) <= max_dist );
}

D2_Straight_Line  D2_Segment::to_straight_line() const
{
	return( D2_Straight_Line( p0 , p1 ) );
}

// private
bool   D2_Segment::check_intersect_on_line( const D2_Vector &  p ) const
{
	if ( this -> point_0().x() == this -> point_1().x() )
	{
		return
		 ( (this -> point_0().y() <= p.y()
					  && p.y() <= this -> point_1().y())
		|| (this -> point_1().y() <= p.y()
					  && p.y() <= this -> point_0().y()) );
	}
	else
	{
		return
		 ( (this -> point_0().x() <= p.x()
					  && p.x() <= this -> point_1().x())
		|| (this -> point_1().x() <= p.x()
					  && p.x() <= this -> point_0().x()) );
	}
}

D2_Vector  D2_Segment::middle_point() const
{
	return( (this -> point_0() + this -> point_1()) / 2 );
}

D2_Vector  D2_Segment::nearest_point( const D2_Vector &  p ) const
{
	const D2_Vector	vec = this -> point_1() - this -> point_0();

	const FLOAT	len_square = vec.r_square();

	if ( len_square == 0.0 )
	{
		return( this -> point_0() );
	}

	FLOAT	inner_product = vec.inner_product( (p - this -> point_0()) );

	//
	// a: p1 - p0
	// b: p - p0
	//
	// check if 0 <= |b|cos(theta) <= |a|
	//       -> 0 <= |a||b|cos(theta) <= |a|^2
	//       -> 0 <= a.b <= |a|^2  // a.b = |a||b|cos(theta)
	//
	if ( inner_product <= 0.0 )
	{
		return( this -> point_0() );
	}
	else if ( inner_product >= len_square )
	{
		return( this -> point_1() );
	}

	return( this -> point_0()
		+ vec * inner_product / len_square );
}

FLOAT  D2_Segment::farthest_distance( const D2_Vector &  p ) const
{
	return( std::max( (this -> point_0() - p).r() ,
			  (this -> point_1() - p).r() ) );
}

FLOAT  D2_Segment::distance( const D2_Vector &  p ) const
{
	FLOAT	len = this -> length();

	if ( len == 0.0 )
	{
		return( (p - this -> point_0()).r() );
	}

	const D2_Vector	vec = this -> point_1() - this -> point_0();

	FLOAT	inner_product = vec.inner_product( (p - this -> point_0()) );

	//
	// a: p1 - p0
	// b: p - p0
	//
	// check if 0 <= |b|cos(theta) <= |a|
	//       -> 0 <= |a||b|cos(theta) <= |a|^2
	//       -> 0 <= a.b <= |a|^2  // a.b = |a||b|cos(theta)
	//
	if ( 0.0 <= inner_product && inner_product <= len * len )
	{
		const FLOAT	perpendicular_dist
				= float_traits<FLOAT>::fabs
				  ( D2_Triangle( *this , p ).signed_area_2()
				    / len );

		return( perpendicular_dist );
	}

	return( std::min( (p - this -> point_0()).r() ,
			  (p - this -> point_1()).r() ) );
}

FLOAT  D2_Segment::distance( const D2_Segment &  seg ) const
{
	// XXX: optimizable?

	if ( this -> intersect( seg ) )
	{
		return( 0.0 );
	}

	return( std::min( std::min( this -> distance( seg.point_0() ) ,
				    this -> distance( seg.point_1() ) ),
			  std::min( seg.distance( this -> point_0() ) ,
				    seg.distance( this -> point_1() ) ) ) );
}

D2_Segment  D2_Segment::get_shift( const D2_Vector &  offset ) const
{
	return( D2_Segment( this -> point_0() + offset ,
			    this -> point_1() + offset ) );
}
