extern "C"
{
	#include  <errno.h>
}
#include  "stream_socket.h"
#include  "system_call_wrapper.h"
#include  "string_extension.h"
#include  "posix_compat_fcntl.h"
#include  <unistd.h>
#include  <cstdio>
#include  <cstring>


// protected:
void   Stream_Socket::system_error( const std::string &  message ,
				    int  err ) const
{
	if ( error_stream != static_cast<std::ostream *>(0) )
	{
#if defined(MINGW)
		// XXX
		static_cast<void>(err);
		(*error_stream) << message << ": error" << std::endl;
#else
		(*error_stream) << message << ": "
				<< std::strerror( err ) << std::endl;
#endif
	}
}


// protected:
void   Stream_Socket::internal_error( const std::string &  message ) const
{
	if ( error_stream != static_cast<std::ostream *>(0) )
	{
		(*error_stream) << message << std::endl;
	}
}


Stream_Socket::Stream_Socket()
{
	this -> error_stream = &(std::cerr);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type   = SOCK_STREAM;
	this -> ipv4_only = false;
}

Stream_Socket::Stream_Socket( Socket_Opened_Socket ,
			      SOCKET_FD_TYTE  sock_fd ,
			      int  sock_domain )
{
	this -> error_stream = &(std::cerr);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = sock_fd;
	this -> domain = sock_domain;
	this -> type   = SOCK_STREAM;
	this -> ipv4_only = false;
}

Stream_Socket::Stream_Socket( Socket_Opened_Socket ,
			      SOCKET_FD_TYTE  sock_fd ,
			      int  sock_domain ,
			      Socket_No_Error )
{
	this -> error_stream = static_cast<std::ostream *>(0);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = sock_fd;
	this -> domain = sock_domain;
	this -> type = SOCK_STREAM;
	this -> ipv4_only = false;
}

Stream_Socket::Stream_Socket( std::ostream &  err )
{
	this -> error_stream = &err;
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type = SOCK_STREAM;
	this -> ipv4_only = false;
}


Stream_Socket::Stream_Socket( Socket_No_Error )
{
	this -> error_stream = static_cast<std::ostream *>(0);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type = SOCK_STREAM;
	this -> ipv4_only = false;
}


Stream_Socket::Stream_Socket( int  sock_domain ,
			      int  sock_type ,
			      int  protocol ,
			      Socket_No_Error )
{
	this -> error_stream = static_cast<std::ostream *>(0);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type = SOCK_STREAM;
	this -> ipv4_only = false;

	this -> socket( sock_domain , sock_type , protocol );
}


Stream_Socket::Stream_Socket( int  sock_domain ,
			      int  sock_type ,
			      int  protocol,
			      std::ostream &  err )
{
	this -> error_stream = &err;
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type = SOCK_STREAM;
	this -> ipv4_only = false;

	this -> socket( sock_domain , sock_type , protocol );
}


Stream_Socket::Stream_Socket( const std::string &  host ,
			      port_number_t  port ,
			      bool  ipv4_only )
{
	this -> error_stream = &(std::cerr);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = BAD_SOCKET_FD;
	this -> domain = AF_INET;
	this -> type   = SOCK_STREAM;
	this -> ipv4_only = ipv4_only;

	this -> make_connection( host , port , ipv4_only );
}


Stream_Socket::~Stream_Socket()
{
	this -> close();
}


int    Stream_Socket::socket( int  sock_domain ,
			      int  sock_type ,
			      int  protocol )
{
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> domain = sock_domain;
	this -> type   = sock_type;

	if ( (sfd = ::socket( sock_domain , sock_type , protocol ))
	     == BAD_SOCKET_FD )
	{
		system_error( "socket" , errno );
	}

	return( sfd );
}


int    Stream_Socket::socket( Socket_Opened_Socket ,
			      SOCKET_FD_TYTE  sock_fd , int  sock_domain )
{
	this -> error_stream = &(std::cerr);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = sock_fd;
	this -> domain = sock_domain;
	this -> type   = SOCK_STREAM;

	return( sfd );
}

int    Stream_Socket::socket( Socket_Opened_Socket ,
			      SOCKET_FD_TYTE  sock_fd , int  sock_domain ,
			      Socket_No_Error )
{
	this -> error_stream = static_cast<std::ostream *>(0);
	this -> connected_flag = false;
	this -> input_end_of_file = false;
	this -> output_end_of_file = false;
	this -> sfd = sock_fd;
	this -> domain = sock_domain;
	this -> type = SOCK_STREAM;

	return( sfd );
}


//
// close
//
int    Stream_Socket::close()
{
	int	ret;

	this -> input_end_of_file = true;
	this -> output_end_of_file = true;
	this -> connected_flag = false;

	if ( sfd == BAD_SOCKET_FD )
	{
		return( 0 );
	}

	if ( (ret = SOCKET_CLOSE_FUNCTION( sfd )) == SOCKET_ERROR_CODE )
	{
		system_error( "close" , errno );
	}
	else
	{
		if ( socket_file != "" )
		{
			if ( ::unlink( socket_file.c_str() ) == -1 )
			{
				system_error( "unlink" , errno );
			}
		}
	}

	sfd = BAD_SOCKET_FD;

	return( ret );
}


//
//  bind
//
int    Stream_Socket::bind( const struct sockaddr *  soc_addr,
			    SOCKLEN_T  soc_addr_len )
{
	const int	one = 1;
	if ( ::setsockopt
	       ( sfd ,
		 SOL_SOCKET , SO_REUSEADDR ,
		 reinterpret_cast<CONST_SOCKOPT_OPTION_VALUE_POINTER_TYPE>
		   ( &one ) ,
		 sizeof(one) ) == -1 )
	{
		system_error( "setsockopt" , errno );
	}


	int	ret;
	if ( (ret = ::bind( sfd , soc_addr , soc_addr_len )) == -1 )
	{
		system_error( "bind" , errno );
	}
#ifdef HAVE_SOCKADDR_UN
	else
	{
		if ( soc_addr -> sa_family == AF_UNIX )
		{
			socket_file = ((const struct sockaddr_un *)soc_addr)
					-> sun_path;
		}
	}
#endif

	return( ret );
}


int    Stream_Socket::bind( const struct sockaddr_in *  soc_addr_in )
{
	int	ret;

	if ( (ret = ::bind( sfd , (const struct sockaddr *)soc_addr_in ,
				  sizeof(struct sockaddr_in) )) == -1 )
	{
		system_error( "bind" , errno );
	}

	return( ret );
}


#ifdef HAVE_SOCKADDR_UN
int    Stream_Socket::bind( const struct sockaddr_un *  soc_addr_un )
{
	int	ret;

	if ( (ret = ::bind( sfd , (const struct sockaddr *)soc_addr_un ,
				  sizeof(struct sockaddr_un) )) == -1 )
	{
		system_error( "bind" , errno );
	}
	else
	{
		socket_file = soc_addr_un -> sun_path;
	}

	return( ret );
}
#endif


#ifdef HAVE_SOCKADDR_UN
int    Stream_Socket::bind( const std::string &  path )
{
	struct sockaddr_un	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sun_family = AF_UNIX;
	std::strncpy( addr.sun_path , path.c_str() ,
		      sizeof(addr.sun_path) );

	int	ret;
	if ( (ret = ::bind( sfd , (const struct sockaddr *)&addr ,
			    sizeof( struct sockaddr_un ) )) == -1 )
	{
		system_error( "bind" , errno );
	}
	else
	{
		socket_file = path;
	}

	return( ret );
}
#endif


int    Stream_Socket::bind( port_number_t  port )
{
	const int	one = 1;
	if ( ::setsockopt
	       ( sfd ,
		 SOL_SOCKET , SO_REUSEADDR ,
		 reinterpret_cast<CONST_SOCKOPT_OPTION_VALUE_POINTER_TYPE>
		   ( &one ) ,
		 sizeof(one) ) == -1 )
	{
		system_error( "setsockopt" , errno );
	}


	struct sockaddr_in	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sin_family      = AF_INET;
	addr.sin_addr.s_addr = htonl( INADDR_ANY );
	addr.sin_port        = htons( port );

	int	ret;
	if ( (ret = ::bind( sfd , (const struct sockaddr *)&addr ,
				  sizeof( addr ) )) == -1 )
	{
		system_error( "bind" , errno );
	}

	return( ret );
}


//
//  connect
//
int    Stream_Socket::connect( const struct sockaddr *  soc_addr ,
			       SOCKLEN_T  soc_addr_len )
{
	int	ret;

	if ( (ret = ::connect( sfd , soc_addr , soc_addr_len )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}

int    Stream_Socket::connect( const struct sockaddr *  soc_addr ,
			       SOCKLEN_T  soc_addr_len ,
			       const std::string &  address_representation )
{
	int	ret;

	if ( (ret = ::connect( sfd , soc_addr , soc_addr_len )) == -1 )
	{
		system_error( std::string("connect to address [")
			      + address_representation + "]" ,
			      errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}

#ifdef HAVE_SOCKADDR_UN
int    Stream_Socket::connect( const struct sockaddr_un *  soc_addr_un )
{
	int	ret;

	if ( (ret = ::connect( sfd , (const struct sockaddr *)soc_addr_un ,
				     sizeof( struct sockaddr_un) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}
#endif


int    Stream_Socket::connect( const struct sockaddr_in *  soc_addr_in )
{
	int	ret;

	if ( (ret = ::connect( sfd , (const struct sockaddr *)soc_addr_in ,
				     sizeof(struct sockaddr_in) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}


#ifdef HAVE_SOCKADDR_UN
int    Stream_Socket::connect( const std::string &  path )
{
	struct sockaddr_un	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sun_family = AF_UNIX;
	std::strncpy( addr.sun_path , path.c_str() , sizeof(addr.sun_path) );

	int	ret;
	if ( (ret = ::connect( sfd , (const struct sockaddr *)&addr ,
			       sizeof( addr ) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}
#endif


#if 0
int    Stream_Socket::connect( ip_address_t  ip_addr ,
			       port_number_t  port )
{
	struct sockaddr_in	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sin_family      = AF_INET;
	addr.sin_addr.s_addr = htonl( ip_addr );
	addr.sin_port        = htons( port );

	int	ret;
	if ( (ret = ::connect( sfd , (const struct sockaddr *)&addr ,
			       sizeof( addr ) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}
#endif


int    Stream_Socket::connect( const std::string &  host ,
			       port_number_t  port )
{
	struct hostent *	hp;
	ip_address_t		address;

	if ( (hp = ::gethostbyname( host.c_str() ))
	     == static_cast<struct hostent *>(0) )
	{
		std::perror( "gethostbyname" );
		return( -1 );
	}
	else
	{
		std::memcpy( &address , hp -> h_addr , hp -> h_length );

		return( this -> connect
			( IPv4_Address( IPv4_Address::Host_Byte_Order ,
					address ) ,
			  port ) );
	}
}

int    Stream_Socket::connect( const IPv4_Address &  ip_addr ,
			       port_number_t  port )
{
	struct sockaddr_in	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sin_family      = AF_INET;
	addr.sin_addr.s_addr = ip_addr.network_byte_order();
	addr.sin_port        = htons( port );

	int	ret;
	if ( (ret = ::connect( sfd , (const struct sockaddr *)&addr ,
			       sizeof( addr ) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}

int    Stream_Socket::connect( port_number_t  port )
{
	struct sockaddr_in	addr;
	std::memset( &addr , 0 , sizeof(addr) );

	addr.sin_family      = AF_INET;
	addr.sin_addr.s_addr = htonl( INADDR_ANY );
	addr.sin_port        = htons( port );

	int	ret;
	if ( (ret = ::connect( sfd , (const struct sockaddr *)&addr ,
			       sizeof( addr ) )) == -1 )
	{
		system_error( "connect" , errno );
	}
	else
	{
		this -> connected_flag = true;
	}

	return( ret );
}


//
// listen
//
int    Stream_Socket::listen( int  backlog )
{
	int	ret;

	if ( (ret = ::listen( sfd , backlog )) == -1 )
	{
		system_error( "listen" , errno );
	}

	return( ret );
}


//
//  accept
//
int    Stream_Socket::accept( struct sockaddr *  soc_addr ,
			      SOCKLEN_T *  soc_addr_len )
{
	int	ret;

	if ( (ret = ::accept( sfd , soc_addr , soc_addr_len ) ) == -1 )
	{
		system_error( "accept" , errno );
	}

	return( ret );
}


#ifdef HAVE_SOCKADDR_UN
int    Stream_Socket::accept( struct sockaddr_un *  soc_addr_un )
{
	int		ret;
	SOCKLEN_T	size = sizeof(struct sockaddr_un);

	if ( (ret = ::accept( sfd , (struct sockaddr *)soc_addr_un ,
				    &size )) == -1 )
	{
		system_error( "accept" , errno );
	}

#if 0
	if ( size != sizeof(soc_addr_un) )
	{
		return( -1 );
	}
#endif

	return( ret );
}
#endif


int    Stream_Socket::accept( struct sockaddr_in *  soc_addr_in )
{
	int		ret;
	SOCKLEN_T	size = sizeof(struct sockaddr_in);

	if ( (ret = ::accept( sfd , (struct sockaddr *)soc_addr_in ,
				    &size )) == -1 )
	{
		system_error( "accept" , errno );
	}

#if 0
	if ( size != sizeof(soc_addr_in) )
	{
		return( -1 );
	}
#endif

	return( ret );
}


//
//  server socket
//
int    Stream_Socket::make_server_socket( port_number_t  port ,
					  const char * hostname ,
					  bool  ipv4_only )
{
	this -> ipv4_only = ipv4_only;

#if ! defined(HAVE_ADDRINFO)
	if ( this -> socket( AF_INET , SOCK_STREAM , 0 ) == -1
	  || this -> bind( this -> port ) == -1 )
	{
		return( -1 );
	}

	return this -> socket.listen( SOMAXCONN );
#else
	if ( this -> ipv4_only )
	{
		if ( this -> socket( AF_INET , SOCK_STREAM , 0 ) == -1
		  || this -> bind( port ) == -1 )
		{
		    return( -1 );
		}

		return this -> listen( SOMAXCONN );
	}

	struct addrinfo	   hints;
	struct addrinfo *  address_set = static_cast<struct addrinfo *>(0);

	std::memset( &hints , 0 , sizeof(hints) );
#if 0
	hints.ai_family   = AF_UNSPEC;
#else
        // XXX: IPv6 and IPv4 only
	hints.ai_family   = AF_INET6; // accecpts both of IPv6 and IPv4
#endif
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags    = AI_PASSIVE;

	std::string	port_string;
	String_Extension::long_to_string( &port_string , port );

	int	err;

	if ( (err = ::getaddrinfo( hostname ,
				   port_string.c_str() ,
				   &hints , &address_set )) != 0 )
	{
		(*error_stream) << gai_strerror( err ) << std::endl;

		if ( address_set )
		{
			::freeaddrinfo( address_set );
		}

		return( -1 );
	}


	bool		found = false;
	const char *	cause = "unknown";

	for ( struct addrinfo *  addr = address_set;
	      addr != static_cast<struct addrinfo *>(0)  ;
	      addr = addr -> ai_next )
	{
		if ( this -> socket( addr -> ai_family ,
				     addr -> ai_socktype ,
				     addr -> ai_protocol ) == -1 )
		{
			cause = "socket";
			continue;
		}

		if ( this -> bind( addr -> ai_addr, addr -> ai_addrlen ) == -1 )
		{
			cause = "bind";
			return( -1 );
		}

		if ( this -> listen( SOMAXCONN ) == -1 )
		{
			cause = "listen";
			return( -1 );
		}

		found = true;

                // XXX: bind first address only
		break;
	}

	if ( ! found )
	{
		(*error_stream) << cause << ": "
				<< "Can't create server socket" << std::endl;
	}

	::freeaddrinfo( address_set );

	return( (found ? 0 : -1) );
#endif
}



//
//  make_connection
//
int    Stream_Socket::make_connection( const std::string &  host ,
				       port_number_t  port ,
				       bool  ipv4_only )
{
	this -> ipv4_only = ipv4_only;

#if ! defined(HAVE_ADDRINFO)
	if ( this -> socket( AF_INET , SOCK_STREAM , 0 ) == -1 )
	{
		return( -1 );
	}

	return( this -> connect( host , port ) );
#else
	if ( this -> ipv4_only )
	{
		if ( this -> socket( AF_INET , SOCK_STREAM , 0 ) == -1 )
		{
			return( -1 );
		}

		return( this -> connect( host , port ) );
	}


	struct addrinfo	   hints;
	struct addrinfo *  address_set = static_cast<struct addrinfo *>(0);

	std::memset( &hints , 0 , sizeof(hints) );
	hints.ai_family   = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;

	std::string	port_string;
	String_Extension::long_to_string( &port_string , port );

	int	err;

	if ( (err = ::getaddrinfo( host.c_str() ,
				   port_string.c_str() ,
				   &hints , &address_set )) != 0 )
	{
		(*error_stream) << gai_strerror( err ) << std::endl;

		if ( address_set )
		{
			::freeaddrinfo( address_set );
		}

		return( -1 );
	}


	bool		found = false;
	const char *	cause = "unknown";

	for ( struct addrinfo *  addr = address_set;
	      addr != static_cast<struct addrinfo *>(0)  ;
	      addr = addr -> ai_next )
	{
		char	address_name[NI_MAXHOST];

		if ( ::getnameinfo
		       ( addr -> ai_addr ,
			 static_cast<SOCKLEN_T>(addr -> ai_addrlen) ,
			 address_name , sizeof(address_name) ,
			 static_cast<char *>(0) , 0 ,
			 NI_NUMERICHOST ) != 0 )
		{
			system_error( "getnameinfo" , errno );
			cause = "getnameinfo";
			continue;
		}

		if ( this -> socket( addr -> ai_family ,
				     addr -> ai_socktype ,
				     addr -> ai_protocol ) == -1 )
		{
			cause = "socket";
			continue;
		}

		if ( this -> connect
		     ( addr -> ai_addr ,
		       static_cast<SOCKLEN_T>(addr -> ai_addrlen) ,
		       address_name ) == -1 )
		{
			cause = "connect";

			this -> close();

			continue;
		}

		found = true;
		break;
	}

	if ( ! found )
	{
		(*error_stream) << cause << ": "
				<< "Can't connect to " << host.c_str()
				<< std::endl;
	}

	::freeaddrinfo( address_set );

	return( (found ? 0 : -1) );
#endif
}


//
//  read
//
ssize_t Stream_Socket::read( void *  p ,  size_t  siz )
{
	ssize_t	ret;

	if ( (ret = ::recv( sfd , p , siz , 0 )) == -1 )
	{
		system_error( "recv" , errno );
	}

	if ( ret == 0 || ret == -1 )
	{
		this -> input_end_of_file = true;
	}

	return( ret );
}

ssize_t Stream_Socket::read_all( std::string *  str )
{
	char	buf[ BUFSIZ /* defined in <cstdio> */ ];

	*str = "";

	for(;;)
	{
		ssize_t	n;

		if ( (n = this -> read( buf , sizeof(buf) )) == -1 )
		{
			this -> input_end_of_file = true;
			return( -1 );
		}

		str -> append( buf , n );

		if ( n == 0 )
		{
			break;
		}
	}

	this -> input_end_of_file = true;

	return( static_cast<ssize_t>( str -> size() ) );
}


//
//  write
//
ssize_t Stream_Socket::write( const void *  p ,  size_t  siz )
{
	ssize_t	ret;

	if ( (ret = ::send( sfd , p , siz , 0 )) == -1 )
	{
		system_error( "send" , errno );
	}

	if ( ret == 0 || ret == -1 )
	{
		this -> output_end_of_file = true;
	}

	return( ret );
}

ssize_t Stream_Socket::write( const std::string &  str )
{
	return( this -> write( str.data() , str.size() ) );
}


//
//  write_all
//
ssize_t Stream_Socket::write_all( const void *  p ,  size_t  siz )
{
	size_t	n_wrote = 0;

	while( n_wrote < siz )
	{
		int	n;

		if ( (n = ::send( sfd , p , siz , 0 )) == -1 )
		{
			system_error( "send" , errno );

			return( -1 );
		}

		n_wrote += n;
	}

	this -> output_end_of_file = true;

	return( n_wrote );
}

ssize_t Stream_Socket::write_all( const std::string &  str )
{
	return( this -> write_all( str.data() , str.size() ) );
}


//
//  recv
//
ssize_t Stream_Socket::recv( void *  p ,  size_t  siz ,  int  flags )
{
	int	ret;

	if ( (ret = ::recv( sfd , p , siz , flags )) == -1 )
	{
		system_error( "recv" , errno );
	}

	if ( ret == 0 || ret == -1 )
	{
		this -> input_end_of_file = true;
	}

	return( ret );
}


//
//  send
//
ssize_t Stream_Socket::send( const void *  p ,  size_t  siz ,  int  flags )
{
	int	ret;

	if ( (ret = ::send( sfd , p , siz , flags )) == -1 )
	{
		system_error( "send" , errno );

		this -> input_end_of_file = true;
	}

	if ( ret == 0 || ret == -1 )
	{
		this -> output_end_of_file = true;
	}

	return( ret );
}


//
//  shutdown
//
int    Stream_Socket::shutdown( int  how )
{
	int	ret;

	if ( (ret = ::shutdown( sfd , how )) == -1 )
	{
		system_error( "shutdown" , errno );
	}

	if ( how == SHUT_RD || how == SHUT_RDWR )
	{
		this -> input_end_of_file = true;
	}

	if ( how == SHUT_WR || how == SHUT_RDWR )
	{
		this -> output_end_of_file = true;
	}

	return( ret );
}


//
// close on exec
//
int    Stream_Socket::set_close_on_exec()
{
#ifdef HAVE_FCNTL
	if ( ::fcntl( sfd , F_SETFD , FD_CLOEXEC ) == -1 )
	{
		std::perror( "fcntl" );
		sfd = -1;
		this -> close();

		return( -1 );
	}

	return( 0 );
#else
	return( -1 );
#endif
}


//
//  poll
//
int    Stream_Socket::poll( int  timeout /* msec */ ) const
{
	if ( sfd == BAD_SOCKET_FD || input_end_of_file )
	{
		return( 0 );
	}


	int	ret;
	if ( (ret = System_Call_Wrapper::poll( this -> sfd , timeout )) == -1 )
	{
		system_error( "select" , errno );

		return( 0 );
	}

	return( ret );
}


//
//  fd
//
int    Stream_Socket::fd() const
{
	return( sfd );
}


//
//  valid
//
bool   Stream_Socket::valid() const
{
	return( sfd != BAD_SOCKET_FD );
}

//
//  operator bool
//
Stream_Socket::operator bool() const
{
	return( this -> valid() );
}

//
// input eof
//
bool   Stream_Socket::input_eof() const
{
	return( this -> input_end_of_file || ! this -> valid() );
}

//
// output eof
//
bool   Stream_Socket::output_eof() const
{
	return( this -> output_end_of_file || ! this -> valid() );
}

//
// connected
//
bool   Stream_Socket::connected() const
{
	return( this -> valid() && this -> connected_flag );
}
