package darumaDump;

import mispClient.MispClient;
import mispClient.MispCommandException;
import mispClient.commands.GetCapabilitiesMispCommand;
import mispClient.commands.DescribeFeatureTypeMispCommand;
import mispClient.commands.GetFeatureDataOnlyMispCommand;

import util.FilePathConverter;

import daruma.storage_manager.type_definition.ElementName;
import daruma.xml.URI;
import daruma.xml.UniversalName;

import daruma.util.CharacerCodeConstant;
import daruma.util.FatalException;

import java.io.OutputStream;
import java.io.PrintStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.IOException;
import java.net.ConnectException;
import java.util.List;
import java.util.ArrayList;


public class DarumaDump
{
	private	static final String	PROGRAM_NAME      = "DarumaDump";

	private	static final String	DEFAULT_MISP_HOST = "localhost";
	private	static final int	DEFAULT_MISP_PORT = 5050;

	private	static final String	DEFAULT_SCHEMA_DUMP_FILE = "schema.xml";

	private	String		host;
	private	int		port;
	private	OutputStream	dataOutputStream;
	private	OutputStream	schemaOutputStream;
	private	String		featureDumpDir;
	private	UniversalName	specifiedFeatureName;
	private boolean		debugFlag;

	public	DarumaDump( String  host ,  int  port ,
			    OutputStream  dataOutputStream ,
			    OutputStream  schemaOutputStream ,
			    String featureDumpDir,
			    UniversalName specifiedFeatureName ,
			    boolean debugFlag )
	{
		this.host = host;
		this.port = port;
		this.dataOutputStream = dataOutputStream;
		this.schemaOutputStream = schemaOutputStream;
		this.featureDumpDir = featureDumpDir;
		this.specifiedFeatureName = specifiedFeatureName;
		this.debugFlag = debugFlag;
	}

	public	void	run() throws FatalException
	{
		try
		{
			MispClient	client = new MispClient( this.host ,
								 this.port );

			//
			// GetCapabilities
			//
			GetCapabilitiesMispCommand	getCapabilities;
			getCapabilities = new GetCapabilitiesMispCommand();

			client.issue( getCapabilities );

			List<ElementName>  features;

			if ( this.specifiedFeatureName != null )
			{
				features = new ArrayList<ElementName>();
				features.add( new ElementName
					      ( this.specifiedFeatureName ) );
			}
			else
			{
				features = getCapabilities.getFeatureList();
			}

			if ( this.debugFlag )
			{
				System.err.println( "=== feature list ===" );
				for ( ElementName f : features )
				{
					System.err.println( f );
				}
				System.err.println( "====================" );
			}


			//
			// DescribeFeatureType
			//
			if ( this.schemaOutputStream != null )
			{
				this.outputRegisterFeatureType
					( client ,
					  this.schemaOutputStream ,
					  features );
			}


			//
			// GetFeature
			//
			if ( this.featureDumpDir == null )
			{
				this.outputAllFeature( client ,
						       this.dataOutputStream ,
						       features );
			}
			else
			{
			    final File dir = new File( this.featureDumpDir );

			    if ( ! dir.exists() )
			    {
				if ( ! dir.mkdirs() )
				{
				    throw new FatalException
					( "Can't create directory ["
					  + this.featureDumpDir + "]." );
				}
			    }
			    else
			    {
				if ( ! dir.isDirectory() )
				{
				    throw new FatalException
					( "[" + this.featureDumpDir + "]"
					  + " is not a directory." );
				}
			    }

			    for( ElementName  f : features )
			    {
				System.err.print( f + " ...");

				OutputStream file
				    = new FileOutputStream
				      ( new File
					( this.featureDumpDir,
					  FilePathConverter.escapePathString
					  ( f.toString() + ".misp" ) ) );
				this.outputFeature( client , file , f );
				file.close();
				System.err.println( " done");
			    }
			}
		}
		catch( MispCommandException  e )
		{
			System.err.println( "command error: "
					    + e.getMessage() );
			System.exit( 1 );
		}
		catch( ConnectException  e )
		{
			System.err.println( "connection error: "
					    + e.getMessage() );
			System.exit( 1 );
		}
		catch( IOException  e )
		{
			e.printStackTrace();
			System.exit( 1 );
		}
	}


	private	static	void	outputFeature
					( MispClient  client ,
					  OutputStream  dataOutputStream ,
					  ElementName  feature )
						throws MispCommandException ,
						       ConnectException ,
						       IOException
	{
		PrintWriter  out = new PrintWriter
				   ( new BufferedWriter
				     ( new OutputStreamWriter
				       ( dataOutputStream ,
					 CharacerCodeConstant.JAVA_UTF8 ) ) );

		out.println( "<misp:Transaction xmlns:misp=\""
						    + URI.MISP + "\">" );
		out.println( "  <misp:Insert>" );
		out.flush();

		client.issue( new GetFeatureDataOnlyMispCommand
			      ( feature ,
				dataOutputStream ,
				CharacerCodeConstant.JAVA_UTF8 ) );

		out.println( "  </misp:Insert>" );
		out.println( "</misp:Transaction>" );
		out.flush();
		out.close();
	}

	private	static	void	outputAllFeature
					( MispClient  client ,
					  OutputStream  dataOutputStream ,
					  List<ElementName>  features )
						throws MispCommandException ,
						       ConnectException ,
						       IOException
	{
		PrintWriter  out = new PrintWriter
				   ( new BufferedWriter
				     ( new OutputStreamWriter
				       ( dataOutputStream ,
					 CharacerCodeConstant.JAVA_UTF8 ) ) );

		out.println( "<misp:Transaction xmlns:misp=\""
						    + URI.MISP + "\">" );
		out.println( "  <misp:Insert>" );
		out.flush();

		for( ElementName  f : features )
		{
			client.issue( new GetFeatureDataOnlyMispCommand
				      ( f ,
					dataOutputStream ,
					CharacerCodeConstant.JAVA_UTF8 ) );
		}

		out.println( "  </misp:Insert>" );
		out.println( "</misp:Transaction>" );
		out.flush();
		out.close();
	}

	private	static	void	outputRegisterFeatureType
				( MispClient  client ,
				  OutputStream  dataOutputStream ,
				  List<ElementName>  features )
						throws MispCommandException ,
						       ConnectException ,
						       IOException
	{
		PrintWriter  out = new PrintWriter
				   ( new BufferedWriter
				     ( new OutputStreamWriter
				       ( dataOutputStream ,
					 CharacerCodeConstant.JAVA_UTF8 ) ) );

		out.println( "<misp:RegisterFeatureType xmlns:misp=\""
						    + URI.MISP + "\">" );
		out.println();
		out.flush();

		DescribeFeatureTypeMispCommand
		  describeFeatureType
		    = new DescribeFeatureTypeMispCommand
			( features ,
			  dataOutputStream ,
			  CharacerCodeConstant.JAVA_UTF8 );

		client.issue( describeFeatureType );

		out.println();
		out.println( "</misp:RegisterFeatureType>" );
		out.flush();
	}


	public	static	void  main( String  args[] )
	{
		DarumaDumpOptionAnalyzer	opt;
		opt = new DarumaDumpOptionAnalyzer
			( args , PROGRAM_NAME ,
			  DEFAULT_MISP_HOST , DEFAULT_MISP_PORT ,
			  DEFAULT_SCHEMA_DUMP_FILE );

		if ( ! opt.analyze() )
		{
			opt.printHelp( System.err );

			System.exit( 1 );
		}
		else if ( opt.getHelp() )
		{
			opt.printHelp( System.out );

			System.exit( 0 );
		}

		OutputStream	schemaOutputStream = null;

		try
		{
			schemaOutputStream = new FileOutputStream
						 ( opt.getSchemaDumpFile() );
		}
		catch( FileNotFoundException  e )
		{
			System.err.println( e.getMessage() );

			System.exit( 1 );
		}


		DarumaDump	darumaDump;

		darumaDump = new DarumaDump( opt.getHost() ,
					     opt.getPort() ,
					     System.out ,
					     schemaOutputStream,
					     opt.getFeatureDumpDir(),
					     opt.getFeatureName(),
					     opt.getDebug() );
		try
		{
			darumaDump.run();
		}
		catch( FatalException e )
		{
			System.err.println( e.getMessage() );
			System.exit( 1 );
		}

		try
		{
			if ( schemaOutputStream != null )
			{
				schemaOutputStream.flush();
				schemaOutputStream.close();
			}
		}
		catch( IOException  e )
		{
			System.err.println( e.getMessage() );
		}
	}


	public static class  DarumaDumpOptionAnalyzer
	{
		private	String[]	args;
		private	String		programName;
		private	String		host;
		private	int		port;
		private UniversalName	featureName = null;

		private	boolean	helpFlag  = false;
		private	boolean	debugFlag = false;

		private	String	schemaDumpFile = null;
		private	String	featureDumpDir = null;

		private	String	defaultSchemaDumpFile = null;

		public	DarumaDumpOptionAnalyzer
			( final String[]  args ,
			  final String  programName ,
			  final String  defaultMispHost ,
			  final int  defaultMispPort ,
			  final String  defaultSchemaDumpFile )
		{
			this.args = args.clone();
			this.programName = programName;
			this.host = defaultMispHost;
			this.port = defaultMispPort;
			this.featureName = null;
			this.schemaDumpFile = defaultSchemaDumpFile;
			this.featureDumpDir = null;

			this.defaultSchemaDumpFile = defaultSchemaDumpFile;
		}

		public	boolean	analyze()
		{
			int	index = 0;
			while( index < args.length )
			{
				final	String	arg = this.args[index];
				int	numArgs = args.length - index - 1;

				if ( arg.equals( "--help" ) )
				{
					this.helpFlag = true;
				}
				else if ( arg.equals( "--debug" ) )
				{
					this.debugFlag = true;
				}
				else if ( arg.equals( "--host" )
				       || arg.equals( "-h" ) )
				{
					if ( numArgs < 1 )
					{
						return( false );
					}

					index ++;

					this.host = this.args[index];
				}
				else if ( arg.equals( "--port" )
				       || arg.equals( "-p" ) )
				{
					if ( numArgs < 1 )
					{
						return( false );
					}

					index ++;

					try
					{
						this.port
						  = Integer.parseInt
						    ( this.args[index] );

					}
					catch( NumberFormatException  e )
					{
						System.err.println
						( "invalid port number: "
						  + this.args[index] );

						return( false );
					}
				}
				else if ( arg.equals( "--schema-dump-file" ) )
				{
					if ( numArgs < 1 )
					{
						return( false );
					}

					index ++;

					this.schemaDumpFile = this.args[index];
				}
				else if ( arg.equals( "--feature-dump-dir" ) )
				{
					if ( numArgs < 1 )
					{
						return( false );
					}

					index ++;

					this.featureDumpDir = this.args[index];
				}
				else if ( arg.equals( "--feature" ) )
				{
					if ( numArgs < 2 )
					{
						return( false );
					}

					this.featureName
					    = new UniversalName
					          ( this.args[index + 1],
						    this.args[index + 2] );

					index += 2;
				}
				else
				{
					System.err.println
						( "invalid option: " + arg );

					return( false );
				}

				index ++;
			}

			return( this.check() );
		}

		public	boolean	check()
		{
			/*
			if ( this.schemaDumpFile == null )
			{
				System.err.println
					( "schema file not specified" );

				return( false );
			}
			*/

			return( true );
		}

		public	void	printHelp( OutputStream  out )
		{
			String	mes = "";

			mes += "Usage: " + this.programName
			       + " [--schema-dump-file FILE]"
			       + " [options] > DUMP_FILE" + "\n";
			mes += "Possible options are:"    + "\n";
			mes += "      --help"             + "\n";
			mes += "      --debug"            + "\n";
			mes += "   -h,--host HOST"        + "\n";
			mes += "   -p,--port PORT"        + "\n";
			mes += "      --schema-dump-file FILE"
			    + " (default: " + this.defaultSchemaDumpFile + ")"
			    + "\n";
			mes += "      --feature-dump-dir DIR"
			    + " (default: none, standard output)"
			    + "\n";
			mes += "      --feature NAMESPACE LOCAL_NAME"
			    + " (default: all features)"
			    + "\n";

			PrintStream	p = new PrintStream( out );
			p.print( mes );
		}

		public	boolean	getHelp()
		{
			return( this.helpFlag );
		}

		public	boolean	getDebug()
		{
			return( this.debugFlag );
		}

		public	String	getHost()
		{
			return( this.host );
		}

		public	int	getPort()
		{
			return( this.port );
		}

		public	UniversalName	getFeatureName()
		{
			return( this.featureName );
		}

		public	String	getSchemaDumpFile()
		{
			return( this.schemaDumpFile );
		}

		public	String	getFeatureDumpDir()
		{
			return this.featureDumpDir;
		}
	}
}
