/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import net.morilib.sh.misc.FileTypeUtils;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2013
 */
public class ShRealFile extends AbstractShFile {

	/**
	 * 
	 */
	protected File file;

	/**
	 * 
	 * @param fs
	 * @param virtualpath
	 * @param file
	 */
	public ShRealFile(ShFileSystem fs, String virtualpath, File file) {
		super(fs, virtualpath);
		this.file = file;
	}

	/**
	 * 
	 * @return
	 */
	public File getRealFile() {
		return file;
	}

	public Collection<ShFile> getFiles() {
		SortedSet<String> z;
		List<ShFile> l;
		String[] a;
		ShFile f;

		if((a = file.list()) == null) {
			return Collections.emptyList();
		} else {
			l = new ArrayList<ShFile>();
			z = new TreeSet<String>(Arrays.asList(a));
			for(String s : z) {
				f = filesystem.getFile(virtualpath, s);
				l.add(f);
			}
			return l;
		}
	}

	public InputStream getInputStream() throws IOException {
		return new FileInputStream(file);
	}

	public String getName() {
		return file.getName();
	}

	public PrintStream getPrintStream(
			boolean append) throws IOException {
		return new PrintStream(new FileOutputStream(file, append));
	}

	public boolean isDirectory() {
		return file.isDirectory();
	}

	public boolean isExecutable() {
		if(file.isDirectory()) {
			return true;
		} else if(file.isFile()) {
			return FileTypeUtils.parseShebang(this) != null;
		} else {
			return false;
		}
	}

	public boolean isExist() {
		return file.exists();
	}

	public boolean isFile() {
		return file.isFile();
	}

	public long getLastModified() {
		return file.lastModified();
	}

	public boolean isReadable() {
		return file.canRead();
	}

	public boolean isWritable() {
		return file.canWrite();
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFile#isHidden()
	 */
	public boolean isHidden() {
		return file.isHidden();
	}

	public boolean isZeroFile() {
		return file.length() == 0;
	}

	public boolean mkdir() {
		return file.mkdir();
	}

	public boolean renameTo(ShFile f) {
		return false;
	}

	public void setLastModified(long time) {
		file.setLastModified(time);
	}

	public boolean rmdir() {
		return file.isDirectory() && file.delete();
	}

	public boolean deleteFile() {
		return file.isFile() && file.delete();
	}

	/**
	 * 
	 * @param t
	 * @param virtualpath
	 * @param file
	 * @return
	 */
	public static ShStat getStat(ShFile t, String virtualpath,
			File file) {
		ShStat r = new ShStat();
		int f = 0;

		r.setFilename(virtualpath);
		r.setMtime(file.lastModified());
		r.setSize(file.length());
		f |= t.isDirectory()  ? ShStat.DIRECTORY : 0;
		f |= t.isExecutable() ? ShStat.EXECUTABLE : 0;
		f |= t.isWritable()   ? ShStat.WRITABLE : 0;
		f |= t.isReadable()   ? ShStat.READABLE : 0;
		r.setFlags(f);
		return r;
	}

	public ShStat getStat() {
		return getStat(this, virtualpath, file);
	}

	public ShProcess getProcess() {
		return null;
	}

	public String toString() {
		return virtualpath;
	}

}
