/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.deculture.translate;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import net.morilib.automata.lr.ContextFreeGrammar;
import net.morilib.automata.lr.ContextFreeRule;
import net.morilib.automata.lr.LR1Table;
import net.morilib.automata.lr.LRConflict;
import net.morilib.deculture.cmd.DecultureOptions;
import net.morilib.deculture.sh.DecultureFileSystem;
import net.morilib.sh.DefaultShRuntime;
import net.morilib.sh.ShBuiltInCommands;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFacade;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShRootEnvironment;
import net.morilib.sh.ShRuntime;
import net.morilib.sh.ShSyntaxException;
import net.morilib.sh.misc.XtraceStream;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2014/01/15
 */
public abstract class AbstractDecultureTranslator
implements DecultureTranslator {

	//
	protected static final String __ERROR = "";

	//
	protected LRObject lr;
	protected Map<ContextFreeRule, Object> actionMap;
	protected Map<Object, Integer> nonterminalMap =
			new HashMap<Object, Integer>();
	private int nonterminalNumber = 0;

	//
	AbstractDecultureTranslator(LRObject lr) {
		this.lr = lr;
		actionMap = new HashMap<ContextFreeRule, Object>(
				lr.getActionMap());
	}

	/**
	 * 
	 * @param o
	 * @return
	 */
	protected int getNonterminalNumber(Object o) {
		if(nonterminalMap.containsKey(o)) {
			return nonterminalMap.get(o);
		} else {
			nonterminalMap.put(o, nonterminalNumber);
			return nonterminalNumber++;
		}
	}

	/**
	 * 
	 * @return
	 */
	public String getDefinition() {
		return lr.getDefinition().getDefinition();
	}

	/**
	 * 
	 * @return
	 */
	public ContextFreeGrammar getGrammar() {
		return lr.getGrammar();
	}

	/**
	 * 
	 * @return
	 */
	public LR1Table getTable() {
		return lr.getTable();
	}

	/**
	 * 
	 * @return
	 */
	public Collection<LRConflict> getConflicts() {
		return lr.getTable().getConflicts();
	}

	/**
	 * 
	 * @param rule
	 * @return
	 */
	public Object getAction(ContextFreeRule rule) {
		return actionMap.get(rule);
	}

	/**
	 * 
	 * @param rule
	 * @param action
	 */
	public void setAction(ContextFreeRule rule, Object action) {
		actionMap.put(rule, action);
	}

	/**
	 * 
	 * @param rule
	 */
	public void removeAction(ContextFreeRule rule) {
		actionMap.remove(rule);
	}

	/**
	 * 
	 * @return
	 */
	protected abstract ShBuiltInCommands getCommands();

	/**
	 * 
	 * @return
	 */
	protected abstract String getResourceName();

	/**
	 * 
	 * @param options
	 * @return
	 */
	public ShEnvironment getenv(DecultureOptions options) {
		ShEnvironment env;
		char[] a;

		env  = new ShRootEnvironment();
		a    = options.getOutputFilename().toCharArray();
		a[0] = Character.toUpperCase(a[0]);
		env.bind("IFS", " \t");
		env.bind("CLASSNAME", new String(a));
		env.bind("FILENAME", options.getFilename());
		env.bind("OUTPUT_FILENAME", options.getOutputFilename());
		env.bind("ERROR_CLASS", "RuntimeException");
		env.bind("INIT_STATE_ID", lr.getTable().getInitialStateID() + "");
		return env;
	}

	/**
	 * 
	 * @param ous
	 * @param options
	 * @throws IOException 
	 */
	public void translate(PrintStream ous,
			DecultureOptions options) throws IOException {
		InputStream ins = null;
		ShBuiltInCommands cmd;
		ShEnvironment env;
		ShFileSystem fs;
		XtraceStream qs;
		ShRuntime run;

		try {
			ins = LRObject.class.getResourceAsStream(getResourceName());
			cmd = getCommands();
			fs  = new DecultureFileSystem(lr.getDefinition().getDefinition(),
					lr.getInput());
			run = new DefaultShRuntime(cmd);
			qs  = new XtraceStream(System.out);
			env = getenv(options);
			ShFacade.execute(env, fs, cmd, run, ins, System.in, ous,
					System.err, qs);
		} catch(ShSyntaxException e) {
			throw new RuntimeException(e);
		} finally {
			if(ins != null)  ins.close();
		}
	}

	/**
	 * 
	 * @return
	 */
	public int howManyShiftReduceConflicts() {
		int r = 0;

		for(LRConflict t : lr.getTable().getConflicts()) {
			r += t.isShiftReduce() ? 1 : 0;
		}
		return r;
	}

	/**
	 * 
	 * @return
	 */
	public int howManyReduceReduceConflicts() {
		int r = 0;

		for(LRConflict t : lr.getTable().getConflicts()) {
			r += t.isReduceReduce() ? 1 : 0;
		}
		return r;
	}

}
