#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <dirent.h>
#include <sys/stat.h>
#include <pthread.h>
#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>
#include <X11/Xaw/Simple.h>

#include "config.h"
#include "dftype.h"
#include "str.h"
#include "list.h"
#include "buffer.h"
#include "buflist.h"
#include "filer.h"
#include "task.h"
#include "info.h"
#include "status.h"
#include "cmd.h"
#include "dfxfunc.h"
#include "mem.h"
#include "xutil.h"
#include "misc.h"
#include "dfval.h"
#include "debug.h"

static int isSizable(DfBuf *b);

enum adjoin_side{
  NONE_SIDE  = 0,
  UP_SIDE    = 1,
  DOWN_SIDE  = 2,
  LEFT_SIDE  = 4,
  RIGHT_SIDE = 8,
};

void *MakeBuffer(size_t sz, DfBuf *parent, df_type type, const DfVerb2 *v)
{
  DfBuf *ptr;

  ptr = bMalloc(sz);
  if(ptr){
    ptr->type = type;
    ptr->parent = parent;
    if(parent){
      ptr->stat = parent->stat & ~(BS_VISIBLE | BS_ENABLE);
      ptr->rc = parent->rc;
    }
    ptr->active = NULL;
    ptr->v = (DfVerb*)v;
    ListClear(&ptr->list);
    ListClear(&ptr->focus);
  }

  return ptr;
}

void DfBufCommon_Init(DfBufCommon *cb)
{
  Str_InitNull(&cb->caption);
  cb->nStart = 0;
  cb->nOffset = 0;
  cb->nCur = 0;
  cb->nItems = 0;
}

void BufComon_Free(DfBufCommon *b)
{
  Str_Free(&b->caption, 0);
}

void *NextBuffer(void *ptr)
{
  DfBuf *buf;

  buf = ptr;

  return CONTENT_OF(DfBuf, list, ListNext(&buf->list));
}

void *PrevBuffer(void *ptr)
{
  DfBuf *buf;

  buf = ptr;

  return CONTENT_OF(DfBuf, list, ListPrev(&buf->list));
}

void *NextFrame(void *ptr)
{
  DfBuf *buf = ptr;

  return CONTENT_OF(DfBuf, focus, ListNext(&buf->focus));
}

void *PrevFrame(void *ptr)
{

  DfBuf *buf = ptr;

  return CONTENT_OF(DfBuf, focus, ListPrev(&buf->focus));
}

int IsSingleBuffer(void)
{
  return dfx->nBuf == 1;
}


int RectIsOverlap(XRectangle *a, XRectangle *b)
{
  int a_w;
  int b_w;
  int a_h;
  int b_h;

  a_w = a->x + a->width;
  b_w = b->x + b->width;
  a_h = a->y + a->height;
  b_h = b->y + b->height;

  if(a->x < b_w && b->x < a_w && a->y < b_h && b->y < a_h){
    return 1;
  }
  return 0;
}

void RectAdd(XRectangle *a, XRectangle *b)
{
  int x, y, cx, cy;
  int a_w;
  int b_w;
  int a_h;
  int b_h;

  a_w = a->x + a->width;
  b_w = b->x + b->width;
  a_h = a->y + a->height;
  b_h = b->y + b->height;

  if(a->x < b->x){
    x = a->x;
  }else{
    x = b->x;
  }

  if(a->y < b->y){
    y = a->y;
  }else{
    y = b->y;
  }

  cx = a_w;
  if(a_w < b_w){
    cx = b_w;
  }
  cy = a_h;
  if(a_h < b_h){
    cy = b_h;
  }

  a->x = x;
  a->y = y;
  a->width = cx - x;
  a->height = cy - y;
}

static int CheckEdge(short int x, unsigned short int x_len, short int y, unsigned short int y_len)
{
  int s;
  int n;
  int m;

  n = x + x_len;
  m = y + y_len;
  
  s = 0;
  if(y <= x && x <= m){
    s |= 1;
  }
  if(y <= n && n <= m){
    s |= 2;
  }
  return s;
}

void RectSub(XRectangle *a, XRectangle *b)
{
  int i;
  int j;
  int cx;
  int cy;

  i = CheckEdge(a->x, a->width, b->x, b->width);
  j = CheckEdge(a->y, a->height, b->y, b->height);

  switch(i<<4 | j){
  case 0x13:
    cx = a->x + a->width;
    a->x = b->x + b->width;
    a->width = cx - a->x;
    break;
  case 0x23:
    a->width = b->x - a->x;
    break;
  case 0x31:
    cy = a->y + a->height;
    a->y = b->y + b->height;
    a->height = cy - a->y;
    break;
  case 0x32:
    a->height = b->y - a->y;
    break;
  case 0x33:
    a->width = 0;
    a->height = 0;
    break;
  }
}

void RectIntersect(XRectangle *a, XRectangle *b)
{
  int a_cx;
  int a_cy;
  int b_cx;
  int b_cy;

  if(!RectIsOverlap(a, b)){
    return;
  }

  a_cx = a->x + a->width;
  a_cy = a->y + a->height;
  b_cx = b->x + b->width;
  b_cy = b->y + b->height;

  if(a->x < b->x){
    a->x = b->x;
  }
  if(a->y < b->y){
    a->y = b->y;
  }

  if(a_cx < b_cx){
    a->width = a_cx - a->x;
  }else{
    a->width = b_cx - a->x;
  }
  if(a_cy < b_cy){
    a->height = a_cy - a->y;
  }else{
    a->height = b_cy - a->y;
  }
}


void RectPos(XRectangle *a, const XRectangle *b)
{
  if(a != b){
    a->x = b->x;
    a->y = b->y;
  }
  a->width = b->x + b->width;
  a->height = b->y + b->height;
}

void RectSizeTop(XRectangle *a, int cy)
{
  a->y -= cy;
  a->height += cy;
}

void RectSizeBottom(XRectangle *a, int cy)
{
  a->height += cy;
}

void RectSizeLeft(XRectangle *a, int cx)
{
  a->x -= cx;
  a->width += cx;
}

void RectSizeRight(XRectangle *a, int cx)
{
  a->width += cx;
}

static enum adjoin_side IsAdjoin(const XRectangle *a, const XRectangle *b)
{
  enum adjoin_side join;
  XRectangle c;
  XRectangle d;

  RectPos(&c, a);
  RectPos(&d, b);

  join = NONE_SIDE;
  if(c.x <= d.width && d.x <= c.width){
    if(c.y == d.height){
      /* upper */
      join |= UP_SIDE;
    }
    if(c.height == d.y){
      /* lower */
      join |= DOWN_SIDE;
    }
  }
  if(c.y <= d.height && d.y <= c.height){
    if(c.x == d.width){
      /* left */
      join |= LEFT_SIDE;
    }
    if(c.width == d.x){
      /* right */
      join |= RIGHT_SIDE;
    }
  }
  return join;
}
 

void GetChangeSize(XRectangle *r, const XRectangle *before, const XRectangle *after)
{
  r->x = before->x - after->x;
  r->y = before->y - after->y;
  r->width = before->width - after->width;
  r->height = before->height - after->height;
}

void PutChangeSize(XRectangle *r, const XRectangle *before, const XRectangle *after)
{
  XRectangle a;
  XRectangle b;

  RectPos(&a, after);
  RectPos(&b, before);
  
  r->x = b.x + a.x;
  r->y = b.y + a.y;
  r->width = b.width + a.width;
  r->height = b.height + a.height;
}

void RedrawFrame(Widget w, DfBuf *b, XRectangle *rc)
{
  struct DfDrawInfo di;

  if(!IS_VISIBLE(b)){
    return;
  }
  di.d = XtDisplay(w);
  di.w = w;
  di.gc = GetBufferGC(w, b);
  di.clip = *rc;
  XClearArea(di.d, XtWindow(w),
	     rc->x, rc->y, rc->width, rc->height, False);
  SetClipRect(di.d, di.gc, rc);
  (*b->v->draw)(b, &di);
  SetClipRect(di.d, di.gc, rc);
  DrawBorder(b, di.d, w, di.gc);
}


static int count_adjoin(DfBuf *b, int dir)
{
  DfBuf *x;
  DfBuf *list;
  int count;

  count = 0;
  list = dfx->lists;
  x = list;
  do{
    if(x != b){
      if(IS_VISIBLE(x) && IsAdjoin(&b->rc, &x->rc) == dir){
	count++;
      }
    }
    x = NextBuffer(x);
  }while(x != list);

  return count;
}

static int choise_killing(DfBuf *b)
{
  DfBuf *x;
  DfBuf *c;
  int dir;
  int check;
  enum adjoin_side side;
  int rev;
  int cnt;

  dir = 0;
  check = 0;
  c = NULL;

  for(x = NextBuffer(b); x != b; x = NextBuffer(x)){
    if(!IS_VISIBLE(x)){
      continue;
    }
    if(!isSizable(x)){
      continue;
    }
    side = IsAdjoin(&b->rc, &x->rc);
    if(side == 0){
      continue;
    }
    switch(side){
    case UP_SIDE:
      rev = DOWN_SIDE;
      break;
    case DOWN_SIDE:
      rev = UP_SIDE;
      break;
    case LEFT_SIDE:
      rev = RIGHT_SIDE;
      break;
    case RIGHT_SIDE:
      rev = LEFT_SIDE;
      break;
    default:
      continue;
    }

    if((dir & side) == 0){
      /* count rectangles adjoining lower direction. */
      cnt = count_adjoin(x, rev);
      switch(cnt){
      case 1:
	c = x;
	check |= side;
	break;
      default:
	/* cann't erase this side */
	dir |= side;
	break;
      }
    }
  }

  if(check & 1 && !(dir & 1)){
    return UP_SIDE;
  }
  if(check & 2 && !(dir & 2)){
    return DOWN_SIDE;
  }
  if(check & 4 && !(dir & 4)){
    return LEFT_SIDE;
  }
  if(check & 8 && !(dir & 8)){
    return RIGHT_SIDE;
  }
  return 0;
}


void KillFrame(Widget w, DfBuf *b)
{
  DfBuf *x;
  enum adjoin_side side;
  int t;
  DfBuf *c;

  if(dfx->nFiler <= 1){
    /* must have least 1 filer buffer */
    return;
  }

  c = NULL;
  side = choise_killing(b);
  if(side == 0){
    /* no killing */
    return;
  }

  for(x = NextBuffer(b); x != b; x = NextBuffer(x)){
    if(!IS_VISIBLE(x)){
      continue;
    }
    t = IsAdjoin(&b->rc, &x->rc);
    if(side == t){
      c = x;
      /* extend size */
      switch(side){
      case UP_SIDE:
	x->stat &= ~BS_DRAWBOTTOMEDGE;
	x->stat |= (b->stat & BS_DRAWBOTTOMEDGE);
	x->rc.height += b->rc.height;
	break;
      case DOWN_SIDE:
	x->rc.y = b->rc.y;
	x->rc.height += b->rc.height;
	break;
      case LEFT_SIDE:
	x->stat &= ~BS_DRAWRIGHTEDGE;
	x->stat |= (b->stat & BS_DRAWRIGHTEDGE);
	x->rc.width += b->rc.width;
	break;
      case RIGHT_SIDE:
	x->rc.x = b->rc.x;
	x->rc.width += b->rc.width;
	break;
      }
      if(x->v->resize){
	(*x->v->resize)(w, x);
      }
      SetupDelayedDraw(w, x);
    }
  }

  if(c == NULL && b == dfx->active){
    c = NextFrame(b);
  }
  b->stat &= ~BS_VISIBLE;
  ListDel(&b->focus);
  if(c){
    SetActiveFrame(w, c);
  }
}

static int enum_update(DfBuf *b, void *ptr)
{
  Widget w = *(Widget *)ptr;
  if(b->stat & BS_NEEDREDRAW){
    b->stat &=  ~BS_NEEDREDRAW;
    RedrawFrame(w, b, &b->rc);
  }
  return 0;
}

static int enum_redraw(DfBuf *b, void *ptr)
{
  Widget w = *(Widget *)ptr;
  if(IS_VISIBLE(b)){
    RedrawFrame(w, b, &b->rc);
  }
  return 0;
}


void KillOtherFrames(Widget w, DfBuf *b)
{
  DfBuf *x;
  int stat;
  XRectangle rc;

  if(dfx->nFiler <= 1){
    /* must have least 1 filer buffer */
    return;
  }
  stat = ~0;
  
  rc = b->rc;
  rc.width += rc.x;
  rc.height += rc.y;
  for(x = NextBuffer(b); x != b; x = NextBuffer(x)){
    if(!IS_VISIBLE(x)){
      continue;
    }
    if(!IS_ENABLE(x)){
      continue;
    }
    stat &= x->stat;
    x->stat &= ~BS_VISIBLE;
    ListClear(&x->focus);
    if(x->rc.x <rc.x){
      rc.x = x->rc.x;
    }
    if(x->rc.y < rc.y){
      rc.y = x->rc.y;
    }
    if(rc.width < x->rc.x + x->rc.width){
      rc.width = x->rc.x + x->rc.width;
    }
    if(rc.height < x->rc.y + x->rc.height){
      rc.height = x->rc.y + x->rc.height;
    }
  }

  b->stat &= stat;
  rc.width -= rc.x;
  rc.height -= rc.y;
  b->rc = rc;

  InvalidateRect(w, &x->rc);
}

static int enum_is_enable(DfBuf *b, void *param)
{
  DfBuf **buf = param;
  DfBuf *act;

  if(b->active){
    act = b->active;
    while(act->active){
      act = act->active;
    }
    if(IS_ENABLE(act)){
      *buf = act;
      return 1;
    }
  }
  if(IS_ENABLE(b)){
    *buf = b;
    return 1;
  }
  return 0;
}

static int enum_is_enable_hidden(DfBuf *b, void *param)
{
  DfBuf **buf = param;
  DfBuf *act;

  if(b->active){
    act = b->active;
    while(act->active){
      act = act->active;
    }
    if(IS_ENABLE(act) && !IS_VISIBLE(act)){
      *buf = act;
      return 1;
    }
  }
  if(IS_ENABLE(b) && !IS_VISIBLE(b)){
    *buf = b;
    return 1;
  }
  return 0;
}

int EnumBuffers(DfBuf *top, int (*callback)(DfBuf*, void*), void *param)
{
  DfBuf *cur = top;

  do{
    if(IS_ENABLE(cur)){
      if((*callback)(cur, param)){
	break;
      }
    }
    cur = NextBuffer(cur);
  }while(top != cur);

  return 0;
}


void UpdateFrames(Widget w, DfBuf *b)
{
  EnumBuffers(b, enum_update, &w);
}

void RedrawFrames(Widget w, DfBuf *b)
{
  EnumBuffers(b, enum_redraw, &w);
}


DfBuf *GetNextEnableHiddenBuffer(DfBuf *cur)
{
  DfBuf *b = cur;

  EnumBuffers(NextBuffer(cur), enum_is_enable_hidden, &b);
  return b;
}


DfBuf *GetNextEnableBuffer(DfBuf *cur)
{
  DfBuf *b = cur;

  EnumBuffers(NextBuffer(cur), enum_is_enable, &b);
  return b;
}

void SwitchBuffer(Widget w, DfBuf *b, DfBuf *org)
{
#ifdef DEBUG
  assert(!IS_VISIBLE(b));
#endif
  org->stat &= ~BS_VISIBLE;

  /* adjust order */
  SetFocusLink(org, b);

  b->stat = org->stat | (BS_VISIBLE | BS_ENABLE);
  b->rc = org->rc;

  dfx->active = b;
  (*b->v->activate)(w, b);
  (*b->v->resize)(w, b);
  RedrawFrame(w, b, &b->rc);
}


void SetActiveFrame(Widget w, DfBuf *b)
{
  DfBuf *ina;

  if(!b){
    return;
  }
  ina = dfx->active;
  dfx->active = b;

  if(ina && IS_VISIBLE(ina)){
    ina->v->inactivate(w, ina);
  }

  b->stat |= BS_VISIBLE;
  b->v->activate(w, b);
}

void SetModalBuffer(Widget w, DfBuf *parent, DfBuf *b)
{
  parent->active = b;
  parent->stat &= ~BS_ENABLE;
  SwitchBuffer(w, b, parent);
}

DfBuf *ChooseBufferSplit(Widget w, DfBuf *buf, void *(*NewBuffer)(Widget,DfBuf*))
{
  DfBuf *n;

  if(IsSingleBuffer()){
    n = (*NewBuffer)(w, buf);
  }else{
    /*    n = NewBuffer(buf);*/
    n = buf;
    do{
      n = NextBuffer(n);
    }while((IS_VISIBLE(n) || !IS_ENABLE(n)) && n != buf);
    if(n == buf){
      /* all buffers are visibled, make new buffer. */
      n = (*NewBuffer)(w, buf);
    }
  }

  return n;
}

static int isSplitable(DfBuf *b, int fVert)
{
  int sz;
  int need;

  if(fVert){
    need = vnFontHeight * 5;
    sz = b->rc.height;
  }else{
    need = vnFontHeight * 8;
    sz = b->rc.width;
  }

  return need < sz;
}

DfBuf *SplitWindow(Widget w, DfBuf *b, int fVert, void *(*NewBuffer)(Widget,DfBuf*))
{
  DfBuf *n;
  XRectangle rc;
  int width;
  int height;

  if(!isSplitable(b, fVert)){
    St_SetMsg(w, vOptions.msg[STM_TOOSMALL]);
    return b;
  }

  rc = b->rc;
  n = ChooseBufferSplit(w, b, NewBuffer);

  ListAdd(&b->focus, &n->focus);
  n->rc = b->rc;
  n->stat = b->stat;
  if(fVert){
    height = b->rc.height;
    b->rc.height /= 2;
    b->stat |= BS_DRAWBOTTOMEDGE;

    n->rc.y = b->rc.y + b->rc.height;
    n->rc.height = height - b->rc.height;
  }else{
    width = b->rc.width;
    b->rc.width /= 2;

    b->stat |= BS_DRAWRIGHTEDGE;
    n->rc.x = b->rc.x + b->rc.width;
    n->rc.width = width - b->rc.width;
  }

  if(n->v->resize){
    n->v->resize(w, n);
  }
  if(b->v->resize){
    b->v->resize(w, b);
  }
  RedrawFrame(w, n, &n->rc);
  RedrawFrame(w, b, &b->rc);

  return n;
}

void AddBuffer(DfBuf *p, DfBuf *c)
{

  if(p == c){
    return;
  }

  ListAdd(&p->list, &c->list);
  dfx->nBuf++;
  BL_Update(c, 1);
}


void KillBuffer(Widget w, DfBuf *list)
{
  DfBuf *b;
  DfBuf *parent;
  DfBuf *child;

  if(dfx->nBuf <= 1){
    /* must have least 1 buffer */
    return;
  }


  if(!IS_VISIBLE(list)){
    goto DELETE;
  }

  b = list->parent;
  if(b && !IS_VISIBLE(b)){
    b->active = list->active;
    if(list->active){
      list->active->parent = b;
    }else{
      SwitchBuffer(w, b, list);
    }
    goto DELETE;
  }

  b = GetNextEnableHiddenBuffer(list);
  if(b == NULL){
    goto DELETE;
  }
  if(b == list){
    KillFrame(w, list);
    goto DELETE;
  }
  b->rc = list->rc;
  list->stat &= ~BS_VISIBLE;

  if(dfx->active == list){
    SwitchBuffer(w, b, list);
    /*SetActiveFrame(w, b);*/
  }
  b->stat = list->stat | (BS_VISIBLE | BS_ENABLE);
  if(b->v->resize){
    (*b->v->resize)(w, b);
  }
  RedrawFrame(w, b, &b->rc);

DELETE:
  /* update */
  if(dfx->lists == list){
    dfx->lists = NextBuffer(list);
  }
  parent = list->parent;
  child = list->active;

  if(parent){
    parent->active = child;
  }
  if(child){
    child->parent = parent;
  }
  BL_Update(list, 0);
  ListDel(list);
  dfx->nBuf--;

  list->v->free(list);
  UpdateFrames(w, dfx->lists);
}

void UnlinkBuffer(Widget w, DfBuf *b)
{
  DfBuf *tmp;

  tmp = NextFrame(b);
  b->stat &= ~BS_VISIBLE;
  ListDel(&b->focus);
  if(dfx->active == b){
    SetActiveFrame(w, tmp);
  }
}

GC GetBufferGC(Widget w, DfBuf *b)
{
  GC gc;
  Display *d;
  XRectangle rc;

  d = XtDisplay(w);
  gc = DefaultGC(d, DefaultScreen(d));
  if(b){
    GetClientRect(b, &rc);
    SetClipRect(d, gc, &rc);
  }else{
    ClearClip(d, gc);
  }

  return gc;
}

static int isSizable(DfBuf *b)
{
  switch(b->type){
  case DT_INFO:
  case DT_PROMPT:
    return 0;
  default:
    return 1;
  }
}

void GetClientRect(DfBuf *b, XRectangle *rc)
{

  *rc = b->rc;
  if(b->stat & BS_DRAWRIGHTEDGE){
    if(rc->width <= vOptions.borderWidth){
      rc->width = 0;
    }else{
      rc->width -= vOptions.borderWidth;
    }
  }
  if(b->stat & BS_DRAWBOTTOMEDGE){
    if(rc->height <= vOptions.borderWidth){
      rc->height = 0;
    }else{
      rc->height -= vOptions.borderWidth;
    }
  }
}

void BF2_CalcCursors(DfBufCommon *b)
{
  int usable;

  /* scrollable ? */
  b->nDrawable =  b->b.rc.height / vnFontHeight - 1;

  if(b->nItems <= b->nDrawable){
    /* no need scroll */
    b->scroll = 0;
    b->usable = 0;
    return;
  }

  b->scroll = 1;
  usable = b->b.rc.height - (vnFontHeight * 4);
  if(usable < 0){
    usable = 0;
  }
  b->usable = usable;
}

int BF_KeyPress(Widget w, void *ptr, KeySym k, char ch, Boolean *cond)
{
  return 0;
}

int BF2_KeyPress(Widget w, void *ptr, KeySym k, char ch, Boolean *cond)
{
  const DfCmdStr *c;
  const DfCmdStr *a;

  a = CmdTypeAheadGetAnchor();

  for(;;){
    c = CmdTypeAhead(k, ch, a);
    if(c == NULL){
      return 0;
    }
    switch(CommandHandler(ptr, w, c, 1/* parse the macro */)){
    case CREQ_IGNORE:
      break;
    default:
      return 0;
    }
    CmdCandNext();
  }

  return 0;
}


int BF_ResizeNop(Widget w, void *ptr)
{
  return 0;
}

int BF2_ResizeDraw(Widget w, void *ptr)
{
  DfBufCommon *b = ptr;
  DfVerb2 *v;

  BF2_CalcCursors(b);
  if(b->nItems == 0){
    return 0;
  }

  v = (DfVerb2*)b->b.v;
  (*v->setcursor)(w, b, b->nCur);

  return 0;
}


int BF2_Draw(void *ptr, struct DfDrawInfo *di)
{
  DfBufCommon *list;
  DfVerb2 *v;
  int y;
  int cy;
  int n;
  int top;
  int bottom;
  XRectangle rc;
  XRectangle cap_rc;

  list = ptr;
  v = (DfVerb2*)list->b.v;

  (*v->draw_caption)(di->w, list, di->d, di->gc);

  if(list->nItems == 0){
    dprintf("draw: return. buffer has no items.\n");
    return 0;
  }

  cy = vnFontHeight;
  bottom = list->b.rc.y + list->b.rc.height;
  if(di->clip.y + di->clip.height < bottom){
    bottom = di->clip.y + di->clip.height;
  }
  bottom += vnFontBase;

  /* Clip caption area */
  cap_rc = list->b.rc;
  cap_rc.height = vnFontHeight;
  rc = di->clip;
  RectSub(&rc, &cap_rc);
  SetClipRect(di->d, di->gc, &rc);

  top = list->b.rc.y + vnFontBase + vnFontHeight + list->nOffset;
  n = list->nStart;
  for(y = top; y < bottom; y += cy){
    (*v->draw_singleitem)(list, di, n);

    n++;
    if(list->nItems <= n){
      goto DRAW_FINISH;
    }
  }

DRAW_FINISH:
  return 0;
}

void BF2_DrawCaption(Widget w, void *ptr, Display *display, GC gc)
{
  DfBufCommon *list;
  char *p;

  list = ptr;

  XSetForeground(display, gc, vOptions.colors[COLOR_FILE]);
  p = Str_Get(&list->caption);
  XmbDrawString(display, XtWindow(w), vFontSet, gc, list->b.rc.x, list->b.rc.y + vnFontBase, p, Str_Length(&list->caption));
}

int BF2_ProcessBusy(Widget w, void *ptr, const DfCmdStr *c, const char **argv)
{
  dprintf("busy.\n");
  return CREQ_DONE;
}

void DrawBorder(void *ptr, Display *display, Widget w, GC gc)
{
  DfBuf *b;
  int x;
  int y;

  b = ptr;

  if(vOptions.borderWidth == 0){
    return;
  }

  if((b->stat & (BS_DRAWRIGHTEDGE | BS_DRAWBOTTOMEDGE)) == 0){
    return;
  }

  XSetForeground(display, gc, vOptions.colors[COLOR_BORDER]);
  x = b->rc.x + b->rc.width - vOptions.borderWidth;
  y = b->rc.y + b->rc.height - vOptions.borderWidth;
  if(b->stat & BS_DRAWRIGHTEDGE){
    XDrawLine(display, XtWindow(w), gc, x, b->rc.y, x, y);
  }
  if(b->stat & BS_DRAWBOTTOMEDGE){
    XDrawLine(display, XtWindow(w), gc, b->rc.x, y, x, y);
  }
}

void calcCursorRect(DfBufCommon *b, int nPos, XRectangle *rc)
{
  int cy;

  /* +1 for cwd */
  nPos++;

  /* adjust scroll offset */
  nPos -= b->nStart;

  GetClientRect(GetBuffer(b), rc);
  cy = rc->y + rc->height;

  rc->y = b->b.rc.y + nPos * vnFontHeight + b->nOffset;
  if(rc->y + vnFontHeight < cy){
    rc->height = vnFontHeight;
  }else{
    rc->height = cy - rc->y;
  }
}

int IsPosition(DfBuf *b)
{
  int x;
  int y;
  int r;

  x = b->rc.x + b->rc.width;
  y = b->rc.y + b->rc.height;

  r = 0;
  if(x < dfx->width){
    r |= POS_LEFT;
  }
  if(y < dfx->height){
    r |= POS_UPPER;
  }

  return r;
}


int DrawGap(struct DfDrawInfo *di, DfBufCommon *b, XRectangle *rc, XRectangle *rc_draw)
{
  DfVerb2 *v;
  int y;
  int n;
  int drew = 0;

  if(rc_draw->width == 0 || rc_draw->height == 0){
    return 0;
  }

  XClearArea(XtDisplay(di->w), XtWindow(di->w),
	     rc_draw->x, rc_draw->y, rc_draw->width, rc_draw->height, False);

  v = (DfVerb2*) b->b.v;
  y = rc_draw->height / vnFontHeight + 2; /* draw lines */
  n = b->nStart + ((rc_draw->y - rc->y - b->nOffset) / vnFontHeight);
  y += n;
  y = y <= b->nItems ? y : b->nItems;
  if(n <= b->nCur && b->nCur < y){
    drew = 1;
  }

  while(n < y){
    (*v->draw_singleitem)(b, di, n);
    n++;
  }

  return drew;
}
#if 0
#define PRINT_RECT(MSG, RC) printf(MSG ": (%d, %d)\n", (RC).y, ((RC).y + (RC).height))
#else
#define PRINT_RECT(MSG, RC)
#endif

struct delayed_draw_info{
  Widget w;
  DfBuf *b;
};

Boolean work_delay_draw(XtPointer ptr)
{
  struct delayed_draw_info *d = ptr;

  dfx->flags &= ~DFX_SCHED_DRAW;

  UpdateFrames(d->w, d->b);
  bFree(d);

  return True;
}

void SetupDelayedDraw(Widget w, DfBuf *b)
{
  struct delayed_draw_info *delay;

  delay = bMalloc(sizeof(struct delayed_draw_info));
  if(delay == NULL){
    return;
  }

  b->stat |= BS_NEEDREDRAW;
  delay->w = w;
  delay->b = b;
  if(!(dfx->flags & DFX_SCHED_DRAW)){
    dfx->flags |= DFX_SCHED_DRAW;
    XtAppAddWorkProc(XtWidgetToApplicationContext(w), work_delay_draw, delay);
  }
}


int SetCursor(Widget w, DfBufCommon *b, int n)
{
  int pre_cursor;
  int pre_start;
  int pre_offset;
  int scroll;
  GC gc;
  XRectangle rc;
  XRectangle rc_top;
  XRectangle rc_bottom;
  XRectangle rc_pre;
  int need_draw_cur = 0;
  int bottom;
  DfVerb2 *v;
  struct DfDrawInfo di;

  vnHoriz = 0;
  pre_cursor = b->nCur;
  pre_start = b->nStart;
  pre_offset = b->nOffset;
  calcCursorRect(b, pre_cursor, &rc_pre);

  if(!CalcBufferOffset(w, b, n)){
    return 0;
  }

  gc = GetBufferGC(w, GetBuffer(b));

  v = (DfVerb2*)b->b.v;

  di.d = XtDisplay(w);
  di.w = w;
  di.gc = gc;
  XSetGraphicsExposures(di.d, di.gc, True);

  if(b->nStart == pre_start && b->nOffset == pre_offset){
    if(0 <= pre_cursor && pre_cursor < b->nItems){
      (*v->draw_singleitem)(b, &di, pre_cursor);
    }
    (*v->draw_singleitem)(b, &di, n);
    goto FINISH;
  }

  scroll = (pre_start - b->nStart) * vnFontHeight + (b->nOffset - pre_offset);

  GetClientRect(GetBuffer(b), &rc);
  /* deduct cwd area */
  if(rc.height < vnFontHeight){
    return 0;
  }
  rc.y += vnFontHeight;
  rc.height -= vnFontHeight;

  SetClipRect(XtDisplay(w), gc, &rc);

  if(rc.height < (scroll < 0 ? -scroll : scroll)){
    DrawGap(&di, b, &rc, &rc);
    goto FINISH;
  }

  rc_top = rc;
  rc_bottom = rc;

  rc_top.height = rc_pre.y < rc_top.y ?  0 : rc_pre.y - rc_top.y;
  rc_bottom.y = rc_pre.y + rc_pre.height;
  bottom = rc.y + rc.height;
  rc_bottom.height = bottom < rc_bottom.y ? 0 : bottom - rc_bottom.y;

  if(rc_top.height){
    ScrollRect(&di, &rc, &rc_top, scroll);
  }
  if(rc_bottom.height){
    ScrollRect(&di, &rc, &rc_bottom, scroll);
  }
  if(scroll < 0){
    rc_top.height += rc_pre.height;
  }else{
    bottom = rc_bottom.height;
    rc_bottom = rc_pre;
    rc_bottom.height += bottom;
  }
  if(!DrawGap(&di, b, &rc, &rc_top)){
    need_draw_cur = 1;
  }

  if(!DrawGap(&di, b, &rc, &rc_bottom)){
    need_draw_cur = 1;
  }

  if(need_draw_cur){
    (*v->draw_singleitem)(b, &di, n);
  }

  if(IsPendingExpose(w)){
    SetupDelayedDraw(w, GetBuffer(b));
  }

FINISH:

  ClearClip(XtDisplay(w), gc);
  
  if(pre_cursor != n){
    RedrawInfo(w);
  }

  return 0;
}

int CalcBufferOffset(Widget w, DfBufCommon *b, int n)
{
  int num;
  int y;
  int new_start;
  int new_offset;
  int max;

  if(b == NULL){
    return 0;
  }
  if(b->nItems == 0){
    b->nCur = 0;
    return 1;
  }

  /* scrollable ? */
  num = b->nDrawable - 1;/* 1 for caption line */
  max = b->nItems ? b->nItems - 1 : 0;

  /* check limit */
  if(max <= n){
    n = max;
  }

  if(!b->scroll){
    /* no need scroll */
    new_offset = 0;
    new_start = 0;
  }else{
    if(n == 0){/* require top */
      y = vnFontHeight;
      new_offset = 0;
      new_start = 0;
    }else{
      if(n == max){ /* bottom */
	y = b->b.rc.height - vnFontHeight;
      }else{
	if(b->usable){
	  y = vnFontHeight + vnFontHeight + (n - 1) * b->usable / (max - 2);
	}else{
	  y = ((b->b.rc.height - vnFontHeight) + vnFontHeight) / 2;
	}
      }
      new_start = n - ((y - 1) / vnFontHeight);
      new_offset = (y % vnFontHeight);
      if(new_offset){
	new_offset = new_offset - vnFontHeight;
      }
    }
  }

  if(new_start < 0 || b->nItems < new_start){
    new_start = 0;
    new_offset = 0;
    n = 0;
  }
  b->nStart = new_start;
  b->nOffset = new_offset;
  b->nCur = n;

  return 1;
}


int upCursor(Widget w, DfBufCommon *b)
{
  int n;
  DfVerb2 *v;

  n = b->nCur;
  if(n){
    v = (DfVerb2*)b->b.v;
    (*v->setcursor)(w, b, n - 1);
  }

  return 0;
}


int downCursor(Widget w, DfBufCommon *b)
{
  int n;
  DfVerb2 *v;

  n = b->nCur;
  if(n < b->nItems - 1){
    v = (DfVerb2*)b->b.v;
    (*v->setcursor)(w, b, n + 1);
  }

  return 0;
}

int upPage(Widget w, DfBufCommon *b)
{
  int n;
  DfVerb2 *v;

  n = b->b.rc.height / vnFontHeight - 1;
  switch(n){/* :D  */
  case 0:
    break;
  default:
    n--;
  }

  if(b->nCur < n){
    n = 0;
  }else{
    n = b->nCur - n;
  }
  v = (DfVerb2*)b->b.v;
  (*v->setcursor)(w, b, n);

  return 0;
}

int downPage(Widget w, DfBufCommon *b)
{
  int n;
  DfVerb2 *v;

  n = b->b.rc.height / vnFontHeight - 1;
  switch(n){/* :D  */
  case 0:
    break;
  default:
    n--;
  }

  if(b->nItems <= b->nCur + n){
    n = b->nItems - 1;
  }else{
    n += b->nCur;
  }
  v = (DfVerb2*)b->b.v;
  (*v->setcursor)(w, b, n);

  return 0;
}


int cursorToTop(Widget w, DfBufCommon *b)
{
  DfVerb2 *v;

  v = (DfVerb2*)b->b.v;
  (*v->setcursor)(w, b, 0);
  return 0;
}

int cursorToBottom(Widget w, DfBufCommon *b)
{
  DfVerb2 *v;

  v = (DfVerb2*)b->b.v;
  (*v->setcursor)(w, b, b->nItems - 1);
  return 0;
}

void ScrollRect(struct DfDrawInfo *di, XRectangle *rc, XRectangle *rc_part, int offset)
{
  XRectangle s;
  XRectangle i;  /* invalid rect */
  int adj;
  int pos;
  int y;

  s = *rc_part;
  i = *rc_part;

  pos = s.y + offset;
  y = s.y + offset;
  if(offset < 0){ /* roll up */
    if(rc_part->height + offset <= 0){
      goto draw;
    }

    if(pos < s.y){
      adj = s.y - pos;
      s.height -= adj;
      y = s.y;
      s.y += adj;
    }
    i.height = -offset;
    i.y = s.y + s.height - i.height;
  }else{/* roll down */
    if(rc_part->height <= offset){
      goto draw;
    }
    s.height -= offset;

    i.y = rc_part->y;
    i.height = offset;
  }

  /*  SetClipRect(XtDisplay(w), gc, 0, 0, rc, 1, Unsorted);*/
  XCopyArea(XtDisplay(di->w), XtWindow(di->w), XtWindow(di->w), di->gc,
	    s.x, s.y, s.width, s.height,
	    s.x, y);
 draw:
  *rc_part = i;
}

DfFileList *GetFilerBuffer(DfBuf *b)
{
  while(b && b->type != DT_FILER){
    b = b->parent;
  }
  return (DfFileList*)b;
}


void KillChildren(Widget w, DfBuf *b)
{
  while(b->active){
    KillBuffer(w, b->active);
  }
}


void SetFocusLink(DfBuf *ina, DfBuf *act)
{
  if(ListIsEmpty(&ina->focus)){
    ListClear(&act->focus);
  }else{
    ListAdd(&ina->focus, &act->focus);
    ListDel(&ina->focus);
  }
}


enum size_edge{
  SIZE_NONE    = 0,
  SIZE_MARKED  = 1,
  SIZE_TOP     = 2,
  SIZE_BOTTOM  = 4,
  SIZE_LEFT    = 8,
  SIZE_RIGHT   = 16,
};

struct sized_buffer_info{
  DfBuf *b;
  enum size_edge e;
};

struct sized_buffer_info2{
  struct sized_buffer_info *bi;
  int n;
  int update;
};

static int enum_count_visibled(DfBuf *b, void *ptr)
{
  int *n = (int *)ptr;

  if(IS_VISIBLE(b) && isSizable(b)){
    (*n)++;
  }
  return 0;
}

static int enum_setup_size_info(DfBuf *b, void *ptr)
{
  struct sized_buffer_info **bi = ptr;

  if(IS_VISIBLE(b) && isSizable(b)){
    (*bi)->b = b;
    (*bi)->e = SIZE_NONE;
    (*bi)++;
  }
  return 0;
}

static int determin_edge(DfBuf *b, struct sized_buffer_info2 *bi, int start, int num)
{
  int update = 1;
  enum adjoin_side side;
  enum size_edge edge;
  struct sized_buffer_info  *bi_ptr = &bi->bi[start];
  int n;
  int effect = 0;

  /* b as current buffer */
  for(;;){
    update--;
    edge = bi_ptr->e;
    for(n = 0; n < num; n++){
      if(b == bi->bi[n].b){
	continue;
      }
      if(bi->bi[n].e != SIZE_NONE){
	continue;
      }

      side = IsAdjoin(&b->rc, &bi->bi[n].b->rc);
      if(side == 0){
	continue;
      }

      switch((side << 4 )| edge){
      case (UP_SIDE << 4) | SIZE_TOP:
	bi->bi[n].e = SIZE_MARKED | SIZE_BOTTOM;
	effect++;
	update++;
	break;
      case (DOWN_SIDE << 4) | SIZE_BOTTOM:
	bi->bi[n].e = SIZE_MARKED | SIZE_TOP;
	effect++;
	update++;
	break;
      case (LEFT_SIDE << 4) | SIZE_LEFT:
	bi->bi[n].e = SIZE_MARKED | SIZE_RIGHT;
	effect++;
	update++;
	break;
      case (RIGHT_SIDE << 4) | SIZE_RIGHT:
	bi->bi[n].e = SIZE_MARKED | SIZE_LEFT;
	effect++;
	update++;
	break;
      default:
	continue;
      }
    }

    if(update == 0){
      break;
    }

    for(n = 0; n < num; n++){
      if(bi->bi[n].e & SIZE_MARKED){
	bi->bi[n].e &= ~SIZE_MARKED;
	bi_ptr = &bi->bi[n];
	b = bi->bi[n].b;
	break;
      }
    }
  }
  return effect;
}

void ChangeFrameSize(Widget w, DfBuf *b, int fVirt, int size)
{
  int num = 0;
  int n;
  struct sized_buffer_info2 bi;
  enum size_edge edge;
  struct sized_buffer_info  *bi_ptr;
  DfBuf *tmp;
  int delta;
  int bottom_edge;

  bottom_edge = dfx->frame_bottom;

  if(0 < size){
    if(fVirt){
      n = dfx->frame_bottom - dfx->frame_y - b->rc.height;
    }else{
      n = dfx->width - b->rc.width;
    }
    if(n < size){
      size = n;
    }
  }else{
    if(fVirt){
      n = -b->rc.height;
    }else{
      n = -b->rc.width;
    }
    if(size < n){
      size = n;
    }
  }
  if(n == 0){
    return;
  }

  EnumBuffers(b, enum_count_visibled, &num);
  bi.bi = bMalloc(sizeof(struct sized_buffer_info) * num);
  if(bi.bi == NULL){
    return;
  }
  bi_ptr = bi.bi;
  EnumBuffers(b, enum_setup_size_info, &bi_ptr);

  bi_ptr = NULL;
  for(n = 0; n < num; n++){
    if(bi.bi[n].b == b){
      if(fVirt){
	if(b->rc.y + b->rc.height == bottom_edge){
	  edge = SIZE_TOP;
	  bi.bi[n].e = SIZE_TOP;
	}else{
	  edge = SIZE_BOTTOM;
	  bi.bi[n].e = SIZE_BOTTOM;
	}
      }else{
	if(b->rc.x + b->rc.width == dfx->width){
	  edge = SIZE_LEFT;
	  bi.bi[n].e = SIZE_LEFT;
	}else{
	  edge = SIZE_RIGHT;
	  bi.bi[n].e = SIZE_RIGHT;
	}
      }
      bi_ptr = &bi.bi[n];
      break;
    }
  }

  if(!bi_ptr){
    bFree(bi.bi);
    return;
  }

  n = determin_edge(b, &bi, n, num);
  if(n == 0){
    bFree(bi.bi);
    return;
  }

  for(n = 0; n < num; n++){
    delta = (edge == bi.bi[n].e) ? size : -size;
    tmp = bi.bi[n].b;
    switch(bi.bi[n].e){
    case SIZE_TOP:
      RectSizeTop(&tmp->rc, delta);
      goto RESIZE;
      break;
    case SIZE_BOTTOM:
      RectSizeBottom(&tmp->rc, delta);
      goto RESIZE;
      break;
    case SIZE_LEFT:
      RectSizeLeft(&tmp->rc, delta);
      goto RESIZE;
      break;
    case SIZE_RIGHT:
      RectSizeRight(&tmp->rc, delta);
    RESIZE:
      if(tmp->rc.width == 0 || tmp->rc.height == 0){
	UnlinkBuffer(w, tmp);
      }else{
	if(tmp->v->resize){
	  (*tmp->v->resize)(w, tmp);
	}
	RedrawFrame(w, tmp, &tmp->rc);
      }
      break;
    default:
      break;
    }
  }

  bFree(bi.bi);
}

#ifdef DEBUG
void DumpBufInfo(void)
{
  DfBuf *b;
  DfBuf *start;
  int n = 1;
  char attr[4];
  int t;
  const static char *type[] = {
    "filer    ",
    "viewer   ",
    "cmd list ",
    "path list",
    "buffers  ",
    "candi    ",
    "prompt   ",
    "info     ",
    "(unknown)"
  };

  start = dfx->lists;
  b = start;
  dprintf("There are %d buffers.\n", dfx->nBuf);
  dprintf("Active is %p.\n", dfx->active);
  do{
    t = b->type;
    if(DT_INFO < t){
      t = DT_INFO + 1;
    }
    attr[0] = IS_VISIBLE(b) ? 'V' : '-';
    attr[1] = IS_ENABLE(b) ? 'E' : '-';
    attr[2] = '\0';

    dprintf("%d: %s %p (p:%p act: %p, %s focus %p, %p)\n", n, type[t], b, b->parent, b->active, attr, b->focus.prev, b->focus.next);
    b = NextBuffer(b);
    n++;
  }while(b != start);
}

#endif
