#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/keysym.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Simple.h>

#include "config.h"
#include "str.h"
#include "dftype.h"
#include "list.h"
#include "buffer.h"
#include "filer.h"
#include "cfg.h"
#include "cmdlist.h"
#include "dialog.h"
#include "task.h"
#include "cmd.h"
#include "xutil.h"
#include "misc.h"
#include "mem.h"
#include "dir.h"
#include "info.h"
#include "status.h"
#include "dfxfunc.h"
#include "debug.h"
#include "dfval.h"


Widget toplevel;
XFontSet vFontSet;
int vnFontHeight;
int vnFontBase;
int vnTabWidth;
int vnHoriz;
int vnScreenWidth;
int vnScreenHeight;
DfWindow *dfx;
DfxAppList *vAppsList;
int mod_keys;		/* referrd from cmd.c */
DfxResources vOptions;
int v_uid;
int v_gid;
int vInUTF8;
Atom WmDelWindow;

struct expose_data{
  int count;
  Display *d;
  XRectangle rc;
};

static void AppInitialize(XtAppContext *app, Widget w);
static int init_file_list(Widget w);
static void postRealized(XtAppContext *app, Widget w);

static int Wnd_Create(Widget parent, int cx, int cy);
static void Wnd_Initialize(Widget w, DfWindow *wnd);
static void handle_event_list(void);

static void wndFocus(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndResize(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndKeyPressProc(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndKeyReleaseProc(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndKeymap(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndExposeEvent(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void wndGraphExposeEvent(Widget w, XEvent *event);
static void wndExpose(Widget w, struct expose_data *ev);
static void wndDrawProc(DfWindow *dfx, Display *display, Widget w, GC gc);

static void wndEvents(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void trapSigchld(int c);

static void actionQuit(Widget w, XEvent *ev, String *str, Cardinal *num);

int DiscardAppsList(DfxAppList *app_list);

static XtResource dfxResourceList[] =
{
  {"background", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_BK]),
    XtRString, "white"},
  {"fileColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_FILE]),
    XtRString, "black"},
  {"dirColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_DIR]),
    XtRString, "navy"},
  {"linkColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_LINK]),
    XtRString, "blue"},
  {"roColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_RO]),
    XtRString, "red"},
  {"executableColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_X]),
    XtRString, "green"},
  {"deviceColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_DEV]),
    XtRString, "yellow"},
  {"otherColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_OTHER]),
    XtRString, "black"},
  {"unknownColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_UNKNOWN]),
    XtRString, "red"},
  {"cursorColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_CURFG]),
    XtRString, "white"},
  {"cursorBkColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_CURBG]),
    XtRString, "black"},
  {"selectedColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_SELFG]),
    XtRString, "white"},
  {"selectedBkColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_SELBG]),
    XtRString, "blue"},
  {"markColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_MARK]),
    XtRString, "yellow"},
  {"borderColor", XtCColor,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(DfxResources, colors[COLOR_BORDER]),
    XtRString, "gray"},
  {"borderWidth", XtCWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(DfxResources, borderWidth),
    XtRString, "1"},
  {"msgReady", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_READY]),
    XtRString, "W E L C O M E !!"},
  {"msgExist", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_ALREADY_EXIST]),
    XtRString, "already exist."},
  {"msgChdir", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_CHDIR]),
    XtRString, "change directory."},
  {"msgGotoparent", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_GOTO_PARENT]),
    XtRString, "go to parent."},
  {"msgOverwrite", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_OVERWRITE]),
    XtRString, "overwrite."},
  {"msgMark", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_MATCH]),
    XtRString, "found."},
  {"msgMarkNomatch", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_NOMATCH]),
    XtRString, "file not found."},
  {"msgCantView", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_CANTVIEW]),
    XtRString, "viewer can't open file."},
  {"msgNoConfigfile", XtCString,
    XtRString, sizeof(String),
   XtOffsetOf(DfxResources, msg[STM_NOCFGFILE]),
    XtRString, "configuration file not found."},
  {"msgExecFail", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_EXECFAIL]),
    XtRString, "can't execute."},
  {"msgTooSmall", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_TOOSMALL]),
    XtRString, "this buffer is too small."},
  {"msgNoMem", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_NOMEM]),
    XtRString, "not enugh memory."},
  {"msgNoFiles", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_NOFILES]),
    XtRString, "no files."},
  {"msgWorking", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_CANTCLOSE]),
    XtRString, "working. can't close."},
  {"msgClearFilter", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_CLEARFILTER]),
    XtRString, "filter cleared."},
  {"msgScan", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_SCANNING]),
    XtRString, "scanning..."},
  {"msgScanDone", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_SCANDONE]),
    XtRString, "scanning...done."},
  {"msgEmpty", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_EMPTY]),
    XtRString, "(no files)"},
  {"msgFound", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_VWFOUND]),
    XtRString, "Found."},
  {"msgNotFound", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[STM_VWNOTFOUND]),
    XtRString, "Not found."},
  {"dlgDelete", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_DELETEP]),
    XtRString, "Delete?"},
  {"dlgOverwrite", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_OVERWRITEP]),
    XtRString, "already exist. overwrite?"},
  {"dlgChmod", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_CHMODP]),
    XtRString, "continue?"},
  {"dlgChown", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_CHOWNP]),
    XtRString, "continue?"},
  {"dlgCopyFail", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_COPY_FAIL]),
    XtRString, "can't copy."},
  {"dlgMoveFail", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_MOVE_FAIL]),
    XtRString, "can't move."},
  {"dlgMoveFailDel", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_MOVE_FAIL_DELETE]),
    XtRString, "can't delete original file."},
  {"dlgMkdirFail", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_MKDIR_FAIL]),
    XtRString, "can't make directory."},
  {"dlgCannotMakeDestination", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[DLG_CANNOT_MKDIR_DEST]),
    XtRString, "can't make destination directory."},
  {"captionCopy", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_COPY]),
    XtRString, "Copy to:"},
  {"captionMove", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_MOVE]),
    XtRString, "Move to:"},
  {"captionLink", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_LINK]),
    XtRString, "Link to:"},
  {"captionSymlink", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_SYMLINK]),
    XtRString, "Symbolic link to:"},
  {"captionChdir", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_CHDIR]),
    XtRString, "chdir:"},
  {"captionIsearch", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_ISEARCH]),
    XtRString, "search:"},
  {"captionMultimark", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_MULTIMARK]),
    XtRString, "Mark:"},
  {"captionMkfile", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_MKFILE]),
    XtRString, "Make file:"},
  {"captionMkdir", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_MKDIR]),
    XtRString, "Make directory:"},
  {"captionExec", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_EXEC]),
    XtRString, "Execute:"},
  {"captionChown", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_CHOWN]),
    XtRString, "chown:"},
  {"captionChgrp", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_CHGRP]),
    XtRString, "chgrp:"},
  {"captionRename", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_RENAME]),
    XtRString, "Rename:"},
  {"captionFilter", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_FILTER]),
    XtRString, "Filter:"},
  {"captionCommands", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_COMMAND]),
    XtRString, "COMMAND:"},
  {"captionPaths", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_PATH]),
    XtRString, "DIRECTORIES:"},
  {"captionBuffres", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_BUFFERS]),
    XtRString, "BUFFERS:"},
  {"captionMacro", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, msg[CAP_MACROINPUT]),
    XtRString, "Macro:"},
  {"fontSet", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, font),
    XtRString, "-*-*-medium-r-normal--12-*"},
  {"dateFormat", XtCString,
    XtRString, sizeof(String),
    XtOffsetOf(DfxResources, dateFormat),
    XtRString, "%a %b %e %H:%M:%S %Z %Y"},
};


XtActionsRec dfxActions[] = {
  {"quit", actionQuit}
};

static char *fallback_resources[] = {
  "*international: True",
  /*
  "*inputMethod: xim",
  "*inputStyle: OverTheSpot,OnTheSpot,Root",
  */
  "*fontSet: -*-*-medium-r-normal--12-*",
  "width: 320",
  "height: 200",
  NULL
};

const DfCmdStr *FL_CmdTab;

DfCmdKey c_strOpen[] ={
  XK_Return,
  0
};
DfCmdKey c_strUp[] ={
  XK_Up,
  0
};
DfCmdKey c_strDown[] ={
  XK_Down,
  0
};
DfCmdKey c_strupPage[] ={
  XK_Prior,
  0
};
DfCmdKey c_strdownPage[] ={
  XK_Next,
  0
};
DfCmdKey c_strSelect[] ={
  XK_space,
  0
};

DfCmdKey c_strSplitH[] ={
  XK_w,  XK_w,
  0
};

DfCmdKey c_strSplitV[] ={
  XK_w,  XK_v,
  0
};

DfCmdKey c_strClose[] ={
  XK_w,  XK_q,
  0
};

DfCmdKey c_strWN[] ={
  XK_Right,
  0
};

DfCmdKey c_strWB[] ={
  XK_Left,
  0
};

DfCmdKey c_strNewBuf[] ={
  DFMOD_CTRL | XK_n,
  0
};

DfCmdKey c_strSwitchBuf[] ={
  XK_Tab,
  0
};

DfCmdKey c_strgotoParent[] ={
  XK_BackSpace,
  0
};

DfCmdKey c_strCtrlH[] ={
  DFMOD_CTRL | XK_h,
  0
};

DfCmdKey c_strCopy[] ={
  XK_c,
  0
};

DfCmdKey c_strQuit[] ={
  XK_q,
  0
};

DfCmdKey c_strI[] ={
  XK_i,
  0
};

DfCmdKey c_strK[] ={
  XK_k,
  0
};

DfCmdKey c_strCtrlK[] ={
  DFMOD_CTRL | XK_k,
  0
};

DfCmdKey c_strMove[] ={
  XK_m,
  0
};

DfCmdKey c_strCtrlS[] ={
  DFMOD_CTRL | XK_s,
  0
};

DfCmdKey c_strR[] ={
  XK_r,
  0
};

DfCmdKey c_strParenL[] ={
  DFMOD_CHAR | '(',
  0
};
DfCmdKey c_strParenR[] ={
  DFMOD_CHAR | ')',
  0
};

DfCmdKey c_strDel[] ={
  XK_d,
  0
};

DfCmdKey c_strL[] ={
  XK_l,
  0
};

DfCmdKey c_strSlash[] ={
  XK_slash,
  0
};

DfCmdKey c_strSN[] ={
  XK_s, XK_n,
  0
};
DfCmdKey c_strSS[] ={
  XK_s, XK_s,
  0
};
DfCmdKey c_strST[] ={
  XK_s, XK_t,
  0
};
DfCmdKey c_strSE[] ={
  XK_s, XK_e,
  0
};
DfCmdKey c_strSA[] ={
  XK_s, XK_m,
  0
};
DfCmdKey c_strAsterisk[] ={
  DFMOD_CHAR | '*',
  0
};
DfCmdKey c_strF5[] ={
  XK_F5,
  0
};
DfCmdKey c_strH[] ={
  XK_h,
  0
};
DfCmdKey c_strF[] ={
  XK_f,
  0
};
DfCmdKey c_strCtrlF[] ={
  DFMOD_CTRL | XK_f,
  0
};

DfCmdKey c_strCtrlL[] ={
  DFMOD_CTRL | XK_l,
  0
};
DfCmdKey c_strShiftL[] ={
  DFMOD_SHIFT | XK_l,
  0
};
DfCmdKey c_strX[] ={
  XK_x,
  0
};
DfCmdKey c_strA[] ={
  XK_a,
  0
};
DfCmdKey c_strN[] ={
  XK_n,
  0
};
DfCmdKey c_strShiftN[] ={
  DFMOD_SHIFT | XK_n,
  0
};
DfCmdKey c_strPeriod[] ={
  XK_period,
  0
};

DfCmdKey c_strColon[] ={
  DFMOD_CHAR | ':',
  0
};

DfCmdKey c_strCtrlColon[] ={
  DFMOD_CTRL | XK_colon,
  0
};

DfCmdKey c_strCtrlXO[] ={
  DFMOD_CTRL | XK_x, XK_o,
  0
};

DfCmdKey c_strCtrlXG[] ={
  DFMOD_CTRL | XK_x, XK_g,
  0
};

DfCmdKey c_strW1[] ={
  XK_w, XK_1,
  0
};

DfCmdKey c_strCtrlC[] ={
  DFMOD_CTRL | XK_c,
  0
};

DfCmdKey c_strShrinkH[] ={
  DFMOD_CHAR | '{',
  0
};

DfCmdKey c_strEnlargeH[] ={
  DFMOD_CHAR | '}',
  0
};

DfCmdKey c_strShrinkV[] ={
  DFMOD_CHAR | '_',
  0
};

DfCmdKey c_strEnlargeV[] ={
  DFMOD_CHAR | '^',
  0
};

#ifdef MEMDEBUG
DfCmdKey c_strDUMP[] ={
  XK_F12,
  0
};
#endif

#ifdef DEBUG
DfCmdKey c_strBUFDUMP[] ={
  XK_F11,
  0
};
#endif

const DfCmdStr FL_DefaultCmdTab[] = {
  {c_strUp,         CMD_ANY,    CURUP, NULL},
  {c_strDown,       CMD_ANY,    CURDOWN},
  {c_strupPage,     CMD_ANY,    PAGEUP},
  {c_strdownPage,   CMD_ANY,    PAGEDOWN},
  {c_strSelect,     CMD_FILER,  MARK},
  {c_strSelect,     CMD_VIEWER, CHCHARSET},
  {c_strAsterisk,   CMD_FILER,  MULTIMARK},
  {c_strCtrlColon,  CMD_FILER,  REVMARK},
  {c_strColon,      CMD_FILER,  CLEARMARK},
  {c_strOpen,       CMD_ANY,    OPEN},
  {c_strOpen,       CMD_VIEWER, CLOSE},
  {c_strSplitH,     CMD_ANY,    SPLIT_H},
  {c_strSplitV,     CMD_ANY,    SPLIT_V},
  {c_strWN,         CMD_ANY,    NEXTFRAME},
  {c_strWB,         CMD_ANY,    PREVFRAME},
  {c_strNewBuf,     CMD_ANY,    NEWBUFFER},
  {c_strSwitchBuf,  CMD_ANY,    SWITCHBUFFER},
  {c_strgotoParent, CMD_ANY,    GOTOPARENT},
  {c_strCtrlH,      CMD_ANY,    GOTOPARENT},
  {c_strCtrlC,      CMD_ANY,    CANCEL},
  {c_strCopy,       CMD_FILER,  COPY},
  {c_strMove,       CMD_FILER,  MOVE},
  {c_strCtrlS,      CMD_FILER,  SYMLINK},
  {c_strDel,        CMD_FILER,  DELETE},
  {c_strR,          CMD_FILER,  RENAME},
  {c_strParenL,     CMD_FILER,  HORIZ_LEFT},
  {c_strParenR,     CMD_FILER,  HORIZ_RIGHT},
  {c_strK,          CMD_FILER,  MKDIR},
  {c_strCtrlK,      CMD_FILER,  MKFILE},
  {c_strF,          CMD_FILER,  FILTER},
  {c_strCtrlF,      CMD_FILER,  IFILTER},
  {c_strL,          CMD_FILER,  GOTO},
  {c_strL,          CMD_VIEWER, CHNEWLINE},
  {c_strI,          CMD_ANY,    INFO},
  {c_strF5,         CMD_FILER,  RELOAD},
  {c_strCtrlL,      CMD_ANY,    REFRESH},
  {c_strSlash,      CMD_FILER,  ISEARCH},
  {c_strSlash,      CMD_VIEWER, VIEWFIND},
  {c_strN,          CMD_VIEWER, VIEWFINDNEXT},
  {c_strShiftN,     CMD_VIEWER, VIEWFINDPREV},
  {c_strA,          CMD_FILER,  CHMOD},
  {c_strCtrlXO,     CMD_FILER,  CHOWN},
  {c_strCtrlXG,     CMD_FILER,  CHGRP},
  {c_strSN,         CMD_FILER,  SORTNAME},
  {c_strST,         CMD_FILER,  SORTTIME},
  {c_strSS,         CMD_FILER,  SORTSIZE},
  {c_strSE,         CMD_FILER,  SORTEXT},
  {c_strSA,         CMD_FILER,  SORTATTR},
  {c_strX,          CMD_FILER,  EXECUTE},
  {c_strPeriod,     CMD_FILER,  SHOWDOTS},
  {c_strH,          CMD_FILER,  CMDLIST},
  {c_strShiftL,      CMD_FILER,  PATHLIST},
  {c_strQuit,       CMD_ANY,    QUIT},
  {c_strClose,      CMD_ANY,    CLOSE},
  {c_strW1,         CMD_ANY,    KILLOTHER},
  {c_strShrinkH,    CMD_ANY,    SHRINK_H},
  {c_strEnlargeH,   CMD_ANY,    ENLARGE_H},
  {c_strShrinkV,    CMD_ANY,    SHRINK_V},
  {c_strEnlargeV,   CMD_ANY,    ENLARGE_V},

#ifdef MEMDEBUG
  {c_strDUMP,       CMD_ANY,    DUMPMEMINFO},
#endif
#ifdef DEBUG
  {c_strBUFDUMP,    CMD_ANY,    DUMPBUFFERINFO},
#endif
  {NULL,            CMD_ANY,    0, NULL},
};

int main(int argc, char **argv)
{
  XtAppContext ac;
  
  if(!XtToolkitThreadInitialize()){
    fprintf(stderr, "system is not support thread.\n");
  }
  XtSetLanguageProc(NULL, NULL, NULL);
  toplevel = XtAppInitialize(&ac, "Dfx", NULL, 0,
			     &argc, argv, fallback_resources, NULL, 0);
  if(!toplevel){
    return ENOMEM;
  }
  AppInitialize(&ac, toplevel);
  if(Wnd_Create(toplevel, 320, 200)){
    return ENOMEM;
  }
  XtRealizeWidget(toplevel);

  postRealized(&ac, toplevel);
  XtAppMainLoop(ac);

  return 0;
}

static void AppInitialize(XtAppContext *app, Widget w)
{
  char **miss;
  int miss_charset;
  char *def;
  XFontSetExtents *fse;
  Display *d;

  signal(SIGCHLD, trapSigchld);
  if(LoadKeyDefine()){
    exit(1);
  }
  LoadConfig();
  if(FL_CmdTab == NULL){
    FL_CmdTab = FL_DefaultCmdTab;
  }

  XtGetApplicationResources(w, &vOptions, dfxResourceList, XtNumber(dfxResourceList), NULL, 0);

  d = XtDisplay(w);

  vFontSet = XCreateFontSet(d,
			    vOptions.font,
			    &miss, &miss_charset, &def);
  if(!vFontSet){
    fprintf(stderr, "can't load font set (%s)\n",
	    vOptions.font);
    exit(1);
  }

  fse = XExtentsOfFontSet(vFontSet);
  vnFontHeight = fse->max_logical_extent.height;  
  vnFontBase = -fse->max_logical_extent.y;
  vnTabWidth =  XmbTextEscapement(vFontSet, " ", 1) * 8;

  vnScreenWidth = DisplayWidth(d, DefaultScreen(d));
  vnScreenHeight = DisplayHeight(d, DefaultScreen(d));

  dprintf("Initialize lists.\n");
  ListClear(&vDlgList);
  ListClear(&vNotifyTaskList);
  pthread_mutex_init(&vMtxNotify, NULL);
  vAppsList = LoadAppsList();

  XtAppAddActions(*app, dfxActions, XtNumber(dfxActions));
  XtOverrideTranslations(toplevel, XtParseTranslationTable("<Message>WM_PROTOCOLS: quit()"));

  v_gid = getgid();
  v_uid = getuid();
  vInUTF8 = InUTF8();
}

int ReloadConfig(void)
{
  DiscardAppsList(vAppsList);

  if(FL_CmdTab != FL_DefaultCmdTab){
    free((void *)FL_CmdTab);
  }

  if(LoadKeyDefine()){
    FL_CmdTab = FL_DefaultCmdTab;
  }

  vAppsList = LoadAppsList();
  XtGetApplicationResources(dfx->w, &vOptions, dfxResourceList, XtNumber(dfxResourceList), NULL, 0);

  return 0;
}

static void postRealized(XtAppContext *app, Widget w)
{
  Display *d;

  init_file_list(w);
  d = XtDisplay(w);
  WmDelWindow = XInternAtom(d, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(d,  XtWindow(w), &WmDelWindow, 1);
}


/* will be use XtGrabKeyboard() */
static void wndKeyPressProc(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  DfBuf *buf;
  KeySym k;
  char ch[1];
  int n;

  n = XLookupString((XKeyEvent*)event,
		    ch, sizeof(ch),
		    &k, NULL);
  k = XLookupKeysym((XKeyEvent*)event, 0);
  if(n == 0){
    ch[0] = 0;
  }

  switch(k){
  case XK_Shift_L:
  case XK_Shift_R:
    mod_keys |= DFMOD_SHIFT;
    break;

  case XK_Control_L:
  case XK_Control_R:
    mod_keys |= DFMOD_CTRL;
    break;

  case XK_Alt_L:
  case XK_Alt_R:
    mod_keys |= DFMOD_ALT;
    break;
  default:
    St_SetMsg(w, NULL);
    buf = dfx->active;
    buf->v->keybd_event(w, buf, k, ch[0], cont);
  }
}

static void wndKeyReleaseProc(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  KeySym k;
  char ch[1];
  int n;

  n = XLookupString((XKeyEvent*)event,
		    ch, sizeof(ch),
		    &k, NULL);
  if(n == 0){
    ch[0] = 0;
  }
  /*  k = XLookupKeysym((XKeyEvent*)event, 0);*/

  switch(k){
  case XK_Shift_L:
  case XK_Shift_R:
    mod_keys &= ~DFMOD_SHIFT;
    break;

  case XK_Control_L:
  case XK_Control_R:
    mod_keys &= ~DFMOD_CTRL;
    break;

  case XK_Alt_L:
  case XK_Alt_R:
    mod_keys &= ~DFMOD_ALT;
    break;
  }
}

static void wndKeymap(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XKeymapEvent *ev;
  KeySym k;
  int i, j;
  int code;
  char *v;

  dprintf("keymap\n");
  ev = (XKeymapEvent*)event;

  v = ev->key_vector;
  code = 0;
  for(i = 0; i < 32; i++){
    for(j = 0; j < 8; j++){
      if(*v & (1<<j)){
	k = XKeycodeToKeysym(ev->display, code, 0);
	switch(k){
	case XK_Shift_L:
	case XK_Shift_R:
	  mod_keys |= DFMOD_SHIFT;
	  break;

	case XK_Control_L:
	case XK_Control_R:
	  mod_keys |= DFMOD_CTRL;
	  break;

	case XK_Alt_L:
	case XK_Alt_R:
	  mod_keys |= DFMOD_ALT;
	  break;
	}
	code++;
      }
    }
    v++;
  }
}

static void wndExpose(Widget w, struct expose_data *ev)
{
  GC gc;

  if(dfx->flags & DFX_CLIPPING){
    AddRect(&dfx->clip, ev->rc.x, ev->rc.y, ev->rc.width, ev->rc.height);
  }else{
    dfx->flags |= DFX_CLIPPING;
    dfx->clip = ev->rc;
  }

  if(ev->count){
    /* add to clip mask and exit. */
    return;
  }

  /* drawing */

  gc = DefaultGC(ev->d, DefaultScreen(ev->d));

  wndDrawProc(dfx, ev->d, w, gc);
}

static void wndExposeEvent(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XExposeEvent *ev;
  struct expose_data data;

  ev = (XExposeEvent*)event;

  data.d = ev->display;
  data.count = ev->count;
  data.rc.x = ev->x;
  data.rc.y = ev->y;
  data.rc.width = ev->width;
  data.rc.height = ev->height;

  wndExpose(w, &data);
}

static void wndGraphExposeEvent(Widget w, XEvent *event)
{
  XGraphicsExposeEvent *ev;
  struct expose_data data;

  ev = (XGraphicsExposeEvent*)event;

  data.d = ev->display;
  data.count = ev->count;
  data.rc.x = ev->x;
  data.rc.y = ev->y;
  data.rc.width = ev->width;
  data.rc.height = ev->height;

  wndExpose(w, &data);
}

static void wndDrawProc(DfWindow *dfx, Display *display, Widget w, GC gc)
{
  DfBuf *start;
  DfBuf *buf;
  XRectangle clip = dfx->clip;
  struct DfDrawInfo di;

  start = buf = dfx->lists;
  di.d = display;
  di.w = w;
  di.gc = gc;
  do{
    if(IS_VISIBLE(buf)){
      if(RectIsOverlap(&clip, &buf->rc)){
	di.clip = clip;
	RectIntersect(&di.clip, &buf->rc);
	SetClipRect(display, gc, &di.clip);
	buf->v->draw(buf, &di);
	SetClipRect(display, gc, &di.clip);
	DrawBorder(buf, display, w, gc);
      }
    }
    buf = NextBuffer(buf);
  }while(start != buf);


  dfx->flags &= ~DFX_CLIPPING;
  ClearClip(display, gc);
}

static void wndFocus(Widget w, XtPointer p, XEvent *event, Boolean *cont){
  XFocusChangeEvent *ev;

  ev = (XFocusChangeEvent*)event;
  mod_keys = 0;
}

static void wndResize(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XConfigureEvent *ev;
  DfBuf *b;
  XRectangle r;
  int draw;
  int width;
  int height;
  int bottom;

  switch(event->type){
  case ConfigureNotify:
    ev = (XConfigureEvent*)event;
    width = ev->width;
    height = ev->height - vnFontHeight;/* for status bar */
    height -= dfx->text_height;
    bottom = dfx->height - vnFontHeight;/* for status bar */
    bottom -= dfx->text_height;

    if(dfx->text){
      XMoveResizeWindow(XtDisplay(dfx->text), XtWindow(dfx->text),
			0, ev->height - dfx->text_height,
			ev->width, dfx->text_height);
      /*
      XtResizeWidget(dfx->text, ev->width, dfx->text_height, 1);
      XtMoveWidget(dfx->text, 0, ev->height - dfx->text_height);
       */
    }

    b = dfx->lists;
    do{
      RectPos(&r, &b->rc);
      switch(b->type){
      case DT_PROMPT:
	b->rc.y = height;
	b->rc.width = width;
	break;

      default:
	draw = 0;
	if(dfx->width != width){
	  if(width <= b->rc.x){
	    b->stat &= ~BS_VISIBLE;
	    continue;
	  }
	  if(width < r.width){
	    b->rc.width = width - r.x;
	    draw = 1;
	  }
	  if(dfx->width != width && dfx->width == r.width){
	    b->rc.width = width - r.x;
	    draw = 1;
	  }
	}
	if(dfx->height != ev->height){
	  if(height <= b->rc.y){
	    b->stat &= ~BS_VISIBLE;
	    continue;
	  }
	  if(height <= r.height){
	    b->rc.height = height - r.y;
	    draw = 1;
	  }
	  if(bottom == r.height){
	    b->rc.height = height - r.y;
	    draw = 1;
	  }
	}
	if(draw){
	  if(b->v->resize){
	    b->v->resize(w, b);
	  }
	  RedrawFrame(w, b, &b->rc);
	}
      }
    }while(b = NextBuffer(b), b != dfx->lists);

    dfx->width = ev->width;
    dfx->height = ev->height;
    break;
  }
}


/* filer window */

static int Wnd_Create(Widget parent, int cx, int cy)
{
  Widget w;

  w = XtVaCreateManagedWidget("main", simpleWidgetClass, parent,
			      XtNbackground, vOptions.colors[COLOR_BK],
			      XtNwidth, cx,
			      XtNheight, cy,
			      NULL);
  if(!w){
    return 1;
  }
  dfx = bMalloc(sizeof(*dfx));
  if(!dfx){
    return 1;
  }
  Wnd_Initialize(w, dfx);

  dfx->width = cx;
  dfx->height = cy;
  dfx->frame_y = 0;
  dfx->frame_bottom = cy;

  return 0;
}


static int init_file_list(Widget w)
{
  DfFileList *list;
  char buf[256];/* for current directory */

  list = bMalloc(sizeof(*list));
  if(!list){
    dfx = NULL;
    bFree(dfx);
    return 1;
  }
  dfx->lists = GetBuffer(list);

  FL_Init(list);
  dprintf("1st filer-1: %d\n", list->b.b.type == DT_FILER);
  list->n = 1;
  DfBufCommon_Init(&list->b);
  list->b.b.rc.x = 0;
  list->b.b.rc.y = 0;
  list->b.b.rc.width = dfx->width;
  list->b.b.rc.height = dfx->height - vnFontHeight;

  list->b.b.stat = BS_VISIBLE | BS_ENABLE | BS_DRAWBOTTOMEDGE;

  dfx->nBuf++;
  dfx->nFiler++;

  if(!getcwd(buf, sizeof(buf))){
    bFree(list);
    bFree(dfx);
    dfx = NULL;
    return 1;
  }
  MakeStatus(toplevel);
  dprintf("1st filer-2: %d\n", list->b.b.type == DT_FILER);
  dprintf("get file list.\n");
  GetFiles(list, w, buf, NULL, NULL, 1);

  list->sort = SORT_NAME;
  SortFileList(list);
  dprintf("setup done.\n");
  St_PresetMsg(vOptions.msg[STM_READY]);
  SetActiveFrame(NULL, GetBuffer(list));
  InvalidateRect(w, &GetBuffer(list)->rc);

  handle_event_list();

  return 0;
}

static void Wnd_Initialize(Widget w, DfWindow *dfx)
{

  /* initialize */
  memset(dfx, 0, sizeof(*dfx));

  dfx->nBuf = 0;
  dfx->w = w;

  /* add event handler */
  XtAddEventHandler(w, KeyPressMask, False, wndKeyPressProc, NULL);
  XtAddEventHandler(w, KeyReleaseMask, False, wndKeyReleaseProc, NULL);
  XtAddEventHandler(w, ExposureMask, False, wndExposeEvent, NULL);
  XtAddEventHandler(w, FocusChangeMask, False, wndFocus, NULL);
  XtAddEventHandler(w, StructureNotifyMask, False, wndResize, NULL);
  XtAddEventHandler(w, KeymapStateMask, False, wndKeymap, NULL);
  XtAddEventHandler(w, NoEventMask, True, wndEvents, NULL);
}

static void handle_event_list(void)
{
  DfTask *t;
  DfList *l;

  dprintf("handleClientMessage: attempt to mutex_lock\n");
  pthread_mutex_lock(&vMtxNotify);
  dprintf("handleClientMessage: mutex_locked\n");
  while(!ListIsEmpty(&vNotifyTaskList)){
    l = ListNext(&vNotifyTaskList);
    t = CONTENT_OF(DfTask, link_notify, l);
    dprintf("get from list %p.\n", t);
    ListDel(l);
    pthread_mutex_unlock(&vMtxNotify);
    dprintf("handleClientMessage: mutex_unlocked\n");
    t->callback(t);
    pthread_mutex_lock(&vMtxNotify);
    dprintf("handleClientMessage: mutex_locked\n");
  }
  pthread_mutex_unlock(&vMtxNotify);
  dprintf("handleClientMessage: mutex_unlocked\n");
}

static void handleClientMessage(Widget w, XClientMessageEvent *ev)
{
  handle_event_list();
  switch(ev->message_type){
  case DRQ_STMSG:
    St_SetMsg(w, ev->data.l[1] ? strerror(ev->data.l[1]) : vOptions.msg[ev->data.l[0]]);
    break;
  default:
    break;
  }
  dprintf("handleClientMessage: done.\n");
}


static void wndEvents(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  switch(event->type){
  case ClientMessage:
    handleClientMessage(w, &event->xclient);
    break;
  case GraphicsExpose:
    dprintf("GraphicsExpose\n");
    wndGraphExposeEvent(w, event);
    break;
  }
}


static void trapSigchld(int c)
{
  int stat;

  signal(SIGCHLD, SIG_IGN);
  wait3(&stat, 0, NULL);
  signal(SIGCHLD, trapSigchld);
}


static void actionQuit(Widget w, XEvent *ev, String *str, Cardinal *num)
{
  exit(0);
}

struct filter_info{
  Window w;
};

static Bool filterEvent_cb(Display *d, XEvent *ev, XPointer ptr)
{
  struct filter_info *i = (void*)ptr;

  if(ev->xany.window != i->w){
    return False;
  }
  if(ev->xany.type != GraphicsExpose){
    return False;
  }

  return True;
}

int IsPendingExpose(Widget w)
{
  struct filter_info i;
  XEvent ev;
  int pend = 0;

  i.w = XtWindow(w);
  while(XCheckIfEvent(XtDisplay(w), &ev, filterEvent_cb, (XPointer)&i)){
    wndGraphExposeEvent(w, &ev);
    pend = 1;
  }
  return pend;
}
