#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <pthread.h>
#include <dirent.h>
#include <unistd.h>

#include <X11/Intrinsic.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "task.h"
#include "status.h"
#include "mem.h"
#include "dir.h"
#include "dfxfunc.h"
#include "dfval.h"
#include "misc.h"
#include "debug.h"

typedef struct _tmpList{
  DfFile *files;
  DfStr s;
  DfStr org_path;
  int num;
  int alloced;
  int cwd_len;
  df_stat ds;
  void (*callback)(void *);
  void *priv;
}tmplist;

static int SetupListing(tmplist *tmp, DfTask *t, void (*callback)(void *), void *ptr);

static int FinishListing(DfFileList *pList, tmplist *tmp);
static int FailedListing(DfFileList *list, tmplist *tmp);

static task_result readdirThreadedInvoke(DfTask *t);
static task_result readdirDone(DfTask *t);
static int readdirScan(DfTask *t);
static int readdirInit(DfTask *t);
static int readdirPredir(DfTask *t, DfStr*, DfStr *);
static int readdirAddlist(DfTask *t, DfStr*, DfStr*, struct stat*);
static int scanDirectory(DfTask *t, DIR *dir, DfStr *cwd);

task_vtable vtable_readdir = {
  readdirThreadedInvoke,
  commonNotify,
  readdirDone,
  readdirInit,
  readdirScan,
  commonProcSubDirectories,
  readdirAddlist,
  readdirPredir,
  NULL
};

static task_result readdirDone(DfTask *t)
{
  DfFileList *list;
  tmplist *tmp;

  list = GetFilerBuffer(t->owner);
  dprintf("readirDone: scan done.\n");
  tmp = t->extend;
  if(tmp->callback){
    dprintf("readirDone: callback.\n");
    (*tmp->callback)(tmp->priv);
  }
  SetFilerState(t->w, list, tmp->ds);
  Str_Free(&tmp->org_path, 0);

  if(t->err){
    St_SetMsg(t->w, strerror(t->err));
  }else{
    St_SetMsg(t->w, vOptions.msg[STM_SCANDONE]);
  }

  return TASK_FREE;
}

static int readdirInit(DfTask *t)
{
  return 0;
}

static int readdirScan(DfTask *t)
{
  int s_idx;
  DfStr src;
  DIR *dir;
  DfFileList *list;

  setupPathes(t, &src, NULL, 1);
  s_idx = Str_Length(&src);

  list = GetFilerBuffer(t->owner);
  dir = opendir(Str_Get(&src));
  if(dir != NULL){
    scanDirectory(t, dir, &src);
    FinishListing(list, t->extend);
    CalcColumnWidth(list);
    ApplyFilter(t->w, list, list->mode & MODE_IFILTER);
  }else{
    t->err = errno;
    dprintf("opendir() is failed. %s.\n", strerror(errno));
    FailedListing(list, t->extend);
  }

  /*  it's temporary cursor fix. */
  if(list->b.nItems <= list->b.nCur){
    list->b.nStart = 0;
    list->b.nCur = 0;
    if(list->b.nItems){
      list->b.nCur = list->b.nItems - 1;
    }
  }

  Str_Free(&src, 0);
  DlgSendEvent(t, taskCbDone);

  return 1;
}

static int scanDirectory(DfTask *t, DIR *dir, DfStr *cwd)
{
  int s_idx;
  int ret;
  struct stat st;
  struct stat *pst;
  struct dirent *d;
  int result = 0;

  s_idx = Str_Length(cwd);
  while((d = readdir(dir)) != NULL){
    if(t->break_flg != 0){
      break;
    }
    do{
      Str_Overwrite(cwd, s_idx, d->d_name);
#if DEBUG
      if(strcmp(d->d_name, "err_stat") == 0){
	t->err = EACCES;
	result = 1;
	pst = NULL;
	break;
      }
#endif
      if(lstat(Str_Get(cwd), &st)){
	t->err = errno;
	result = 1;
	pst = NULL;
	break;
      }
      pst = &st;
      if(S_ISDIR(st.st_mode)){
	if(t->flag & DLGF_SUBDIR){
	  if(initSubDir(t, Str_Get(cwd), NULL)){
	    t->err = 0;
	    result = 0;
	  }else{
	    t->state = STATE_FAIL;
	    result = 1;
	  }
	}
      }
    }while(0);

    ret = t->v->fileop(t, cwd, NULL, pst);
    if(ret != 0){
      t->err = ret;
      result = 1;
      break;
    }
    t->update = 1;
  }

  closedir(dir);

  return result;
}

static int readdirPredir(DfTask *t, DfStr *src, DfStr *dst)
{
  if(t->sub){
    return 1;
  }
  return 0;
}

static int readdirAddlist(DfTask *t, DfStr *src, DfStr *dst, struct stat *st)
{
  tmplist *tmp = t->extend;
  DfFile *tmp_files;
  DfFile *f;
  int n;
  int c;
  DfFile *ra;
  mode_t mode;
  char buf[32];
  int len;
  struct stat link_st;
  const char *path;
  const char *filename;

  path = Str_Get(src);
  filename = path + tmp->cwd_len;

#if DEBUG
  if(strcmp(filename, "err_add") == 0){
    t->state = STATE_FAIL;
    return 1;
  }
#endif

  /* extend list */
  n = tmp->num;
  tmp_files = tmp->files;
  if(tmp->alloced <= n){
    ra = bReAlloc(tmp_files, (sizeof(*tmp_files) * (tmp->alloced * 2)));
    if(!ra){
      t->state = STATE_FAIL;
      return 1;
    }
    tmp_files = ra;
    tmp->alloced *= 2;
  }

  f = &tmp_files[n];
  f->link = NULL;
  f->flags = 0;
  if(st){
    f->r_attr = st->st_mode;

    c = COLOR_FILE;
    mode = GetMode(st);
    switch(st->st_mode & S_IFMT){
    case S_IFREG:
      c = COLOR_FILE;
      if(mode & 1){
        c = COLOR_X;
      }else if((mode & 2) == 0){
        c = COLOR_RO;
      }
      break;
    case S_IFDIR:
      c = COLOR_DIR;
      break;
    case S_IFLNK:
      c = COLOR_LINK;
      Str_Extend(&tmp->s, 512);

      f->link = (char *)(Str_Length(&tmp->s));/* hold index */
      stat(path, &link_st);
      f->r_attr = link_st.st_mode;
      len = readlink(path, Str_Get(&tmp->s) + Str_Length(&tmp->s), 512);
      if(len < 0){
        fprintf(stderr, "readlink %s:%s\n", path, strerror(errno));
      }else{
        Str_SetLength(&tmp->s, Str_Length(&tmp->s) + len);
        Str_AddChar(&tmp->s, '\0');
      }
      break;

    case S_IFBLK:
    case S_IFCHR:
      c = COLOR_DEV;
      break;
    }
    f->attr = st->st_mode;
    f->size = st->st_size;
    f->owner = st->st_uid;
    f->group = st->st_gid;
    f->atime = st->st_atime;
    f->mtime = st->st_mtime;
    f->ctime = st->st_ctime;
  }else{
    f->flags |= DFFL_NOSTAT;
    c = COLOR_UNKNOWN;
    f->attr = 0;
    f->size = 0;
    f->owner = 0;
    f->group = 0;
    f->atime = 0;
    f->mtime = 0;
    f->ctime = 0;
  }

  f->ext = NULL;
  f->color = vOptions.colors[c];
  f->nSelect = 0;
  f->name = (char *)(Str_Length(&tmp->s));/* hold index */

#if DRAW_SIZE
  f->cx[FILECX_NAME] = XmbTextEscapement(vFontSet, filename, strlen(filename));
  len = StrSize(buf, sizeof(buf), f->size);
  f->cx[FILECX_SIZE] = XmbTextEscapement(vFontSet, buf, len);
  len = StrDate(buf, sizeof(buf), &f->mtime);
  f->cx[FILECX_DATE] = XmbTextEscapement(vFontSet, buf, len);
#endif
  Str_Add(&tmp->s, filename);
  Str_AddChar(&tmp->s, '\0');

  tmp->files = tmp_files;

  tmp->num++;

  return 0;
}

static void *operation_thread(void *ptr)
{
  DfTask *t;

  t = ptr;

  readdirScan(t);

  return NULL;
}

static task_result readdirThreadedInvoke(DfTask *t)
{

  St_SetMsg(t->w, vOptions.msg[STM_SCANNING]);
  t->flag |= DLGF_THREAD;
  pthread_create(&t->t_info, NULL, operation_thread, t);

  return TASK_LEAVE;
}

int GetFiles(DfFileList *list, Widget w, const char *new_dir, void (*callback)(void *), void *ptr, int sync)
{
  DfTask *t;
  DfStr cwd;
  int idx;
  char numstr[10];
  tmplist *tmp;

  t = MakeTask(w, GetBuffer(list), DO_READDIR, 0, sizeof(tmplist));
  dprintf("1st filer: %d\n", t->owner->type == DT_FILER);
  dprintf("1st filer: %d\n", list->b.b.type == DT_FILER);

  if(new_dir){
    if(IsAbsPath(new_dir)){
      Str_InitStr(&cwd, new_dir);
    }else{
      Str_InitStr(&cwd, list->cwd);
      SetLastSlash(&cwd);
      Str_Add(&cwd, new_dir);
    }
  }else{
    Str_InitStr(&cwd, list->cwd);
  }

  Str_RegPath(&cwd, 1/* add slash */, 0);

  /* backup current directory and caption */
  tmp = t->extend;
  Str_InitStr(&tmp->org_path, Str_Get(&list->b.caption));

  /* edit captin */
  dprintf("GetFiles: scan %s.\n", Str_Get(&cwd));
  if(Str_Get(&list->b.caption)){
    dprintf("phase 1\n");
    idx = list->cwd - Str_Get(&list->b.caption);
    Str_Overwrite(&list->b.caption, idx, Str_Get(&cwd));
  }else{
    dprintf("initial scan.\n");
    list->b.caption = t->cwd;
    Str_InitNull(&t->cwd);
    Str_InitStr(&list->b.caption, Str_Get(&cwd));
    dprintf("%s\n", Str_Get(&t->cwd));
    idx = sprintf(numstr, "%d:", list->n);
    Str_Insert(&list->b.caption, 0, numstr);
  }
  list->cwd = Str_Get(&list->b.caption) + idx;
  dprintf("GetFiles/setup. now caption [%s].\n", Str_Get(&list->b.caption));
  dprintf("GetFiles/setup. cwd is [%s].\n", list->cwd);

  t->list = list;
  Str_InitStr(&t->cwd, Str_Get(&cwd));
  t->flag = DLGF_UPDATECWD;
  t->files = Str_Get(&cwd);
  t->now_proc = t->files;

  t->v = &vtable_readdir;

  SetupListing(t->extend, t, callback, ptr);

  dprintf("GetFiles attempt invoke.\n");
  if(sync){
    readdirScan(t);
  }else{
    t->v->invoke(t);
  }
  dprintf("GetFiles invoke done.\n");

  return 1;
}

static int SetupListing(tmplist *tmp, DfTask *t, void (*callback)(void *), void *ptr)
{
  DfFileList *list;

  Str_Init(&tmp->s);

  tmp->files = bMalloc(sizeof(DfFile) * 32);
  tmp->alloced = 32;
  tmp->num = 0;
  tmp->cwd_len = Str_Length(&t->cwd);
  tmp->callback = callback;
  tmp->priv = ptr;

  dprintf("1st filer: %d\n", t->owner->type == DT_FILER);
  list = GetFilerBuffer(t->owner);
  dprintf("setuplist: %p %p.\n", list, t->owner);
  tmp->ds = SetFilerState(t->w, list, DS_WAIT_DIR);
  dprintf("listing start.\n");

  return 0;
}

static int FinishListing(DfFileList *list, tmplist *tmp)
{
  int i;
  int max;
  DfFile *f;
  void *p;

  bFree(list->raw);
  list->raw = NULL;

  max = tmp->num;

  if(tmp->num){
#if 0
    list->raw = tmp->files;
#else
    p = bReAlloc(tmp->files, sizeof(DfFile) * tmp->num);
    list->raw  = p ? p : tmp->files;
#endif
  }else{
    bFree(tmp->files);
    list->raw = NULL;
  }

  if(list->files == NULL){
    if(max){
      list->files = bMalloc(sizeof(DfFile*) * max);
    }
  }else{
    if(max){
      p = bReAlloc(list->files, sizeof(DfFile*) * max);
      list->files = p ? p : list->files;
    }else{
      bFree(list->files);
      list->files = NULL;
    }
  }

  Str_Shrink(&tmp->s);
  bFree(list->pszFilenames);
  list->pszFilenames = Str_Get(&tmp->s);

  /*
   * Adjust index
   */
  f = list->raw;
  for(i = max; i; i--){
    f->name += (long int)list->pszFilenames;
    if(f->link){
      f->link += (long int)list->pszFilenames;
    }
    f->ext = FindLastChar(f->name, '.');
    f++;
  }

  list->nHasFiles = max;
  list->b.nItems = max;
  list->nSelects = 0;
  list->nHoriz = 0;

  dprintf("listing done.\n");
  return 1;
}

static int FailedListing(DfFileList *list, tmplist *tmp)
{
  int idx;

  Str_Free(&tmp->s, 0);
  bFree(tmp->files);
  idx = list->cwd - Str_Get(&list->b.caption);
  Str_Overwrite(&list->b.caption, 0, Str_Get(&tmp->org_path));
  list->cwd = Str_Get(&list->b.caption) + idx;

  return 0;
}

int CalcColumnWidth(DfFileList *list)
{
  DfFile *f;
  int i;
  int max;
  int c;
  int cx[FILECX_MAX] = {0};

  max = list->nHasFiles;

  f = list->raw;

  for(i = 0; i < max; i++){
    c = FILECX_MAX;
    do{
      c--;
      if(cx[c] < f->cx[c]){
	cx[c] = f->cx[c];
      }
    }while(c);
    f++;
  }

  memcpy(list->cx, cx, sizeof cx);

  return 0;
}

int ApplyFilter(Widget w, DfFileList *list, int ifilter)
{
  DfFile *f;
  int n;
  int i;
  int max;
  int cursor_moved = 0;

  max = list->nHasFiles;

  f = list->raw;
  n = 0;
  for(i = 0; i < max; i++){
    do{
      if(!(list->mode & SHOW_DOTS) && f->name[0] == '.') {
	break;
      }

      if(list->filter){
	if(!S_ISDIR(f->attr)){
	  if(ifilter == 0){
	    /* require match */
	    if(IsMatch(list->filter, f->name) != MATCH){
	      break;
	    }
	  } else {
	    /* require i-filter */
	    if(IsMatch(list->filter, f->name) == UNMATCH){
	      break;
	    }
	  }
	}
      }

      list->files[n] = f;
      n++;
    }while(0);
    f++;
  }
  if(list->b.nItems != n){
    cursor_moved = 1;
  }
  list->b.nItems = n;

  if(list->b.nItems <= list->b.nCur){
    cursor_moved = 1;
  }

  BF2_CalcCursors(&list->b);
  if(cursor_moved){
    CalcBufferOffset(w, &list->b, list->b.nCur);
    SetCursor(w, &list->b, list->b.nCur);
  }
  return cursor_moved;
}

unsigned int GetMode(struct stat *st)
{
  unsigned int mode;

  mode = st->st_mode & 0007;

  if(st->st_uid == v_uid){
    mode |= (st->st_mode >> 6) & 7;
  }
  if(st->st_gid == v_gid){
    mode |= (st->st_mode >> 3) & 7;
  }

  return mode;
}


