#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "task.h"
#include "info.h"
#include "cmd.h"
#include "cmdlist.h"
#include "mem.h"
#include "cfg.h"
#include "extmacro.h"
#include "dfxfunc.h"
#include "xutil.h"
#include "misc.h"
#include "debug.h"
#include "dfval.h"


enum{
  BLANK,
  NAME,
  PRECOMMAND,
  COMMAND,
  IGN
};

extern const DfCmdStr *FL_CmdTab;

static int Cl_Free(void *list);
static int Cl_Active(Widget w, void *ptr);
static int Cl_Inactive(Widget w, void *ptr);
static int Cl_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv);

static int doOpen(Widget w, DfxCmdList *list, const char **param);

static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur);

int listInit(void *ptr);
int listFree(void *ptr);
static int parseFile(int fd, DfxList *clist);


static const DfVerb2 CL_Verb = {
  {
    Cl_Free,
    Cl_Active,
    Cl_Inactive,
    BF2_KeyPress,
    BF2_ResizeDraw,
    BF2_Draw,
  },
  BF2_DrawCaption,
  drawSingleItem,
  Cl_Process,
  SetCursor,
};


static int Cl_Free(void *ptr)
{
  DfxCmdList *list;

  list = ptr;

  listFree(&list->list);
  bFree(list);

  return 0;
}

static int Cl_Active(Widget w, void *ptr)
{
  XRectangle rc;
  DfxCmdList *list;
  struct DfDrawInfo di;

  list = ptr;

  if(GetBuffer(ptr)->type == DT_CMDLIST){
    CmdSetTable(CMD_CMDLIST);
  }else{
    CmdSetTable(CMD_PATHLIST);
  }

  calcCursorRect(&list->b, list->b.nCur, &rc);
  if(w){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
  }
  return 0;
}

static int Cl_Inactive(Widget w, void *ptr)
{
  DfxCmdList *list;
  struct DfDrawInfo di;

  list = ptr;

  if(w && IS_VISIBLE(list)){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
  }
  return 0;
}


static int Cl_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv)
{
  DfxCmdList *list;
  int redraw;
  int pre_cur;
  int cmd;

  cmd = c->builtin;
  redraw = 0;

  list = ptr;
  pre_cur = list->b.nCur;

  switch(cmd){
  case QUIT:
    KillBuffer(w, &list->b.b);
    break;
  case CLOSE:
    KillBuffer(w, &list->b.b);
    break;
  case CURUP:
    upCursor(w, &list->b);
    break;
  case CURDOWN:
    downCursor(w, &list->b);
    break;
  case OPEN:
    redraw = doOpen(w, list, argv);
    break;
  case PAGEUP:
    upPage(w, &list->b);
    break;
  case PAGEDOWN:
    downPage(w, &list->b);
    break;
  case TOP:
    cursorToTop(w, &list->b);
    break;
  case BOTTOM:
    cursorToBottom(w, &list->b);
    break;
  case INFO:
    ToggleInfo(w, GetFilerBuffer(GetBuffer(list)));
    SetCursor(w, &list->b, list->b.nCur);
    break;
  default:
    return CREQ_TOPARENT;
  }

  if(redraw){
    RedrawInfo(w);
    RedrawFrame(w, GetBuffer(list), &list->b.b.rc);
  }

  return CREQ_DONE;
}


#if DEBUG
static void  put_strs(char *p)
{
  int len;
  int n;

  n = 0;
  while(*p){
    len = strlen(p) + 1;
    dprintf("%d:[%s]\n", ++n, p);
    p += len;
  }
}
#endif

DfBuf *MakeCmdList(DfBuf *parent, const char *filename)
{
  DfxCmdList *buf;
  DfBuf *b;

  if(!parent){
    return NULL;
  }

  buf = MakeBuffer(sizeof(DfxCmdList), parent, DT_CMDLIST, &CL_Verb);
  if(!buf){
    return NULL;
  }

  b = GetBuffer(buf);

  SetupCmdList(buf, filename);

  if(buf->b.nItems == 0){
    Cl_Free(buf);
    return NULL;
  }

  AddBuffer(parent, b);

  return GetBuffer(buf);
}

DfBuf *MakePathList(DfBuf *parent, const char *filename)
{
  DfxCmdList *buf;
  DfBuf *b;

  if(!parent){
    return NULL;
  }

  buf = MakeBuffer(sizeof(DfxCmdList), parent, DT_PATHLIST, &CL_Verb);
  if(!buf){
    return NULL;
  }

  b = &buf->b.b;
  b->stat &= ~(BS_VISIBLE | BS_ENABLE);
  b->rc = parent->rc;

  SetupCmdList(buf, filename);

  if(buf->b.nItems == 0){
    Cl_Free(buf);
    return NULL;
  }

  BF2_CalcCursors(&buf->b);
  AddBuffer(parent, b);

  return GetBuffer(buf);
}



int SetupCmdList(DfxCmdList *clist, const char *menufile)
{
  DfBufCommon *c;
  int fd;
  const char *filename;
  int n;

  switch(clist->b.b.type){
  case DT_CMDLIST:
    filename = "cmd";
    n = CAP_COMMAND;
    break;
  case DT_PATHLIST:
    filename = "path";
    n = CAP_PATH;
    break;
  default:
    dprintf("%s:%s(%d)\n", __FILE__, __FUNCTION__, __LINE__);
    return 0;
  }

  listInit(&clist->list);
  c = &clist->b;
  DfBufCommon_Init(c);
  Str_InitPsz(&c->caption, vOptions.msg[n]);

  if(menufile && *menufile){
    dprintf("menufile [%s]\n", menufile);
    fd = open(menufile, O_RDONLY);
  }else{
    fd = CfgOpenFile(filename);
  }

  if(fd == -1){
    return 0;
  }

  /* allocate for commands memories */
  Str_Init(&clist->list.name);
  Str_Init(&clist->list.cmds);

  clist->b.nItems = parseFile(fd, &clist->list);
  close(fd);
  BF2_CalcCursors(&clist->b);

#if DEBUG
  if(clist->b.nItems){
    put_strs(Str_Get(&clist->list.name));
    put_strs(Str_Get(&clist->list.cmds));
  }
#endif

  return 0;
}


static void callbackOpen(DfTask *t)
{
  struct parse_extention *pe = t->extend;
  int builtin;
  int fail;

  dprintf("cmdlist: callback %s[%s]\n", pe->immediate ? "immediate " : "", Str_Get(&pe->cmd));
  if(pe->immediate){
    fail = DfxExec(t->w, t->owner, Str_Get(&t->cwd), Str_Get(&pe->cmd), &builtin);
    Str_Free(&pe->cmd, 0);
    if(fail == 0 && builtin == 0){
      KillChildren(t->w, GetBuffer(GetFilerBuffer(t->owner)));
    }
  }else{
    DoExecStr(t->w, t->owner, &pe->cmd);
  }
}

static int doOpen(Widget w, DfxCmdList *list, const char **param)
{
  int cur;
  char *p;
  DfStr c;
  DfFileList *l;

  cur = list->b.nCur;

  p = Str_Get(&list->list.cmds);
  if(list->b.nItems <= cur){
    return 0;
  }

  p = skipPsz(p, cur);

  if(!*p){
    return 0;
  }

  l = GetFilerBuffer(GetBuffer(list));
  switch(list->b.b.type){
  case DT_CMDLIST:
    dprintf("[CMD-1]%d:%s\n", cur,  p);

    if(l == NULL){
      break;
    }

    Str_InitStr(&c, p);
    ParseMacro2(w, &c, GetBuffer(list), callbackOpen, NULL);
    break;

  case DT_PATHLIST:
    if(l){
      chDir(w, l, p);
    }
    KillBuffer(w, GetBuffer(list));
    break;
  default:
    dprintf("%s:%s(%d)\n", __FILE__, __FUNCTION__, __LINE__);
    return 0;
  }

  return 0;
}


static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur)
{
  DfxCmdList *list;
  long int fg;
  long int bg;
  XRectangle rc;
  int draw_line;
  char *cmd;

  dprintf("cmdlist:draw\n");

  list = p;

  if(!list->list.idx){
    return;
  }
  cmd = list->list.idx[cur];
  if(!*cmd){
    return;
  }

  draw_line = 0;
  fg = vOptions.colors[COLOR_FILE];
  bg = vOptions.colors[COLOR_BK];
  if(cur == list->b.nCur){
    if(dfx->active == GetBuffer(list)){
      fg = vOptions.colors[COLOR_CURFG];
      bg = vOptions.colors[COLOR_CURBG];
    }else{
      draw_line = 1;
    }
  }
  calcCursorRect(&list->b, cur, &rc);
  XSetForeground(di->d, di->gc, bg);
  XFillRectangle(di->d, XtWindow(di->w), di->gc, rc.x, rc.y, rc.width, rc.height);

  XSetForeground(di->d, di->gc, fg);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, rc.x + 4, rc.y + vnFontBase, cmd, strlen(cmd));
  if(draw_line){
    XSetForeground(di->d, di->gc, vOptions.colors[COLOR_CURBG]);
    XDrawLine(di->d, XtWindow(di->w), di->gc, rc.x, rc.y + rc.height - 1, rc.x + rc.width - 1, rc.y + rc.height - 1);
  }
}

DfxAppList *LoadAppsList(void)
{
  int fd;
  const char *filename = "apps";
  DfxAppList *Apps;

  fd = CfgOpenFile(filename);
  if(fd == -1){
    return NULL;
  }

  /* allocate for commands memories */
  Apps = malloc(sizeof(DfxAppList));
  if(!Apps){
    return NULL;
  }

  listInit(&Apps->list);
  Str_Init(&Apps->list.name);
  Str_Init(&Apps->list.cmds);

  Apps->num = parseFile(fd, &Apps->list);
  close(fd);

  return Apps;
}

int DiscardAppsList(DfxAppList *apps)
{
  listFree(&apps->list);
  free(apps);
  
  return 0;
}

int listInit(void *ptr)
{
  DfxList *list;

  list = ptr;
  Str_InitNull(&list->name);
  Str_InitNull(&list->cmds);
  list->idx = NULL;

  return 0;
}

int listFree(void *ptr)
{
  DfxList *list;

  list = ptr;

  Str_Free(&list->name, 0);
  Str_Free(&list->cmds, 0);
  bFree(list->idx);

  return 0;
}


static int parseFile(int fd, DfxList *clist)
{
  char *b;
  int len;
  MbStr mb;
  char *p;
  char *start;
  int left;
  int stat;
  DfStr tmp;
  DfStr *name;
  DfStr *cmds;
  int num;
  int i;

  b = bMalloc(4096);
  name = &clist->name;
  cmds = &clist->cmds;
  num = 0;

  mb.len = 0;
  mb.str = b;
  mb.c_len = 0;
  start = NULL;
  stat = BLANK;
  Str_Init(&tmp);
  do{
    len = read(fd, b, 4096 - mb.len);
    mb.str = b;
    mb.len = mb.len + len;
    mb.c_len = mblen(b, mb.len);
    p = b;
    if(len){
      left = 16;/* support for max 16bytes charset. EUC: 3, DBCS(Shift_JIS): 2 iso8859-1: 1 */
    }else{
      left = 0;/* EOF */
    }

    /* read buffer */
    while(left < mb.len){
      switch(stat){
      case BLANK:
	if(!IsSkipChar(&mb)){
	  stat = NAME;
	  start = p;
	  continue;
	}
	break;
      case NAME:
	if(mb.c_len == 1 && p[0] == ':'){
	  Str_AddLen(&tmp, start, p - start);
	  Str_Trim(&tmp);
	  Str_AddNUL(name, Str_Get(&tmp));
	  Str_SetLength(&tmp, 0);
	  stat = PRECOMMAND;
	  start = NULL;
	}
	break;
      case PRECOMMAND:
	if(!IsSkipChar(&mb)){
	  stat = COMMAND;
	  start = p;
	  continue;
	}
	break;
      case COMMAND:
	if(mb.c_len == 1 && p[0] == '\n'){
	  Str_AddLen(&tmp, start, p - start);
	  Str_Trim(&tmp);
	  Str_AddNUL(cmds, Str_Get(&tmp));
	  Str_SetLength(&tmp, 0);
	  stat = BLANK;
	  start = NULL;
	  num++;
	}
	break;
      case IGN:
	if(mb.c_len == 1 && p[0] == '\n'){
	  stat = BLANK;
	  start = NULL;
	}
      }
      p = MbNextChar(&mb);
    }
    if(start){
      Str_AddLen(&tmp, start, p - start);
      start = b;
    }
    memmove(b, mb.str, mb.len);
  }while(len);

  bFree(b);

  if(stat == COMMAND){
    Str_Trim(&tmp);
    Str_AddNUL(cmds, Str_Get(&tmp));
    num++;
  }
  Str_AddChar(name, '\0');
  Str_Shrink(name);
  Str_AddChar(cmds, '\0');
  Str_Shrink(cmds);
  clist->idx = bMalloc(sizeof(char*) * num);
  p = Str_Get(name);
  for(i = 0; i < num; i++){
    clist->idx[i] = p;
    len = strlen(p) + 1;
    p += len;
  }
  
  Str_Free(&tmp, 0);

  return num;
}

