#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h> 
#include <sys/stat.h> 

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "mem.h"
#include "cfg.h"
#include "debug.h"

typedef struct read_list{
  int *key_idx;
  int *val_idx;
  int num;
  int size;
  DfStr strKey;
  DfStr strVal;
}tmpReadList;

struct values{
  char *k;/* symbol */
  char *v;/* value */
};
typedef struct value_list{
  struct values *idx;
  int num;
  DfStr strKey;
  DfStr strVal;
  int refcnt;
}ValueList;

ValueList *vEnv;

static tmpReadList *parseFile(int fd);
static void init_tmp_list(tmpReadList *t);
static int extend(tmpReadList *t);

static ValueList *setup_value_list(tmpReadList *t);

typedef enum{
  NONE,
  SYM,
  QUOTE,
  PREVALUE,
  VALUE,
  IGN
}conf_stat;

static void dump_config(ValueList *t)
{
  int n;

  for(n = 0; n < t->num; n++){
    dprintf("%d: [%s],[%s]\n",
	    n,
	    t->idx[n].k,
	    t->idx[n].v);
  }
}

int LoadConfig(void)
{
  int fd;
  tmpReadList *t;
  ValueList *r;

  fd = CfgOpenFile("config");

  if(fd == -1){
    return 0;
  }

  t = parseFile(fd);

  close(fd);

  if(t == NULL){
    return 0;
  }
  r = setup_value_list(t);
  bFree(t);

  dump_config(r);
  r->refcnt = 1;
  vEnv = r;
  return 0;
}

static tmpReadList *parseFile(int fd)
{
  int len;
  conf_stat s;
  DfStr tmp;
  int found;
  MbStr mb;
  char *b;
  char *p;
  int left;
  char *start;
  int stat;
  tmpReadList *t;

  /* init */

  b = bMalloc(4096);

  s = NONE;
  found = 0;

  t = bMalloc(sizeof(tmpReadList));
  init_tmp_list(t);

  Str_Init(&tmp);

  mb.len = 0;
  mb.str = b;
  mb.c_len = 0;
  start = NULL;
  stat = NONE;
  do{
    len = read(fd, b, 4096 - mb.len);
    mb.str = b;
    mb.len = mb.len + len;
    mb.c_len = mblen(b, mb.len);
    p = b;
    if(len){
      left = 16;
    }else{
      left = 0;
    }
    while(left < mb.len){
      switch(stat){
      case NONE:
	if(mb.c_len != 1){
	  if(t->size <= t->num){
	    if(extend(t) != 0){
	      goto ERROR;
	    }
	  }
	  /* hold the beginning of symbol. */
	  t->key_idx[t->num] = Str_Length(&t->strKey);
	  stat = SYM;
	  start = p;
	  break;
	}
	switch(*p){
	case '\"':
	  stat = QUOTE;
	  start = p;
	  break;
	case '\r':
	case '\n':
	  break;
	default:
	  if(!IsSkipChar(&mb)){
	    if(t->size <= t->num){
	      if(extend(t) != 0){
		return NULL;
	      }
	    }
	    /* hold the beginning of symbol. */
	    t->key_idx[t->num] = Str_Length(&t->strKey);
	    stat = SYM;
	    start = p;
	  }
	  break;
	}
	break;

      case SYM:
	if(mb.c_len == 1){
	  switch(p[0]){
	  case '=':
	    Str_AddLen(&tmp, start, p - start);
	    Str_Trim(&tmp);
	    Str_AddNUL(&t->strKey, Str_Get(&tmp));
	    Str_SetLength(&tmp, 0);
	    stat = PREVALUE;
	    start = NULL;
	    break;
	  case '\"':
	    stat = QUOTE;
	    break;
	  }
	}
	break;
      case QUOTE:
	if(mb.c_len == 1 && p[0] == '\"'){
	  stat = SYM;
	}
	break;
      case PREVALUE:
	stat = VALUE;
	start = p;
	break;
      case VALUE:
	if(mb.c_len == 1){
	  switch(p[0]){
	  case '\r':
	  case '\n':
	    Str_AddLen(&tmp, start, p - start);
	    Str_Trim(&tmp);
	    /* hold the beginning of value. */
	    t->val_idx[t->num] = Str_Length(&t->strVal);
	    Str_AddNUL(&t->strVal, Str_Get(&tmp));
	    t->num++;
	    Str_SetLength(&tmp, 0);
	    stat = NONE;
	    start = NULL;
	  }
	}
	break;
      case IGN:
	if(mb.c_len == 1 && p[0] == '\n'){
	  stat = NONE;
	  start = NULL;
	}
      }
      p = MbNextChar(&mb);
    }
    if(start){
      Str_AddLen(&tmp, start, p - start);
      start = b;
    }
    memmove(b, mb.str, mb.len);
  }while(len);

  if(stat == VALUE){
    Str_Trim(&tmp);
    /* hold the beginning of value. */
    t->val_idx[t->num] = Str_Length(&t->strVal);
    Str_AddNUL(&t->strVal, Str_Get(&tmp));
    t->num++;
  }

  if(t->num == 0){
    goto ERROR;
  }
  Str_Free(&tmp, 0);
  bFree(b);

  return t;

ERROR:
  Str_Free(&tmp, 0);
  bFree(b);
  bFree(t->key_idx);
  Str_Free(&t->strKey, 0);
  bFree(t);

  return NULL;
}

static void init_tmp_list(tmpReadList *t)
{
  t->size = 32;
  t->key_idx = bMalloc(sizeof(int *) * 32);
  t->val_idx = bMalloc(sizeof(int *) * 32);
  t->num = 0;
  Str_Init(&t->strKey);
  Str_Init(&t->strVal);
}

static int extend(tmpReadList *t)
{
  int *k;
  int *v;
  int new_size;

  new_size = t->size + 32;
  k = bReAlloc(t->key_idx, sizeof(int*) * new_size);
  if(k == NULL){
    return 1;
  }
  v = bReAlloc(t->key_idx, sizeof(int*) * new_size);
  if(v == NULL){
    bFree(k);
    return 1;
  }

  t->size += 32;
  t->key_idx = k;
  t->val_idx = v;
  return 0;
}

static int comp_idx(const struct values *a, const struct values *b)
{
  return strcmp(a->k, b->k);
}

static ValueList *setup_value_list(tmpReadList *tmp)
{
  ValueList *t;
  struct values *idx;
  int i;
  int n;
  char *key;
  char *val;

  t = bMalloc(sizeof(*t) + sizeof(struct values) * tmp->num);
  t->idx = (void *)&t[1];

  idx = t->idx;
  n = tmp->num;
  key = Str_Get(&tmp->strKey);
  val = Str_Get(&tmp->strVal);
  for(i = 0; i < n; i++){
    idx[i].k = key + tmp->key_idx[i];
    idx[i].v = val + tmp->val_idx[i];
  }
  t->strKey = tmp->strKey;
  t->strVal = tmp->strVal;
  t->num  = n;
  dump_config(t);
  qsort(idx, n, sizeof(*idx), (int(*)(const void*, const void*))comp_idx);
  dump_config(t);

  return t;
}

static int comp_bsearch(const void *akey, const void *aobj)
{
  const char *key = (const char*)akey;
  const struct values *obj = aobj;

  return strcmp(key, obj->k);
}

const char *LookupValue(const char *key)
{
  struct values *p;

  if(!vEnv){
    return NULL;
  }

  p = bsearch(key, vEnv->idx, vEnv->num, sizeof(struct values), comp_bsearch);

  return p ? p->v : NULL;
}

