#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <utime.h>
#include <errno.h>

#include <strings.h>

#include <sys/types.h> 
#include <sys/stat.h> 

#include "config.h"
#include "str.h"
#include "fop.h"
#include "mem.h"
#include "debug.h"

#define FOP_BUFSIZE (16 * 1024)

static int deleteSubDir(DfStr *path, volatile int *flag);

int CopyFile(const char *src, const char *dst, volatile int *flag)
{
  struct stat st;
  int fin, fout;
  char *buf;
  char *p;
  int ret;
  int r;
  int w;
  struct utimbuf ut;

#if DEBUG
  buf = FindLastChar(src, '/');
  if(strcmp(buf, "/nocopy") == 0){
    dprintf("[TEST] no copying.\n");
    return EPERM;
  }
#endif

  dprintf("[COPY] %s to %s.\n", src, dst);
  ret = 0;
  fin = -1;
  fout = -1;
  buf = bMalloc(FOP_BUFSIZE);
  if(buf == NULL){
    ret = ENOMEM;
    goto CLEANUP;
  }
  if(lstat(src, &st) != 0){
    ret = errno;
    goto CLEANUP;
  }

  fin = open(src, O_RDONLY);
  if(fin == -1){
    ret = errno;
    goto CLEANUP;
  }

  fout = open(dst, O_CREAT | O_TRUNC | O_WRONLY, 0600);
  if(fout == -1){
    ret = errno;
    goto CLEANUP2;
  }

  while((r = read(fin, buf, FOP_BUFSIZE)) != 0){
    if(*flag){
      goto CLEANUP3;
    }
    if(r == -1){
      ret = errno;
      goto CLEANUP3;
    }

    for(p = buf; r; r -= w, p += w){
      w = write(fout, p, r);
      if(w == -1){
	ret = errno;
	goto CLEANUP3;
      }
    }
  }
  bFree(buf);
  close(fin);
  close(fout);

  ut.actime = st.st_atime;
  ut.modtime = st.st_mtime;
  utime(dst, &ut);

  chmod(dst, st.st_mode);

  return ret;

CLEANUP3:
  close(fout);
  DeleteFile(dst);

CLEANUP2:
  close(fin);

CLEANUP:
  bFree(buf);

  return ret;
}

int CopyLink(const char *src, const char *dst)
{
  struct stat st;
  char *buf;
  int ret;
  int r;

  ret = 0;

  if(lstat(src, &st) != 0){
    return errno;
  }
  if(!S_ISLNK(st.st_mode)){
    return EINVAL;
  }

  buf = bMalloc(st.st_size + 1);
  if(buf == NULL){
    return errno;
  }

  r = readlink(src, buf, st.st_size + 1);
  if(r != st.st_size){
    ret = errno;
    goto CLEANUP;
  }
  buf[st.st_size] = '\0';

  if(symlink(buf, dst)){
    ret = errno;
    goto CLEANUP;
  }
  dprintf("link [%s]\n", buf);
  bFree(buf);

  return ret;

CLEANUP:
  bFree(buf);

  return ret;
}


int DeleteFile(const char *name)
{
#if DEBUG
  char *p;
#endif

#if DEBUG
  p = FindLastChar(name, '/');
  if(strcmp(p, "/nodelete") == 0){
    dprintf("[TEST] no delete.\n");
    return EPERM;
  }
#endif
  if(unlink(name) != 0){
    return errno;
  }

  return 0;
}

int CreateDirectory(char *path)
{
  if(mkdir(path, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) == 0){
    return 0;
  }

  return errno;
}


int CreateNestDirectory(char *path)
{
  MbStr mb;
  char *p;
  int ret;
  char pre;
  struct stat st;

  InitMbStr(&mb, path);

  p = path;
  pre = '\0';

  if(*p == '/'){
    p = MbNextChar(&mb);
  }

  ret = 0;
  while(*p){
    if(*p != '/'){
      goto NEXT;
    }
    *p = '\0';
    if(lstat(path, &st) == 0){
      if(S_ISDIR(st.st_mode)){
	*p = '/';
	goto NEXT;
      }
    }

    ret = CreateDirectory(path);
    *p = '/';
    if(ret){
      return ret;
    }
NEXT:
    pre = *p;
    p = MbNextChar(&mb);
  }
  if(pre != '/'){
    ret = CreateDirectory(path);
  }
  return ret;
}

int DeleteDirectory(const char *path)
{
  int ret;

  ret = rmdir(path);
  if(ret){
    ret = errno;
  }
  return ret;
}

int DeleteSubDirectories(const char *path, volatile int *flag)
{
  DfStr s;
  int ret;

  Str_InitStr(&s, path);
  SetLastSlash(&s);
  ret = deleteSubDir(&s, flag);

  Str_Free(&s, 0);
  return ret;
}

static int deleteSubDir(DfStr *path, volatile int *flag)
{
  DIR *dir;
  struct dirent *de;
  struct stat st;
  int ret;
  int org_len;
  int add;

  if(*flag != 0){
    return 0;
  }
  org_len = Str_Length(path);
  ret = 0;
  SetLastSlash(path);
  add = Str_Length(path);

  dprintf("[STUB]del subdir [%s]\n", Str_Get(path));
  dir = opendir(Str_Get(path));
  if(dir){
    while((*flag == 0) && (de = readdir(dir)) != NULL){
      dprintf("[STUB] - %s\n", de->d_name);
      Str_Overwrite(path, add, de->d_name);
      lstat(Str_Get(path), &st);

      if(S_ISDIR(st.st_mode)){
	if(!IsDots(de->d_name)){
	  ret = deleteSubDir(path, flag);
	  if(ret){
	    goto EXIT;
	  }
	}
      }else{
	ret = DeleteFile(Str_Get(path));
      }
      if(ret){
	goto EXIT;
      }
    }
  EXIT:
    closedir(dir);
  }

  Str_SetLength(path, org_len);
  if(ret == 0 && *flag == 0){
    dprintf("[STUB]- %s\n", Str_Get(path));
    add = 0;
    if(1 < org_len && IsEndSlash(path)){
      Str_SetLength(path, org_len - 1);
      add = 1;
    }
    ret = DeleteDirectory(Str_Get(path));
    if(add){
      Str_AddChar(path, '/');
    }
  }
  return ret;
}


int ChMod(const char *path, int mode)
{
  int ret;

  ret = chmod(path, mode);

  if(ret){
    return errno;
  }

  return 0;
}

