#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <dirent.h>
#include <errno.h>
#include <sys/stat.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>
#include <X11/Shell.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "status.h"
#include "dialog.h"
#include "task.h"
#include "mem.h"
#include "misc.h"
#include "fop.h"
#include "dfxfunc.h"
#include "dfval.h"
#include "xutil.h"
#include "debug.h"

extern Widget toplevel;

/* callback handler of key input for Xt */
static void defaultKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void dlgKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont);
static void chmodKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont);

static void inputConfigureProc(Widget w, XtPointer p, XEvent *event, Boolean *cont);

static void closeDialog(DfDialog *t);

static void callbackOk(Widget w, XtPointer p, XtPointer q);
static void callbackYes(Widget w, XtPointer p, XtPointer q);
static void callbackNo(Widget w, XtPointer p, XtPointer q);
static void callbackOverwrite(Widget w, XtPointer p, XtPointer q);
static void callbackRetry(Widget w, XtPointer p, XtPointer q);
static void callbackSkip(Widget w, XtPointer p, XtPointer q);
static void callbackRename(Widget w, XtPointer p, XtPointer q);
static void callbackCancel(Widget w, XtPointer p, XtPointer q);

static void confirm(Widget w, DfDialog *d);
static void confirmInput(Widget w, DfDialog *d);
static void confirmButton(Widget w, DfDialog *d);
static void confirmChmodDialog(Widget w, DfDialog *d);

static void destroyDialog(DfDialog *d);
static void destroyInputDialog(DfDialog *d);

static void chmodLabelProc(Widget w, XtPointer p, XEvent *event, Boolean *cont);

static void makeCandidateList(DfInputDialog *d, Widget w);
static void forwardCandidateList(DfInputDialog *d, Widget w);

static void clearCandidateList(XtPointer p);
static int isSeparate(char p, const char *seps);

static struct dialog_vtable msg_vtable = {
  destroyDialog,/*destroy*/
  confirmButton/*confirm*/
};
static struct dialog_vtable input_vtable = {
  destroyInputDialog,/*destroy*/
  confirmInput/*confirm*/
};
static struct dialog_vtable chmod_vtable = {
  destroyDialog,/*destroy*/
  confirmChmodDialog/*confirm*/
};

DfMessageBox *MakeMessageBox(DfTask *t)
{
  DfMessageBox *d;

  d = bMalloc(sizeof(DfMessageBox));
  if(d == NULL){
    return NULL;
  }

  d->b.v = &msg_vtable;
  d->b.task = t;
  d->b.w = t->w;

  return d;
}

DfInputDialog *MakeInputDialog(DfTask *t, const char *str, int (*keyproc)(DfDialog *, Widget,  KeySym, char, Boolean *))
{
  DfInputDialog *d;

  d = bMalloc(sizeof(DfInputDialog));
  if(d == NULL){
    return NULL;
  }

  d->b.task = t;
  d->b.w = t->w;
  d->b.owner = t->owner;

  if(str){
    Str_InitStr(&d->input, str);
  }else{
    Str_InitNull(&d->input);
  }
  d->cand_ptr = NULL;
  Str_InitNull(&d->cand);
  d->b.v = &input_vtable;
  d->keyproc = keyproc;

  return d;
}

DfChmodDialog *MakeChmodDialog(DfTask *t)
{
  DfChmodDialog *d;

  d = bMalloc(sizeof(DfChmodDialog));
  if(d == NULL){
    return NULL;
  }

  d->b.v = &chmod_vtable;
  d->b.task = t;
  d->b.w = t->w;

  return d;
}

int ShowInputDialog(DfInputDialog *d, int prompt)
{
  char *p;
  int len;
  XtWidgetGeometry gmText;
  Widget text;

  p = Str_Get(&d->input);
  if(!p){
    p = "";
  }

  len = Str_Length(&d->input);

#if CACHE_TEXT
  if(text == NULL){
    text = XtVaCreateManagedWidget("text", asciiTextWidgetClass, toplevel,
				   XtNeditType, XawtextEdit,
				   XtNstring, p,
				   XtNinsertPosition, len,
				   XtNwidth, dfx->width,
				   NULL);
  }else{
    XtVaSetValues(text,
		  XtNstring, p,
		  XtNwidth, dfx->width,
		  NULL);
    XtVaSetValues(text,
		  XtNinsertPosition, len,
		  NULL);
  }
#else
  text = XtVaCreateWidget("text", asciiTextWidgetClass, toplevel,
			  XtNeditType, XawtextEdit,
			  XtNstring, p,
			  XtNinsertPosition, len,
			  XtNwidth, dfx->width,
 			  NULL);
#endif
  XtQueryGeometry(text, NULL, &gmText);
  dfx->text_height = gmText.height;
  XtVaSetValues(text,
		XtNy, dfx->height - gmText.height,
		NULL);

  dprintf("d->text is %p\n", text);
  d->text = text;
  dfx->text = text;

  d->b.task->flag |= DLGF_INPUT;

  XtAddEventHandler(text, KeyPressMask, False, defaultKeyHandler, (void*)d);
  XtAddEventHandler(text, StructureNotifyMask, False, inputConfigureProc, (void*)d);

  St_MoveUp(d->b.w, gmText.height, prompt);
  XtManageChild(text);
  XtSetKeyboardFocus(toplevel, text);

  return 0;
}


static void centeringShell(DfDialog *t, Widget dialog)
{
  XtWidgetGeometry base;
  int x;
  int y;
  Window root;
  unsigned int cx;
  unsigned int cy;
  unsigned int border;
  unsigned int depth;

  if(t->rc.width == 0 && t->rc.height == 0){
    /* not required */
    x = 0;
    y = 0;
  }else{
    XtQueryGeometry(toplevel, NULL, &base);
    XtRealizeWidget(t->shell);
    XGetGeometry(XtDisplay(t->shell), XtWindow(t->shell),
		 &root,
		 &x, &y,
		 &cx, &cy,
		 &border, &depth);
    x = base.x + t->rc.x;
    y = base.y + t->rc.y;
    if((base.y + base.height) < (y + cy)){
      y = base.y + base.height - cy;
    }

    if((base.x + base.width) < (x + cx)){
      x = base.x + base.width - cx;
    }
  }

  XtVaSetValues(t->shell,
		XtNx, x,
		XtNy, y,
		NULL);
}


int ShowDialog(DfBuf *owner, DfMessageBox *d, const char *label, int fButtons)
{
  Widget dialog;
  Widget shell;

  shell = XtVaCreatePopupShell("dlg", transientShellWidgetClass, d->b.w, NULL);
  dialog = XtVaCreateManagedWidget("dialog", dialogWidgetClass, shell, NULL);

  d->b.owner = owner;
  d->b.shell = shell;
  d->buttons = fButtons;

  if(fButtons & DLG_OK){
    XawDialogAddButton(dialog, "OK", callbackOk, d);
  }
  if(fButtons & DLG_YES){
    XawDialogAddButton(dialog, "Yes", callbackYes, d);
  }
  if(fButtons & DLG_NO){
    XawDialogAddButton(dialog, "No", callbackNo, d);
  }
  if(fButtons & DLG_OVERWRITE){
    XawDialogAddButton(dialog, "Overwrite", callbackOverwrite, d);
  }
  if(fButtons & DLG_RETRY){
    XawDialogAddButton(dialog, "Retry", callbackRetry, d);
  }
  if(fButtons & DLG_SKIP){
    XawDialogAddButton(dialog, "Skip", callbackSkip, d);
  }
  if(fButtons & DLG_RENAME){
    XawDialogAddButton(dialog, "Rename", callbackRename, d);
  }
  if(fButtons & DLG_CANCEL){
    XawDialogAddButton(dialog, "Cancel", callbackCancel, d);
  }

  /* label */
  XtVaSetValues(dialog, XtNlabel, label, NULL);
  XtAddEventHandler(dialog, KeyPressMask, False, dlgKeyHandler, (void*)d);

  centeringShell((DfDialog*)d, dialog);

  XtPopup(d->b.shell, XtGrabNonexclusive);

  return 0;
}

static char *findCompletionPath(char *str, int cur, const char *seps)
{
  MbStr mb;
  const char *limit;
  char *p;
  int separator;
  char *path;

  InitMbStr(&mb, str);

  separator = 0;
  limit = str + cur;
  path = str;
  p = str;

  while(p < limit){
    if(*p == '\\'){
      p = MbNextChar(&mb);
    }else if(isSeparate(*p, seps)){
      separator = 1;
    }else{
      if(separator){
	path = p;
      }
      separator = 0;
    }

    p = MbNextChar(&mb);
  }
  if(separator){
    path = p;
  }

  return path;
}


static int findCompletionPoint(const char *str, int cur, const char *seps)
{
  int point;
  MbStr mb;
  const char *limit;
  const char *p;
  int separator;

  InitMbStr(&mb, str);

  point = 0;
  separator = 0;
  limit = str + cur;
  p = str;

  while(p < limit){
    if(*p == '\\'){
      p = MbNextChar(&mb);
    }else if(isSeparate(*p, seps)){
      separator = 1;
    }else{
      if(separator){
	point = p - str;
      }
      separator = 0;
    }

    p = MbNextChar(&mb);
  }
  if(separator){
    point = p - str;
  }

  return point;
}

const int cand_item_unit = 8;

struct cand_list_item{
  char *idx;
  int dir;
};

struct cand_list_info{
  DfInputDialog *d;
  int num;
  int err;
  struct cand_list_item items[8];
};

static struct cand_list_info *gatherCandidate(struct cand_list_info *cand, DfStr *path, const char *pattern)
{
  DIR *dir;
  struct dirent *de;
  struct stat st;
  int path_len;
  int i;
  int num = 0;
  int alloced = cand_item_unit;
  int allow_dot = 0;;
  int left = cand_item_unit;
  void *tmp;

  cand->err = 0;
  path_len = Str_Length(path);
  dir = opendir(Str_Get(path));
  if(!dir){
    cand->err = errno;
    return cand;
  }

  if(pattern && pattern[0] == '.'){
    allow_dot = 1;
  }else if(GetFilerBuffer(cand->d->b.owner)->mode & SHOW_DOTS){
    allow_dot = 1;
  }

  /* make candidate list */
  while((de = readdir(dir)) != 0){
    if(IsDots(de->d_name)){
      continue;
    }
    if(pattern){
      if(IsMatch(pattern, de->d_name) == UNMATCH){
	continue;
      }
    }else{
      if(de->d_name[0] == '.' && allow_dot == 0){
	continue;
      }
    }

    Str_Overwrite(path, path_len, de->d_name);
    dprintf("[%s]\n", Str_Get(path));
    lstat(Str_Get(path), &st);

    if(left == 0){
      left = alloced;
      alloced *= 2;
      tmp = bReAlloc(cand,
		     sizeof(struct cand_list_info)
		     + sizeof(struct cand_list_item) * alloced);
      if(tmp == NULL){
	cand->err = ENOMEM;
	break;
      }
      cand = tmp;
      left = cand_item_unit;
    }
    cand->items[num].idx = (char *)Str_Length(&cand->d->cand);
    Str_AddNUL(&cand->d->cand, de->d_name);

    cand->items[num].dir = S_ISDIR(st.st_mode);
    left--;
    num++;
  }
  closedir(dir);

  cand->num = num;
  for(i = 0; i < num; i++){
    cand->items[i].idx = Str_Get(&cand->d->cand) + (int)cand->items[i].idx;
  }

  return cand;
}

static int comp_cand_item(const void *a, const void *b)
{
  const struct cand_list_item *x = a;
  const struct cand_list_item *y = b;

  return strcmp(x->idx, y->idx);
}

static void sortCandidateItems(struct cand_list_info *cand, int escape)
{
  char *back;
  DfStr *cand_str = &cand->d->cand;
  DfStr strEscape;
  struct cand_list_item *ptr;
  char *add;
  int loop;

  qsort(cand->items, cand->num, sizeof(struct cand_list_item), comp_cand_item);
  back = Str_Get(cand_str);
  Str_Init(cand_str);

  if(escape){
    Str_Init(&strEscape);
  }

  loop = cand->num;
  ptr = cand->items;
  while(loop){
    if(escape){
      Str_Overwrite(&strEscape, 0, ptr->idx);
      Str_Escape(&strEscape);
      add =  Str_Get(&strEscape);
    }else{
      add = ptr->idx;
    }
    Str_AddNUL(cand_str, add);
    if(ptr->dir){
      Str_Get(cand_str)[Str_Length(cand_str) - 1] = '/';
      Str_AddChar(cand_str, '\0');
    }
    ptr++;
    loop--;
  }

  if(escape){
    Str_Free(&strEscape, 0);
  }
  Str_AddChar(cand_str, '\0');
  bFree(back);
}

static void makeCandidateList(DfInputDialog *d, Widget w)
{
  struct cand_list_info *c;
  DfStr path;
  DfStr strFile;
  char *file;
  char *tmp;
  int path_len;
  int n;
  char *seps[2];
  Arg a;

  XtVaGetValues(d->text,
		XtNstring, &tmp,
		XtNinsertPosition, &n,
		NULL);

  if(d->b.flag & DLGF_MULTIPARAM){
    seps[0] = " /";
    seps[1] = " ";
  }else{
    seps[0] = "/";
    seps[1] = NULL;
  }
  d->cand_insert = findCompletionPoint(tmp, n, seps[0]);

  file = findCompletionPath(tmp, n, seps[1]);
  if(strcmp(file, "~") == 0){
    return;
  }
  if(IsAbsPath(file)){
    Str_InitStr(&path, file);
  }else{
    Str_InitStr(&path, Str_Get(&d->b.task->cwd));
    SetLastSlash(&path);
    Str_Add(&path, file);
  }

  Str_RegPath(&path, 0, 1);

  file = FindLastChar(Str_Get(&path), '/');

  path_len =  0;
  if(file[0]){
    file++;
    path_len = file - Str_Get(&path);
    if(!file[0]){
      file = NULL;
    }
  }

  if(file){
    Str_InitStr(&strFile, file);
    Str_SetLength(&path, path_len);
    file = Str_Get(&strFile);
  }else{
    Str_InitNull(&strFile);
  }

  dprintf(">> %s - %s\n", Str_Get(&path), file ? file : "<empty>");

  Str_Descape(&path);
  SetLastSlash(&path);

  c = bMalloc(sizeof(struct cand_list_info));
  if(c == NULL){
    St_SetMsg(w, strerror(errno));
    Str_Free(&strFile, 0);
    Str_Free(&path, 0);
    return;
  }

  c->d = d;
  c = gatherCandidate(c, &path, file);
  if(c->num == 0 && c->err){
    St_SetMsg(w, strerror(c->err));
    bFree(c);
    Str_Free(&strFile, 0);
    Str_Free(&path, 0);
    return;
  }

  if(c->num){
    sortCandidateItems(c, d->b.flag & DLGF_MULTIPARAM);
    Str_AddChar(&d->cand, '\0');

    d->cand_idx = 0;
    d->cand_ptr = Str_Get(&d->cand);

    Str_Overwrite(&path, 0, tmp);
    Str_Overwrite(&path, d->cand_insert, d->cand_ptr);
    tmp = Str_Get(&path);
    dprintf("result [%s]\n", tmp);
    dprintf("d is %p, d->text is %p\n", d, d->text);
    DumpMemInfo();
    XtSetArg(a, XtNstring, tmp);
    XtSetValues(d->text, &a, 1);
    XtSetArg(a, XtNinsertPosition, Str_Length(&path));
    XtSetValues(d->text, &a, 1);
/*
    XtVaSetValues(d->text, XtNstring, tmp, NULL);
    XtVaSetValues(d->text, XtNinsertPosition, strlen(tmp), NULL);
    dprintf("set position.\n");
*/
    if(c->num == 1){
      clearCandidateList(d);
    }
  }

  bFree(c);
  Str_Free(&strFile, 0);
  Str_Free(&path, 0);
}

static void forwardCandidateList(DfInputDialog *d, Widget w)
{
  char *tmp;
  DfStr s;
  char *cand;

  XtVaGetValues(d->text, XtNstring, &tmp, NULL);

  Str_InitStr(&s, tmp);
  cand = d->cand_ptr;
  cand += (strlen(cand) + 1);
  if(!*cand){
    /* rewind */
    cand = Str_Get(&d->cand);
  }
  d->cand_ptr = cand;
  Str_Overwrite(&s, d->cand_insert, cand);
  tmp = Str_Get(&s);
  XtVaSetValues(d->text, XtNstring, tmp, NULL);
  XtVaSetValues(d->text, XtNinsertPosition, strlen(tmp), NULL);

  Str_Free(&s, 0);
}

static void clearCandidateList(XtPointer p)
{
  DfInputDialog *d = p;

  Str_Free(&d->cand, 0);
  Str_InitNull(&d->cand);
  d->cand_ptr = NULL;
}

void CandidateList(DfInputDialog *d, Widget w)
{
  if(d->cand_ptr){
    forwardCandidateList(d, w);
  }else{
    makeCandidateList(d, w);
  }
}

static void defaultKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XKeyEvent *ev;
  KeySym k;
  DfInputDialog *d;
  DfTask *t;
  char ch[1];

  d = p;
  t = d->b.task;
  ev = (XKeyEvent*)event;
/*  k = XLookupKeysym(ev, 0); */
  XLookupString(ev, ch, 1, &k, NULL);
  switch(k){
  case XK_Return:
    *cont = False;
    confirm(w, &d->b);
    break;
  case XK_Escape:
    *cont = False;
    closeDialog(&d->b);
    break;
  default:
    if(d && d->keyproc){
      (*d->keyproc)(&d->b, w, k, ch[0], cont);
      dprintf("defaultKeyHandler %s\n", *cont ? "continue" : "discard");
    }
  }
}


static void dlgKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XKeyEvent *ev;
  KeySym k;
  DfMessageBox *t;

  ev = (XKeyEvent*)event;
  k = XLookupKeysym(ev, 0);
  t = p;
  switch(k){
  case XK_e:
    if((t->buttons & (DLG_RETRY | DLG_RENAME)) == (DLG_RETRY | DLG_RENAME)){
      *cont = False;
      callbackRename(w, p, NULL);
    }
    break;
  case XK_n:
    if(t->buttons & DLG_NO){
      *cont = False;
      callbackNo(w, p, NULL);
    }
    break;

  case XK_o:
    if(t->buttons & DLG_OVERWRITE){
      *cont = False;
      callbackOverwrite(w, p, NULL);
    }
    break;

  case XK_r:
    if(t->buttons & DLG_RETRY){
      *cont = False;
      callbackRetry(w, p, NULL);
    }else if(t->buttons & DLG_RENAME){
      *cont = False;
      callbackRename(w, p, NULL);
    }
    break;

  case XK_s:
    if(t->buttons & DLG_SKIP){
      *cont = False;
      callbackSkip(w, p, NULL);
    }
    break;
  case XK_y:
    if(t->buttons & DLG_YES){
      callbackYes(w, p, NULL);
    }
    break;
  case XK_Return:
    if(t->buttons & DLG_OK){
      *cont = False;
      callbackOk(w, p, NULL);
    }else if(t->buttons & DLG_YES){
      callbackYes(w, p, NULL);
    }
    break;
  case XK_Escape:
    if(t->buttons & DLG_CANCEL){
      *cont = False;
      callbackCancel(w, p , NULL);
    }
    break;
  }
}


static void confirm(Widget w, DfDialog *d)
{
  if(d && d->v->confirm){
    (*d->v->confirm)(w, d);
  }
}

static void confirmInput(Widget w, DfDialog *d)
{
  DfInputDialog *id = (void*)d;
  DfTask *t;
  Arg a;
  char *p;

  if(!d){
    return;
  }
  t = d->task;

  if(id->text){
    XtSetArg(a, XtNstring, &p);
    XtGetValues(id->text, &a, 1);
    if(Str_IsNull(&t->input)){
      Str_InitStr(&t->input, p);
    }else{
      Str_Overwrite(&t->input, 0, p);
    }
  }
  clearCandidateList(id);

  (*d->v->destroy)(d);

  DlgSendEvent(t, taskCbInvoke);
}
static void confirmButton(Widget w, DfDialog *d)
{
  DfTask *t;

  if(!d){
    return;
  }

  t = d->task;

  (*d->v->destroy)(d);
  DlgSendEvent(t, taskCbInvoke);
}

static void confirmChmodDialog(Widget w, DfDialog *d)
{
  DfTask *t;
  DfChmodDialog *cd = (void*)d;
  struct dftask_chmod_info *ext;
 
  if(!d){
    return;
  }

  t = d->task;
  ext = t->extend;

  /* ToDo: propagetion user choosen button to task */
  ext->mode = cd->mode;

  (*d->v->destroy)(d);

  DlgSendEvent(t, taskCbInvoke);
}

static void closeDialog(DfDialog *d)
{
  DfTask *t;
  if(!d){
    return;
  }

  t = d->task;
  (*d->v->destroy)(d);
  DlgSendEvent(t, taskCbDone);
}

static void callbackOk(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  d->task->state = CMD_OK;
  confirmButton(w, p);
}


static void callbackYes(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_YES;
  (*d->v->confirm)(w, d);
}

static void callbackNo(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_NO;
  (*d->v->confirm)(w, d);
}

static void callbackOverwrite(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_OVERWRITE;
  (*d->v->confirm)(w, d);
}

static void callbackRetry(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_RETRY;
  (*d->v->confirm)(w, d);
}

static void callbackSkip(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_SKIP;
  (*d->v->confirm)(w, d);
}
static void callbackRename(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_RENAME;
  (*d->v->confirm)(w, d);
}
static void callbackCancel(Widget w, XtPointer p, XtPointer q)
{
  DfDialog *d = p;

  
  d->task->state = CMD_CANCEL;
  (*d->v->confirm)(w, d);
}

int ShowChmodDialog(DfBuf *owner, DfChmodDialog *d)
{
  Widget shell;
  Widget dialog;
  Widget label;
  Widget w;
  Widget left;
  Widget rwx[4][4];
  XtWidgetGeometry gmButton;
  static const int permit[3] = {4, 2, 1};
  static const char *widget_name[4][4] = {
    {"St",   "Owner", "Group", "Other"},
    {"UID",  "r",     "gr",    "or"},
    {"GID",  "w",     "gw",    "ow"},
    {"Swap", "x",     "gx",    "ox"},
  };
  int x;
  int y;
  int cx;
  int cy;
  Arg args[3];
  int mode[4];
  void *widget_class;
  Boolean state;

  shell = XtVaCreatePopupShell("dlg", transientShellWidgetClass, d->b.w, NULL);
  dialog = XtVaCreateManagedWidget("dialog", formWidgetClass, shell, NULL);
  XawFormDoLayout(dialog, False);
  w = NULL;
  left = NULL;
  cx = 0;
  cy = 0;

  label = XtVaCreateManagedWidget("perm", simpleWidgetClass, dialog,
				  XtNwidth, vnFontHeight,
				  XtNheight, vnFontHeight,
				  XtNfromHoriz, NULL,
				  XtNfromVert, NULL,
				  NULL);
  d->chk = XtVaCreateManagedWidget("subdir", toggleWidgetClass, dialog,
				   XtNfromHoriz, label,
				   XtNfromVert, NULL,
				   NULL);

  for(x = 0; x < 4; x++){
    w = label;
    widget_class = labelWidgetClass;
    for(y = 0; y < 4; y++){
      w = XtVaCreateManagedWidget(widget_name[y][x], widget_class, dialog,
				  XtNfromHoriz, left,
				  XtNfromVert, w,
				  NULL);
      rwx[y][x] = w;

      XtQueryGeometry(w, NULL, &gmButton);
      if(cx < gmButton.width){
	cx = gmButton.width;
      }
      if(cy < gmButton.height){
	cy = gmButton.height;
      }
      widget_class = toggleWidgetClass;
    }
    left = rwx[0][x];
  }
  XtSetArg(args[0], XtNwidth , (XtArgVal)cx);
  XtSetArg(args[1], XtNheight, (XtArgVal)cy);

  XtSetValues(label, args, 2);
  mode[0] = d->mode >> 9;
  mode[1] = d->mode >> 6;
  mode[2] = d->mode >> 3;
  mode[3] = d->mode;

  for(x = 0; x < 4; x++){
    XtSetValues(rwx[0][x], args, 2);
    for(y = 0; y < 3; y++){
      state = (mode[x] & (permit[y])) ? True : False;
      XtSetArg(args[2], XtNstate, state);

      XtSetValues(rwx[y + 1][x], args, 3);
      d->buttons[y][x] = rwx[y + 1][x];
    }
  }
  XawFormDoLayout(dialog, True);

  d->b.shell = shell;
  d->b.owner = owner;
  d->b.dialog = dialog;
  d->label = label;
  d->cx = cx;
  d->cy = cy;
  d->subdir = 0;


  XtAddEventHandler(dialog, KeyPressMask, False, chmodKeyHandler, (void*)d);
  XtAddEventHandler(d->label, ExposureMask, False, chmodLabelProc, (void*)d);

  centeringShell((DfDialog*)d, dialog);
  XtPopup(d->b.shell, XtGrabNonexclusive);

  return 0;
}

static void chmodLabelProc(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XExposeEvent *ev;
  char mode[5];
  GC gc;
  DfTask *t;
  DfChmodDialog *d;
  int len;
  int x;
  int y;
  int cx[4];
  int n;
  int fReverse;

  ev = (XExposeEvent*)event;

  if(ev->count){
    return;
  }

  d = p;
  t = d->b.task;
  gc = DefaultGC(ev->display, DefaultScreen(ev->display));
  len = sprintf(mode, "%4.4o", d->mode & 07777);
  y = (d->cy - vnFontHeight) / 2 + vnFontBase;
  XSetForeground(ev->display, gc, vOptions.colors[COLOR_BK]);
  XFillRectangle(ev->display, XtWindow(w), gc, 0, 0, d->cx, d->cy);
  XSetForeground(ev->display, gc, vOptions.colors[COLOR_FILE]);

  x = d->cx;
  for(n = 0; n < 4; n++){
    cx[n] = XmbTextEscapement(vFontSet, mode + n, 1) + 2;
    x -= cx[n];
  }

  fReverse = 0;
  for(n = 0; n < 4; n++){
    if(d->colum == n){
      XSetForeground(ev->display, gc, vOptions.colors[COLOR_FILE]);
      XFillRectangle(ev->display, ev->window, gc, x, 0, cx[n], d->cy);
      XSetForeground(ev->display, gc, vOptions.colors[COLOR_BK]);
      fReverse = 1;
    }else if(fReverse){
      XSetForeground(ev->display, gc, vOptions.colors[COLOR_FILE]);
    }
    XmbDrawString(ev->display, XtWindow(w), vFontSet, gc, x + 1, y, mode + n, 1);
    x += cx[n];
  }
}

static void chmodKeyHandler(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  static const int mask[4] = {00777, 07077, 07707, 07770};
  static const int shift[4] = {9, 6, 3, 0};
  static const int rwx[3] = {4, 2, 1};

  XKeyEvent *ev;
  KeySym k;
  DfChmodDialog *d = p;
  char ch[1];

  Arg args[1];
  Boolean state;
  int mode;
  int x;
  int y;

  dprintf("chmdo key handler.\n");
  ev = (XKeyEvent*)event;
/*  k = XLookupKeysym(ev, 0); */
  XLookupString(ev, ch, 1, &k, NULL);

  switch(k){
  case XK_Return:
    *cont = False;
    confirm(w, &d->b);
    return;

  case XK_Escape:
    *cont = False;
    closeDialog(&d->b);
    return;

  case XK_0:
  case XK_1:
  case XK_2:
  case XK_3:
  case XK_4:
  case XK_5:
  case XK_6:
  case XK_7:
    mode = k - XK_0;
    mode <<= shift[d->colum];
    d->mode &= mask[d->colum];
    d->mode |= mode;
    *cont = False;
    if(d->colum < 3){
      d->colum++;
    }
    break;
  case XK_s:
    d->subdir ^= 1;
    state = d->subdir ? True : False;
    XtSetArg(args[0], XtNstate, state);
    XtSetValues(d->chk, args, 1);
    *cont = False;
    break;
  case XK_r:
    if(d->colum == 0){
      break;
    }
    x = 4 << shift[d->colum];
    d->mode ^= x;
    *cont = False;
    break;
  case XK_w:
    if(d->colum == 0){
      break;
    }
    x = 2 << shift[d->colum];
    d->mode ^= x;
    *cont = False;
    break;
  case XK_x:
    if(d->colum == 0){
      break;
    }
    x = 1 << shift[d->colum];
    d->mode ^= x;
    *cont = False;
    break;
  case XK_Left:
    if(d->colum){
      d->colum--;
    }
    *cont = False;
    break;
  case XK_Right:
    if(d->colum < 3){
      d->colum++;
    }
    break;
  case XK_BackSpace:
    if(d->colum){
      d->colum--;
    }
    break;
  default:
    *cont = False;

    return;
  }

  InvalidateWindow(d->label);
  for(x = 0; x < 4; x++){
    mode = d->mode >> shift[x];
    for(y = 0; y < 3; y++){
      state = (mode & (rwx[y])) ? True : False;
      XtSetArg(args[0], XtNstate, state);
      XtSetValues(d->buttons[y][x], args, 1);
    }
  }

  return;
}


static void destroyDialog(DfDialog *d)
{
  dprintf("destroy dialog.\n");
  XtPopdown(d->shell);
  XtDestroyWidget(d->shell);
  bFree(d);
}


static void destroyInputDialog(DfDialog *d)
{
  DfInputDialog *id = (void*)d;
  St_MoveDown(dfx->text_height);

#if CACHE_TEXT
  XtRemoveEventHandler(d->text, KeyPressMask, False, defaultKeyHandler, (void*)d);
  XtUnmapWidget(id->text);
  XtQueryGeometry(text, NULL, &wg);
#else
  XtDestroyWidget(id->text);
#endif
  dfx->text = NULL;
  dfx->text_height = 0;
  XtSetKeyboardFocus(toplevel, dfx->w);

  Str_Free(&id->input, 0);
  clearCandidateList(d);

  bFree(d);
  return;
}

static void inputConfigureProc(Widget w, XtPointer p, XEvent *event, Boolean *cont)
{
  XConfigureEvent *ev;
  DfTask *t;

  t = p;

  ev = (XConfigureEvent*)event;
  switch(event->type){
  case ConfigureNotify:
    if(ev->height != dfx->text_height){
      XtVaSetValues(w,
		    XtNheight, dfx->text_height,
		    NULL);
    }
  }
}

static int isSeparate(char p, const char *seps)
{
  if(!seps){
    return 0;
  }

  while(*seps){
    if(p == *seps){
      return 1;
    }
    seps++;
  }
  return 0;
}

int inputKeyProc(DfDialog *d, Widget w, KeySym k, char ch, Boolean *cont)
{
  DfInputDialog *id = (void*)d;

  switch(k){
  case XK_Tab:
    if(id->text){
      CandidateList(id, w);
      *cont = False;
    }
    break;
  default:
    if(id->cand_ptr && ch){
      clearCandidateList(d);
    }
  }
  return 0;
}

int copyKeyProc(DfDialog *d, Widget w, KeySym k, char ch, Boolean *cont)
{
  DfInputDialog *id = (void*)d;
  DfFileList *l;
  Arg a[2];
  char *p;

  switch(k){
  case XK_Up:/* temporary for copy/move dialog */
    l = GetNextFileBuf(d->task->list);

    p = l->cwd;
    XtSetArg(a[0], XtNstring, p);
    XtSetArg(a[1], XtNinsertPosition, strlen(p));

    XtSetValues(id->text, a, 1);
    XtSetValues(id->text, a + 1, 1);
    d->task->list = l;
    *cont = False;
    break;

  case XK_Tab:
    if(id->text){
      CandidateList(id, w);
      *cont = False;
    }
    break;

  default:
    if(id->cand_ptr && ch){
      clearCandidateList(d);
    }
  }
  return 0;
}

int isearchKeyProc(DfDialog *d, Widget w, KeySym k, char ch, Boolean *cont)
{
  int skip = 1;
  char *p;
  DfInputDialog *id = (DfInputDialog *)d;
  DfTask *t = d->task;

  switch(k){
  case XK_Shift_L:
  case XK_Shift_R:
    *cont = False;
    skip = 0;
    break;
  case XK_Tab:
  case XK_Down:
    *cont = False;
    t->state = CMD_ISEARCHNEXT;
    break;
  case XK_Up:
    t->state = CMD_ISEARCHPREV;
    break;
  default:
    switch(ch){
    case 0:
      skip = 0;
      break;
    case 8:
      XtVaGetValues(id->text, XtNstring, &p, NULL);
      if(*p){
	skip = 0;
      }else{
	t->state = CMD_ISEARCHPARENT;
	*cont = False;
      }
      break;
    case '/':
      t->state = CMD_ISEARCHOPEN;
      *cont = False;
      break;
    default:
      t->state = STATE_NONE;
      break;
    }
  }
  if(skip){
    DlgSendEvent(t, taskCbNotify);
  }

  return 0;
}



