#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h> 
#include <sys/stat.h> 

#include <X11/Intrinsic.h>
#include <X11/keysym.h>
#include <X11/StringDefs.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "cmd.h"
#include "dfxfunc.h"
#include "cmdinput.h"
#include "mem.h"
#include "cfg.h"
#include "dfval.h"

/*
 Shift   : +
 Contrl  : ^
 Alt     : %
 Char    : #
*/

typedef struct cmd_item{
  int cmd;
  const char *name;
}CmdItem;

typedef struct key_name{
  KeySym k;
  const char *name;
}CmdKeyName;


typedef enum{
  NONE,
  COMMAND,
  QUOTE_CMD,
  PREKEYDEF,
  KEYDEF,
  IGN
}kcfg_stat;


typedef struct _tmpCmdIdx{
  int key;
  int cmd;
  int allow;
}tmpCmdIdx;

typedef struct _tmpKeystroke{
  int num;
  int size;
  tmpCmdIdx *idx;
  DfStr cmds;
  DfCmdKey *keys;
}tmpKeyStroke;

typedef struct _stroke{
  DfCmdKey *keys;
  int len;
  int size;
}CmdStroke;


/* finally */
typedef struct _keyDefine{
  char **cmd;
  DfCmdKey *keys;
}KeyDefie;

#if DEBUG
void testCmdTable(const DfCmdStr *cmdTable);
#endif

static const CmdItem CmdTable[] =
{
  {OPEN,		"open"},
  {CURUP,		"up"},
  {CURDOWN,		"down"},
  {PAGEUP,		"pageup"},
  {PAGEDOWN,		"pagedown"},
  {TOP,			"top"},
  {BOTTOM,		"bottom"},
  {MARK,		"mark"},
  {MULTIMARK,		"multimark"},
  {CLEARMARK,		"clearmark"},
  {REVMARK,		"revmark"},
  {FILTER,		"filter"},
  {IFILTER,		"ifilter"},
  {SHOWDOTS,		"showdots"},
  {GOTO,		"cd"},
  {GOTOPARENT,		"gotoparent"},
  {CANCEL,		"cancel"},
  {COPY,		"cp"},
  {MOVE,		"mv"},
  {LINK,		"ln"},
  {SYMLINK,		"symln"},
  {DELETE,		"rm"},
  {CHMOD,		"chmod"},
  {CHOWN,		"chown"},
  {CHGRP,		"chgrp"},
  {MKFILE,		"mkfile"},
  {MKDIR,		"mkdir"},
  {RENAME,		"rename"},
  {HORIZ_LEFT,		"horiz_l"},
  {HORIZ_RIGHT,		"horiz_r"},
  {RELOAD,		"reload"},
  {REFRESH,		"refresh"},
  {INFO,		"info"},
  {EXECUTE,		"exec"},
  {QUIT,		"quit"},
  {ISEARCH,		"isearch"},
  {SORTNAME,		"sortname"},
  {SORTSIZE,		"sortsize"},
  {SORTTIME,		"sorttime"},
  {SORTEXT,		"sortext"},
  {SORTATTR,		"sortmode"},
  {CMDLIST,		"cmdlist"},
  {PATHLIST,		"pathlist"},
  {BUFLIST,		"blist"},
  {SPLIT_H,		"split_h"},
  {SPLIT_V,		"split_v"},
  {ENLARGE_H,		"enlarge_h"},
  {ENLARGE_V,		"enlarge_v"},
  {SHRINK_H,		"shrink_h"},
  {SHRINK_V,		"shrink_v"},
  {CLOSE,		"close"},
  {KILL,		"kill"},
  {KILLOTHER,		"killother"},
  {NEWBUFFER,		"newbuffer"},
  {SWITCHBUFFER,	"switchbuffer"},
  {NEXTFRAME,		"nextframe"},
  {PREVFRAME,		"prevframe"},
  {VIEW,		"view"},
  {CHCHARSET,		"chcharset"},
  {CHNEWLINE,		"chline"},
  {VIEWFIND,		"find"},
  {VIEWFINDNEXT,	"findnext"},
  {VIEWFINDPREV,	"findprev"},
  {DUMPMEMINFO,		"DumpMemInfo"},
  {DUMPBUFFERINFO,		"DumpBufInfo"},
  {0,			NULL},
};

static const CmdKeyName KeyTable[] = {
  {XK_slash,		"/"},
  {XK_space,		"Space"},
  {XK_Return,		"Enter"},
  {0,			NULL},
};

static tmpKeyStroke *parseFile(int fd);
int findCommand(const char *cmd);
static int convertKeyDef(CmdStroke *ks, const char *keydef, tmpKeyStroke *t);
static KeySym findKeyString(const char *key);
static int addStrokeKey(CmdStroke *ks, DfCmdKey k);
static const DfCmdStr *buildCommand(tmpKeyStroke *t);
static int extend(tmpKeyStroke *t);
static int getAllowBuffer(DfStr *allow);

int LoadKeyDefine(void)
{
  int fd;
  tmpKeyStroke *t;
  const DfCmdStr *cmdTable;

  fd = CfgOpenFile("keys");

  if(fd == -1){
    return 0;
  }

  t = parseFile(fd);

  close(fd);

  if(t == NULL){
    return 0;
  }

  cmdTable = buildCommand(t);
  if(cmdTable){
#if DEBUG
    testCmdTable(cmdTable);
#endif
    FL_CmdTab = cmdTable;
  }
  return 0;
}

static tmpKeyStroke *parseFile(int fd)
{
  int len;
  kcfg_stat s;
  DfStr tmp;
  int found;
  MbStr mb;
  char *b;
  char *p;
  int left;
  char *start;
  int stat;
  int allow;
  int fail;

  tmpKeyStroke *t;
  CmdStroke ks;

  /* init */

  b = bMalloc(4096);
  ks.keys = bMalloc(sizeof(DfCmdKey) * 256);
  ks.len = 0;
  ks.size = 256;

  s = NONE;
  found = 0;

  t = bMalloc(sizeof(tmpKeyStroke));
  t->size = 32;
  t->idx = bMalloc(sizeof(tmpCmdIdx) * 32);
  t->num = 0;
  Str_Init(&t->cmds);

  Str_Init(&tmp);

  mb.len = 0;
  mb.str = b;
  mb.c_len = 0;
  start = NULL;
  stat = NONE;
  do{
    len = read(fd, b, 4096 - mb.len);
    mb.str = b;
    mb.len = mb.len + len;
    mb.c_len = mblen(b, mb.len);
    p = b;
    if(len){
      left = 16;
    }else{
      left = 0;
    }
    while(left < mb.len){
      switch(stat){
      case NONE:
	if(mb.c_len != 1){
	  if(t->size <= t->num){
	    if(extend(t) != 0){
	      goto ERROR;
	    }
	  }
	  t->idx[t->num].cmd = Str_Length(&t->cmds);
	  stat = COMMAND;
	  start = p;
	  break;
	}
	switch(*p){
	case '\"':
	  stat = QUOTE_CMD;
	  start = p;
	  break;
	case '\r':
	case '\n':
	  break;
	default:
	  if(!IsSkipChar(&mb)){
	    if(t->size <= t->num){
	      if(extend(t) != 0){
		return NULL;
	      }
	    }
	    t->idx[t->num].cmd = Str_Length(&t->cmds);
	    allow = CMD_ANY;
	    stat = COMMAND;
	    start = p;
	  }
	  break;
	}
	break;

      case COMMAND:
	if(mb.c_len == 1){
	  switch(p[0]){
	  case ',':
	    Str_AddLen(&tmp, start, p - start);
	    allow = getAllowBuffer(&tmp);
	    Str_SetLength(&tmp, 0);
	    stat = NONE;
	    start = NULL;
	    break;
	  case ':':
	    Str_AddLen(&tmp, start, p - start);
	    Str_Trim(&tmp);
	    Str_AddNUL(&t->cmds, Str_Get(&tmp));
	    Str_SetLength(&tmp, 0);
	    stat = PREKEYDEF;
	    start = NULL;
	    t->idx[t->num].allow = allow;
	    break;
	  case '\"':
	    stat = QUOTE_CMD;
	    break;
	  }
	}
	break;
      case QUOTE_CMD:
	if(mb.c_len == 1 && p[0] == '\"'){
	  stat = COMMAND;
	}
	break;
      case PREKEYDEF:
	if(!IsSkipChar(&mb)){
	  stat = KEYDEF;
	  start = p;
	  continue;
	}
	break;
      case KEYDEF:
	if(mb.c_len == 1){
	  switch(p[0]){
	  case '\r':
	  case '\n':
	    Str_AddLen(&tmp, start, p - start);
	    Str_Trim(&tmp);
	    t->idx[t->num].key = ks.len;
	    fail = convertKeyDef(&ks, Str_Get(&tmp), t);
	    Str_SetLength(&tmp, 0);
	    stat = NONE;
	    start = NULL;
	    if(!fail){
	      t->num++;
	    }
	  }
	}
	break;
      case IGN:
	if(mb.c_len == 1 && p[0] == '\n'){
	  stat = NONE;
	  start = NULL;
	}
      }
      p = MbNextChar(&mb);
    }
    if(start){
      Str_AddLen(&tmp, start, p - start);
      start = b;
    }
    memmove(b, mb.str, mb.len);
  }while(len);

  if(stat == KEYDEF){
    Str_AddLen(&tmp, start, p - start);
    Str_Trim(&tmp);
    t->idx[t->num].key = ks.len;
    if(convertKeyDef(&ks, Str_Get(&tmp), t) == 0){
      t->num++;
    }
  }

  if(t->num == 0){
    goto ERROR;
  }
  t->keys = ks.keys;
  Str_Free(&tmp, 0);
  bFree(b);

  return t;

ERROR:
  Str_Free(&tmp, 0);
  bFree(b);
  free(ks.keys);
  bFree(t->idx);
  Str_Free(&t->cmds, 0);
  bFree(t);

  return NULL;
}


int findCommand(const char *cmd)
{
  const CmdItem *i;

  for(i = CmdTable; i->name; i++){
    if(strcmp(i->name, cmd) == 0){
      return i->cmd;
    }
  }
  return NOP;
}

static DfCmdKey getKeycode(char *key, int len, DfCmdKey mod)
{
  DfCmdKey k;

 if(mod & DFMOD_CHAR){
    k = key[0];
   mod = DFMOD_CHAR;
  }else{
    key[len] = '\0';
    k = findKeyString(key);
  }
  if(k == 0){
    return 0;
  }
  return mod | k;
}

static int convertKeyDef(CmdStroke *ks, const char *keydef, tmpKeyStroke *t)
{
  int idx;
  DfCmdKey mod;
  DfCmdKey k;
  char key[32];
  int i;

  /* init */
  mod = 0;
  k = 0;
  idx = ks->len;
  i = 0;

  /* find */
  while(*keydef){
    switch(*keydef){
    case '+':
      mod |= DFMOD_SHIFT;
      break;
    case '%':
      mod |= DFMOD_ALT;
      break;
    case '^':
      mod |= DFMOD_CTRL;
      break;
    case '#':
      mod |= DFMOD_CHAR;
      break;
    case ' ':
    case '\t':
      if(i){
	k = getKeycode(key, i, mod);
	if(k != 0){
	  if(addStrokeKey(ks, k)){
	    return 1;
	  }
	}
      }
      k = 0;
      mod = 0;
      i = 0;
      break;
    default:
      key[i] = *keydef;
      i++;
      break;
      /* clear */
    }
    keydef++;
  }
  if(i){
    k = getKeycode(key, i, mod);
    if(k != 0){
      if(addStrokeKey(ks, k)){
	return 1;
      }
    }
  }

  if(idx == ks->len){
    fprintf(stderr, "keys (%d): invalid key sequence.\n", t->num + 1);
    return 1;
  }

  /* termination  */
  if(addStrokeKey(ks, 0)){
    return 1;
  }

  return 0;
}

static KeySym findKeyString(const char *key){
  const CmdKeyName *i;
  KeySym k;

  for(i = KeyTable; i->name; i++){
    if(strcmp(i->name, key) == 0){
      return i->k;
    }
  }

  k = XStringToKeysym(key);
  if(k == NoSymbol){
    k = 0;
  }
  return k;
}

static int addStrokeKey(CmdStroke *ks, DfCmdKey k)
{
  DfCmdKey *p;
  int size;

  if(ks->size <= ks->len){
    size = ks->size + 32;
    p = bReAlloc(ks->keys, size);
    if(!p){
      return 1;
    }

    ks->keys = p;
    ks->size = size;
  }

  ks->keys[ks->len] = k;
  ks->len++;

  return 0;
}


static const DfCmdStr *buildCommand(tmpKeyStroke *t)
{
  DfCmdStr *cmd_list;
  int num;
  int i;
  int cmd;
  char *cmds;
  DfCmdKey *keys;
  InputCmd *arg;

  num = t->num;
  cmd_list = bMalloc(sizeof(DfCmdStr) * (num + 1));
  cmds = Str_Get(&t->cmds);
  keys = t->keys;
  arg = NULL;

  for(i = 0; i < num; i++){
    cmd_list[i].key = keys + t->idx[i].key;

    arg = BuildArgs2(cmds + t->idx[i].cmd, arg);

    cmd = findCommand(arg->argv[0]);
    cmd_list[i].builtin = cmd;
    cmd_list[i].allow = t->idx[i].allow;
    cmd_list[i].cmd = NULL;
    if(cmd == 0 || 1 < arg->argc){
      fprintf(stderr, "not found %s\n", arg->argv[0]);
      cmd_list[i].cmd = cmds + t->idx[i].cmd;
    }
    if(!*cmd_list[i].key){
      fprintf(stderr, "invalid key seq. %s\n", arg->argv[0]);
    }
  }
  FreeInputCmd(arg);
  /* sentinel */
  cmd_list[num].key = NULL;
  cmd_list[num].cmd = NULL;

  return cmd_list;
}


static int extend(tmpKeyStroke *t)
{
  tmpCmdIdx *t2;

  t2 = bReAlloc(t->idx, sizeof(tmpCmdIdx) * (t->size + 32));
  if(t2 == NULL){
    return 1;
  }
  t->size += 32;
  t->idx = t2;
  return 0;
}

static int getAllowBuffer(DfStr *allow_str)
{
  char *p;
  int allow;

  allow = 0;

  p = Str_Get(allow_str);

  while(*p){
    switch(*p){
    case '*':
      return 0;
    case 'f':
      allow |= CMD_FILER;
      break;

    case 'v':
      allow |= CMD_VIEWER;
      break;

    case 'c':
      allow |= CMD_CMDLIST;
      break;

    case 'p':
      allow |= CMD_PATHLIST;
      break;
    }
    p++;
  }

  return allow;
}

const char *CmdNoToStr(int cmd)
{
  const CmdItem *ptr;

  ptr = CmdTable;
  while(ptr->name){
    if(ptr->cmd == cmd){
      return ptr->name;
    }
    ptr++;
  }

  return "not defined.";
}

#if DEBUG
void testCmdTable(const DfCmdStr *cmdTable)
{
  const DfCmdStr *p;
  int i;

  i = 0;
  p = cmdTable;
  while(p->key){
    p++;
    i++;
  }
}

#endif
