#include <stdio.h>
#include <stdlib.h>
#include <strings.h>

#include <X11/Intrinsic.h>

#include "config.h"
#include "str.h"
#include "mem.h"
#include "debug.h"

#define BUF_SIZE (512)

#if MEMDEBUG
#undef Str_Init
#undef Str_InitStr
#undef Str_InitPsz
#undef Str_Add
#undef Str_AddNUL
#undef Str_AddChar
#undef Str_Overwrite
#undef Str_Shrink
#undef Str_Dup
#undef Str_Free
#endif

static int extend(DfStr *s, int len);

int Str_Init(DfStr *s)
{
  s->str = bMalloc(BUF_SIZE);
  s->len = 0;
  s->str[0] = '\0';
  s->alloced_len = BUF_SIZE;

  return 0;
}

int Str_InitNull(DfStr *s)
{
  s->str = NULL;
  s->len = 0;
  s->alloced_len = 0;

  return 0;
}

int Str_Move(DfStr *d, DfStr *s)
{
  *d = *s;
  Str_InitNull(s);

  return 0;
}


int Str_InitStr(DfStr *s, const char *str)
{
  int len;
  int a_len;

  if(!str){
    return Str_Init(s);
  }
  len = strlen(str);
  a_len = ((len + BUF_SIZE) / BUF_SIZE) * BUF_SIZE;

  s->str = bMalloc(a_len);
  strcpy(s->str, str);
  s->len = len;
  s->alloced_len = a_len;

  return 0;
}

int Str_InitPsz(DfStr *s, const char *str)
{
  int len;

  len = str ? strlen(str) : 0;
  s->str = (char*)str;
  s->len = len;
  s->alloced_len = len;

  return 0;
}

int Str_Trim(DfStr *s)
{
  char *p;
  MbStr mb;
  int phase;
  char *src;
  char *last;
  int blank;
  int len;

  p = s->str;
  InitMbStr(&mb, p);
  src = s->str;
  last = src + s->len;

  phase = 0;
  while(*p){
    blank= 0;
    if(mb.c_len == 1){
      switch(*p){
      case ' ':
      case '\t':
      case '\r':
	blank = 1;
      }
    }

    switch(phase){
    case 0:
      if(!blank){
	phase = 1;
	src = p;
      }
      break;
    case 1:
      if(blank){
	phase = 2;
	last = p;
      }
      break;
    case 2:
      if(!blank){
	phase = 1;
      }
      break;
    }
    p = MbNextChar(&mb);
  }
  if(phase == 1){
    last = p;
  }
  if(phase == 0){
    return 0;
  }
  len = last - src;
  memmove(s->str, src, len);
  s->len = len;
  s->str[len] = '\0';

  return 0;
}

int Str_Free(DfStr *s, int fFree)
{
  if(s == NULL){
    return 1;
  }

  if(s->str){
    bFree(s->str);
  }
  if(fFree){
    bFree(s);
  }else{
    s->str = NULL;
    s->len = 0;
    s->alloced_len = 0;
  }

  return 0;
}

void Str_SetLength(DfStr *s, int len)
{

  if(s->alloced_len < len){
    if(extend(s, len - s->len) == 0){
      return;
    }
  }

  s->str[len] = '\0';
  s->len = len;
}

void Str_Reserve(DfStr *s, int len)
{
  if(s->alloced_len < len){
    if(extend(s, len - s->len) == 0){
      return;
    }
  }
}

#ifdef STR_DECLARE
char *Str_Get(const DfStr *s)
{
  return s->str;
}

int Str_Length(const DfStr *s)
{
  return s->len;
}
int Str_Size(const DfStr *s)
{
  return s->alloced_len;
}
#endif

int Str_Add(DfStr *s, const char *str)
{
  int add;

  add = strlen(str);

  if(extend(s, add) == 0){
    return 0;
  }

  strcpy(s->str + s->len, str);
  s->len = s->len + add;

  if(s->str == NULL || s->str[0] == '\0'){
    dprintf("what?\n");
  }
  return 1;
}

int Str_AddLen(DfStr *s, const char *str, int len)
{
  if(extend(s, len) == 0){
    return 0;
  }

  memcpy(s->str + s->len, str, len);
  s->len = s->len + len;
  s->str[s->len] = '\0';

  return 1;
}


int Str_AddNUL(DfStr *s, const char *str)
{
  int add;

  add = strlen(str) + 1;

  if(extend(s, add) == 0){
    return 0;
  }

  strcpy(s->str + s->len, str);
  s->str[s->len + add - 1] = '\0';
  s->len = s->len + add;

  if(s->str == NULL || s->str[0] == '\0'){
    dprintf("what?\n");
  }
  return 1;
}


int Str_AddChar(DfStr *s, char ch)
{
  if(extend(s, 1) == 0){
    return 0;
  }

  s->str[s->len] = ch;
  s->len++;
  s->str[s->len] = '\0';

  return 1;
}

int Str_Overwrite(DfStr *s, int idx, const char *str)
{
  int len;
  int req;

  len = strlen(str);
  req = idx + len + 1;

  if(s->alloced_len <= req){
    if(extend(s, req - s->len) == 0){
      return 0;
    }
  }

  strcpy(s->str + idx, str);
  s->len = idx + len;

  return 1;
}

int Str_Insert(DfStr *s, int idx, const char *str)
{
  int len;
  int req;

  len = strlen(str);
  req = s->len + len + 1;

  if(extend(s, req) == 0){
      return 0;
  }

  memmove(s->str + idx + len, s->str + idx, s->len - idx + 1);
  memcpy(s->str + idx, str, len);
  s->len += len;

  return 1;
}

int Str_InsertChar(DfStr *s, int idx, char ch)
{
  int req;

  req = s->len + 1;

  if(extend(s, 1) == 0){
    return 0;
  }

  s->len++;
  memmove(s->str + idx + 1, s->str + idx, s->len - idx);
  s->str[idx] = ch;

  return 1;
}


int IsEscape(int ch)
{
  switch(ch){
  case ' ':
  case '[':
  case ']':
  case '<':
  case '>':
  case '*':
  case '?':
  case '\\':
  case ':':
  case ';':
  case '+':
  case '\"':
  case '\'':
    return 1;
  }
  return 0;
}

  
int count_has_escape(const char *str)
{
  MbStr mb;
  int n;

  n = 0;
  InitMbStr(&mb, str);
  while(*str){
    if(IsEscape(*str)){
      ++n;
    }
    str = MbNextChar(&mb);
  }
  return n;
}

void strcpy_escape(char *dst, const char *src, int len)
{
  MbStr mb;
  InitMbStr(&mb, src);

  while(*src){
    switch(mb.c_len){
    case 0:
    case -1:
      return;

    case 1:
      if(IsEscape(*src)){
	*dst = '\\';
	++dst;
      }
      *dst = *src;
      dst++;
      break;
    default:
      memcpy(dst, src, mb.c_len);
      dst += mb.c_len;
    }
    src = MbNextChar(&mb);
  }
}

int Str_InsertEscape(DfStr *s, int idx, const char *str)
{
  int len;
  int req;

  len = strlen(str) + count_has_escape(str);
  req = s->len + len + 1;

  if(extend(s, req) == 0){
    return 0;
  }


  memmove(s->str + idx + len, s->str + idx, s->len - idx + 1);
  strcpy_escape(s->str + idx, str, len);

  s->len += len;

  return 1;
}

int Str_CatEscape(DfStr *s, const char *str)
{
  int len;
  int req;

  len = strlen(str) + count_has_escape(str);
  req = s->len + len + 1;

  if(extend(s, req) == 0){
    return 0;
  }

  strcpy_escape(s->str + s->len, str, len);

  s->len += len;
  s->str[s->len] = '\0';

  return 1;
}

int Str_Delete(DfStr *s, int idx, int len)
{
  char *src;
  char *dst;
  int mov;

  src = s->str + idx + len;
  dst = s->str + idx;
  s->len -= len;
  mov = s->len - idx + 1;
  memmove(dst, src, mov);

  return 1;
}



int Str_Shrink(DfStr *s)
{
  char *p;

  p = bReAlloc(s->str, s->len + 1);
  if(p){
    s->str = p;
    s->alloced_len = s->len + 1;
  }
  return 1;
}

int Str_IsNull(DfStr *s)
{
  if(s->str == NULL){
    return 1;
  }

  return 0;
}

int Str_RegPath(DfStr *path, int bSlash, int mode)
{
  char *p;
  char *s;
  int len;
  char *home;
  MbStr mb;

  p = path->str;
  switch(p[0]){
  case '~':
    switch(p[1]){
    case '/':
    case '\0':
      home = getenv("HOME");
      memmove(path->str, path->str + 1, path->len);
      if(home){
	path->len--;
	Str_Insert(path, 0, home);
      }
      dprintf("path %s\n", path->str);
      break;
    }
  }

  p = path->str;
  s = p;
  InitMbStr(&mb, p);

  while(*p){
    switch(*p){
    case '/':
      switch(p[1]){
      case '.':		/* /.? */
	switch(p[2]){
	case '/':	/* /./ */
	case '\0':	/* /. */
	  if(mode == 1 && p[2] != '/'){
	  }else{
	    p = MbSkipChar(&mb, 2);
	    continue;
	  }
	  break;
	case '.':	/* /..? */
	  if(p[3] == '/' || p[3] == '\0'){
	    /* /../ or /.. */
	    if(mode == 1 && p[3] != '/'){
	    }else{
	      p = MbSkipChar(&mb, 3);
	      /* back */
	      *s = '\0';
	      s = FindLastChar(path->str, '/');
	      continue;
	    }
	  }
	  break;
	}
	break;
      case '/':		/* // */
	while(p[1] == '/')
	  p = MbSkipChar(&mb, 1);
	break;
      }
      break;
    }

    /* store */
    switch(mb.c_len){
    case 0:
    case -1:
      break;
    case 1:
      *s = *p;
      s++;
      break;
    default:
      memcpy(s, p, mb.c_len);
      s += mb.c_len;
    }
    p = MbNextChar(&mb);
  }

  len = s - path->str;
  if(len == 0){
    len = 1;
  }
  Str_SetLength(path, len);
  if(bSlash){
    SetLastSlash(path);
  }

  return p != s;
}

int Str_RelativePath(DfStr *result, DfStr *src, DfStr *target)
{
  int level;
  int match;
  const char *p;
  const char *q;
  const char *diff;
  const char *hold;
  char *ret;
  MbStr mbSrc;

  p = src->str;
  q = target->str;
  while(*p == *q){
    p++;
    q++;
  }
  diff = p;

  level = 0;
  match =0;
  hold = NULL;

  p = src->str;
  InitMbStr(&mbSrc, p);
  while(*p){
    if(*p == '/'){
      if(p < diff){
	match++;
	hold = p;
      }
      level++;
    }
    p = MbNextChar(&mbSrc);
  }

  if(match < 1){
    Str_InitStr(result, target->str);
    return 0;
  }
  if(hold){
    hold = target->str + (hold - src->str);
  }else{
    hold = target->str;
  }

  Str_Init(result);
  ret = malloc(512);
  ret[0] = '\0';
  while(match < level){
    Str_Add(result, "../");
    match++;
  }
  if(hold[0] == '/'){
    hold++;
  }
  Str_Add(result, hold);

  return 0;
}


DfStr *Str_Dup(const DfStr *s)
{
  DfStr *p;

  p = bMalloc(sizeof(*p));
  if(p == NULL){
#if MEMDEBUG
      dprintf("no enough memory.\n");
#endif
      return NULL;
  }

  p->str = bMalloc(s->len + 1);
  if(p->str == NULL){
    bFree(p);
#if MEMDEBUG
      dprintf("no enough memory.\n");
#endif
      return NULL;
  }

  p->len = s->len;
  p->alloced_len = s->len + 1;

  memcpy(p->str, s->str, s->len + 1);

  return p;
}

DfStr *Str_Copy(DfStr *d, const DfStr *s)
{
  if(d->alloced_len < s->len){
    extend(d, s->len - d->alloced_len);
  }
  memcpy(d->str, s->str, s->len + 1);
  d->len = s->len;

  return d;
}

void InitMbStrLen(MbStr *p, const char *str, int len)
{
  p->str = str;
  p->len = len;
  p->c_len = p->len == 0 ? 0 : mblen(p->str, p->len);
}

void InitMbStr(MbStr *p, const char *str)
{
  p->str = str;
  p->len = strlen(str);
  p->c_len = p->len == 0 ? 0 : mblen(p->str, p->len);
}

char *MbNextChar(MbStr *p)
{
  int fwd;
  if(!p->str[0]){
    return (char *)p->str;
  }
  fwd = p->c_len;
  p->str += fwd;
  p->len -= fwd;

  fwd = mblen(p->str, p->len);
  if(fwd == -1){
    fwd = 1;
  }
  p->c_len = fwd;
  return (char *)p->str;
}

char *MbSkipChar(MbStr *p, int skip)
{
  if(!p->str[0]){
    return (char *)p->str;
  }
  p->str += skip;
  p->len -= skip;
  p->c_len = mblen(p->str, p->len);
  return (char *)p->str;
}


int MbFind(MbStr *p, MbStr *s)
{
  while(p->str[0]){
    if(StrEqChar(p, s)){
      return 1;
    }
    MbNextChar(p);
  }
  return 0;
}

char *FindChar(const char *p, char c)
{
  MbStr mb;

  InitMbStr(&mb, p);

  while(*p){
    if(*p == c){
      return (char *)p;
    }
    p = MbNextChar(&mb);
  }
  return (char *)p;
}


char *FindLastChar(const char *p, char c)
{
  const char *f;
  MbStr mb;

  InitMbStr(&mb, p);

  f = p + mb.len;
  while(*p){
    if(*p == c){
      f = p;
    }
    p = MbNextChar(&mb);
  }
  return (char *)f;
}


int IsSkipChar(MbStr *mb){
  if(mb->c_len != 1){
    return 0;
  }

  switch(mb->str[0]){
  case ' ':
  case '\r':
  case '\t':
    return 1;
    break;
  }

  return 0;
}

int IsAbsPath(const char *path)
{
  switch(path[0]){
  case '/':
    return 1;
  case '~':
    switch(path[1]){
    case '/':
    case '\0':
      return 1;
    }
  }
  return 0;
}

int IsDots(const char *path)
{
  if(path[0] != '.'){
    return 0;
  }
  switch(path[1]){
  case '\0':
    return 1;
  case '.':
    if(path[2] == '\0'){
      return 1;
    }
  }
  return 0;
}

int StrEqChar(MbStr *a, MbStr *b)
{
  if(a->c_len != b->c_len){
    return 1;
  }

  if(a->c_len == 1){
    return a->str[0] == b->str[0];
  }

  return memcmp(a->str, b->str, a->c_len) == 0;
}


int IsEndSlash(DfStr *path)
{
  char *slash;

  slash = FindLastChar(Str_Get(path), '/');
  if(!slash[0]){
    return 0;
  }

  if(slash[1] == '\0'){
    return 1;
  }
  return 0;
}

void SetLastSlash(DfStr *path)
{
  if(Str_Length(path) != 0 && !IsEndSlash(path)){
    Str_AddChar(path, '/');
  }
  return;
}

int Str_Escape(DfStr *s)
{
  char *p;
  MbStr mb;
  int idx;

  p = s->str;
  InitMbStr(&mb, p);
  do{
    switch(*p){
    case ' ':
    case '[':
    case ']':
    case '\'':
    case '\"':
    case ':':
    case '/':
    case '?':
    case '*':
    case '\\':
      idx = p - s->str;
      Str_InsertChar(s, idx, '\\');
      mb.str = s->str + idx + 1;
      break;
    }
    p = MbNextChar(&mb);
  }while(*p);

  return 1;
}

int Str_Descape(DfStr *s)
{
  char *p;
  MbStr mb;
  int idx;

  p = s->str;
  InitMbStr(&mb, p);
  do{
    switch(*p){
    case '\\':
      idx = p - s->str;
      memmove(p, p + 1, s->len - idx);
      s->len--;
      mb.len--;
      break;
    }
    p = MbNextChar(&mb);
  }while(*p);

  return 1;
}



int Str_Extend(DfStr *s, int len)
{
  return extend(s, len);
}

static int extend(DfStr *s, int len)
{
  char *p;

  len = s->len + len;

  if(s->alloced_len <= len){
    /* extend */
    len += BUF_SIZE;
    len = (len / BUF_SIZE) * BUF_SIZE;
    p = bReAlloc(s->str, len);
    if(p == NULL){
#if MEMDEBUG
      dprintf("no enough memory.\n");
#endif
      return 0;
    }
#if MEMDEBUG
    if(!FindMemInfo(p)){
      dprintf("extend()\n");
    }
#endif
    s->str = p;
    s->alloced_len = len;
  }
  return 1;
}

#if MEMDEBUG

int dbgStr_Init(DfStr *s, const char *file, const char *func, int line)
{
  int r;

  r = Str_Init(s);
  if(!FindMemInfo(s->str)){
    dprintf("Str_Init()\n");
  }

  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_InitStr(DfStr *s, const char *str, const char *file, const char *func, int line)
{
  int r;

  r = Str_InitStr(s, str);
  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_Add(DfStr *s, const char *str, const char *file, const char *func, int line)
{
  int r;

  r = Str_Add(s, str);
  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_AddNUL(DfStr *s, const char *str, const char *file, const char *func, int line)
{
  int r;

  r = Str_AddNUL(s, str);
  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_AddChar(DfStr *s, char ch, const char *file, const char *func, int line)
{
  int r;

  r = Str_AddChar(s, ch);
  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_Overwrite(DfStr *s, int idx, const char *str, const char *file, const char *func, int line)
{
  int r;

  r = Str_Overwrite(s, idx, str);
  MemModInfo(s->str, file, func, line);
  return r;
}

int dbgStr_Shrink(DfStr *s, const char *file, const char *func, int line)
{
  int r;

  r = Str_Shrink(s);
  if(!FindMemInfo(s->str)){
    dprintf("Str_Shrink()\n");
  }
  MemModInfo(s->str, file, func, line);
  return r;
}

DfStr *dbgStr_Dup(DfStr *s, const char *file, const char *func, int line)
{
  DfStr *r;

  r = Str_Dup(s);
  if(!FindMemInfo(r->str)){
    dprintf("Str_Dup()\n");
  }

  MemModInfo(r->str, file, func, line);
  MemModInfo(r, file, func, line);
  return r;
}

int dbgStr_Free(DfStr *s, int f, const char *file, const char *func, int line)
{

  if(!s){
    return 1;
  }
  if(s->str && is_trap(s->str)){
    fprintf(stderr, "TRAP: Str_Free() called with watch pointer %p. %s %s %d\n",
	    s->str, file, func, line);
  }
  if(s == NULL){
    return 1;
  }

  if(s->str){
    dbgFree(s->str, file, func, line);
  }
  if(f){
    dbgFree(s, file, func, line);
  }else{
    s->str = NULL;
    s->len = 0;
    s->alloced_len = 0;
  }
#if 0
  if(!FindMemInfo(s->str)){
    dprintf("Str_Free()\n");
  }

  if(f && !FindMemInfo(s)){
    dprintf("Str_Free() 2\n");
  }

  MemModInfo(s->str, file, func, line);
  if(f){
    MemModInfo(s, file, func, line);
  }
#endif
  return 0;
}


#endif
