package org.blueforest.rockhouse.encoding_converter.test.popup.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.blueforest.rockhouse.encoding_converter.popup.actions.LineDelimiter;
import org.blueforest.rockhouse.encoding_converter.popup.actions.LineDelimiterConvertInputStream;
import org.junit.Test;

public class LineDelimiterConvertInputStreamTest {
	private LineDelimiterConvertInputStream createStream(String source, LineDelimiter code) {
		return new LineDelimiterConvertInputStream(new ByteArrayInputStream(source.getBytes()), code);
	}
	
	private String readAll(InputStream is) throws IOException {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		while (true) {
			int c = is.read();
			if (c == -1) break;
			
			os.write(c);
		}
		return os.toString();
	}

	private String repeat(char c, int count) {
		StringBuilder builder = new StringBuilder();
		for(int i = 0; i < count; ++i) {
			builder.append('*');
		}
		return builder.toString();
	}

	@Test
	public void testRead_NoLineDelimiter() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("test", LineDelimiter.CR);
		assertEquals("test", this.readAll(stream));
	}

	@Test
	public void testRead_CrToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r222", LineDelimiter.CR);
		assertEquals("111\r222", this.readAll(stream));
	}

	@Test
	public void testRead_CrToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r222", LineDelimiter.LF);
		assertEquals("111\n222", this.readAll(stream));
	}

	@Test
	public void testRead_CrToCrLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r222", LineDelimiter.CRLF);
		assertEquals("111\r\n222", this.readAll(stream));
	}

	@Test
	public void testRead_LfToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n222", LineDelimiter.CR);
		assertEquals("111\r222", this.readAll(stream));
	}

	@Test
	public void testRead_LfToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n222", LineDelimiter.LF);
		assertEquals("111\n222", this.readAll(stream));
	}

	@Test
	public void testRead_LfToCrLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n222", LineDelimiter.CRLF);
		assertEquals("111\r\n222", this.readAll(stream));
	}

	@Test
	public void testRead_CrLfToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n222", LineDelimiter.CR);
		assertEquals("111\r222", this.readAll(stream));
	}

	@Test
	public void testRead_CrLfToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n222", LineDelimiter.LF);
		assertEquals("111\n222", this.readAll(stream));
	}

	@Test
	public void testRead_CrLfToCrLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n222", LineDelimiter.CRLF);
		assertEquals("111\r\n222", this.readAll(stream));
	}

	@Test
	public void testRead_lastCr_ToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r", LineDelimiter.CR);
		assertEquals("111\r", this.readAll(stream));
	}

	@Test
	public void testRead_lastCr_ToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r", LineDelimiter.LF);
		assertEquals("111\n", this.readAll(stream));
	}

	@Test
	public void testRead_lastCr_ToCrlf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r", LineDelimiter.CRLF);
		assertEquals("111\r\n", this.readAll(stream));
	}

	@Test
	public void testRead_lastLf_ToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n", LineDelimiter.CR);
		assertEquals("111\r", this.readAll(stream));
	}

	@Test
	public void testRead_lastLf_ToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n", LineDelimiter.LF);
		assertEquals("111\n", this.readAll(stream));
	}

	@Test
	public void testRead_lastLf_ToCrlf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n", LineDelimiter.CRLF);
		assertEquals("111\r\n", this.readAll(stream));
	}

	@Test
	public void testRead_lastCrLf_ToCr() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n", LineDelimiter.CR);
		assertEquals("111\r", this.readAll(stream));
	}

	@Test
	public void testRead_lastCrLf_ToLf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n", LineDelimiter.LF);
		assertEquals("111\n", this.readAll(stream));
	}

	@Test
	public void testRead_lastCrLf_ToCrlf() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\r\n", LineDelimiter.CRLF);
		assertEquals("111\r\n", this.readAll(stream));
	}

	@Test
	public void testRead_SequentialDelimiter() throws Exception {
		LineDelimiterConvertInputStream stream = createStream("111\n\n\r\r\n\r\n", LineDelimiter.CRLF);
		assertEquals("111\r\n\r\n\r\n\r\n\r\n", this.readAll(stream));
	}

	@Test
	public void testRead_BufferSizeOver() throws Exception {
		String base = this.repeat('*', 4096);
		
		LineDelimiterConvertInputStream stream = createStream(base + "\r\n" + base, LineDelimiter.LF);
		assertEquals(base + "\n" + base, this.readAll(stream));
	}


	@Test
	public void testRead_DelimitorOnBufferBound() throws Exception {
		String base = this.repeat('*', 4095);
		
		LineDelimiterConvertInputStream stream = createStream(base + "\r\n", LineDelimiter.LF);
		assertEquals(base + "\n", this.readAll(stream));
	}

	@Test
	public void testAvailable() throws IOException {
		String base = this.repeat('*', 4095);
		
		LineDelimiterConvertInputStream stream = createStream("\r\n" + base, LineDelimiter.LF);
		assertEquals(4097, stream.available());

		stream.read();
		assertEquals(4095, stream.available());
	}

	@Test
	public void testLineDelimiterConvertInputStream() {
		//tested on other case
	}

	@Test
	public void testHasCr() throws IOException {
		String source = "111\r222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertTrue(stream.hasCr());
	}

	@Test
	public void testHasCr_NotExists() throws IOException {
		String source = "111\r\n222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertFalse(stream.hasCr());
	}

	@Test
	public void testHasLf() throws IOException {
		String source = "111\n222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertTrue(stream.hasLf());
	}

	@Test
	public void testHasLf_NotExists() throws IOException {
		String source = "111\r\n222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertFalse(stream.hasLf());
	}

	@Test
	public void testHasCrLf() throws IOException {
		String source = "111\r\n222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertTrue(stream.hasCrLf());
	}

	@Test
	public void testHasCrLf_NotExists() throws IOException {
		String source = "111\r222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CR);
		this.readAll(stream);
		assertFalse(stream.hasCrLf());
	}

	@Test
	public void testComplexDelimiter() throws IOException {
		String source = "111\n\r\r\n222";
		LineDelimiterConvertInputStream stream = createStream(source, LineDelimiter.CRLF);
		assertEquals("111\r\n\r\n\r\n222", this.readAll(stream));
		assertTrue(stream.hasCr());
		assertTrue(stream.hasLf());
		assertTrue(stream.hasCrLf());
	}
}
