package Egg::Model::DBIC;
#
# Masatoshi Mizuno E<lt>lusheE<64>cpan.orgE<gt>
#
# $Id: DBIC.pm 176 2007-06-27 05:56:06Z lushe $
#
use strict;
use warnings;
use UNIVERSAL::require;
use base qw/Egg::Model/;

our $VERSION = '2.03';

=head1 NAME

Egg::Model::DBIC - DBIx::Class for Egg Model.

=head1 SYNOPSIS

  __PACKAGE__->egg_startup(
    ...
    .....
  
    MODEL=> [
      [ DBIC => { schema_names => [qw/ MySchema /] } } ],
     ],
  
    );

  # The object of MyApp::Model::DBIC::MySchema is acquired.
  my $model= $e->model('MySchema');
  
  # If the transaction is effective.
  $model->storage->txn_begin;
  
  # The object of MyApp::Model::DBIC::MySchema::Moniker is acquired.
  my $db = $model->resultset('Moniker');
     or
  my $db = $e->model('myscema:moniker');
  
  $db->search( ... );
  
  $model->storage->txn_commit;
    or
  $model->storage->txn_rollback;

=head1 DESCRIPTION

It is MODEL to use DBIx::Class.

Appropriate Schema is generated by using the helper for use.

  % perl myapp_helper.pl Model:DBIC MySchema -d dbi:Pg:dbname=dbname -u user -p passwd

/path/to/MyApp/lib/MyApp/Model/DBIC/MySchema is generated with this.

* Please refer to the document of L<Egg::Helper::Model::DBIC > for details
  for helper's use.

And, the setting of following MODEL is added.

  MODEL => [ [ DBIC => {} ] ],

All modules under the control of/path/to/MyApp/lib/MyApp are read by this as
Schema.  * Specification becomes empty HASH.

If the module such as Schema for the test that doesn't want to be read exists
together, the name of Schema is specified.

  MODEL=> [
    [ DBIC => { schema_names => [qw/ MySchema /] } ],
   ],

Schema can be acquired for the small letter specification.

  my $schema = $e->model('myschema');

When the source is acquired, it specifies it by the small letter without fail.

  my $source = $e->mode('myschema:source');

When the object is acquired, it only has to operate it L<DBIx::Class> originally.

Please refer to the document of L<DBIx::Class> for details.

=cut
sub _setup {
	my($class, $e, $conf)= shift->SUPER::_setup(@_);

	if ($e->debug && ! defined($ENV{DBIC_TRACE})) {
		$ENV{DBIC_TRACE} = 1;
		$ENV{DBIC_TRACE}.= "=$conf->{trace_file}" if $conf->{trace_file};
	}

	my $names= $conf->{schema_names} ||= do {
		my @names;
		my $schema_dir= $e->path('lib_project', '/DBIC');
		for (<$schema_dir/*>) {  ## no critic
			m{([^\\\/\:]+)\.pm$} || next;
			push @names, $1;
		}
		@names || die qq{ Schema module is not found from '$schema_dir'. };
		if ($e->debug) {
			$e->debug_out("# + $e->{namespace}"
			            . " - DBIC load schema: ". join(', ', @names));
		}
		\@names;
	  };

	for my $name (ref($names) eq 'ARRAY' ? @$names: $names) {
		my $schema_class= qq{$e->{namespace}::DBIC::${name}};

		my $model_name= lc($name);
		$e->regist_model($model_name, $schema_class, 1);

		my $conf= $schema_class->config;
		$conf->{dsn}  || die qq{ I want setup '$schema_class'-> 'dsn'.  };
		$conf->{user} || die qq{ I want setup '$schema_class'-> 'user'. };
		$conf->{password} ||= "";
		$conf->{options}  ||= {};

		no strict 'refs';  ## no critic;
		no warnings 'redefine';
		*{"${schema_class}::new"}= $class->_mk_schema_closure
		   ($schema_class, @{$conf}{qw{ dsn user password options }});

		my $schema= $schema_class->new
		         || die qq{ Schema of '$schema_class' cannot be acquired. };
		for my $moniker ($schema->sources) {
			my $moniker_class= "$schema_class\::$moniker";
			$e->regist_model(lc("${name}:${moniker}"), $moniker_class);
			*{"${moniker_class}::ACCEPT_CONTEXT"}= sub {
				$_[1]->model($model_name)->resultset($moniker);
			  };
		}
	}

	$class;
}
sub _mk_schema_closure {
	my($class, $schema_class, @source)= @_;
	my $schema;
	sub {
		return $schema if ( $schema
		  && $schema->storage->dbh->{Active}
		  && $schema->storage->dbh->ping
		  );
		$schema= $schema_class->connect(@source);
	 };
}

=head1 CREATE SCHEMA SOURCE

Please note the following respect when you make the Schema file by oneself.

=over 4

=item * The Schema module succeeds to Egg::Model::DBIC::Schema.

a necessary method for Schema to operate on Egg such as 'config' is added.

  package MyApp::MySchema;
  use strict;
  use warnings;
  ## use base 'DBIx::Class::Schema';
  
  use base qw/Egg::Model::DBIC::Schema/;
  
  __PACKAGE__->config(
    dsn      => 'dbi:Pg:dbname=mydata',
    user     => 'user',
    password => 'passwd',
    );
  
  ...
  .....

=back

=head1 SEE ALSO

L<DBIx::Class>,
L<DBIx::Class::Schema>,
L<Egg::Model::DBIC::Schema>,
L<Egg::Model::DBIC::Moniker>,
L<Egg::Helper::Model::DBIC>,
L<Egg::Release>,

=head1 AUTHOR

Masatoshi Mizuno, E<lt>lusheE<64>cpan.orgE<gt>

=head1 COPYRIGHT

Copyright (C) 2007 by Bee Flag, Corp. E<lt>L<http://egg.bomcity.com/>E<gt>, All Rights Reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.

=cut

1;
