/**
 * @file discinfo.c
 * @brief ディスク情報取得関連
 * @author BananaJinn
 * @version $Id: discinfo.c,v 1.11 2010/11/02 15:23:51 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#if defined(WIN32)
# include <io.h>
#endif

#include "mem.h"
#include "aspi.h"
#include "struct.h"
#include "cmd.h"
#include "ui.h"
#include "text.h"
#include "log.h"
#include "discinfo.h"
#include "emg.h"
#include "ebstring.h"

/**
 * ディスク情報を設定(イメージファイルに記録する)
 * @param[in]	drive	書込装置構造体
 * @param[in]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int SetDiscInformation(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	int ret;

	if(drive->type == CMDDRVTYPE_ISO){
		/* ISOイメージの場合は、1トラック/1セッションのみ */
		if(discinfo->tracks > 1){
			UIDispMessage(MSG_CANT_CREATE_MULTITRACK_ISO,
						  UIDMT_ERROR);
			return RET_NG;
		}
	}
	else if(drive->type == CMDDRVTYPE_IMAGE){
		/* 円盤複写屋オリジナルイメージ */
		/* イメージファイルにディスク情報を書き込む */
		ret = EmgWriteDiscInformation(&drive->u.image, discinfo);
		if(ret!=RET_OK)
			return ret;
	}
	
	return RET_OK;
}


/**
 * TAO記録すべきかどうかを判別する
 * @param[in]	drive	ドライブ構造体
 * @param[in]	discinfo  ディスク情報構造体
 * @param[in]	sess_num  セッション番号
 * @param[out]	result    判別結果(TAO=TRUE)
 * @retval	RET_OK  正常終了
 * @retval	RET_NG	エラー
 */
static int JudgeTAO(CMDDRIVE *drive, CPDISCINFO *discinfo,
					WORD sess_num, BOOL *result)
{
	if(!REALDRIVE(drive))
		return RET_OK;

	*result = FALSE;
	if(!DT_CD_FAMILY(drive->disc_type)){
		*result = TRUE;	/* DVD系の場合はTAOとする */
		DebugLog("JudgeTAO: TAO(Disc is not CD.)\n");
		return RET_OK;
	}

	if(sess_num == discinfo->sessions){
		/* 最終セッションの場合はオープンセッションかどうかを確認 */
		if(discinfo->last_sess_stat == SESSSTAT_EMPTY){
			*result = FALSE;
			DebugLog("JudgeTAO: Session is empty.\n");
			return RET_OK;
		}
		else if(discinfo->last_sess_stat == SESSSTAT_INCOMPLETE){
			/* 未完了セッションの場合はTAOとする */
			*result = TRUE;
			DebugLog("JudgeTAO: TAO(Last session is incomplete.)\n");
			return RET_OK;
		}
	}

	/*
	 * 以前はセッションの最後に Run-out があるかどうかを判別していたが、
	 * もはや Run-out がリードエラーにならないドライブもあり、判別が不可能!?
	 * セッションがクローズされているなら、SAOで記録することにする。
	 */
	*result = FALSE;
	return RET_OK;
}
	
/**
 * 指定されたセッションのリードアウト開始アドレスを求める。
 * @param[in]	drive	ドライブ構造体
 * @param[in]	discinfo  ディスク情報構造体
 * @param[in]	sess_num  セッション番号
 * @param[out]	lout_start  Lead-out開始アドレス
 * @retval	RET_OK  正常終了
 * @retval	RET_NG	エラー
 */
static int GetLeadOutStart(CMDDRIVE *drive, CPDISCINFO *discinfo,
						   WORD sess_num, DWORD *lout_start)
{
	int ret;
	int i;
	struct _FULLTOCDESC *ftd;
	DWORD lba=0UL;
	WORD tno;
	WORD sno;
	CPTRACKINFO *cpti;

	/* TOC情報から該当セッションのリードアウト開始アドレスを求める */
	ret = SendReadToc(drive, sess_num, 1, RTF_FULLTOC);
	if(ret==RET_OK){
		for(i=sizeof(struct _TOCHEADER); i<drive->bufsize; i+=sizeof(struct _FULLTOCDESC)){
			ftd = (struct _FULLTOCDESC *)(drive->data_buf+i);
			if(ftd->point == 0xa2){
				lba = MSF2LBA(ftd->pmin, ftd->psec, ftd->pframe, FALSE);
				break;
			}
		}
	}
	else{
		/* ReadTOCがエラーになる場合はトラック情報から求める */
		for(tno=1; tno<=discinfo->tracks; tno++){
			cpti = &discinfo->trackinfo[tno-1];
			sno = (WORD)cpti->trackinfo.session_number_msb<<8 |
				cpti->trackinfo.session_number_lsb;
			if(sno == sess_num){
				lba = Get4bytes(cpti->trackinfo.track_start) +
					Get4bytes(cpti->trackinfo.track_size);
			}
		}
		/*
		 * Run-outがあると仮定しておいて +2 する。
		 * 実際に読んでアドレスエラーなら Run-out は無かったことにする。
		 */
		lba += 2;
	}
  
	if(lba==0UL){
		/* 取得できなかった */
		*lout_start = lba;
		return RET_OK;
	}
  
	/* リードアウト開始アドレスの直前2ブロックを読んでみる */
	ret = SendReadCD(drive, lba-2, 2);
	if(ret != RET_OK){
		if(SD_ASC(drive)==0x21){
			/* Run-out は無かった */
			lba -= 2;
		}
	}

	*lout_start = lba;

	return RET_OK;
}

/**
 * 読取装置の指定トラックのプリギャップ長を取得する
 * @param[in]	drive	読取装置構造体
 * @param[in/out] cpti	トラック情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetGapLen(CMDDRIVE *drive, CPTRACKINFO *cpti)
{
	int ret;
	DWORD search_start, search_end, lba=0, track_start;
	DWORD real_start;
	BYTE track_num = cpti->trackinfo.track_number_lsb;

	track_start = Get4bytes(cpti->trackinfo.track_start);
	search_end = track_start;
	search_start = search_end - 151;	/* GAP長=150が一般的なので、その手前から検索 */
	real_start = search_start;

	while(TRUE){
		/*
		 * ２分探索するよりもシーケンシャルでアクセスした方がたぶん速い
		 */
		for(lba=search_start; lba<search_end; lba++){
			ret = GetSubQ(drive, lba, 1);
			if(ret!=RET_OK){
				search_start++;
				continue;
			}
			if((drive->data_buf[0] & 0x0f)!=0x01){
				/* たぶんISRCかMediaCatalogNumberが記録されている */
				search_start++;
				continue;
			}
			if(track_num==drive->data_buf[1]){
				break;
			}
		}
		if(lba==real_start){
			/* 検索した先頭が既にギャップなら、もっと前から検索する */
			search_end = search_start;
			search_start -= 150;
			real_start = search_start;
			continue;
		}
		break;
	}
	if(lba >= track_start)
		cpti->pause_len = 0;
	else if(track_start-lba > 0xffff)
		cpti->pause_len = 0xffff;
	else
		cpti->pause_len = (WORD)(track_start - lba);

	if(cpti->pause_len>0){
		/* 念のため、ギャップの最後も確認する */
		ret = GetSubQ(drive, Get4bytes(cpti->trackinfo.track_start)-1, 1);
		if(ret==RET_OK){
			if((drive->data_buf[0] & 0x0f)==0x01 &&
			   drive->data_buf[1]==track_num &&
			   drive->data_buf[2]>0){
				cpti->pause_len--;
			}
		}
	}

	return RET_OK;
}

/**
 * トラック情報取得
 * @param[in]	drive	読取装置構造体
 * @param[in]	track_num	トラック番号
 * @param[out]	cpti	トラック情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetTrackInformation(CMDDRIVE *drive, WORD track_num, CPTRACKINFO *cpti)
{
	int ret;

	ret = SendReadTrackInfo(drive, track_num);
	if(ret!=RET_OK){
		DispCommandError(drive);
		return ret;
	}
	memcpy(&cpti->trackinfo, drive->data_buf, sizeof(struct _TRACKINFO));
	cpti->mode2 = 0;
	cpti->pause_len = 0;
	if(cpti->trackinfo.blank)
		return RET_OK;
	if(DT_CD_FAMILY(drive->disc_type)){
		memset(cpti->isrc, 0, 12+1);
		if(IS_TRACKMODE_DATA(cpti->trackinfo.track_mode)){
			/* MODE2かどうかを調べる */
			ret = SendReadCD(drive, Get4bytes(cpti->trackinfo.track_start), 1);
			if(ret!=RET_OK){
				DispCommandError(drive);
				return ret;
			}
			if(drive->data_buf[15]==2)
				cpti->mode2 = 1;
		}
		else{
			ret = SendReadSubchannel(drive, (BYTE)track_num, 0, 1, RSCF_ISRC);
			if(ret!=RET_OK){
				DispCommandError(drive);
				return ret;
			}
			if(drive->data_buf[8] & 0x80){
				memcpy(cpti->isrc, drive->data_buf+9, 12);
			}
			if(track_num>1){
				/* ギャップ長を調べる */
				ret = GetGapLen(drive, cpti);
				if(ret!=RET_OK){
					return ret;
				}
			}
		}
	}

	DebugLog("GetTrackInformation: start=0x%08lX size=0x%08lX tmode=0x%X mode2=%d gap=0x%X\n",
			 Get4bytes(cpti->trackinfo.track_start),
			 Get4bytes(cpti->trackinfo.track_size),
			 cpti->trackinfo.track_mode,
			 cpti->mode2, cpti->pause_len);

	return RET_OK;
}

/**
 * トラックサイズを調整する
 * @param[in]	drive	読取装置構造体
 * @param[in/out] discinfo ディスク情報
 * @param[in]	track_num  トラック番号
 * @param[in]	lout_start Lead-out開始アドレス
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int AdjustTrackSize(CMDDRIVE *drive, CPDISCINFO *discinfo,
						   WORD track_num, DWORD lout_start)
{
	DWORD next_start=0;
	CPTRACKINFO *cpti, *ncpti=NULL;
	DWORD lba, track_size;
	int ret;
#if 0
	char buf[80];
#endif

	if(!REALDRIVE(drive))
		return RET_OK;

	if(DT_DVD_FAMILY(drive->disc_type))
		return RET_OK;

	next_start = lout_start;
	cpti = &discinfo->trackinfo[track_num-1];
	if(cpti->tao){
		/* 完了トラックならトラックの最後2ブロックを読んでみる */
		if(Get4bytes(cpti->trackinfo.free_blocks)==0){
			lba = Get4bytes(cpti->trackinfo.track_start) + 
				Get4bytes(cpti->trackinfo.track_size) -2;
			ret = SendReadCD(drive, lba, 2);
			if(ret!=RET_OK){
				/* 読めないならたぶん Run-out でしょう */
				track_size = Get4bytes(cpti->trackinfo.track_size) -2;
				Set4bytes(cpti->trackinfo.track_size, track_size);
			}
		}
		return RET_OK;
	}
	if(track_num < discinfo->tracks){
		ncpti = &discinfo->trackinfo[track_num];
		if(((WORD)cpti->trackinfo.session_number_msb<<8 | cpti->trackinfo.session_number_lsb)==
		   ((WORD)ncpti->trackinfo.session_number_msb<<8 | ncpti->trackinfo.session_number_lsb)){
			/* 次のトラックも同じセッション */
			next_start = Get4bytes(ncpti->trackinfo.track_start);
			if(IS_TRACKMODE_DATA(ncpti->trackinfo.track_mode)){
				next_start -= 150;
				if(!IS_TRACKMODE_DATA(cpti->trackinfo.track_mode) ||
				   ncpti->mode2 != cpti->mode2){
					/* 2part */
					next_start -= 75;
				}
			}
			else{
				next_start -= ncpti->pause_len;
			}
		}
	}
	if(next_start==0)
		return RET_OK;

	/* 実際に読めるとこまで読んでみる? */
	for(lba=next_start-4; lba<next_start; lba++){
		ret = SendReadCD(drive, lba, 1);
		if(ret!=RET_OK)
			break;
	}

	track_size = lba-Get4bytes(cpti->trackinfo.track_start);
#if 0
	if(track_size != Get4bytes(cpti->trackinfo.track_size)){
		sprintf(buf, "トラック%d のサイズは 0x%08lX から 0x%08lX に変更されました。",
				track_num, Get4bytes(cpti->trackinfo.track_size), track_size);
		UIDispMessage(buf, UIDMT_INFORMATION);
	}
#endif
	Set4bytes(cpti->trackinfo.track_size, track_size);

	return RET_OK;
}

/**
 * CD-TEXT情報取得
 * @param[in]	drive	読取装置構造体
 * @param[out]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetCDText(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	int ret;
	WORD len;

	ret = SendReadToc(drive, 0, 0, RTF_CDTEXT);
	if(ret!=RET_OK){
		return RET_OK;
	}
	len = Get2bytes(drive->data_buf);
	if(len<=2){	/* header(4bytes) - length field(2bytes) */
		/* CD-TEXTなし */
		return RET_OK;
	}
	len -= 2;
	discinfo->cdtext = (BYTE *)MemNew(len);
	if(discinfo->cdtext==NULL)
		return RET_MEMERR;
	memcpy(discinfo->cdtext, drive->data_buf+4, len);
	discinfo->cdtext_size = len;
	return RET_OK;
}

/**
 * @brief 1セッション1トラックのディスク情報を作成する
 * @param[in] track_size トラックサイズ
 * @param[out] discinfo ディスク情報
 * @retval RET_OK 正常終了
 * @retval RET_MEMERR メモリ確保エラー
 */
static int CreateDummyDiscInformation(DWORD track_size, CPDISCINFO *discinfo)
{
	/* ディスク情報設定 */
	discinfo->disc_stat = DISCSTAT_COMPLETE;
	discinfo->last_sess_stat = SESSSTAT_COMPLETE;
	discinfo->last_addr = (DWORD)(track_size > 0 ? track_size-1 : 0);
	discinfo->sessions = 1;
	discinfo->tracks = 1;
	discinfo->trackinfo = (CPTRACKINFO *)MemNew(sizeof(CPTRACKINFO));
	if(discinfo->trackinfo==NULL){
		return RET_MEMERR;
	}
	memset(discinfo->trackinfo, 0, sizeof(CPTRACKINFO));
	discinfo->trackinfo->mode2 = 0;
	discinfo->trackinfo->tao = FALSE;
	discinfo->trackinfo->trackinfo.track_number_lsb = 1;
	discinfo->trackinfo->trackinfo.session_number_lsb = 1;
	discinfo->trackinfo->trackinfo.track_mode = 4;
	discinfo->trackinfo->trackinfo.data_mode = 0x0f;
	Set4bytes(discinfo->trackinfo->trackinfo.track_start, 0);
	Set4bytes(discinfo->trackinfo->trackinfo.track_size, track_size);
	return RET_OK;
}

/**
 * @brief ディスク情報取得(mkisofs)
 * @param[in] drive 読取装置構造体
 * @param[in] discinfo ディスク情報
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 * @retval RET_MEMERR メモリ確保エラー
 */
static int GetDiscInformationMkisofs(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	DWORD track_size = 0;
	UIDispInfo(MSG_READING);
	if(UICheckAbort()){
		return RET_ABORT;
	}
	track_size = MIFGetBlockCount(&drive->u.mkisofs);
	if(track_size == 0){
		char *message = EbStringNewWithFormat(
			MSG_MKISOFS_ERROR_,
			MIFGetErrorMessage(&drive->u.mkisofs));
		UIDispMessage(message, UIDMT_ERROR);
		EbStringFree(message);
		return RET_NG;
	}
	UIDispInfo("");
	return CreateDummyDiscInformation(track_size, discinfo);
}

/**
 * ディスク情報取得(ISOイメージ)
 * @param[in]	drive	読取装置構造体
 * @param[out]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetDiscInformationISO(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	DWORD track_size;
#ifdef WIN32
	__int64 file_size;
#else
	off_t file_size;
#endif

	/* ファイルサイズ取得 */
#ifdef WIN32
	file_size = _lseeki64(_fileno(drive->u.image.fp), 0, SEEK_END);
	_lseeki64(_fileno(drive->u.image.fp), 0, SEEK_SET);
#else
	fseeko(drive->u.image.fp, 0, SEEK_END);
	file_size = ftello(drive->u.image.fp);
	fseeko(drive->u.image.fp, 0, SEEK_SET);
#endif
	track_size = (DWORD)((file_size+2047)/2048);
		
	return CreateDummyDiscInformation(track_size, discinfo);
}

/**
 * ディスク情報取得(イメージファイル)
 * @param[in]	drive	読取装置構造体
 * @param[out]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetDiscInformationImage(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	int ret;

	ret = EmgReadDiscInformation(&drive->u.image, discinfo);
	if(ret!=RET_OK)
		return ret;

	return RET_OK;
}

/**
 * ディスク情報取得(実ドライブ)
 * @param[in]	drive	読取装置構造体
 * @param[out]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
static int GetDiscInformationDrive(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	WORD track_num;
	WORD sess_num, last_sess_num;
	BOOL tao=FALSE;
	struct _DISCINFO *di;
	CPTRACKINFO *cpti;
	DWORD last_addr=0, lout_start=0;
	int ret;

	ret = SendReadDiscInfo(drive);
	if(ret!=RET_OK){
		DispCommandError(drive);
		return ret;
	}
	di = (struct _DISCINFO *)drive->data_buf;
	discinfo->disc_stat = di->disc_status;
	discinfo->last_sess_stat = di->sess_status;
	discinfo->sessions = (WORD)di->numsess_msb<<8 | di->numsess_lsb;
	discinfo->tracks = (WORD)di->last_track_ls_msb<<8 | di->last_track_ls_lsb;
	discinfo->disc_type = di->disc_type;

	DebugLog("GetDiscInformationDrive: disc_status=%d\n",
			 discinfo->disc_stat);
	DebugLog("GetDiscInformationDrive: last_sess_status=%d\n",
			 discinfo->last_sess_stat);
	DebugLog("GetDiscInformationDrive: tracks=%d\n",
			 discinfo->tracks);
	DebugLog("GetDiscInformationDrive: disc_type=%d\n",
			 discinfo->disc_type);

	if(discinfo->disc_stat == DISCSTAT_EMPTY){
		UIDispMessage(MSG_SOURCE_IS_BLANK
					  /*"複写元ディスクはブランクです。複写できません。"*/,
					  UIDMT_INFORMATION);
		return RET_ABORT;
	}	  
	discinfo->cdtext = NULL;
	discinfo->cdtext_size = 0;
	memset(discinfo->media_catalog_number, 0, 13+1);
	if(DT_CD_FAMILY(drive->disc_type)){
		ret = SendReadSubchannel(drive, 0, 0, 1, RSCF_MCN);
		if(ret!=RET_OK){
			DispCommandError(drive);
			return ret;
		}
		if(drive->data_buf[8] & 0x80){
			memcpy(discinfo->media_catalog_number, drive->data_buf+9, 13);
		}
		/* CD-TEXT */
		ret = GetCDText(drive, discinfo);
		if(ret!=RET_OK){
			return ret;
		}
	}

	/* トラックごとに情報を取得 */
	UIMeter1Initialize(MSG_READING);
	if(drive->type==CMDDRVTYPE_NET){
		NAUIMeter1Initialize(&drive->u.net, MSG_READING);
	}
	discinfo->trackinfo = (CPTRACKINFO *)MemNew(sizeof(CPTRACKINFO)*discinfo->tracks);
	for(track_num=1; track_num<=discinfo->tracks; track_num++){
		UIMeter1Update((float)track_num*100/discinfo->tracks/2);
		if(drive->type==CMDDRVTYPE_NET){
			NAUIMeter1Update(&drive->u.net, (float)track_num*100/discinfo->tracks/2);
		}
		if(UICheckAbort()){
			MemFree(discinfo->cdtext);
			discinfo->cdtext = NULL;
			MemFree(discinfo->trackinfo);
			discinfo->trackinfo = NULL;
			return RET_ABORT;
		}
		cpti = &discinfo->trackinfo[track_num-1];
		ret = GetTrackInformation(drive, track_num, cpti);
		if(ret!=RET_OK){
			MemFree(discinfo->cdtext);
			discinfo->cdtext = NULL;
			MemFree(discinfo->trackinfo);
			discinfo->trackinfo = NULL;
			return ret;
		}
	}

	last_sess_num = 0;
	for(track_num=1; track_num<=discinfo->tracks; track_num++){
		UIMeter1Update((float)track_num*100/discinfo->tracks/2+50);
		if(drive->type==CMDDRVTYPE_NET){
			NAUIMeter1Update(&drive->u.net, (float)track_num*100/discinfo->tracks/2+50);
		}
		if(UICheckAbort()){
			MemFree(discinfo->cdtext);
			discinfo->cdtext = NULL;
			MemFree(discinfo->trackinfo);
			discinfo->trackinfo = NULL;
			return RET_ABORT;
		}
		cpti = &discinfo->trackinfo[track_num-1];
		/* セッションがTAO記録かどうかを調べる */
		sess_num = (WORD)cpti->trackinfo.session_number_msb<<8 |
			cpti->trackinfo.session_number_lsb;
		if(sess_num != last_sess_num){
			/* TAOかSAOか判断 */
			ret = JudgeTAO(drive, discinfo, sess_num, &tao);
			if(ret!=RET_OK){
				MemFree(discinfo->cdtext);
				discinfo->cdtext = NULL;
				MemFree(discinfo->trackinfo);
				discinfo->trackinfo = NULL;
				return ret;
			}
			/* リードアウト開始アドレスを取得 */
			ret = GetLeadOutStart(drive, discinfo, sess_num, &lout_start);
			if(ret!=RET_OK){
				MemFree(discinfo->cdtext);
				discinfo->cdtext = NULL;
				MemFree(discinfo->trackinfo);
				discinfo->trackinfo = NULL;
				return ret;
			}
			last_sess_num = sess_num;
		}
		cpti->tao = tao;
		/* セッションの最終トラックサイズ調整 */
		ret = AdjustTrackSize(drive, discinfo, track_num, lout_start);
		if(ret!=RET_OK){
			MemFree(discinfo->cdtext);
			discinfo->cdtext = NULL;
			MemFree(discinfo->trackinfo);
			discinfo->trackinfo = NULL;
			return ret;
		}
		/* ディスク最終アドレス更新 */
		if(cpti->trackinfo.nwa_valid){
			if(Get4bytes(cpti->trackinfo.next_writable_addr) > Get4bytes(cpti->trackinfo.track_start))
				last_addr = Get4bytes(cpti->trackinfo.next_writable_addr);
		}
		else{
			last_addr = Get4bytes(cpti->trackinfo.track_start) +
				Get4bytes(cpti->trackinfo.track_size);
		}
	}
	discinfo->last_addr = last_addr;
  
	if(DT_DVD_FAMILY(drive->disc_type) && discinfo->tracks>1){
		/* DVDのDAOでは複数トラック記録できない */
		if(GetOption()->dao){
			/* トラックが複数存在するので DAO 記録できません。\nTAO で記録を続行しますか?" */
			ret = UIDispMessage(MSG_CANT_DAO,
								UIDMT_QUESTION);
			if(ret!=UIDMRET_OK){
				MemFree(discinfo->cdtext);
				discinfo->cdtext = NULL;
				MemFree(discinfo->trackinfo);
				discinfo->trackinfo = NULL;
				return RET_ABORT;
			}
			GetOption()->dao = FALSE;
		}
	}

	UIMeter1Initialize(MSG_TOTAL /*"全体"*/);
	if(drive->type==CMDDRVTYPE_NET){
		NAUIMeter1Initialize(&drive->u.net, MSG_TOTAL /*"全体"*/);
	}

	return RET_OK;
}

/**
 * ディスク情報取得
 * @param[in]	drive	読取装置構造体
 * @param[out]	discinfo  ディスク情報
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int GetDiscInformation(CMDDRIVE *drive, CPDISCINFO *discinfo)
{
	int ret;

	memset(discinfo, 0, sizeof(CPDISCINFO));
	if(!REALDRIVE(drive)){
		if(drive->type == CMDDRVTYPE_MKISOFS){
			/* フォルダからのISOイメージ(mkisofs)の場合 */
			ret = GetDiscInformationMkisofs(drive, discinfo);
			if(ret != RET_OK){
				return ret;
			}
		}
		else if(drive->type == CMDDRVTYPE_ISO){
			/* ISOイメージファイルの場合 */
			ret = GetDiscInformationISO(drive, discinfo);
			if(ret != RET_OK){
				return ret;
			}
		}
		else if(drive->type == CMDDRVTYPE_IMAGE){
			/* イメージファイル(emg)からディスク情報を得る */
			ret = GetDiscInformationImage(drive, discinfo);
			if(ret != RET_OK){
				return ret;
			}
		}
	}
	else{
		/* 実ドライブから情報取得 */
		ret = GetDiscInformationDrive(drive, discinfo);
		if(ret != RET_OK){
			return ret;
		}
	}

	return RET_OK;
}
