/*
 * @file drive.c
 * @brief ドライブアクセス
 * @author BananaJinn
 * @version $Id: drive.c,v 1.16 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */

#include <stdio.h>
#if !defined(WIN32)
# include <stdlib.h>
# include <string.h>
#endif

#include "mem.h"
#include "drive.h"
#include "cmdlog.h"
#include "cmd.h"
#include "ui.h"
#include "text.h"
#include "log.h"

/**
 * @brief ドライブにコマンドを送信する
 * @param[in] drive 送信先ドライブ
 * @parma[in] cdb CDB
 * @param[in] buflen コマンドに付随するデータの転送サイズ
 * @param[in] reqflag データ転送方向
 * (REQ_NODATA:データ転送なし, REQ_DATAOUT:データ送信, REQ_DATAIN:データ受信)
 * @retval RET_OK コマンド成功
 * @retval RET_NG コマンド失敗
 */
int SendCmd(CMDDRIVE *drive, BYTE *cdb, DWORD buflen, BYTE reqflag)
{
	int ret;

	if(drive->cmd_type == CMDDRVCTYPE_UNKNOWN){
		drive->cmd_type = CMDDRVCTYPE_CHECKING;
		CheckAtapi(drive);
	}

	drive->cmdcode = cdb[0];
	if(drive->type == CMDDRVTYPE_DRIVE){
		ret = SendAspiCmd(&drive->u.drive, cdb, drive->data_buf, buflen,
						  drive->sense_data, sizeof(drive->sense_data), reqflag);
	}
	else if(drive->type == CMDDRVTYPE_NET){
		ret = SendNetCmd(&drive->u.net, cdb, drive->data_buf, buflen,
						 drive->sense_data, sizeof(drive->sense_data), reqflag);
	}
	else{
		return RET_NG;
	}
	CmdLog(drive, cdb, buflen, reqflag, ret);
#ifdef DEBUGCMD
	DebugLog("SendCmd: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X len=0x%08lX io=%d\n",
			 cdb[0], cdb[1], cdb[2], cdb[3], cdb[4], cdb[5],
			 cdb[6], cdb[7], cdb[8], cdb[9], cdb[10], cdb[11],
			 buflen, reqflag);
#endif

	if(ret != RET_OK){
		DebugLog("SendCmd: Command error. cmd=%02Xh %X-%02X-%02X\n",
				 cdb[0], SD_SENSEKEY(drive), SD_ASC(drive), SD_ASCQ(drive));
	}

	return ret;
}

/**
 * @brief 長い読み込み要求を送信する
 * @param[in] drive 送信先ドライブ
 * @param[in] cdb CDB
 * @param[in] buflen データ転送サイズ
 * @param[in] reqflag データ転送方向
 * (REQ_NODATA:データ送信なし, REQ_DATAOUT:データ送信, REQ_DATAIN:データ受信)
 * @param[in] total_blocks 転送ブロック数
 * @retval RET_OK コマンド成功
 * @retval RET_NG コマンド失敗
 */
int SendCmdLongRead(CMDDRIVE *drive, BYTE *cdb, DWORD buflen, BYTE reqflag,
					DWORD total_blocks)
{
	int ret;

	if(drive->cmd_type == CMDDRVCTYPE_UNKNOWN){
		drive->cmd_type = CMDDRVCTYPE_CHECKING;
		CheckAtapi(drive);
	}

	if(drive->type == CMDDRVTYPE_NET){
		ret = SendNetCmdLongRead(&drive->u.net, cdb, drive->data_buf, buflen,
								 drive->sense_data, sizeof(drive->sense_data),
								 reqflag, total_blocks);
	}
	else{
		return RET_NG;
	}

	CmdLog(drive, cdb, buflen, reqflag, ret);
	
	return ret;
}

/**
 * @brief ドライブに対する初期設定を行う
 * @param[in] drive ドライブ
 * @param[in] bufsize 転送バッファーサイズ(バイト数)
 * @param[in] cmdlog_size ログサイズ(レコード数)
 * @param[in] type ドライブ種別(CMDDRVTYPE_*)
 * @retval RET_OK 成功
 * @retval RET_NG 失敗
 */
int InitializeCmdDrive(CMDDRIVE *drive, int bufsize, int cmdlog_size, int type)
{
	int ret = RET_NG;

	if(type != CMDDRVTYPE_DRIVE && type != CMDDRVTYPE_NET &&
	   type != CMDDRVTYPE_IMAGE && type != CMDDRVTYPE_ISO &&
	   type != CMDDRVTYPE_MKISOFS){
		return RET_NG;
	}

	FreeCmdDrive(drive);
	drive->cmd_type = CMDDRVCTYPE_UNKNOWN;

	if(CmdLogInit(drive, cmdlog_size)!=RET_OK){
		return RET_NG;
	}

	if(bufsize>0){
		drive->data_buf = (BYTE *)MemNew(bufsize);
		if(drive->data_buf==NULL){
			return RET_NG;
		}
		drive->bufsize = bufsize;
		drive->own_alloc = TRUE;
	}

	ret = RET_NG;
	drive->type = type;
	if(type == CMDDRVTYPE_DRIVE){
		ret = InitializeDrive(&drive->u.drive);
	}
	else if(type == CMDDRVTYPE_NET){
		ret = InitializeSOCKCB(&drive->u.net);
	}
	else if(type == CMDDRVTYPE_IMAGE){
		ret = InitializeImage(&drive->u.image);
	}
	else if(type == CMDDRVTYPE_ISO){
		ret = InitializeImage(&drive->u.image);
	}
	else if(type == CMDDRVTYPE_MKISOFS){
		ret = MIFInitialize(&drive->u.mkisofs);
	}
	if(ret != RET_OK){
		if(drive->own_alloc){
			MemFree(drive->data_buf);
			drive->data_buf=NULL;
		}
		MemFree(drive->cmdlog);
		drive->cmdlog=NULL;
		return ret;
	}

	return RET_OK;
}

/**
 * @brief 使用済みドライブの開放処理を行う
 * @param[in] drive ドライブ
 */
void FreeCmdDrive(CMDDRIVE *drive)
{
	CmdLogFree(drive);
	if(drive->data_buf!=NULL){
		if(drive->own_alloc){
			MemFree(drive->data_buf);
		}
		drive->data_buf=NULL;
	}

	if(drive->type == CMDDRVTYPE_DRIVE){
		FreeDrive(&drive->u.drive);
	}
	else if(drive->type == CMDDRVTYPE_NET){
		FreeSOCKCB(&drive->u.net);
	}
	else if(drive->type == CMDDRVTYPE_IMAGE){
		FreeImage(&drive->u.image);
	}
	else if(drive->type == CMDDRVTYPE_ISO){
		FreeImage(&drive->u.image);
	}
	else if(drive->type == CMDDRVTYPE_MKISOFS){
		MIFFree(&drive->u.mkisofs);
	}
}

/**
 * @brief ATAPIドライブかどうかを判断する
 * 判断結果はdriveの構造体に保持される。
 * @param[in] drive ドライブ
 */
void CheckAtapi(CMDDRIVE *drive)
{
	BYTE cdb[12];
	BYTE buf[12];
	BYTE *keep;
	int ret;

	keep = drive->data_buf;
	drive->data_buf = buf;

	memset(cdb, 0, sizeof(cdb));
	cdb[0] = CMD_MODE_SENSE10;
	cdb[2] = 0x2a;
	cdb[8] = 12;
	ret = SendCmd(drive, cdb, 12, REQ_DATAIN);
	if(ret!=RET_OK){
		drive->cmd_type = CMDDRVCTYPE_SCSI;
	}
	else{
		drive->cmd_type = (drive->data_buf[8]==0x2a ? CMDDRVCTYPE_ATAPI:CMDDRVCTYPE_SCSI);
	}

	drive->data_buf = keep;
}

/**
 * @brief ドライブにホストアダプタNoやターゲットID、タイムアウトを指定する
 * @param[in] drive ドライブ
 * @param[in] ha ホストアダプタNo
 * @param[in] tg ターゲットID
 * @param[in] timeout タイムアウト(秒)
 * @retval RET_OK 成功
 * @retval RET_NG 失敗
 */
int SetDriveAspiSetting(CMDDRIVE *drive, int *ha, int *tg, DWORD *timeout)
{
	drive->cmd_type = CMDDRVCTYPE_UNKNOWN;
	return SetAspiSetting(&drive->u.drive, ha, tg, timeout);
}

/**
 * @brief ドライブが使用する転送用バッファを設定する
 * @param[in] drive ドライブ
 * @param[in] data_buf データバッファ
 * @param[in] bufsize バッファサイズ
 */
void SetDriveBuffer(CMDDRIVE *drive, BYTE *data_buf, int bufsize)
{
	drive->data_buf = data_buf;
	drive->bufsize = bufsize;
}

/**
 * @brief 装置をオープンする
 * @param[out]	drive	装置構造体
 * @param[in]	id	装置ID
 * @param[in]	reader	読取装置かどうか(読取装置=TRUE)
 * @param[in]	bWaitDisc  ディスク準備待ち有無(有=TRUE)
 * @param[in]	data_buf   データバッファ
 * @param[in]	bufsize    データバッファサイズ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int OpenDevice(CMDDRIVE *drive, int hid, int tid,
			   BOOL reader, BOOL bWaitDisc,
			   BYTE *data_buf, int bufsize)
{
	int ret;
	DWORD timeout=5*60;

	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_DRIVE);
	if(ret!=RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	ret = SetDriveAspiSetting(drive, &hid, &tid, &timeout);
	if(ret!=RET_OK){
		FreeCmdDrive(drive);
			return ret;
	}

	ret = GetDiscType(drive, &drive->disc_type, bWaitDisc);
	if(ret!=RET_OK){
		if(ret!=RET_ABORT && bWaitDisc){
			UIDispMessage(MSG_CANT_GET_DISC_TYPE, UIDMT_ERROR);
		}
		FreeCmdDrive(drive);
		return ret;
	}
	ret = SendPreventAllow(drive, 1);
	if(ret!=RET_OK){
		DispCommandError(drive);
		FreeCmdDrive(drive);
		return ret;
	}

	UIDispInfo("");
	return RET_OK;
}

/**
 * @brief オープンされた装置をクローズする
 * @param[in]	drive	クローズする装置構造体
 */
void CloseDevice(CMDDRIVE *drive)
{
	if(drive->type == CMDDRVTYPE_DRIVE){
		SendPreventAllow(drive, 0);
	}

	FreeCmdDrive(drive);
}

/**
 * @brief 一時ファイル装置をオープン
 * @param[in]	drive	装置構造体
 * @param[in]	reader	読取装置かどうか(読取装置=TRUE)
 * @param[in]	data_buf   データバッファ
 * @param[in]	bufsize    データバッファサイズ
 * @param[in]	tmpfile    一時ファイルパス
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int OpenTempImageDevice(CMDDRIVE *drive, BOOL reader,
						BYTE *data_buf, int bufsize, const char *tmpfile)
{
	int ret;

	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_IMAGE);
	if(ret!=RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	ret = OpenImage(&drive->u.image, tmpfile, reader);
	if(ret!=RET_OK){
		FreeCmdDrive(drive);
			return ret;
	}
  
	return RET_OK;
}

/**
 * @brief mkisofs装置をオープン
 * @param[in] drive 装置構造体
 * @param[in] data_buf データバッファ
 * @param[in] bufsize データバッファサイズ
 * @retval RET_OK 正常終了
 * @retval RET_NG エラー
 */
int OpenMkisofsDevice(CMDDRIVE *drive, BYTE *data_buf, int bufsize)
{
#if defined(linux) || defined(WIN32)
	char folder[_MAX_PATH];
	int ret;

	ret = UIFolderDialog(folder, sizeof(folder));
	if(ret != UIDMRET_OK){
		return RET_ABORT;
	}
	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_MKISOFS);
	if(ret != RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	MIFSetSrcPath(&drive->u.mkisofs, folder);
	return RET_OK;
#else
	return RET_NG;
#endif
}

/**
 * @brief ISOイメージファイル装置をオープン
 * @param[in]	drive	装置構造体
 * @param[in]	reader	読取装置かどうか(読取装置=TRUE)
 * @param[in]	data_buf   データバッファ
 * @param[in]	bufsize    データバッファサイズ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int OpenISOImageDevice(CMDDRIVE *drive, BOOL reader,
					   BYTE *data_buf, int bufsize)
{
	char tmpfile[_MAX_PATH];
	int ret;

	if(reader){
		tmpfile[0] = '\0';
	}
	else{
		strcpy(tmpfile, "image.iso");
	}
  
	ret = UIFileDialog(reader, tmpfile, sizeof(tmpfile), "iso");
	if(ret!=UIDMRET_OK){
		return RET_ABORT;
	}
	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_ISO);
	if(ret!=RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	ret = OpenImage(&drive->u.image, tmpfile, reader);
	if(ret!=RET_OK){
		FreeCmdDrive(drive);
		return ret;
	}

	return RET_OK;
}



/**
 * @brief オリジナルイメージファイル装置をオープン
 * @param[in]	drive	装置構造体
 * @param[in]	reader	読取装置かどうか(読取装置=TRUE)
 * @param[in]	data_buf   データバッファ
 * @param[in]	bufsize    データバッファサイズ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int OpenEnbanImageDevice(CMDDRIVE *drive, BOOL reader,
						 BYTE *data_buf, int bufsize)
{
	char tmpfile[_MAX_PATH];
	int ret;

	if(reader){
		tmpfile[0] = '\0';
	}
	else{
		strcpy(tmpfile, "image.emg");
	}
  
	ret = UIFileDialog(reader, tmpfile, sizeof(tmpfile), "emg");
	if(ret!=UIDMRET_OK){
		return RET_ABORT;
	}
	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_IMAGE);
	if(ret!=RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	ret = OpenImage(&drive->u.image, tmpfile, reader);
	if(ret!=RET_OK){
		FreeCmdDrive(drive);
			return ret;
	}

	return RET_OK;
}


/**
 * @brief ネットワーク装置をオープン
 * @param[in]	drive	装置構造体
 * @param[in]	server	読取装置(server側)かどうか(読取装置=TRUE)
 * @param[in]	data_buf   データバッファ
 * @param[in]	bufsize    データバッファサイズ
 * @retval	RET_OK	正常終了
 * @retval	RET_NG	エラー
 */
int OpenNetDevice(CMDDRIVE *drive, BOOL server,
				  BYTE *data_buf, int bufsize)
{
	int ret;
	char remote_host[128];
	int port_number;

	ret = UINetDialog(server, remote_host, sizeof(remote_host), &port_number);
	if(ret!=UIDMRET_OK){
		return RET_ABORT;
	}
	ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_NET);
	if(ret!=RET_OK){
		return ret;
	}
	SetDriveBuffer(drive, data_buf, bufsize);
	if(server){
		/* サーバモード */
		drive->u.net.u.server.port = (WORD)port_number;
		ret = NACreateServer(&drive->u.net);
		if(ret!=RET_OK){
			DispSocketError(&drive->u.net, ret);
			return ret;
		}
		UIClearAbort();
		while(1){
			if(UICheckAbort()){
				ret = RET_ABORT;
				break;
			}
			ret = NAWaitConnect(&drive->u.net, 1000);
			if(ret==RET_OK){
				break;
			}
			if(ret!=RET_TIMEOUT){
				break;
			}
		}
		if(ret!=RET_OK){
			DispSocketError(&drive->u.net, ret);
			FreeCmdDrive(drive);
			return ret;
		}
	}
	else{
		/* クライアントモード */
		drive->u.net.remote_host = MemNew(strlen(remote_host)+1);
		strncpy(drive->u.net.remote_host, remote_host, strlen(remote_host)+1);
		drive->u.net.u.client.remote_port = (WORD)port_number;
		ret = NAConnect(&drive->u.net);
		if(ret!=RET_OK){
			DispSocketError(&drive->u.net, ret);
			FreeCmdDrive(drive);
			return ret;
		}

		ret = GetDiscType(drive, &drive->disc_type, TRUE);
		if(ret!=RET_OK){
			if(ret!=RET_ABORT){
				UIDispMessage(MSG_CANT_GET_DISC_TYPE
							  /*"ディスクタイプ取得に失敗しました。"*/, UIDMT_ERROR);
			}
			FreeCmdDrive(drive);
			return ret;
		}
		ret = SendPreventAllow(drive, 1);
		if(ret!=RET_OK){
			DispCommandError(drive);
			FreeCmdDrive(drive);
			return ret;
		}
	}
  
	return RET_OK;
}
