/**
 * @file ebstring.c
 * @brief 文字列操作
 * @author BananaJinn
 * @version $Id: ebstring.c,v 1.4 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋 - EnbanFukusyaYa
 * Copyright (C) 2005-2006 Kagetani Hideto
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include "ebstring.h"

/**
 * @brief 文字列を複製する
 * @param[in] string 文字列
 * @return 複製した文字列を返す。
 */
char *EbStringNew(const char *string)
{
	return strdup(string);
}

/**
 * @brief 書式付きで文字列を作成する
 * @param[in] format 書式
 * @param[in] ... 書式に対するパラメータ
 * @return 作成した文字列を返す。
 */
char *EbStringNewWithFormat(const char *format, ...)
{
	va_list args;
	int bufsize=80;
	char *ret_string=NULL, *new_ptr;
	int ret;

	while(1){
		new_ptr = realloc(ret_string, bufsize);
		if(new_ptr == NULL){
			if(ret_string != NULL){
				free(ret_string);
			}
			return NULL;
		}
		ret_string = new_ptr;
		va_start(args, format);
#if defined(WIN32)
		ret = _vsnprintf(ret_string, bufsize, format, args);
#else
		ret = vsnprintf(ret_string, bufsize, format, args);
#endif
		va_end(args);
		if((ret > -1) && (ret < bufsize)){
			break;
		}
		if(ret > -1){
			bufsize = ret+1;
		}
		else{
			bufsize *= 2;
		}
	}
	return ret_string;
}

/**
 * @brief 文字列を結合する
 * @param[in] ebstring 結合される側の文字列
 * @param[in] append_string 結合する文字列
 * @return 結合された文字列を返す。
 */
char *EbStringAppend(char *ebstring, const char *append_string)
{
	char *ret;
	int len;

	if(ebstring == NULL){
		return EbStringNew(append_string);
	}
	len = strlen(ebstring);
	ret = (char *)realloc(ebstring, len+strlen(append_string)+1);
	if(ret == NULL){
		return ebstring;
	}

	strcpy(ret+len, append_string);
	return ret;
}

/**
 * @brief 書式付きで文字列を結合する
 * @param[in] ebstring 結合される側の文字列
 * @param[in] format 書式
 * @param[in] ... 書式に対するパラメータ
 * @return 結合された文字列を返す。
 */
char *EbStringAppendWithFormat(char *ebstring, const char *format, ...)
{
	va_list args;
	int offset=0;
	int bufsize=80;
	char *ret_string=ebstring, *new_ptr;
	int ret;

	if(ebstring != NULL){
		offset=strlen(ebstring);
		bufsize=strlen(ebstring)+80;
	}
	while(1){
		new_ptr = realloc(ret_string, bufsize);
		if(new_ptr == NULL){
			if(ret_string != NULL){
				free(ret_string);
			}
			return NULL;
		}
		ret_string = new_ptr;
		va_start(args, format);
#if defined(WIN32)
		ret = _vsnprintf(ret_string+offset, bufsize-offset, format, args);
#else
		ret = vsnprintf(ret_string+offset, bufsize-offset, format, args);
#endif
		va_end(args);
		if((ret > -1) && (ret < bufsize)){
			break;
		}
		if(ret > -1){
			bufsize = ret+1;
		}
		else{
			bufsize *= 2;
		}
	}
	return ret_string;
}

/**
 * @brief パス区切りを付けて文字列を結合する
 * @param[in] ebstring 結合される側の文字列
 * @param[in] path 結合する文字列
 * @return 結合された文字列を返す。
 */
char *EbStringAppendPath(char *ebstring, const char *path)
{
	int len;
	
	if(ebstring == NULL){
		return EbStringNew(path);
	}
	len = strlen(ebstring);
	if(len > 0){
		if(ebstring[len-1] == PATH_SEPARATOR){
			ebstring[len-1] = '\0';
		}
	}
	return EbStringAppendWithFormat(ebstring, "%c%s", PATH_SEPARATOR, path);
}
	
/**
 * @brief 文字列のメモリを解放する
 * @param[in] ebstring 解放する文字列
 * @return NULLを返す。
 */
char *EbStringFree(char *ebstring)
{
	free(ebstring);
	return NULL;
}

/**
 * @brief 文字列の先頭から切り出す
 * @param[in] string 切り出される文字列
 * @param[in] len 切り出すバイト数
 * @param[in] trim 末尾をトリミングする文字('\0'で省略)
 * @return 切り出した文字列を返す。
 */
char *EbStringGetString(const char *string, int len, char trim)
{
	char *ret_string = (char *)malloc(len+1);
	int i;

	strncpy(ret_string, string, len);
	ret_string[len] = '\0';

	if(trim != '\0'){
		for(i=len-1; i>=0; i--){
			if(ret_string[i] != trim){
				break;
			}
			ret_string[i] = '\0';
		}
	}

	return ret_string;
}


