/**
 * @file image.c
 * @brief イメージファイル
 * @author BananaJinn
 * @version $Id: image.c,v 1.14 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <string.h>
#include <stdio.h>
#if !defined(WIN32)
# include <stdlib.h>
#endif
#include "mem.h"
#include "image.h"
#include "ui.h"
#include "text.h"
#include "ebstring.h"

static int OpenImageFile(IMAGEFILE *image, BOOL reader);
static int OpenNextImageFile(IMAGEFILE *image, BOOL reader);
static int CloseImageFile(IMAGEFILE *image);

#define IMGFILEMAXSIZE	0x40000000	/* 1giga-bytes */


static int OpenImageFile(IMAGEFILE *image, BOOL reader)
{
	char *filename = NULL;

	CloseImageFile(image);
	if(image->fileindex>=10 || image->fileindex<0){
		UIDispMessage(MSG_CANT_OPEN_IMAGE
				/*"イメージファイルのオープンに失敗しました。"*/, UIDMT_ERROR);
		return RET_NG;
	}
	filename = EbStringNew(image->filename);
	if(image->fileindex>0){
		filename = EbStringAppendWithFormat(filename, ".%d",
											image->fileindex);
	}
#if defined(WIN32)
	image->fp = fopen(filename, reader ? "rb" : "wb");
#else
	image->fp = fopen(filename, reader ? "r" : "w");
#endif
	if(image->fp==NULL){
		UIDispMessage(MSG_CANT_OPEN_IMAGE
			/*"イメージファイルのオープンに失敗しました。"*/, UIDMT_ERROR);
		EbStringFree(filename);
		return RET_NG;
	}
	EbStringFree(filename);
	image->bytecount = 0;
	
	return RET_OK;
}

static int OpenNextImageFile(IMAGEFILE *image, BOOL reader)
{
	image->fileindex++;
	return OpenImageFile(image, reader);
}

int ReadImageFile(IMAGEFILE *image, void *buf, DWORD size, BOOL split)
{
	int ret;
	BYTE *bufp = (BYTE *)buf;
	size_t readsize;

	if(split){
	  if(size > IMGFILEMAXSIZE-image->bytecount){
		fread(bufp, 1, (IMGFILEMAXSIZE-image->bytecount), image->fp);
		bufp += (IMGFILEMAXSIZE-image->bytecount);
		size -= (IMGFILEMAXSIZE-image->bytecount);
		ret = OpenNextImageFile(image, TRUE);
		if(ret!=RET_OK)
		  return ret;
	  }
	}
	readsize = fread(bufp, 1, size, image->fp);
	if(readsize < size){
	  UIDispMessage(MSG_CANT_READ_IMAGE
			/*"イメージファイルの読込に失敗しました。"*/, UIDMT_ERROR);
	  return RET_NG;
	}
	image->bytecount += size;

	return RET_OK;
}

int WriteImageFile(IMAGEFILE *image, void *buf, DWORD size, BOOL split)
{
	int ret;
	BYTE *bufp = (BYTE *)buf;
	size_t writesize;

	if(split){
	  if(size > IMGFILEMAXSIZE-image->bytecount){
		fwrite(bufp, 1, (IMGFILEMAXSIZE-image->bytecount), image->fp);
		bufp += (IMGFILEMAXSIZE-image->bytecount);
		size -= (IMGFILEMAXSIZE-image->bytecount);
		ret = OpenNextImageFile(image, FALSE);
		if(ret!=RET_OK)
		  return ret;
	  }
	}
	writesize = fwrite(bufp, 1, size, image->fp);
	if(writesize < size){
	  UIDispMessage(MSG_CANT_WRITE_IMAGE
			/*"イメージファイルの書込に失敗しました。"*/, UIDMT_ERROR);
	  return RET_NG;
	}
	
	image->bytecount += size;

	return RET_OK;
}


void RemoveImageFile(const char *basefilename)
{
	char *filename;
	int i;

	remove(basefilename);
	for(i=1; i<=9; i++){
		filename = EbStringNewWithFormat("%s.%d", basefilename, i);
		remove(filename);
		filename = EbStringFree(filename);
	}
	MemFree(filename);
}


static int CloseImageFile(IMAGEFILE *image)
{
  if(image->fp!=NULL){
    fclose(image->fp);
    image->fp = NULL;
    image->bytecount = 0;
  }
  return RET_OK;
}


int InitializeImage(IMAGEFILE *image)
{
  memset(image, 0, sizeof(IMAGEFILE));
  return RET_OK;
}

void FreeImage(IMAGEFILE *image)
{
  CloseImageFile(image);
  if(image->filename != NULL){
    MemFree(image->filename);
    image->filename = NULL;
  }
  image->fileindex = 0;
  image->bytecount = 0;
}

int OpenImage(IMAGEFILE *image, const char *filename, BOOL reader)
{
  int ret;

  if(filename == NULL){
    return RET_NG;
  }
  image->filename = MemNew(strlen(filename)+1);
  strncpy(image->filename, filename, strlen(filename)+1);
  image->fileindex = 0;
  image->fp = NULL;
  ret = OpenImageFile(image, reader);
  if(ret != RET_OK){
    return ret;
  }

  return RET_OK;
}

