/**
 * @file mem.c
 * @brief メモリ関係
 * @author BananaJinn
 * @version $Id: mem.c,v 1.4 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2007 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#if !defined(MACOSX)
# include <malloc.h>
#endif
#include "mem.h"

#if defined(DEBUGMEM)
struct _MEMDEBUG {
	void *ptr;
	size_t size;
	char fname[16];
	long line;
} *g_MemArray = NULL;
int g_NumMemArray = 0;
int g_MemDebug = 0;

# if defined(WIN32)
#  include <windows.h>
#  define PRINT_MEMDEBUG Trace
# else
#  define PRINT_MEMDEBUG printf
# endif
# define MEMDUMP_MAX 48
#endif

#if defined(DEBUGMEM) && defined(WIN32)
static void Trace(const char *format, ...)
{
	va_list args;
	char buf[512];

	va_start(args, format);
	_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);
	OutputDebugString(buf);
}
#endif /* DEBUGMEM */


#if defined(DEBUGMEM)
static void MemDebugStore(struct _MEMDEBUG *md,
						  void *ptr, size_t size, const char *fname, long line)
{
	md->ptr = ptr;
	md->size = size;
	strncpy(md->fname, fname, sizeof(md->fname));
	md->fname[sizeof(md->fname)-1] = '\0';
	md->line = line;
}
#endif /* DEBUGMEM */


#if defined(DEBUGMEM)
static void MemDebug(const char *fname, long line,
					 void *old_ptr, void *new_ptr, size_t new_size)
{
	int index;
	char *wp;

	if(!g_MemDebug){
		return;
	}

#if defined(WIN32)
	wp = strrchr(fname, '\\');
#else
	wp = strrchr(fname, '/');
#endif
	if(wp != NULL){
		fname = wp+1;
	}

	if(old_ptr == NULL){
#if defined(DUMP_MEMDEBUG)
		PRINT_MEMDEBUG("Add %p(%d bytes) by %s(%ld)\n", new_ptr, new_size,
					   fname, line);
#endif
		for(index=0; index<g_NumMemArray; index++){
			if(g_MemArray[index].ptr == NULL){
				MemDebugStore(&g_MemArray[index],
							  new_ptr, new_size, fname, line);
				return;
			}
		}
		g_MemArray = realloc(g_MemArray,
							 sizeof(struct _MEMDEBUG)*(g_NumMemArray+1));
		if(g_MemArray != NULL){
			MemDebugStore(&g_MemArray[g_NumMemArray],
						  new_ptr, new_size, fname, line);
			g_NumMemArray++;
		}
		return;
	}

    for(index=0; index<g_NumMemArray; index++){
		if(g_MemArray[index].ptr == old_ptr){
			if(new_ptr==NULL){
#if defined(DUMP_MEMDEBUG)
				PRINT_MEMDEBUG("Delete %p(%d bytes) by %s(%ld)\n",
							   old_ptr, g_MemArray[index].size,
							   fname, line);
#endif
				MemDebugStore(&g_MemArray[index], NULL, 0, "", 0);
				return;
			}
			else{
#if defined(DUMP_MEMDEBUG)
				PRINT_MEMDEBUG("Resize %p(%d bytes) => %p(%d bytes) by %s(%ld)\n",
							   old_ptr, g_MemArray[index].size,
							   new_ptr, new_size,
							   fname, line);
#endif
				MemDebugStore(&g_MemArray[index], new_ptr, new_size,
							  fname, line);
				return;
			}
		}
    }
    PRINT_MEMDEBUG("ERROR! %p is invalid address. %s(%ld)\n",
				   old_ptr, fname, line);
    return;
}
#endif /* DEBUGMEM */

#if defined(DEBUGMEM)
static void MemDump(void *ptr, size_t size)
{
	size_t index, index16;
	char ascii[16+1];
	unsigned char *bytep = (unsigned char *)ptr;

	for(index=0; index<size && index<MEMDUMP_MAX; index++){
		index16 = index & 15;
		if((bytep[index] >= 0x20) && (bytep[index] < 0x7f)){
			ascii[index16] = bytep[index];
		}
		else{
			ascii[index16] = '.';
		}
		if(index16 == 0){
			PRINT_MEMDEBUG("0x%08lX ", bytep+index);
		}
		else if(index16 == 8){
			PRINT_MEMDEBUG(" ");
		}
		PRINT_MEMDEBUG("%02X ", bytep[index]);
		if(index16 == 15){
			ascii[16] = '\0';
			PRINT_MEMDEBUG("%s\n", ascii);
		}
	}

	index16 = index & 15;
	if(index16 != 0){
		ascii[index16] = '\0';
		PRINT_MEMDEBUG("%*s %s\n",
					   3*index16+((index16 >= 8) ? 1:0),
					   "", ascii);
	}
}
#endif /* DEBUGMEM */


/**
 * @brief メモリデバッグ開始
 * @note DEBUGMEMがデファインされていない場合は何もしない
 */
void MemDebugStart()
{
#if defined(DEBUGMEM)
	if(g_MemDebug){
		MemDebugEnd();
	}
	g_MemDebug = 1;
#endif /* DEBUGMEM */
}


/**
 * @brief メモリデバッグ終了
 * @note DEBUGMEMがデファインされていない場合は何もしない
 */
void MemDebugEnd()
{
#if defined(DEBUGMEM)
	if(g_MemArray){
		free(g_MemArray);
		g_MemArray = NULL;
	}
	g_NumMemArray = 0;
	g_MemDebug = 0;
#endif /* DEBUGMEM */
}

/**
 * @brief メモリリークを表示
 * @note DEBUGMEMがデファインされていない場合は何もしない
 */
void MemDumpLeaks()
{
#if defined(DEBUGMEM)
	int index;
	int leakcnt=0;

	for(index=0; index<g_NumMemArray; index++){
		if(g_MemArray[index].ptr != NULL){
			PRINT_MEMDEBUG("Detect memory leaks! %p(%d bytes) by %s(%ld)\n",
						   g_MemArray[index].ptr, g_MemArray[index].size,
						   g_MemArray[index].fname, g_MemArray[index].line);
			MemDump(g_MemArray[index].ptr, g_MemArray[index].size);
			leakcnt++;
		}
	}
	if(leakcnt == 0){
		PRINT_MEMDEBUG("No memory leaks!\n");
	}
#endif /* DEBUGMEM */
}


#if defined(DEBUGMEM)
/**
 * @brief メモリ確保
 * @param[in] fname 呼び出し元ソースファイル名
 * @param[in] line 呼び出し元ソースファイル行番号
 * @param[in] size メモリサイズ(バイト数)
 * @return 確保したメモリの先頭アドレスを返す。エラーの場合はNULLを返す。
 */
void *MemNewF(const char *fname, long line, size_t size)
{
	void *ptr = malloc(size);
	MemDebug(fname, line, NULL, ptr, size);
	return ptr;
}
#endif /* DEBUGMEM */

#if defined(DEBUGMEM)
/**
 * @brief メモリサイズ拡大
 * @param[in] fname 呼び出し元ソースファイル名
 * @param[in] line 呼び出し元ソースファイル行番号
 * @param[in] ptr 拡大するメモリの先頭アドレス
 * @param[in] size 新しいメモリサイズ(バイト数)
 * @return 拡大されたメモリの先頭アドレスを返す。エラーの場合はNULLを返す。
 */
void *MemResizeF(const char *fname, long line, void *ptr, size_t new_size)
{
	void *new_ptr;

	new_ptr = realloc(ptr, new_size);
	MemDebug(fname, line, ptr, new_ptr, new_size);

	return new_ptr;
}
#endif /* DEBUGMEM */


#if defined(DEBUGMEM)
/**
 * @brief 確保されたメモリの解放
 * @param[in] fname 呼び出し元ソースファイル名
 * @param[in] line 呼び出し元ソースファイル行番号
 * @param[in] ptr 解放するメモリの先頭アドレス
 * @return NULLを返す。
 */
void *MemFreeF(const char *fname, long line, void *ptr)
{
	MemDebug(fname, line, ptr, NULL, 0);
	free(ptr);

	return NULL;
}
#endif /* DEBUGMEM */

