/**
 * @file netaccess.c
 * @brief ネットワークアクセス
 * @author BananaJinn
 * @version $Id: netaccess.c,v 1.23 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#if defined(__APPLE__) && defined(__MACH__)
# include <sys/time.h>
#endif
#if defined(linux) || defined(MACOSX)
# include <errno.h>
# include <stdlib.h>
# include <string.h>
# include <unistd.h>
# include <netinet/in.h>
# include <arpa/inet.h>
# include <netdb.h>
# define INVALID_SOCKET (-1)
# define SOCKET_ERROR (-1)
# define SD_SEND SHUT_WR
# define SD_RECEIVE SHUT_RD
# define SD_BOTH SHUT_RDWR
# define closesocket close
#endif
#if defined(__MINGW32__)
# include <errno.h>
#endif

#include "mem.h"
#include "cmd.h"
#include "netaccess.h"
#include "ui.h"
#include "text.h"
#include "ebstring.h"

static void SetErrorCode(SOCKCB *scb)
{
#if defined(WIN32)
	scb->error_code = WSAGetLastError();
#else
	scb->error_code = errno;
#endif
}

/**
 * @brief	ソケット待ち
 * @param[out]	read_fds	読み込み用
 * @param[out]	write_fds	書き込み用
 * @param[out]	except_fds	例外用
 * @param[in]	sd			ソケットデスクリプタ
 * @param[in]	timeout		タイムアウト(ミリ秒)
 * @retval	RET_OK	ソケット入出力有り
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_NG	エラー
 */
static int WaitFDS(fd_set *read_fds, fd_set *write_fds, fd_set *except_fds,
				   SOCKET sd, DWORD timeout)
{
	struct timeval timeout_val;
	int ret;
  
	if(read_fds != NULL){
		FD_ZERO(read_fds);
	}
	if(write_fds != NULL){
		FD_ZERO(write_fds);
	}
	if(except_fds != NULL){
		FD_ZERO(except_fds);
	}

	if(read_fds != NULL){
		FD_SET(sd, read_fds);
	}
	if(write_fds != NULL){
		FD_SET(sd, write_fds);
	}
	if(except_fds != NULL){
		FD_SET(sd, except_fds);
	}

	timeout_val.tv_sec = timeout/1000;
	timeout_val.tv_usec = (timeout%1000)*1000;
	ret = select(sd+1, read_fds, write_fds, except_fds, &timeout_val);
#if defined(WIN32)
	if(ret == SOCKET_ERROR){
		return RET_NG;
	}
#else
	if(ret < 0){
		return RET_NG;
	}
#endif
	if(ret == 0){
		return RET_TIMEOUT;
	}
	return RET_OK;
}

  
/**
 * @brief ソケット制御ブロック初期化
 * @param[out] scb 初期化するソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 */
int InitializeSOCKCB(SOCKCB *scb)
{
#if defined(WIN32)
	WSADATA wsadata;
#endif

	memset(scb, 0, sizeof(SOCKCB));

#if defined(WIN32)
	if(WSAStartup(MAKEWORD(1,1), &wsadata) != 0){
		SetErrorCode(scb);
		return RET_SOCKET;
	}
#endif
  
	return RET_OK;
}

/**
 * @brief ソケット制御ブロック解放
 * @param[in] scb 解放するソケット制御ブロック
 * @retval RET_OK 正常終了
 */
int FreeSOCKCB(SOCKCB *scb)
{
	NAClose(scb);
	if(scb->remote_host != NULL){
		MemFree(scb->remote_host);
		scb->remote_host = NULL;
	}
#if defined(linux) || defined(__MINGW32__)
	if(scb->iconv_desc != NULL){
		iconv_close(scb->iconv_desc);
	}
#endif
	memset(scb, 0, sizeof(SOCKCB));

#if defined(WIN32)
	WSACleanup();
#endif
  
	return RET_OK;
}


/**
 * @brief サーバソケット作成
 * @param[in,out] scb ソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 */
int NACreateServer(SOCKCB *scb)
{
	int ret;
	struct sockaddr_in sin;

	scb->server_mode = TRUE;

	/* Setup listener */
	scb->u.server.listen_socket = socket(AF_INET, SOCK_STREAM, 0);
	if(scb->u.server.listen_socket == INVALID_SOCKET){
		SetErrorCode(scb);
		return RET_SOCKET;
	}
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(scb->u.server.port);
	ret = bind(scb->u.server.listen_socket,
			   (struct sockaddr *)&sin,
			   sizeof(struct sockaddr_in));
	if(ret == SOCKET_ERROR){
		SetErrorCode(scb);
		return RET_SOCKET;
	}

	listen(scb->u.server.listen_socket, 1);

#if defined(__MINGW32__)
	/* SJIS => UTF-8 */
	scb->iconv_desc = iconv_open("UTF-8", "SJIS");
#elif defined(linux)
	/* SJIS => local charset */
	scb->iconv_desc = iconv_open("", "SJIS");
#endif
  
	return RET_OK;
}

/**
 * @brief ソケット接続要求待ち
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] timeout タイムアウト時間(ミリ秒)
 * @retval RET_OK 正常終了
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_NG その他のエラー
 */
int NAWaitConnect(SOCKCB *scb, DWORD timeout)
{
	struct sockaddr_in sin_remote;
#if defined(linux)
	socklen_t addr_size;
#else
	int addr_size;
#endif
	fd_set read_fds;
	int ret;
	char *remote_host;

	if(scb->server_mode == FALSE){
		scb->error_code = 0;
		return RET_NG;
	}

	ret = WaitFDS(&read_fds, NULL, NULL, scb->u.server.listen_socket, timeout);
	if(ret != RET_OK){
		if(ret==RET_TIMEOUT){
			return ret;
		}
		SetErrorCode(scb);
		return RET_SOCKET;
	}

	if(!FD_ISSET(scb->u.server.listen_socket, &read_fds)){
		SetErrorCode(scb);
		return RET_SOCKET;
	}
	
	/* Accept connection */
	addr_size = sizeof(sin_remote);
	scb->u.server.socket_desc = accept(scb->u.server.listen_socket,
									   (struct sockaddr *)&sin_remote,
									   &addr_size);
	if(scb->u.server.socket_desc == INVALID_SOCKET){
		SetErrorCode(scb);
		return RET_SOCKET;
	}

	remote_host = inet_ntoa(sin_remote.sin_addr);
	scb->remote_host = MemNew(strlen(remote_host)+1);
	strncpy(scb->remote_host, remote_host, strlen(remote_host)+1);

	return RET_OK;
}


/**
 * @brief ソケット接続
 * @param[in,out] scb ソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_NG その他のエラー
 */
int NAConnect(SOCKCB *scb)
{
	DWORD remaddr;
	struct hostent *hent;
	struct sockaddr_in sin_remote;

	if(scb->remote_host == NULL){
		return RET_NG;
	}
  
	/* lookup remote address */
	remaddr = inet_addr(scb->remote_host);
	if(remaddr == INADDR_NONE){
		hent = gethostbyname(scb->remote_host);
		if(hent == NULL){
			SetErrorCode(scb);
			return RET_SOCKET;
		}
		remaddr = *((DWORD*)hent->h_addr_list[0]);
	}
	scb->u.client.remote_address = remaddr;

	scb->u.client.socket_desc = socket(AF_INET, SOCK_STREAM, 0);
	if(scb->u.client.socket_desc == INVALID_SOCKET){
		SetErrorCode(scb);
		return RET_SOCKET;
	}

	memset(&sin_remote, 0, sizeof(sin_remote));
	sin_remote.sin_family = AF_INET;
	sin_remote.sin_addr.s_addr = scb->u.client.remote_address;
	sin_remote.sin_port = htons(scb->u.client.remote_port);
	if(connect(scb->u.client.socket_desc,
			   (struct sockaddr*)&sin_remote,
			   sizeof(sin_remote)) == SOCKET_ERROR){
		SetErrorCode(scb);
		shutdown(scb->u.client.socket_desc, SD_BOTH);
		closesocket(scb->u.client.socket_desc);
		scb->u.client.socket_desc = 0;
		return RET_SOCKET;
	}

#if defined(__MINGW32__)
	/* UTF-8 => SJIS */
	scb->iconv_desc = iconv_open("SJIS", "UTF-8");
#elif defined(linux)
	/* local charset => SJIS */
	scb->iconv_desc = iconv_open("SJIS", "");
#endif
  
	return RET_OK;
}


/**
 * @brief ソケットクローズ
 * @param[in,out] scb ソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 */
int NAClose(SOCKCB *scb)
{
	SOCKET sd;
	int read_bytes;
	char buf[256];
	fd_set read_fds;
	int ret;

	if(scb->server_mode){
		sd = scb->u.server.socket_desc;
	}
	else{
		sd = scb->u.client.socket_desc;
	}

	if(sd > 0){
		shutdown(sd, SD_SEND);

		while(1){
			ret = WaitFDS(&read_fds, NULL, NULL, sd, 30*1000);
			if(ret == RET_TIMEOUT){
				continue;
			}
			else if(ret != RET_OK){
				break;
			}
			read_bytes = recv(sd, buf, sizeof(buf), 0);
			if(read_bytes == SOCKET_ERROR){
				break;
			}
			else if(read_bytes == 0){
				break;
			}
		}

		shutdown(sd, SD_RECEIVE);

		if(closesocket(sd) == SOCKET_ERROR){
			SetErrorCode(scb);
			return RET_SOCKET;
		}
	}

	if(scb->server_mode){
		closesocket(scb->u.server.listen_socket);
	}

	return RET_OK;
}


/**
 * @brief ソケット送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] buffer 送信データバッファ
 * @param[in] length 送信データバイト数
 * @param[in] tmieout タイムアウト(ミリ秒)
 * @param[in] check_readfd READ用fdを監視するかどうか
 * @retval RET_OK 正常終了
 * @retval RET_READFD check_readfd指定時にREAD用のfdが読み取り可能となった
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 接続断念
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NASend(SOCKCB *scb, const void *buffer, int length, DWORD timeout,
		   BOOL check_readfd)
{
	fd_set read_fds;
	fd_set write_fds;
	int ret;
	SOCKET sd;
	int write_bytes;
	const char *bufp = (const char *)buffer;

	if(scb->server_mode){
		sd = scb->u.server.socket_desc;
	}
	else{
		sd = scb->u.client.socket_desc;
	}

	while(length > 0){
		ret = WaitFDS(check_readfd ? &read_fds : NULL,
					  &write_fds, NULL, sd, timeout);
		if(ret != RET_OK){
			if(ret==RET_TIMEOUT){
				return ret;
			}
			SetErrorCode(scb);
			return RET_SOCKET;
		}

		if(check_readfd){
			if(FD_ISSET(sd, &read_fds)){
				return RET_READFD;
			}
		}
		if(FD_ISSET(sd, &write_fds)){
			write_bytes = send(sd, bufp, length, 0);
			if(write_bytes == SOCKET_ERROR){
				SetErrorCode(scb);
				return RET_SOCKET;
			}
			else if(write_bytes == 0){
				/* connection closed */
				return RET_ABORT;
			}
			length -= write_bytes;
			bufp += write_bytes;
		}
	}

	return RET_OK;
}

/**
 * @brief	ソケットからデータ受信
 * @param[in,out]	scb		ソケット制御ブロック
 * @param[out]	buffer	受信データ格納領域(NULL指定でデータを捨てる)
 * @param[in]	length	受信データ長
 * @param[in]	timeout	タイムアウト(ミリ秒)
 * @retval	RET_OK	正常終了
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_ABORT	中断(相手側より切断された)
 * @retval	RET_SOCKET	ソケットエラー
 */
int NAReceive(SOCKCB *scb, void *buffer, int length, DWORD timeout)
{
	fd_set read_fds;
	int ret;
	SOCKET sd;
	int read_bytes;
	char dummy_buffer[64];
	int len;
	char *bufp = (char *)buffer;

	if(scb->server_mode){
		sd = scb->u.server.socket_desc;
	}
	else{
		sd = scb->u.client.socket_desc;
	}

	while(length > 0){
		ret = WaitFDS(&read_fds, NULL, NULL, sd, timeout);
		if(ret != RET_OK){
			if(ret==RET_TIMEOUT){
				return ret;
			}
			SetErrorCode(scb);
			return RET_SOCKET;
		}

		if(FD_ISSET(sd, &read_fds)){
			if(bufp==NULL){
				len = (length < sizeof(dummy_buffer))
					? length : sizeof(dummy_buffer);
				read_bytes = recv(sd, dummy_buffer, len, 0);
			}
			else{
				read_bytes = recv(sd, bufp, length, 0);
			}
			if(read_bytes == SOCKET_ERROR){
				SetErrorCode(scb);
				return RET_SOCKET;
			}
			else if(read_bytes == 0){
				/* connection closed */
				return RET_ABORT;
			}
			length -= read_bytes;
			bufp += read_bytes;
		}
	}

	return RET_OK;
}


/**
 * @brief コマンド応答を受信
 * @param[in,out] scb ソケット制御ブロック
 * @param[out] buf 応答データ格納バッファ
 * @param[in] buflen バッファサイズ(バイト数)
 * @param[out] sensedata センスデータ格納バッファ
 * @param[in] senselen センスデータ格納バッファサイズ(バイト数)
 * @param[in] reqflag 要求フラグ(REQ_NODATA/REQ_DATAIN/REQ_DATAOUT)
 * @param[out] cmd_header コマンドヘッダー
 * @retval RET_OK 正常終了
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_NG その他のエラー
 */
static int ReceiveCmdResponse(SOCKCB *scb, void *buf, DWORD buflen,
							  BYTE *sensedata, DWORD senselen,
							  int reqflag, NETCMDHEADER *cmd_header)
{
	BYTE netdata_common[8];
	NETCMDHEADER recv_header;
	int ret;

	memset(sensedata, 0, senselen);

	if(cmd_header == NULL){
		cmd_header = &recv_header;
	}
  
	/* 共通部(データ長とデータタイプ)を受信する */
	ret = NAReceive(scb, (char *)netdata_common, 8, NETTIMEOUT);
	if(ret != RET_OK){
		return ret;
	}
  
	if(netdata_common[4]!=NETDATATYPE_RESPONSE){
		return RET_ABORT;
	}

	/* receive command result */
	ret = NAReceive(scb, (char *)cmd_header, sizeof(NETCMDHEADER), NETTIMEOUT);
	if(ret != RET_OK){
		return ret;
	}
	/* check command result */
	if((cmd_header->sense_data[2]&0x0f) != 0x00 ||
	   cmd_header->sense_data[12] != 0x00 ||
	   cmd_header->sense_data[13] != 0x00){
		memcpy(sensedata, cmd_header->sense_data, senselen);
		return RET_NG;
	}
  
	if(reqflag==REQ_DATAIN){
		/* receive_data */
		ret = NAReceive(scb, buf, buflen, NETTIMEOUT);
		if(ret != RET_OK){
			return ret;
		}
	}

	return RET_OK;
}


/**
 * @brief コマンドデータ送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] cdb コマンド列(12バイト)
 * @param[in,out] buf 送信コマンドに付随するデータ / 受信データ
 * @param[in] buflen バッファサイズ(バイト数)
 * @param[out] sensedata センスデータ格納バッファ
 * @param[in] senselen センスデータ格納バッファサイズ(バイト数)
 * @param[in] reqflag 要求フラグ(REQ_NODATA/REQ_DATAIN/REQ_DATAOUT)
 * @retval RET_OK 正常終了
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_NG その他のエラー
 */
int SendNetCmd(SOCKCB *scb, BYTE *cdb, void *buf, DWORD buflen,
			   BYTE *sensedata, DWORD senselen, int reqflag)
{
	BYTE netdata_common[8];
	NETCMDHEADER header;
	int ret;
	DWORD length;

	length = sizeof(NETCMDHEADER);
	if(reqflag==REQ_DATAOUT){
		length += buflen;
	}
	Set4bytes(netdata_common, length);
	netdata_common[4] = NETDATATYPE_COMMAND;
	header.reqflag = reqflag;
	memcpy(header.cdb, cdb, 12);
	Set4bytes(header.data_length, buflen);

	/* 共通部(データ長とデータタイプ)を送る */
	ret = NASend(scb, netdata_common, 8, NETTIMEOUT, FALSE);
	if(ret != RET_OK){
		return ret;
	}

	/* 個別部を送る */
	ret = NASend(scb, &header, sizeof(NETCMDHEADER), NETTIMEOUT, FALSE);
	if(ret != RET_OK){
		return ret;
	}
  
	if(reqflag==REQ_DATAOUT){
		/* send data */
		ret = NASend(scb, buf, buflen, NETTIMEOUT, FALSE);
		if(ret != RET_OK){
			return ret;
		}
	}

	ret = ReceiveCmdResponse(scb, buf, buflen, sensedata, senselen, reqflag,
							 NULL);
	if(ret != RET_OK){
		return ret;
	}

	return RET_OK;
}

/**
 * @brief Network経由でLONGREADを送信する。または、LONGREAD結果を受信する。
 *        total_blocksが0の場合に結果受信を行い、それ以外の場合は、送信を行う。
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] cdb コマンド列(12バイト)
 * @param[out] buf 受信バッファ
 * @param[in] buflen 受信バッファサイズ(バイト数)
 * @param[out] sensedata センスデータ格納バッファ
 * @param[in] senselen センスデータ格納バッファサイズ(バイト数)
 * @param[in] reqflag 要求フラグ(REQ_NODATA/REQ_DATAIN/REQ_DATAOUT)
 * @param[in] total_blocks READブロック数
 * @retval RET_OK 正常終了
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_NG その他のエラー
 */
int SendNetCmdLongRead(SOCKCB *scb, BYTE *cdb, void *buf, DWORD buflen,
					   BYTE *sensedata, DWORD senselen,
					   int reqflag, DWORD total_blocks)
{
	BYTE netdata_common[8];
	NETCMDHEADER header, recv_header;
	BYTE total_blocks_data[4];
	int ret;
	DWORD length;

	if(total_blocks == 0){
		/* LONGREAD結果受信 */
		ret = ReceiveCmdResponse(scb, buf, buflen, sensedata, senselen, reqflag,
								 &recv_header);
		if(ret != RET_OK){
			return ret;
		}
	
		if(memcmp(recv_header.cdb, cdb, 12)){
			/* コマンド相違 */
			return RET_NG;
		}
	}
	else{
		/* LONGREAD送信 */
		length = sizeof(NETCMDHEADER);
		if(reqflag==REQ_DATAOUT){
			length += buflen;
		}
		Set4bytes(netdata_common, length);
		netdata_common[4] = NETDATATYPE_LONGREAD;
		header.reqflag = reqflag;
		memcpy(header.cdb, cdb, 12);
		Set4bytes(header.data_length, buflen);

		/* 共通部(データ長とデータタイプ)を送る */
		ret = NASend(scb, (char *)netdata_common, 8, NETTIMEOUT, FALSE);
		if(ret != RET_OK){
			return ret;
		}

		/* 個別部を送る */
		ret = NASend(scb, (char *)&header, sizeof(NETCMDHEADER), NETTIMEOUT,
					 FALSE);
		if(ret != RET_OK){
			return ret;
		}
		Set4bytes(total_blocks_data, total_blocks);
		ret = NASend(scb, (char *)total_blocks_data, 4, NETTIMEOUT, FALSE);
		if(ret!=RET_OK){
			return ret;
		}
	}

	return RET_OK;
}


#if defined(linux) || defined(__MINGW32__)
/**
 * @brief 文字コード変換
 * @param[in] iconv_desc iconvディスクリプタ
 * @param[in] in_string 変換文字列
 * @return 変換結果の文字列。エラーが発生した場合は NULL を返す。
 * @note 変換結果文字列は使用後 MemFree() が必要。
 */
char *ConvertCharcode(iconv_t iconv_desc, const char *in_string)
{
	char *out_string = NULL;
	int bufsize = strlen(in_string)+1;
	size_t inbytes = strlen(in_string);
	size_t outbytes = strlen(in_string);
	char *inbufp = (char *)in_string;
	char *outbufp;
	int out_offset = 0;
	int iconv_ret;

	out_string = (char *)MemNew(bufsize);
  
	if(strlen(in_string)==0){
		out_string[0] = '\0';
		return out_string;
	}

	iconv(iconv_desc, NULL, NULL, NULL, NULL);
	outbufp = out_string;
	while(inbytes > 0){
		iconv_ret = iconv(iconv_desc,
						  (char **)&inbufp, &inbytes,
						  &outbufp, &outbytes);
		if(iconv_ret != (size_t)-1){
			continue;
		}
		if(errno != E2BIG){
			/* 変換失敗 */
			MemFree(out_string);
			return NULL;
		}
		/* 領域不足 */
		out_offset = (int)(outbufp-out_string);
		bufsize += 64;
		outbytes += 64;
		out_string = (char *)MemResize(out_string, bufsize);
		if(out_string == NULL){
			return NULL;
		}
		outbufp = out_string+out_offset;
	}
	*outbufp = '\0';
  
	return out_string;
}
#endif


/**
 * @brief 表示データ送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] area 表示領域<br>
 *   0:画面下メッセージ<br>
 *   1:プログレスバー1<br>
 *   2:プログレスバー2
 * @param[in] percent パーセンテージ
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NASendDispCmd(SOCKCB *scb, BYTE area, BYTE percent, const char *message)
{
	int ret;
	BYTE netdata_common[8];
	NETDISPHEADER header;
	DWORD length;
#if defined(linux) || defined(__MINGW32__)
	char *message_sjis = NULL;
#endif
  
	header.area = area;
	header.percent = percent;

	length = sizeof(header);
	if(message != NULL){
#if defined(linux) || defined(__MINGW32__)
		/* local charset => SJIS */
		message_sjis = ConvertCharcode(scb->iconv_desc, message);
		if(message_sjis == NULL){
			return RET_MEMERR;
		}
		message = message_sjis;
#endif
		length += strlen(message);
	}
  
	Set4bytes(netdata_common, length);
	netdata_common[4] = NETDATATYPE_DISPLAY;
	ret = NASend(scb, (char *)netdata_common, 8, NETTIMEOUT, FALSE);
	if(ret != RET_OK){
#if defined(linux) || defined(__MINGW32__)
		if(message_sjis != NULL){
			MemFree(message_sjis);
		}
#endif
		return ret;
	}
	ret = NASend(scb, (char *)&header, sizeof(header), NETTIMEOUT, FALSE);
	if(ret != RET_OK){
#if defined(linux) || defined(__MINGW32__)
		if(message_sjis != NULL){
			MemFree(message_sjis);
		}
#endif
		return ret;
	}
	if(message != NULL){
		ret = NASend(scb, message, strlen(message), NETTIMEOUT, FALSE);
		if(ret != RET_OK){
#if defined(linux) || defined(__MINGW32__)
			if(message_sjis != NULL){
				MemFree(message_sjis);
			}
#endif
			return ret;
		}
	}
  
#if defined(linux) || defined(__MINGW32__)
	if(message_sjis != NULL){
		MemFree(message_sjis);
	}
#endif
	return RET_OK;
}


/**
 * @brief 中断を送信
 * @param[in,out] scb ソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NASendAbort(SOCKCB *scb)
{
	int ret;
	BYTE netdata_common[8];
  
	Set4bytes(netdata_common, 0);
	netdata_common[4] = NETDATATYPE_ABORT;
	ret = NASend(scb, (char *)netdata_common, 8, NETTIMEOUT, FALSE);
	if(ret != RET_OK){
		return ret;
	}
	return RET_OK;
}

/**
 * @brief 完了を送信
 * @param[in,out] scb ソケット制御ブロック
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NASendComplete(SOCKCB *scb)
{
	int ret;
	BYTE netdata_common[8];
  
	Set4bytes(netdata_common, 0);
	netdata_common[4] = NETDATATYPE_COMPLETE;
	ret = NASend(scb, (char *)netdata_common, 8, NETTIMEOUT, FALSE);
	if(ret != RET_OK){
		return ret;
	}
	return RET_OK;
}



/**
 * @brief 情報表示データを送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NAUIDispInfo(SOCKCB *scb, const char *message)
{
	return NASendDispCmd(scb, 0, 0, message);
}

/**
 * @brief プログレスバー1の初期化を送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NAUIMeter1Initialize(SOCKCB *scb, const char *message)
{
	return NASendDispCmd(scb, 1, 0, message);
}

/**
 * @brief プログレスバー2の初期化を送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NAUIMeter2Initialize(SOCKCB *scb, const char *message)
{
	return NASendDispCmd(scb, 2, 0, message);
}

/**
 * @brief プログレスバー1の更新を送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NAUIMeter1Update(SOCKCB *scb, float percentage)
{
	return NASendDispCmd(scb, 1, (BYTE)(percentage), NULL);
}

/**
 * @brief プログレスバー2の更新を送信
 * @param[in,out] scb ソケット制御ブロック
 * @param[in] message 表示メッセージ
 * @retval RET_OK 正常終了
 * @retval RET_SOCKET ソケットエラー
 * @retval RET_ABORT 中断
 * @retval RET_TIMEOUT タイムアウト発生
 * @retval RET_NG その他のエラー
 */
int NAUIMeter2Update(SOCKCB *scb, float percentage)
{
	return NASendDispCmd(scb, 2, (BYTE)(percentage), NULL);
}

#if defined(WIN32)
static const struct SOCKERRMESSAGE {
	DWORD code;
	char *message;
} g_ErrorMessage[] = {
	{ 0,                  "No error"},
	{ WSAEINTR,           "Interrupted system call" },
	{ WSAEBADF,           "Bad file number" },
	{ WSAEACCES,          "Permission denied" },
	{ WSAEFAULT,          "Bad address" },
	{ WSAEINVAL,          "Invalid argument" },
	{ WSAEMFILE,          "Too many open sockets" },
	{ WSAEWOULDBLOCK,     "Operation would block" },
	{ WSAEINPROGRESS,     "Operation now in progress" },
	{ WSAEALREADY,        "Operation already in progress" },
	{ WSAENOTSOCK,        "Socket operation on non-socket" },
	{ WSAEDESTADDRREQ,    "Destination address required" },
	{ WSAEMSGSIZE,        "Message too long" },
	{ WSAEPROTOTYPE,      "Protocol wrong type for socket" },
	{ WSAENOPROTOOPT,     "Bad protocol option" },
	{ WSAEPROTONOSUPPORT, "Protocol not supported" },
	{ WSAESOCKTNOSUPPORT, "Socket type not supported" },
	{ WSAEOPNOTSUPP,      "Operation not supported on socket" },
	{ WSAEPFNOSUPPORT,    "Protocol family not supported" },
	{ WSAEAFNOSUPPORT,    "Address family not supported" },
	{ WSAEADDRINUSE,      "Address already in use" },
	{ WSAEADDRNOTAVAIL,   "Can't assign requested address" },
	{ WSAENETDOWN,        "Network is down" },
	{ WSAENETUNREACH,     "Network is unreachable" },
	{ WSAENETRESET,       "Net connection reset" },
	{ WSAECONNABORTED,    "Software caused connection abort" },
	{ WSAECONNRESET,      "Connection reset by peer" },
	{ WSAENOBUFS,         "No buffer space available" },
	{ WSAEISCONN,         "Socket is already connected" },
	{ WSAENOTCONN,        "Socket is not connected" },
	{ WSAESHUTDOWN,       "Can't send after socket shutdown" },
	{ WSAETOOMANYREFS,    "Too many references, can't splice" },
	{ WSAETIMEDOUT,       "Connection timed out" },
	{ WSAECONNREFUSED,    "Connection refused" },
	{ WSAELOOP,           "Too many levels of symbolic links" },
	{ WSAENAMETOOLONG,    "File name too long" },
	{ WSAEHOSTDOWN,       "Host is down" },
	{ WSAEHOSTUNREACH,    "No route to host" },
	{ WSAENOTEMPTY,       "Directory not empty" },
	{ WSAEPROCLIM,        "Too many processes" },
	{ WSAEUSERS,          "Too many users" },
	{ WSAEDQUOT,          "Disc quota exceeded" },
	{ WSAESTALE,          "Stale NFS file handle" },
	{ WSAEREMOTE,         "Too many levels of remote in path" },
	{ WSASYSNOTREADY,     "Network system is unavailable" },
	{ WSAVERNOTSUPPORTED, "Winsock version out of range" },
	{ WSANOTINITIALISED,  "WSAStartup not yet called" },
	{ WSAEDISCON,         "Graceful shutdown in progress" },
	{ WSAHOST_NOT_FOUND,  "Host not found" },
	{ WSANO_DATA,         "No host data of that type was found" },
};
#endif	/* WIN32 */

/**
 * @brief エラーコードに対応するメッセージを取得
 * @param[in] scb ソケット制御ブロック
 * @return 取得したメッセージを返す。エラーの場合はNULLを返す。
 * @note 取得したメッセージはメモリ解放する必要なし。
 */
const char *NAGetErrorMessage(SOCKCB *scb)
{
	const char *message = NULL;
  
#if defined(WIN32)
	int index;
	for(index=0; index<sizeof(g_ErrorMessage)/sizeof(struct SOCKERRMESSAGE); index++){
		if(g_ErrorMessage[index].code == scb->error_code){
			message = g_ErrorMessage[index].message;
			break;
		}
	}
#else
	message = strerror(scb->error_code);
#endif
  
	return message;
}

/**
 * ソケット関連エラーならエラーを表示
 * @param[in]	drive	装置構造体
 * @param[in]	ret_value 関数返却値
 * @return	ソケット関連エラーかどうか
 */
BOOL DispSocketError(SOCKCB *scb, int ret_value)
{
	if(ret_value==RET_SOCKET){
		const char *message = NAGetErrorMessage(scb);
		if(message != NULL){
			char *net_err = EbStringNewWithFormat(
				MSG_NETWORK_ERROR_ /*"ネットワークエラー %ld : "*/,
				scb->error_code);
			net_err = EbStringAppend(net_err, message);
			UIDispMessage(message, UIDMT_ERROR);
			net_err = EbStringFree(net_err);
			return TRUE;
		}
	}
	return FALSE;
}
