/*
 * @file netaccess.h
 * @brief ネットワークアクセス
 * @author BananaJinn
 * @version $Id: netaccess.h,v 1.12 2010/11/01 14:34:11 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#ifndef __NETACCESS_H__
#define __NETACCESS_H__

#if defined(WIN32)
# include <winsock.h>
# define SD_RECEIVE      0x00
# define SD_SEND         0x01
# define SD_BOTH         0x02
#else
# include <sys/socket.h>
  typedef int SOCKET;
#endif
#if defined(linux) || defined(__MINGW32__)
# include <iconv.h>
#endif

#define NETTIMEOUT (60*1000)

typedef struct {
  BOOL server_mode;
  union {
	struct {
	  WORD port;
	  SOCKET listen_socket;
	  SOCKET socket_desc;
	} server;
	struct {
	  DWORD remote_address;
	  WORD remote_port;
	  SOCKET socket_desc;
	} client;
  } u;
  char *remote_host;
  DWORD error_code;
#if defined(linux) || defined(__MINGW32__)
  iconv_t iconv_desc;
#endif
} SOCKCB;



/*
 * データ転送形式
 * +-----------------------+ ----
 * | length[0]          MSB|   ^
 * +-----------------------+   |
 * | length[1]             |   |
 * +-----------------------+   |
 * | length[2]             |   | 共通部
 * +-----------------------+   | (8 bytes)
 * | length[3]          LSB|   |
 * +-----------------------+   |
 * | data type             |   |
 * +-----------------------+   |
 * | reserved              |   |
 * +-----------------------+   |
 * | reserved              |   |
 * +-----------------------+   |
 * | reserved              |   v
 * +-----------------------+ ----
 * | ...                   |   ^
 * |                       |   | 個別部(data typeによる)
 * |                       |   v
 * +-----------------------+ ----
 *
 * lengthは個別部のデータ長を表す(共通部は含まない)
 */

/*
 * ATAPI/ASPIコマンド(Client => Server)
 * 個別データ部は NETCMDHEADER + コマンドデータ(あれば)
 */
#define NETDATATYPE_COMMAND		1
/*
 * ATAPI/ASPIコマンドに対する応答(Server => Client)
 * 個別データ部は NETCMDHEADER + コマンドデータ(あれば)
 */
#define NETDATATYPE_RESPONSE	2
/*
 * ATAPI/ASPIコマンドの長く続くREAD(Client => Server)
 * READ(12)またはREADCDの形式でServerから連続してデータを返す
 * 個別データ部は NETCMDHEADER + トータルlength(4bytes)
 */
#define NETDATATYPE_LONGREAD	3

typedef struct {
  BYTE reqflag;
  BYTE reserved[3];
  BYTE cdb[12];
  BYTE data_length[4];
  BYTE sense_data[0x20];
} NETCMDHEADER;

/*
 * 表示更新命令(Client => Server)
 * 個別データ部は NETDISPHEADER + メッセージ文字列(\0まで)
 * メッセージ文字列長が 0 の場合はプログレスバーのみ更新したりする
 */
#define NETDATATYPE_DISPLAY		4

typedef struct {
  BYTE area;	/* 0:画面下メッセージ/1:プログレスバー1/2:プログレスバー2 */
  BYTE percent;	/* プログレスバーの進捗(0-100) */
} NETDISPHEADER;

/*
 * 処理中断(Client <=> Server)
 * 個別データ部はなし。即座に処理を中断する。
 */
#define NETDATATYPE_ABORT		5

/*
 * 処理完了(Client => Server)
 * 個別データ部はなし。
 */
#define NETDATATYPE_COMPLETE	6

extern int InitializeSOCKCB(SOCKCB *scb);
extern int FreeSOCKCB(SOCKCB *scb);

extern int NACreateServer(SOCKCB *scb);
extern int NAWaitConnect(SOCKCB *scb, DWORD timeout);
extern int NAConnect(SOCKCB *scb);
extern int NAClose(SOCKCB *scb);
extern int NASend(SOCKCB *scb, const void *buffer, int length, DWORD timeout,
				  BOOL check_readfd);
extern int NAReceive(SOCKCB *scb, void *buffer, int length, DWORD timeout);

extern int SendNetCmd(SOCKCB *scb, BYTE *cdb, void *buf, DWORD buflen,
					  BYTE *sensedata, DWORD senselen, int reqflag);
extern int SendNetCmdLongRead(SOCKCB *scb, BYTE *cdb, void *buf, DWORD buflen,
							  BYTE *sensedata, DWORD senselen,
							  int reqflag, DWORD total_blocks);

extern int NASendDispCmd(SOCKCB *scb, BYTE area, BYTE percent,
						 const char *message);
extern int NASendAbort(SOCKCB *scb);
extern int NASendComplete(SOCKCB *scb);
extern int NAUIDispInfo(SOCKCB *scb, const char *message);
extern int NAUIMeter1Initialize(SOCKCB *scb, const char *message);
extern int NAUIMeter2Initialize(SOCKCB *scb, const char *message);
extern int NAUIMeter1Update(SOCKCB *scb, float percentage);
extern int NAUIMeter2Update(SOCKCB *scb, float percentage);
extern const char *NAGetErrorMessage(SOCKCB *scb);
#if defined(linux) || defined(__MINGW32__)
extern char *ConvertCharcode(iconv_t iconv_desc, const char *in_string);
#endif
extern BOOL DispSocketError(SOCKCB *scb, int ret_value);

#endif /* !__NETACCESS_H__ */
