/**
 * @file netserver.c
 * @brief サーバ処理
 * @author BananaJinn
 * @version $Id: netserver.c,v 1.10 2010/11/05 17:24:03 bananajinn Exp $
 * 円盤複写屋
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#ifdef linux
# include <fcntl.h>
#endif
#include <time.h>
#if defined(WIN32)
# include <io.h>
#endif
#include "mem.h"
#include "aspi.h"
#include "struct.h"
#include "cmd.h"
#include "ui.h"
#include "copydisc.h"
#include "cmdlog.h"
#include "text.h"

static int ReceiveNetCmd(CMDDRIVE *netdrive, CMDDRIVE *realdrive,
						 BOOL enable_longread);

/**
 * @brief	リモート要求によるコマンド実行
 * @param[in]	netdrive	ネットワークドライブ(リモートドライブ)
 * @param[in]	realdrive	実ドライブ(ローカルドライブ)
 * @retval	RET_OK	正常終了
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_SOCKET	ソケットエラー
 * @retval	RET_NG	エラー
 */
static int ExecuteCommand(CMDDRIVE *netdrive, CMDDRIVE *realdrive)
{
  int ret;
  DWORD length;
  NETCMDHEADER cmdheader;
  BYTE netdata_common[8];

  memset(netdata_common, 0, sizeof(netdata_common));
  
	  /* コマンド受信 */
  ret = NAReceive(&netdrive->u.net, (char *)&cmdheader, sizeof(cmdheader),
				  NETTIMEOUT);
  if(ret != RET_OK){
	return ret;
  }
  length = Get4bytes(cmdheader.data_length);
  if(length > (DWORD)netdrive->bufsize){
	length = (DWORD)netdrive->bufsize;
  }
  
  if((length > 0) && (cmdheader.reqflag==REQ_DATAOUT)){
		/* コマンドデータ部を受信 */
	ret = NAReceive(&netdrive->u.net, (char *)netdrive->data_buf,
					(int)length, NETTIMEOUT);
	if(ret != RET_OK){
	  return ret;
	}
  }
  ret = SendCmd(realdrive, cmdheader.cdb, length, cmdheader.reqflag);
	  /* エラーが発生しても、エラー情報は返却する */
	  /* データ返送 */
  if(cmdheader.reqflag == REQ_DATAIN){
	Set4bytes(netdata_common, length+sizeof(NETCMDHEADER));
  }
  else{
	Set4bytes(netdata_common, sizeof(NETCMDHEADER));
  }
  netdata_common[4] = NETDATATYPE_RESPONSE;
  ret = NASend(&netdrive->u.net, (char *)netdata_common, 8,
			   NETTIMEOUT, FALSE);
  if(ret != RET_OK){
	return ret;
  }
  memcpy(cmdheader.sense_data, realdrive->sense_data,
		 sizeof(cmdheader.sense_data));
  ret = NASend(&netdrive->u.net, (char *)&cmdheader, sizeof(cmdheader),
			   NETTIMEOUT, FALSE);
  if(ret != RET_OK){
	return ret;
  }
  if(cmdheader.reqflag == REQ_DATAIN){
	ret = NASend(&netdrive->u.net, (char *)netdrive->data_buf, length,
				 NETTIMEOUT, FALSE);
	if(ret != RET_OK){
	  return ret;
	}
  }

  return RET_OK;
}



/**
 * @brief	リモート要求による連続READ実行
 * @param[in]	netdrive	ネットワークドライブ(リモートドライブ)
 * @param[in]	realdrive	実ドライブ(ローカルドライブ)
 * @param[in]	org_header	受けた要求のコマンドヘッダー
 * @retval	RET_OK	正常終了
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_SOCKET	ソケットエラー
 * @retval	RET_NG	エラー
 */
static int ExecuteLongRead(CMDDRIVE *netdrive, CMDDRIVE *realdrive)
{
  DWORD lba, start_lba;
  DWORD length, total_length;
  DWORD blocksize;
  int ret;
  BYTE netdata_common[8];
  NETCMDHEADER org_header, header;
  BYTE work4[4];

	  /* コマンド受信 */
  ret = NAReceive(&netdrive->u.net, (char *)&org_header, sizeof(org_header),
				  NETTIMEOUT);
  if(ret != RET_OK){
	return ret;
  }
	  /* 合計ブロック数を受信 */
  ret = NAReceive(&netdrive->u.net, (char *)work4, sizeof(work4),
				  NETTIMEOUT);
  if(ret != RET_OK){
	return ret;
  }
  total_length = Get4bytes(work4);
  start_lba = Get4bytes(org_header.cdb+2);
  
  memcpy(header.cdb, org_header.cdb, 12);
  lba = start_lba;
  if(header.cdb[0]==CMD_READ12){
	length = Get4bytes(header.cdb+6);
	blocksize = 2048;
  }
  else if(header.cdb[0]==CMD_READ_CD){
	length = Get3bytes(header.cdb+6);
	blocksize = 2352;
  }
  else{
	return RET_NG;
  }

  while(total_length > 0){
	if(UICheckAbort()){
	  return RET_ABORT;
	}
	if(length > total_length){
	  length = total_length;
	  if(header.cdb[0]==CMD_READ12){
		Set4bytes(header.cdb+2, lba);
		Set4bytes(header.cdb+6, length);
	  }
	  else /* CMD_READ_CD */{
		Set4bytes(header.cdb+2, lba);
		Set3bytes(header.cdb+6, length);
	  }
	}
		/* 実ドライブからREAD */
	Set4bytes(header.cdb+2, lba);
	ret = SendCmd(realdrive, header.cdb, length*blocksize, REQ_DATAIN);
	if(ret != RET_OK){
	  return ret;
	}
	memcpy(header.sense_data, realdrive->sense_data,
		   sizeof(header.sense_data));

		/* READデータ返送 */
	Set4bytes(netdata_common, length*blocksize+sizeof(header));
	netdata_common[4] = NETDATATYPE_RESPONSE;
	while(1){
	  ret = NASend(&netdrive->u.net, (char *)netdata_common, 8,
				   NETTIMEOUT, TRUE);
	  if(ret==RET_OK){
		break;
	  }
	  if(ret!=RET_READFD){
		return ret;
	  }
	  ret = ReceiveNetCmd(netdrive, realdrive, FALSE);
	  if(ret != RET_OK){
		return ret;
	  }
	}
	ret = NASend(&netdrive->u.net, (char *)&header, sizeof(header),
				 NETTIMEOUT, FALSE);
	if(ret != RET_OK){
	  return ret;
	}
	ret = NASend(&netdrive->u.net, (char *)netdrive->data_buf,
				 length*blocksize, NETTIMEOUT, FALSE);
	if(ret != RET_OK){
	  return ret;
	}

	lba += length;
	total_length -= length;
  }

  return RET_OK;
}

/**
 * @brief	リモート要求によるメッセージ表示
 * @param[in]	netdrive	ネットワークドライブ(リモートドライブ)
 * @param[in]	receive_bytes	受けた要求のバイト長
 * @retval	RET_OK	正常終了
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_NG	エラー
 * @retval	RET_MEMERR	メモリ不足
 */
static int DisplayMessage(CMDDRIVE *netdrive, DWORD receive_bytes)
{
  NETDISPHEADER dispheader;
  int ret;
  char *message=NULL;
  DWORD message_len, ignore_len=0;
  
  ret = NAReceive(&netdrive->u.net, (char *)&dispheader, sizeof(dispheader),
				  NETTIMEOUT);
  if(ret != RET_OK){
	return ret;
  }
  if(receive_bytes > (DWORD)sizeof(dispheader)){
	message_len = receive_bytes-sizeof(dispheader);
		/* 最大255文字とする */
	if(message_len > 255){
	  ignore_len = message_len-255;
	  message_len = 255;
	}
	message = (char *)MemNew(message_len+1);
	if(message==NULL){
	  UIDispMessage(MSG_MEM_ALLOC_ERROR
			/*"メモリ確保に失敗しました。"*/, UIDMT_ERROR);
	  return RET_MEMERR;
	}
	/* メッセージ文字列を受信 */
	ret = NAReceive(&netdrive->u.net, message, message_len, NETTIMEOUT);
	if(ret != RET_OK){
	  MemFree(message);
	  return ret;
	}
	ret = NAReceive(&netdrive->u.net, NULL, ignore_len, NETTIMEOUT);
	if(ret != RET_OK){
	  MemFree(message);
	  return ret;
	}
	message[message_len] = '\0';
#if defined(linux) || defined(__MINGW32__)
	{
	  char *message_euc = ConvertCharcode(netdrive->u.net.iconv_desc,
										 message);
	  if(message_euc == NULL){
	    MemFree(message);
	    UIDispMessage(MSG_CODE_CONV_ERROR
			  /*"文字コード変換に失敗しました。"*/, UIDMT_ERROR);
	    return RET_MEMERR;
	  }
	  MemFree(message);
	  message = message_euc;
	}
#endif
  }
  switch(dispheader.area){
  case 0:	/* 画面下メッセージ */
    UIDispInfo(message);
    break;
  case 1:	/* プログレスバー1 */
	if(message!=NULL)
	  UIMeter1Initialize(message);
	else
	  UIMeter1Update((float)dispheader.percent);
	break;
  case 2:	/* プログレスバー2 */
	if(message!=NULL)
	  UIMeter2Initialize(message);
	else
	  UIMeter2Update((float)dispheader.percent);
	break;
  }
  MemFree(message);
  
  return RET_OK;
}


/**
 * @brief	リモート要求処理
 * @param[in]	netdrive	ネットワークドライブ(リモートドライブ)
 * @param[in]	realdrive	実ドライブ(ローカルドライブ)
 * @param[in]	enable_longread	連続READ許可
 * @retval	RET_OK	正常終了
 * @retval	RET_TIMEOUT	タイムアウト
 * @retval	RET_ABORT	処理中断
 * @retval	RET_COMPLETE	処理完了
 * @retval	RET_SOCKET	ソケットエラー
 * @retval	RET_NG	エラー
 */
static int ReceiveNetCmd(CMDDRIVE *netdrive, CMDDRIVE *realdrive,
						 BOOL enable_longread)
{
  int ret;
  BYTE netdata_common[8];
  DWORD receive_bytes;
  
  /* 共通部(データ長とデータタイプ)を受信する */
  ret = NAReceive(&netdrive->u.net, (char *)netdata_common, 8, 1000);
  if(ret != RET_OK){
    return ret;
  }

  receive_bytes = Get4bytes(netdata_common);

  switch(netdata_common[4]){
  case NETDATATYPE_COMMAND:
    ret = ExecuteCommand(netdrive, realdrive);
    if(ret != RET_OK){
      return ret;
    }
    break;
    
  case NETDATATYPE_RESPONSE:
    /* あり得ない */
    break;
	
  case NETDATATYPE_LONGREAD:
    if(enable_longread){
      ret = ExecuteLongRead(netdrive, realdrive);
      if(ret != RET_OK){
	return ret;
      }
    }
    else{
      UIDispMessage(MSG_UNKNOWN_COMMAND
		    /*"不正なコマンドを受信しました。"*/, UIDMT_ERROR);
      return RET_NG;
    }
    break;
	
  case NETDATATYPE_DISPLAY:
    ret = DisplayMessage(netdrive, receive_bytes);
    if(ret != RET_OK){
      return ret;
    }
    break;
	
  case NETDATATYPE_ABORT:
    return RET_ABORT;
    
  case NETDATATYPE_COMPLETE:
    return RET_COMPLETE;

  default:
    /* 不正なコマンド */
    UIDispMessage(MSG_UNKNOWN_COMMAND
		  /*"不正なコマンドを受信しました。"*/, UIDMT_ERROR);
    return RET_NG;
  }
  return RET_OK;
}


/**
 * @brief	リモート要求受付処理
 * @param[in]	netdrive	ネットワークドライブ(リモートドライブ)
 * @param[in]	realdrive	実ドライブ(ローカルドライブ)
 * @retval	RET_OK	正常終了
 * @retval	RET_ABORT	処理中断
 * @retval	RET_COMPLETE	処理完了
 * @retval	RET_SOCKET	ソケットエラー
 * @retval	RET_MEMERR	メモリ不足
 * @retval	RET_NG	エラー
 */
int WaitingNetCmd(CMDDRIVE *netdrive, CMDDRIVE *realdrive)
{
  BOOL done=FALSE;
  int ret;
  
  while(!done){
    if(UICheckAbort()){
      return RET_ABORT;
    }
    ret = ReceiveNetCmd(netdrive, realdrive, TRUE);
    if(ret == RET_COMPLETE){
      done=TRUE;
    }
    else if((ret != RET_OK) && (ret != RET_TIMEOUT)){
      return ret;
    }
  }
  
  return RET_OK;
}

