/**
 * ~Օʉ - EnbanKensa
 * Copyright (C) 2005 Kagetani Hideto
 * check.c - 
 * $Date: 2006/12/02 14:00:19 $
 * $Revision: 1.14 $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(__APPLE__) && defined(__MACH__)
# include <Carbon/Carbon.h>
#else
# include <sys/timeb.h>
#endif

#include "check.h"
#include "drive.h"
#include "cmd.h"
#include "struct.h"
#include "option.h"
#include "ui.h"
#include "text.h"
#include "ebstring.h"

#if defined(WIN32)
# define snprintf _snprintf
#endif

#define D_MARGIN_LEFT	(8*4)
#define D_MARGIN_RIGHT	(8*6)
#define D_MARGIN_TOP	(8*2)
#define D_MARGIN_BOTTOM	(8*2)

#define D_COL_BASELINE			0xffff, 0xffff, 0xffff
#define D_COL_TRANSRATE			0x0000, 0x7fff, 0xffff
#define D_COL_TRANSRATE_GRID	0x0000, 0x3fff, 0x7fff
#define D_COL_ERRORRATE			0x7fff, 0xffff, 0x0000
#define D_COL_ERRORRATE_GRID	0x3fff, 0x7fff, 0x0000
#define D_COL_ERRORLINE			0xffff, 0x0000, 0x0000
#define D_COL_ADDRESSLINE		0x7fff, 0x7fff, 0x7fff
#define D_COL_TEXT				0xffff, 0xffff, 0xffff
#define D_COL_BACK				0x0000, 0x0000, 0x0000
#define D_COL_DISABLED			0x2fff, 0x4fff, 0x4fff

typedef struct {
	WORD track;
	WORD session;
	DWORD start_lba;
	DWORD end_lba;
	DWORD blocks;
	BYTE track_mode;
	BYTE data_mode;
BYTE fp:1;
BYTE packet:1;
	BYTE damage;
BYTE blank:1;
BYTE res:4;
	DWORD packet_size;
} TRACKINFO;

typedef struct {
	TRACKINFO *tracks;
	WORD num_track;
	DWORD last_lba;
} DISCINFO;

typedef struct {
	DWORD lba;
	float trans_rate;
	float error_rate;
} RATES;

typedef struct {
	RATES *rates;
	int num_rates;
} TRACKRESULT;

typedef struct {
	TRACKRESULT *track_result;
	int num_track;
	float max_trans_rate;	/* \ő{ */
	float max_error_rate;	/* \őG[(%) */
	BOOL c2_error;		/* C2G[ */
	BOOL read_error;		/* ǂݎG[ */
} RESULTINFO;

static struct _DRAWAREA {
	int width;
	int height;
} g_drawArea;

static BYTE g_zero294[294];

static int OpenDevice(CMDDRIVE *drive, DRIVEID *id, BOOL bWaitDisc)
{
	int ret;
	DWORD timeout=5*60;

	ret = InitializeCmdDrive(drive, 0x10000);
	if(ret!=RET_OK){
		return ret;
	}
	ret = SetDriveAspiSetting(drive, &id->hid, &id->tid, &timeout);
	if(ret!=RET_OK){
		return ret;
	}

	ret = GetDiscType(drive, &drive->disc_type, bWaitDisc);
	if(ret!=RET_OK){
		if(ret!=RET_ABORT && bWaitDisc){
			UIDispMessage(MSG_CANT_GET_DISC_TYPE
						  /*"fBXN^Cv擾Ɏs܂B"*/, UIDMT_ERROR);
		}
		FreeCmdDrive(drive);
		return ret;
	}
	ret = SendPreventAllow(drive, 1);
	if(ret!=RET_OK){
		DispCommandError(drive);
		FreeCmdDrive(drive);
		return ret;
	}

	return RET_OK;
}

static void CloseDevice(CMDDRIVE *drive)
{
	SendPreventAllow(drive, 0);
	FreeCmdDrive(drive);
}

static DWORD GetEndLbaByTrackInfo(CMDDRIVE *drive, struct _TRACKINFO *ti)
{
	int ret;
	DWORD start_lba;
	DWORD end_lba;

	start_lba = Get4bytes(ti->track_start);
  
	if(ti->blank != 0){
		end_lba = start_lba;
	}
	else{
		end_lba =
			Get4bytes(ti->track_start) +
			Get4bytes(ti->track_size) -
			Get4bytes(ti->free_blocks) -1;

		if(DT_CD_FAMILY(drive->disc_type)){
				/* CD̏ꍇAŌ2ubN READ  Run-out ǂmF */
			ret = SendReadCD(drive, end_lba-1, 2);
			if(ret == RET_CMDERR){
				end_lba -= 2;
			}
		}
	}
  
	return end_lba;
}

static RESULTINFO *FreeResultInfo(RESULTINFO *resultInfo)
{
	WORD track;
  
	if(resultInfo == NULL){
		return NULL;
	}

	for(track=0; track<resultInfo->num_track; track++){
		if(resultInfo->track_result[track].rates != NULL){
			free(resultInfo->track_result[track].rates);
		}
	}
	free(resultInfo->track_result);
	free(resultInfo);
	return NULL;
}

static RESULTINFO *AllocInitResultInfo(DISCINFO *discInfo, DWORD readBlock)
{
	RESULTINFO *resultInfo;
	WORD trackIndex;
	TRACKINFO *trackInfo;
	RATES *rates;
	int numRates;

	resultInfo = (RESULTINFO *)malloc(sizeof(RESULTINFO));
	if(resultInfo == NULL){
		return NULL;
	}
  
	memset(resultInfo, 0, sizeof(RESULTINFO));

	resultInfo->track_result =
		(TRACKRESULT *)malloc(sizeof(TRACKRESULT)*discInfo->num_track);
	if(resultInfo->track_result == NULL){
		free(resultInfo);
		return NULL;
	}
	memset(resultInfo->track_result, 0, sizeof(TRACKRESULT)*discInfo->num_track);
	resultInfo->num_track = discInfo->num_track;

	for(trackIndex=0; trackIndex<discInfo->num_track; trackIndex++){
		trackInfo = &discInfo->tracks[trackIndex];
		if(trackInfo->start_lba < trackInfo->end_lba){
			numRates =
				((trackInfo->end_lba-trackInfo->start_lba)+readBlock-1)/readBlock;
				/* 1߂ɃmۂĂ */
			rates = (RATES *)malloc(sizeof(RATES)*(numRates+1));
			if(rates == NULL){
				FreeResultInfo(resultInfo);
				return NULL;
			}
			memset(rates, 0, sizeof(RATES)*numRates);

			resultInfo->track_result[trackIndex].rates = rates;
			resultInfo->track_result[trackIndex].num_rates = numRates;
		}
	}

	return resultInfo;
}


static DISCINFO *FreeDiscInfo(DISCINFO *discInfo)
{
	if(discInfo == NULL){
		return NULL;
	}

	if(discInfo->tracks != NULL){
		free(discInfo->tracks);
	}

	free(discInfo);
	return NULL;
}  

static DISCINFO *AllocInitDiscInfo()
{
	DISCINFO *discInfo;

	discInfo = (DISCINFO *)malloc(sizeof(DISCINFO));
	if(discInfo == NULL){
		return NULL;
	}
  
	memset(discInfo, 0, sizeof(DISCINFO));

	return discInfo;
}

static int MakeDiscInfo(CMDDRIVE *drive, DISCINFO **discInfoRet)
{
	int ret;
	DISCINFO *discInfo;
	TRACKINFO *trackInfo;
	struct _DISCINFO *di;
	struct _TRACKINFO ti;
	WORD last_track;
	WORD curr_track;

	discInfo = AllocInitDiscInfo();
	if(discInfo==NULL){
		return RET_MEMERR;
	}

	*discInfoRet = discInfo;

	ret = SendReadDiscInfo(drive);
	if(ret != RET_OK){
		return ret;
	}
	di = (struct _DISCINFO *)drive->data_buf;
	if(di->disc_status == DISCSTAT_EMPTY){
		UIDispMessage(MSG_INS_MEDIA_IS_BLANK
					  /*"}ꂽfBXN͖L^fBXNłB"*/, UIDMT_ERROR);
		return RET_ABORT;
	}
	last_track =
		((WORD)di->last_track_ls_msb << 8) |
		(WORD)di->last_track_ls_lsb;
	trackInfo = (TRACKINFO *)malloc(sizeof(TRACKINFO)*last_track);
	if(trackInfo == NULL){
		*discInfoRet = FreeDiscInfo(discInfo);
		return RET_MEMERR;
	}
	memset(trackInfo, 0, sizeof(TRACKINFO)*last_track);
	discInfo->num_track = last_track;
	discInfo->tracks = trackInfo;

	for(curr_track=1; curr_track<=last_track; curr_track++){
		UIDispInfo(MSG_MAKING_DISCINFO_, curr_track, last_track);
		if(UICheckAbort()){
			*discInfoRet = FreeDiscInfo(discInfo);
			return RET_ABORT;
		}
		ret = SendReadTrackInfo(drive, curr_track);
		if(ret != RET_OK){
			*discInfoRet = FreeDiscInfo(discInfo);
			return RET_MEMERR;
		}
		memcpy(&ti, drive->data_buf, sizeof(struct _TRACKINFO));
		trackInfo[curr_track-1].session =
			((WORD)ti.session_number_msb << 8) |
			(WORD)ti.session_number_lsb;
		trackInfo[curr_track-1].track =
			((WORD)ti.track_number_msb << 8) |
			(WORD)ti.track_number_lsb;
		trackInfo[curr_track-1].start_lba = Get4bytes(ti.track_start);
		trackInfo[curr_track-1].end_lba = GetEndLbaByTrackInfo(drive, &ti);
		trackInfo[curr_track-1].blocks = Get4bytes(ti.track_size);
		trackInfo[curr_track-1].track_mode = ti.track_mode;
		trackInfo[curr_track-1].data_mode = ti.data_mode;
		trackInfo[curr_track-1].fp = ti.fp;
		trackInfo[curr_track-1].packet = ti.packet;
		trackInfo[curr_track-1].damage = ti.damage;
		trackInfo[curr_track-1].blank = ti.blank;
		trackInfo[curr_track-1].packet_size = Get4bytes(ti.packet_size);
		if(trackInfo[curr_track-1].start_lba < trackInfo[curr_track-1].end_lba){
			discInfo->last_lba = trackInfo[curr_track-1].end_lba;
		}
	}
  
	return RET_OK;
} 

static void DrawGridLines(CMDDRIVE *drive, DISCINFO *discInfo,
						  float maxTransRate, float maxErrorRate)
{
	DWORD maxAddress = discInfo->last_lba;
	int x0, y0, x1, y1;
	int width;
	int xx, yy;
	int i, skipMask;
	char textBuf[16];
	UICOLOR baseLineColor = { D_COL_BASELINE };
	UICOLOR transRateColor = { D_COL_TRANSRATE_GRID };
	UICOLOR errorRateColor = { D_COL_ERRORRATE_GRID };
	UICOLOR addressLineColor = { D_COL_ADDRESSLINE };
	UICOLOR textColor = { D_COL_TEXT };
	UICOLOR backColor = { D_COL_BACK };
	UICOLOR disableColor = { D_COL_DISABLED };

		/* S */
	UIDrawBox(0, 0, g_drawArea.width-1, g_drawArea.height-1, &backColor, TRUE);

		/* {Wݒ */
	x0 = D_MARGIN_LEFT;
	y0 = D_MARGIN_TOP;
	x1 = g_drawArea.width-D_MARGIN_RIGHT;
	y1 = g_drawArea.height-D_MARGIN_BOTTOM;
	width = g_drawArea.width-D_MARGIN_LEFT-D_MARGIN_RIGHT;

		/* READs̈̕\ */
	for(i=1; i<discInfo->num_track; i++){
		if((discInfo->tracks[i-1].end_lba+1 < discInfo->tracks[i].start_lba) &&
		   (discInfo->tracks[i].start_lba < discInfo->tracks[i].end_lba)){
			UIDrawBox(x0+width*discInfo->tracks[i-1].end_lba/maxAddress, y0,
					  x0+width*discInfo->tracks[i].start_lba/maxAddress, y1,
					  &disableColor, TRUE);
		}
	}
  
		/* X̖ڐ */
	for(i=1; i<=4; i++){
		xx = x0+(x1-x0)*i/4;
		UIDrawLine(xx, y0, xx, y1, &addressLineColor);
	}
		/* ]x̖ڐ */
	if(maxTransRate > 64){
		skipMask = 7;
	}
	else if(maxTransRate > 32){
		skipMask = 3;
	}
	else if(maxTransRate > 16){
		skipMask = 1;
	}
	else{
		skipMask = 0;
	}
	for(i=1; i<=maxTransRate; i++){
		if((i&skipMask) != 0){
			continue;
		}
		yy = (int)(y1-i*(y1-y0)/maxTransRate);
		UIDrawLine(x0, yy, x1, yy, &transRateColor);
		sprintf(textBuf, "%2dx", i);
		UIDrawText(0, yy, &textColor, textBuf);
	}
	if(maxErrorRate > 0){
			/* G[[g̖ڐ */
		for(i=1; i<=4; i++){
			yy = y1-(y1-y0)*i/4;
			UIDrawLine(x0, yy, x1, yy, &errorRateColor);
			sprintf(textBuf, "%.3f%%", maxErrorRate*i/4);
			UIDrawText(x1, yy, &textColor, textBuf);
		}
	}

		/* { */
	UIDrawLine(x0, y1, x1, y1, &baseLineColor);
	UIDrawLine(x0, y0, x0, y1, &baseLineColor);

	UIDrawText(x0-8, y1, &textColor, "0");
	sprintf(textBuf, "0x%08lX", maxAddress);
	UIDrawText(x1-80, y1, &textColor, textBuf);
  
}

static void DrawTransRateLine(RATES *r1, RATES *r2,
							  float maxTransRate, DWORD maxAddress)
{
	int x0, y0, x1, y1;
	int baseY = g_drawArea.height-D_MARGIN_BOTTOM;
	int width, height;
	UICOLOR color = { D_COL_TRANSRATE };
	UICOLOR errColor = { D_COL_ERRORLINE };

	if((r1->lba == 0) || (r2->lba == 0)){
		return;
	}

	width = g_drawArea.width-D_MARGIN_LEFT-D_MARGIN_RIGHT;
	height = g_drawArea.height-D_MARGIN_TOP-D_MARGIN_BOTTOM;
	x0 = (int)(D_MARGIN_LEFT + width*r1->lba/maxAddress);
	x1 = (int)(D_MARGIN_LEFT + width*r2->lba/maxAddress);
	y0 = (int)(baseY - height*r1->trans_rate/maxTransRate);
	y1 = (int)(baseY - height*r2->trans_rate/maxTransRate);
	UIDrawLine(x0, y0, x1, y1,
			   (r2->trans_rate==0) ? &errColor : &color);
}

static void DrawErrorRateLine(RATES *r1, RATES *r2,
							  float maxErrorRate, DWORD maxAddress)
{
	int x0, y0, x1, y1;
	int baseY = g_drawArea.height-D_MARGIN_BOTTOM;
	int width, height;
	UICOLOR color = { D_COL_ERRORRATE };
	UICOLOR errColor = { D_COL_ERRORLINE };

	if(maxErrorRate==0){
		return;
	}
	if((r1->lba == 0) || (r2->lba == 0)){
		return;
	}
  
	width = g_drawArea.width-D_MARGIN_LEFT-D_MARGIN_RIGHT;
	height = g_drawArea.height-D_MARGIN_TOP-D_MARGIN_BOTTOM;

	x0 = (int)(D_MARGIN_LEFT + width*r1->lba/maxAddress);
	x1 = (int)(D_MARGIN_LEFT + width*r2->lba/maxAddress);
	y0 = (int)(baseY - height*r1->error_rate/maxErrorRate);
	y1 = (int)(baseY - height*r2->error_rate/maxErrorRate);
	UIDrawLine(x0, y0, x1, y1,
			   (r2->trans_rate==0) ? &errColor : &color);
}

static void DrawTransRateLineTrack(RATES *rates, int count,
								   float maxTransRate, DWORD maxAddress)
{
	int i;

	for(i=0; i<count; i++){
		DrawTransRateLine(&rates[i], &rates[i+1], maxTransRate, maxAddress);
	}
}

static void DrawTransRateLineAll(RESULTINFO *resultInfo, WORD trackCount,
								 DWORD maxAddress)
{
	WORD trackIndex;

	for(trackIndex=0; trackIndex<trackCount; trackIndex++){
		DrawTransRateLineTrack(resultInfo->track_result[trackIndex].rates,
							   resultInfo->track_result[trackIndex].num_rates-1,
							   resultInfo->max_trans_rate, maxAddress);
	}
}

static void DrawErrorRateLineTrack(RATES *rates, int count,
								   float maxErrorRate, DWORD maxAddress)
{
	int i;

	if(maxErrorRate==0){
		return;
	}
  
	for(i=0; i<count; i++){
		DrawErrorRateLine(&rates[i], &rates[i+1], maxErrorRate, maxAddress);
	}
}

static void DrawErrorRateLineAll(RESULTINFO *resultInfo, WORD trackCount,
								 DWORD maxAddress)
{
	WORD trackIndex;

	for(trackIndex=0; trackIndex<trackCount; trackIndex++){
		DrawErrorRateLineTrack(resultInfo->track_result[trackIndex].rates,
							   resultInfo->track_result[trackIndex].num_rates-1,
							   resultInfo->max_error_rate, maxAddress);
	}
}

static void UpdateScreen(CMDDRIVE *drive, DISCINFO *discInfo,
						 RESULTINFO *resultInfo,
						 WORD trackIndex, int current, BOOL drawCurrent)
{
	DWORD maxAddress = discInfo->last_lba;
	BOOL resetGrids = FALSE;
	RATES *rates = resultInfo->track_result[trackIndex].rates;

	if(rates[current].trans_rate > resultInfo->max_trans_rate){
			/* ʏ㕔͂ݏoꍇ */
		resultInfo->max_trans_rate = (float)((int)rates[current].trans_rate + 2);
		resetGrids = TRUE;
	}
	if(resultInfo->max_error_rate > 0){
		if(rates[current].error_rate > resultInfo->max_error_rate){
				/* ʏ㕔͂ݏoꍇ */
			resultInfo->max_error_rate = (float)(rates[current].error_rate + 0.01);
			resetGrids = TRUE;
		}
	}
  
	if(resetGrids){
		DrawGridLines(drive, discInfo,
					  resultInfo->max_trans_rate, resultInfo->max_error_rate);
		DrawTransRateLineAll(resultInfo, trackIndex, maxAddress);
		DrawTransRateLineTrack(rates, current,
							   resultInfo->max_trans_rate, maxAddress);
		if(resultInfo->max_error_rate > 0){
			DrawErrorRateLineAll(resultInfo, trackIndex, maxAddress);
			DrawErrorRateLineTrack(rates, current,
								   resultInfo->max_error_rate, maxAddress);
		}
	}
	else{
		if((current > 0) && drawCurrent){
			DrawTransRateLine(&rates[current-1], &rates[current],
							  resultInfo->max_trans_rate, maxAddress);
			if(resultInfo->max_error_rate > 0){
				DrawErrorRateLine(&rates[current-1], &rates[current],
								  resultInfo->max_error_rate, maxAddress);
			}
		}
	}
}

static DWORD GetMilliSec()
{
#if defined(__APPLE__) && defined(__MACH__)
	UnsignedWide gms;
	Microseconds(&gms);
	return gms.lo/1000;
#elif defined(WIN32)
	return timeGetTime();
#else
	struct timeb t;
	ftime(&t);
	return (DWORD)t.time*1000+t.millitm;
#endif
}

static DWORD CountC2ErrorBits(const BYTE *ptr)
{
	DWORD count=0;
	int index;
	const BYTE bitCount[] = {
		0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
		1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
		1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
		1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
		2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
		3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
		3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
		4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8
	};

	for(index=0; index<294; index++, ptr++){
		count += bitCount[*ptr];
	}

	return count;
}

static int GetErrorRate(CMDDRIVE *drive, DWORD lba, DWORD len,
						float *errorRate)
{
	int ret;
	DWORD block;
	BYTE cdb[12];
	DWORD errorCount=0;

	*errorRate = 0.0;
	memset(cdb, 0, sizeof(cdb));
	cdb[0] = CMD_READ_CD;
	Set4bytes(cdb+2, lba);
	Set3bytes(cdb+6, len);
	cdb[9] = 0x04;
	ret = SendCmd(drive, cdb, len*296, REQ_DATAIN);
	if(ret == RET_OK){
		for(block=0; block<len; block++){
#if 0
			if(memcmp(drive->data_buf+294*block+2, g_zero294, 294)!=0){
				errorCount += 2352;
			}
#else
			errorCount += CountC2ErrorBits(drive->data_buf+294*block+2);
#endif
		}
		if(errorCount > 0){
			*errorRate = (float)(100.0*errorCount/(len*2352));
		}
	}

	return ret;
}

static int ReadCheck(CMDDRIVE *drive, DWORD lba, DWORD blocks,
					 DWORD *readTime, float *errorRate, DWORD *lastTime)
{
	int ret;
	DWORD startTime;
	BOOL measureErrorRate=(DT_CD_FAMILY(drive->disc_type) && (errorRate!=NULL));
	DWORD len = 20;
	DWORD totalBlocks = blocks;
	float totalErrorRate = 0;

	if(*lastTime != 0){
		startTime = *lastTime;
	}
	else{
		startTime = GetMilliSec();
	}
  
	while(blocks > 0){
		if(blocks < len){
			len = blocks;
		}
		if(measureErrorRate){
			ret = GetErrorRate(drive, lba, len, errorRate);
			if(ret != RET_OK){
				return ret;
			}
			totalErrorRate += (*errorRate) * len;
		}
		else if(DT_CD_FAMILY(drive->disc_type)){
			ret = SendReadCD(drive, lba, len);
			if(ret != RET_OK){
				return ret;
			}
		}
		else{
			ret = SendRead10(drive, lba, (WORD)len, len*2048);
			if(ret != RET_OK){
				return ret;
			}
		}
	
		lba += len;
		blocks -= len;
	}

	*lastTime = GetMilliSec();
	*readTime = *lastTime-startTime;

	if(measureErrorRate){
		*errorRate = totalErrorRate/totalBlocks;
	}
  
	return RET_OK;
}


static int ReadLoopRange(CMDDRIVE *drive,
						 DISCINFO *discInfo, RESULTINFO *resultInfo,
						 WORD trackIndex, int start_pos,
						 DWORD startLba, DWORD endLba, DWORD readBlockParam,
						 DWORD dummyStart)
{
	int ret;
	DWORD readBlock = readBlockParam;
	DWORD lba = startLba;
	float errorRate=0;
	int baseRateKBS = DT_CD_FAMILY(drive->disc_type) ? 150 : 1385;
	TRACKINFO *trackInfo = &discInfo->tracks[trackIndex];
	TRACKRESULT *trackResult = &resultInfo->track_result[trackIndex];
	RATES *rates = trackResult->rates;
	DWORD readTime, lastTime=0;
	char buf[80];
	int current = start_pos;
  
	ret = ReadCheck(drive, dummyStart,
					((dummyStart < lba) ? lba-dummyStart : 1),
					&readTime, &errorRate, &lastTime);
	if(dummyStart >= lba){
		lastTime=0;
	}
	for(; current<trackResult->num_rates; current++){
		if(UICheckAbort()){
			return RET_ABORT;
		}
		if(lba >= endLba){
			trackResult->num_rates = current;
			break;
		}
	
		if((endLba-lba) < readBlock){
			readBlock = endLba-lba;
		}
		else if((endLba-lba-readBlock) < readBlock/2){
				/*
				 * c肪ʏǂݎTCY̔ɂȂꍇA
				 * cSēǂݍށB
				 */
			readBlock = endLba-lba;
		}
	
		ret = ReadCheck(drive, lba, readBlock, &readTime, &errorRate, &lastTime);
		if((ret != RET_OK) && (ret != RET_CMDERR)){
			return ret;
		}
		rates[current].lba = lba+readBlock/2;
		if(ret == RET_CMDERR){
			rates[current].trans_rate = 0;
			readTime = 0;
			resultInfo->read_error = TRUE;
		}
		else{
			if((current > 0) && (readBlock < readBlockParam/16)){
					/* ǂݎ蕝ꍇ͌덷傫̂Ō떂 */
				rates[current].trans_rate = rates[current-1].trans_rate;
			}
			else{
				rates[current].trans_rate =
					(float)(readBlock*2.0*1000/readTime/baseRateKBS);
			}
		}
		rates[current].error_rate = errorRate;
		if(errorRate > 0){
			resultInfo->c2_error = TRUE;
		}
		UpdateScreen(drive, discInfo,
					 resultInfo, trackIndex, current, (current>start_pos));

			/* ǂݎ茋ʂɂĕ\񕶎쐬 */
		if(ret != RET_OK){
			snprintf(buf, sizeof(buf), MSG_STATUS_STRING_READ_NG_
					 /*"gbN=%d Ԓn=0x%08lX ǂݎ莸s"*/,
					 trackInfo->track, rates[current].lba);
		}
		else{
			snprintf(buf, sizeof(buf), MSG_STATUS_STRING_
					 /*"gbN=%d Ԓn=0x%08lX %.1f{ ǎ掞=%ldms"*/,
					 trackInfo->track,
					 rates[current].lba,
					 rates[current].trans_rate, readTime);
			if(rates[current].error_rate > 0){
				snprintf(buf+strlen(buf), sizeof(buf)-strlen(buf), " C2Error=%.2f",
						 rates[current].error_rate);
			}
		}
	
		UIDispInfo(buf);

		lba += readBlock;
	}

	return RET_OK;
}

static int ReadLoop(CMDDRIVE *drive,
					DISCINFO *discInfo, RESULTINFO *resultInfo,
					WORD trackIndex, DWORD readBlock)
{
	int ret;
	TRACKINFO *trackInfo;
	TRACKRESULT *trackResult, *lastTrackResult;
		/*  DWORD maxAddress = discInfo->last_lba;*/
	DWORD lba, dummyStart;
	int current;
	int rangeIndex;
	struct _LBARANGE {
		DWORD start, end;
		int start_pos;	/* RATESz̈ʒu */
	} lbaRange[3];

	if(trackIndex >= discInfo->num_track){
		return RET_OK;
	}
	if(trackIndex >= resultInfo->num_track){
		return RET_OK;
	}
	trackInfo = &discInfo->tracks[trackIndex];
	trackResult = &resultInfo->track_result[trackIndex];
	if(trackInfo->start_lba >= trackInfo->end_lba){
		return RET_OK;
	}
	lba = trackInfo->start_lba;
  
	memset(g_zero294, 0, sizeof(g_zero294));

	current=0;
	if(trackIndex > 0){
			/* ÕgbNƂȂĂ邩ǂmF */
		if(trackInfo->start_lba == (discInfo->tracks[trackIndex-1].end_lba+1)){
				/*
				 * AREAD\ȂA\͐ŌԂ߂
				 * OgbN̍ŏIʂ̍ŏ̌ʂƂB
				 */
			lastTrackResult = &resultInfo->track_result[trackIndex-1];
			if(lastTrackResult->num_rates > 0){
				memcpy(&trackResult->rates[current],
					   &lastTrackResult->rates[lastTrackResult->num_rates-1],
					   sizeof(RATES));
				current++;
			}
		}
	}

	if(GetOption()->out2in){
		lbaRange[0].start = trackInfo->start_lba;
		lbaRange[0].end =
			(trackInfo->end_lba - trackInfo->start_lba)/3/readBlock*readBlock-1;
		lbaRange[0].start_pos = current;
		lbaRange[1].start = lbaRange[0].end+1;
		lbaRange[1].end =
			(trackInfo->end_lba - trackInfo->start_lba)*2/3/readBlock*readBlock-1;
		lbaRange[1].start_pos = (lbaRange[1].start-trackInfo->start_lba)/readBlock;
		lbaRange[2].start = lbaRange[1].end+1;
		lbaRange[2].end = trackInfo->end_lba;
		lbaRange[2].start_pos = (lbaRange[2].start-trackInfo->start_lba)/readBlock;
			/* overlap */
		lbaRange[0].end += readBlock;
		lbaRange[1].end += readBlock;
	
		for(rangeIndex=2; rangeIndex>=0; rangeIndex--){
			if(rangeIndex == 0){
				dummyStart = lbaRange[rangeIndex].start;
			}
			else if((lbaRange[rangeIndex].start - trackInfo->start_lba) > 0x200){
				dummyStart = lbaRange[rangeIndex].start-0x200;
			}
			else{
				dummyStart = trackInfo->start_lba;
			}
	  
			ret = ReadLoopRange(drive, discInfo, resultInfo, trackIndex,
								lbaRange[rangeIndex].start_pos,
								lbaRange[rangeIndex].start,
								lbaRange[rangeIndex].end,
								readBlock, dummyStart);
			if(ret != RET_OK){
				return ret;
			}
		}
	}
	else{
		ret = ReadLoopRange(drive, discInfo, resultInfo, trackIndex, current,
							trackInfo->start_lba, trackInfo->end_lba, readBlock,
							trackInfo->start_lba);
		if(ret != RET_OK){
			return ret;
		}
	}

	return RET_OK;
}

static int SetReadRetryCount(CMDDRIVE *drive,
							 BYTE count, BYTE *currentCountReturn)
{
	int ret;
	struct _MODEPAGE01 *mp01;

	ret = SendModeSense(drive, MSPC_CURRENT, 1);
	if(ret != RET_OK){
		return ret;
	}
	mp01 = (struct _MODEPAGE01 *)
		(drive->data_buf + (drive->cmd_type==CMDDRVCTYPE_ATAPI ? 8 : 16));
	if(currentCountReturn != NULL){
		*currentCountReturn = mp01->rd_retry_count;
	}
	mp01->rd_retry_count = count;
	ret = SendModeSelect(drive, 1);
	if(ret != RET_OK){
		return ret;
	}

	return RET_OK;
}


/**
 * `FbNs
 * @param[in]	drive	u
 * @param[out]	c2Error	C2G[ǂ
 * @param[out]	readError	ǂݎG[ǂ
 * @retval	RET_OK	I
 * @retval	̑	G[
 */
static int ExecCheck(CMDDRIVE *drive, BOOL *c2Error, BOOL *readError)
{
	int ret;
	int width, height;
	BYTE oldRetryCount;
	DISCINFO *discInfo=NULL;
	RESULTINFO *resultInfo=NULL;
	WORD trackIndex;
	DWORD readBlock;

		/* fBXN\̂̏ */
	ret = MakeDiscInfo(drive, &discInfo);
	if(ret != RET_OK){
		return ret;
	}

	ret = SetOption(drive, discInfo->num_track);
	if(ret != RET_OK){
		discInfo = FreeDiscInfo(discInfo);
		return ret;
	}
  
	readBlock = DT_CD_FAMILY(drive->disc_type) ? 1024 : 4096;
  
		/* vʊi[̈̏ */
	resultInfo = AllocInitResultInfo(discInfo, readBlock);
	if(resultInfo == NULL){
		discInfo = FreeDiscInfo(discInfo);
		return RET_MEMERR;
	}  
  
		/* Yől̏l */
	if(DT_CD_FAMILY(drive->disc_type)){
		resultInfo->max_trans_rate = 24/*16*/;
		resultInfo->max_error_rate = (float)0.01;
	}
	else {
		resultInfo->max_trans_rate = 5;
		resultInfo->max_error_rate = (float)0.0;	/* vȂ */
	}
  
	UIGetDrawableSize(&width, &height);
	g_drawArea.width = width;
	g_drawArea.height = height;

		/* ǎ摬xőݒ */
	if(DT_CD_FAMILY(drive->disc_type)){
		SendSetCdSpeed(drive, 0xffff, 0xffff, 0);
	}

		/* ǎ惊gC񐔐ݒ */
	ret = SetReadRetryCount(drive, GetOption()->read_retry, &oldRetryCount);
	if(ret != RET_OK){
		discInfo = FreeDiscInfo(discInfo);
		resultInfo = FreeResultInfo(resultInfo);
		return ret;
	}

		/* Obh` */
	DrawGridLines(drive, discInfo,
				  resultInfo->max_trans_rate, resultInfo->max_error_rate);

	for(trackIndex=0; trackIndex<discInfo->num_track; trackIndex++){
		ret = ReadLoop(drive, discInfo, resultInfo, trackIndex, readBlock);
		if(ret != RET_OK){
			discInfo = FreeDiscInfo(discInfo);
			resultInfo = FreeResultInfo(resultInfo);
			SetReadRetryCount(drive, oldRetryCount, NULL);
			return ret;
		}
	}

	*c2Error = resultInfo->c2_error;
	*readError = resultInfo->read_error;
  
	discInfo = FreeDiscInfo(discInfo);
	resultInfo = FreeResultInfo(resultInfo);
	SetReadRetryCount(drive, oldRetryCount, NULL); /* ǎ惊gC񐔂߂ */
  
	return RET_OK;
}


static void DispEndMessage(CMDDRIVE *drive, int retcode,
						   BOOL c2Error, BOOL readError)
{
	char *msg;
  
	if(retcode==RET_OK){
		if(readError){
				/*"Ɋ܂AǂݎɎs܂B\n	\
				  sȂꍇ͂Ɠǂݎ萫\̗ǂuŃobNAbv
				  킵܂傤B\
				  sȂȂꍇ͎cOȂx̉\łB";*/
			msg = MSG_COMPLETE_WITH_READ_ERROR;
		}
		else if(c2Error){
			msg = MSG_COMPLETE_WITH_C2_ERROR;
				/*"Ɋ܂Bǂݎ莸s͂܂񂪁AC2G[	\
				  Ă镔܂B				\
				  ԂXɈȂɃobNAbv߂܂B";*/
		}
		else{
			msg = MSG_COMPLETE_WITHOUT_ERROR;
				/*"Ɋ܂BSȂǂݎ肪ł܂B";*/
		}
		UIDispMessage(msg, UIDMT_INFORMATION);
	}
	else if(retcode==RET_ABORT){
		UIDispMessage(MSG_ABORTED/*"f܂B"*/, UIDMT_INFORMATION);
	}
	else if(retcode==RET_CMDERR){
		DispCommandError(drive);
	}
	else{
		UIDispMessage(MSG_ABORTED_BECAUSE_ERROR
					  /*"G[ɂ蒆f܂B"*/, UIDMT_ERROR);
	}
}


int CheckDisc(DRIVEID *driveid)
{
	CMDDRIVE drv;
	int ret;
	BOOL c2Error=FALSE, readError=FALSE;

	if(UICheckAbort()){
		return RET_ABORT;
	}
  
	memset(&drv, 0, sizeof(drv));
	ret = OpenDevice(&drv, driveid, TRUE);
	if(ret != RET_OK){
		if(ret != RET_ABORT){
			UIDispMessage(MSG_DRIVE_INIT_ERROR
						  /*"ȕɎs܂B"*/, UIDMT_ERROR);
		}
		UIDispInfo("");
		return ret;
	}

	ret = ExecCheck(&drv, &c2Error, &readError);
	DispEndMessage(&drv, ret, c2Error, readError);
  
	CloseDevice(&drv);
	UIDispInfo("");
	return RET_OK;
}

/**
 * Volume񕶎A
 */
static char *AppendVolumeInfo(char *string, const char *label,
							  const char *buf, int len)
{
	char *tmp;
	tmp = EbStringGetString(buf, len, ' ');
	if(tmp==NULL){
		return string;
	}

	if(strlen(tmp) == 0){
		free(tmp);
		return string;
	}

	if(string!=NULL){
		if(strlen(string) > 0){
			string = EbStringAppend(string, "\n");
		}
	}
	string = EbStringAppend(string, label);
	string = EbStringAppend(string, "=");
	string = EbStringAppend(string, tmp);
	tmp = EbStringFree(tmp);

	return string;
}

/**
 * VolumeA
 * `:YYYYmmddHHMMSS
 * TZ: -48(-1200)`52(+1300)
 */
static char *AppendVolumeInfoDateTime(char *string, const char *label,
									  const char *buf)
{
	char *tmp;
	char tz[6];
	char datetime[23]; /* YYYY/mm/dd HH:MM:SS.FF */
	int n;

	tmp = EbStringGetString(buf, 17, '\0');
	if(tmp==NULL){
		return string;
	}

	if(string!=NULL){
		if(strlen(string) > 0){
			string = EbStringAppend(string, "\n");
		}
	}
	string = EbStringAppend(string, label);
	string = EbStringAppend(string, "=");
  
		/* date & time */
	memcpy(datetime+0, tmp+0, 4);
	datetime[4] = '/';
	memcpy(datetime+5, tmp+4, 2);
	datetime[7] = '/';
	memcpy(datetime+8, tmp+6, 2);
	datetime[10] = ' ';
	memcpy(datetime+11, tmp+8, 2);
	datetime[13] = ':';
	memcpy(datetime+14, tmp+10, 2);
	datetime[16] = ':';
	memcpy(datetime+17, tmp+12, 2);
	datetime[19] = '.';
	memcpy(datetime+20, tmp+14, 2);
	datetime[22] = '\0';

		/* TZ */
	tz[0] = (tmp[16] < 0) ? '-' : '+';
	n = (int)((tmp[16] < 0) ? -tmp[16] : tmp[16]);
	snprintf(tz+1, 5, "%02d%02d", n/4, (n%4)*15);
	string = EbStringAppend(string, datetime);
	string = EbStringAppend(string, tz);
	tmp = EbStringFree(tmp);

	return string;
}
    

/**
 * Volume擾
 */
static char *GetVolume(CMDDRIVE *drive, WORD track, DWORD start_lba)
{
	int ret;
	char *vol=NULL;

	ret = SendRead10(drive, start_lba+16, 1, 2048);
	if(ret != RET_OK){
		return NULL;
	}
	if(drive->data_buf[0] != 0x01){
		return NULL;
	}
	if(strncmp(drive->data_buf+1, "CD001", 5) != 0){
		return NULL;
	}

		/* System */
	vol = AppendVolumeInfo(vol, MSG_SYSTEM, drive->data_buf+0x08, 0x20);
		/* Volume */
	vol = AppendVolumeInfo(vol, MSG_VOLUME, drive->data_buf+0x28, 0x20);
		/* {[W */
	vol = AppendVolumeInfo(vol, MSG_VOLUMESET, drive->data_buf+0xbe, 0x80);
		/* oŎ */
	vol = AppendVolumeInfo(vol, MSG_PUBLISHER, drive->data_buf+0x13e, 0x80);
		/* ҏW */
	vol = AppendVolumeInfo(vol, MSG_PREPARER, drive->data_buf+0x1be, 0x80);
		/* pVXe */
	vol = AppendVolumeInfo(vol, MSG_APPLICATION, drive->data_buf+0x23e, 0x80);
		/* 쐬 */
	vol = AppendVolumeInfoDateTime(vol, MSG_CREATION_DATE_TIME, drive->data_buf+0x32d);
		/* ̑ */
	vol = AppendVolumeInfo(vol, MSG_OTHER, drive->data_buf+0x374, 0x1ff);

	return vol;
}

/**
 * f[^[h擾
 */
static char *GetDataMode(CMDDRIVE *drive, TRACKINFO *trackInfo)
{
	int ret;
	char *mode=NULL;

	if((trackInfo->data_mode==1) || (trackInfo->data_mode==2)){
		mode = EbStringNewWithFormat("Mode%d", trackInfo->data_mode);
	}
	else if(DT_CD_FAMILY(drive->disc_type)){
		ret = SendReadCD(drive, trackInfo->start_lba, 1);
		if(ret == RET_OK){
			mode = EbStringNewWithFormat("Mode%d", drive->data_buf[15]);
		}
	}
  
	if(mode == NULL){
		mode = EbStringNew("Data");
	}

	return mode;
}


/**
 * ISRC擾
 */
static char *GetISRC(CMDDRIVE *drive, BYTE track)
{
	int ret;
	char *isrc;

	ret = SendReadSubchannel(drive, track, 0, 1, 3/*ISRC*/);
	if(ret != RET_OK){
		return NULL;
	}
	isrc = (char *)malloc(13);
	if(isrc == NULL){
		UIDispMessage(MSG_MEM_ALLOC_ERROR, UIDMT_ERROR);
		return NULL;
	}
	memcpy(isrc, drive->data_buf+9, 12);
	isrc[12] = '\0';

	return isrc;
}

/**
 * hCufBXN擾ĕ\
 */
int DisplayDiscInformation(DRIVEID *driveid)
{
	CMDDRIVE drv;
	DISCINFO *discInfo=NULL;
	UITRACKINFO *uiTrackInfo=NULL, *uiInfo;
	TRACKINFO *trackInfo;
	WORD trackIdx;
	int ret;
	BOOL aborted=FALSE;

	if(UICheckAbort()){
		return RET_ABORT;
	}

	memset(&drv, 0, sizeof(drv));
	ret = OpenDevice(&drv, driveid, TRUE);
	if(ret != RET_OK){
		if(ret != RET_ABORT){
			UIDispMessage(MSG_DRIVE_INIT_ERROR, UIDMT_ERROR);
		}
		UIDispInfo("");
		return ret;
	}

	ret = MakeDiscInfo(&drv, &discInfo);
	if(ret != RET_OK){
		CloseDevice(&drv);
		UIDispInfo("");
		return ret;
	}

	uiTrackInfo = (UITRACKINFO *)malloc(discInfo->num_track *
										sizeof(UITRACKINFO));
	if(uiTrackInfo == NULL){
		UIDispMessage(MSG_MEM_ALLOC_ERROR, UIDMT_ERROR);
		FreeDiscInfo(discInfo);
		CloseDevice(&drv);
		UIDispInfo("");
		return RET_NG;
	}

	memset(uiTrackInfo, 0, discInfo->num_track * sizeof(UITRACKINFO));
	for(trackIdx=0; trackIdx<discInfo->num_track; trackIdx++){
		UIDispInfo(MSG_READING_TRACK_, trackIdx+1, discInfo->num_track);
		if(UICheckAbort()){
			aborted = TRUE;
			break;
		}
		
		uiInfo = &uiTrackInfo[trackIdx];
		trackInfo = &discInfo->tracks[trackIdx];
		if(trackInfo->blank){
			continue;
		}
		uiInfo->session = trackInfo->session;
		uiInfo->track = trackInfo->track;
		uiInfo->start_lba = trackInfo->start_lba;
		uiInfo->end_lba = trackInfo->end_lba;
		uiInfo->blocks = trackInfo->blocks;
		uiInfo->packet_size = trackInfo->packet_size;
		if((trackInfo->track_mode & 0x0c) == 4){
				/* Data Track */
			char *vol;
			uiInfo->mode = GetDataMode(&drv, trackInfo);
			vol = GetVolume(&drv, uiInfo->track, trackInfo->start_lba);
			if(vol != NULL){
				uiInfo->comment = EbStringNew(vol);
				free(vol);
			}
		}
		else{
				/* Audio Track */
			DWORD blocks = uiInfo->end_lba-uiInfo->start_lba+1;
			char *isrc;
			uiInfo->mode = EbStringNew("Audio");
			uiInfo->comment = EbStringNewWithFormat("Time=%ld:%02ld",
													blocks/75/60,
													(blocks/75)%60);
			isrc = GetISRC(&drv, (BYTE)uiInfo->track);
			if(isrc != NULL){
				uiInfo->comment = EbStringAppendWithFormat(uiInfo->comment,
														   "\nISRC=%s", isrc);
				free(isrc);
			}
		}
    
			/* Damage? */
		if(trackInfo->damage){
			if(uiInfo->comment != NULL){
				uiInfo->comment = EbStringAppend(uiInfo->comment, " ");
			}
			uiInfo->comment = EbStringAppend(uiInfo->comment, "Damage!!");
		}
	}

	UIDispInfo("");
	if(aborted == FALSE){
		UIDispDiscInfo(uiTrackInfo, (int)discInfo->num_track);
	}

	for(trackIdx=0; trackIdx<discInfo->num_track; trackIdx++){
		EbStringFree(uiTrackInfo[trackIdx].mode);
		EbStringFree(uiTrackInfo[trackIdx].comment);
	}
	free(uiTrackInfo);
	uiTrackInfo = NULL;

	discInfo = FreeDiscInfo(discInfo);
	CloseDevice(&drv);
	UIDispInfo("");
	return RET_OK;
}
