/* ~Օʉ
 * Copyright (C) 2004 Kagetani Hideto
 */
#include <stdio.h>
#include <time.h>
#include <stddef.h>

#define ASPIBUF_CHECK 1

#include "aspi.h"
#include "cmd.h"
#include "cmdlog.h"
#include "scsipt.h"

extern void DebugLog(char *fmt, ...);


static HINSTANCE hAspiInstance=NULL;
static int g_NumHa=0;


typedef void *LPSRB;

DWORD	(*pfnGetASPI32SupportInfo)(void);
DWORD	(*pfnSendASPI32Command)(LPSRB);
BOOL	(*pfnGetASPI32Buffer)(PASPI32BUFF);
BOOL	(*pfnFreeASPI32Buffer)(PASPI32BUFF);
BOOL	(*pfnTranslateASPI32Address)(PDWORD, PDWORD);

#if USE_SCSIPT
static BOOL g_bUseScsiPT=FALSE;
#endif


static void DispASPI32Error(BYTE status)
{
	char *message = NULL;
	switch(status){
	case SS_ABORTED:				message = "SRB aborted";	break;
	case SS_ABORT_FAIL:				message = "Unable to abort SRB";	break;
	case SS_ERR:					message = "SRB completed with error";	break;
	case SS_ABORTED_QUEUE_FROZEN:	message = "SRB aborted, target queue frozen";	break;
	case SS_ERR_QUEUE_FROZEN:		message = "SRB completed with error, target queue frozen";	break;
	case SS_BUFFTOBIG_QUEUE_FROZEN:	message = "SRB buffer too big, target queue frozen";	break;
	case SS_INVALID_CMD:			message = "Invalid ASPI command";	break;
	case SS_INVALID_HA:				message = "Invalid host adapter number";	break;
	case SS_NO_DEVICE:				message = "SCSI device not installed";	break;
	case SS_INVALID_SRB:			message = "Invalid parameter set in SRB";	break;
	case SS_BUFFER_ALIGN:			message = "Buffer not aligned (replaces OLD_MANAGER in Win32)";	break;
	case SS_ILLEGAL_MODE:			message = "Unsupported Windows mode";	break;
	case SS_NO_ASPI:				message = "No ASPI managers resident";	break;
	case SS_FAILED_INIT:			message = "ASPI for windows failed init";	break;
	case SS_ASPI_IS_BUSY:			message = "No resources available to execute cmd";	break;
	case SS_BUFFER_TOO_BIG:			message = "Buffer size toobig to handle!";	break;
	case SS_MISMATCHED_COMPONENTS:	message = "The DLLs/EXEs of ASPI don't version check";	break;
	case SS_NO_ADAPTERS:			message = "No host adapters to manage";	break;
	case SS_INSUFFICIENT_RESOURCES:	message = "Couldn't allocate resources needed to init";	break;
	case SS_ASPI_IS_SHUTDOWN:		message = "Call came to ASPI after PROCESS_DETACH";	break;
	case SS_BAD_INSTALL:			message = "The DLL or other components are installed wrong";	break;
	default:						message = "Unknown error";
	}
	MessageBox(NULL, message, NULL, MB_ICONERROR);
}

#if USE_SCSIPT
static BOOL IsPlatformWinNT()
{
	OSVERSIONINFO osver;

	ZeroMemory(&osver, sizeof(OSVERSIONINFO));
	osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if(GetVersionEx(&osver)==FALSE){
		return FALSE;
	}
	return osver.dwPlatformId==VER_PLATFORM_WIN32_NT;
}
#endif /* USE_SCSIPT */

int OpenAspi()
{
	DWORD dwParam;

#if USE_SCSIPT
	if(IsPlatformWinNT()){
		if(strstr(GetCommandLine(), "-aspi")==NULL){
			g_bUseScsiPT=TRUE;
			g_NumHa = 1;
			return 0;
		}
	}
#endif /* USE_SCSIPT */

	hAspiInstance = LoadLibrary("wnaspi32.dll");
	if(hAspiInstance==NULL){
		MessageBox(NULL, "Cannot load wnaspi32.dll", NULL, MB_ICONERROR);
		return -1;
	}

	*(FARPROC *)&pfnGetASPI32SupportInfo	= GetProcAddress(hAspiInstance, "GetASPI32SupportInfo");
	*(FARPROC *)&pfnSendASPI32Command	= GetProcAddress(hAspiInstance, "SendASPI32Command");
	*(FARPROC *)&pfnGetASPI32Buffer		= GetProcAddress(hAspiInstance, "GetASPI32Buffer");
	*(FARPROC *)&pfnFreeASPI32Buffer		= GetProcAddress(hAspiInstance, "FreeASPI32Buffer");
	*(FARPROC *)&pfnTranslateASPI32Address = GetProcAddress(hAspiInstance, "TranslateASPI32Address");

	dwParam = pfnGetASPI32SupportInfo();
	switch(HIBYTE(LOWORD(dwParam))){
	case SS_COMP:
		g_NumHa = LOBYTE(LOWORD(dwParam));
		break;
	default:
		DispASPI32Error(HIBYTE(LOWORD(dwParam)));
		FreeLibrary(hAspiInstance);
		return -1;
	}

	return 0;
}


void CloseAspi()
{
	if(hAspiInstance!=NULL){
		FreeLibrary(hAspiInstance);
		hAspiInstance = NULL;
	}
#if CMDLOGFILE
	if(g_CmdLogFP!=NULL)
		fclose(g_CmdLogFP);
#endif
}

int InitializeDrive(DRIVE *drive, int bufsize, int cmdlog_size)
{
	FreeDrive(drive);
	drive->hid = -1;
	drive->tid = -1;

	if(CmdLogInit(drive, cmdlog_size)!=RET_OK){
		return RET_NG;
	}

	if(drive->data_buf!=NULL){
		free(drive->data_buf);
		drive->data_buf = NULL;
	}
	if(bufsize>0){
		drive->data_buf = (BYTE *)malloc(bufsize);
		if(drive->data_buf==NULL){
			return RET_NG;
		}
		drive->bufsize = bufsize;
	}

#if USE_SCSIPT
	drive->hDrive=NULL;
	if(g_bUseScsiPT){
		drive->srb=NULL;
	}
	else{
#endif
		drive->srb = (SRB32_ExecSCSICmd *)malloc(sizeof(SRB32_ExecSCSICmd)+SENSEDATA_SIZE);
		if(drive->srb==NULL){
			free(drive->data_buf);
			drive->data_buf=NULL;
			free(drive->cmdlog);
			drive->cmdlog=NULL;
			return RET_NG;
		}
		memset(drive->srb, 0, sizeof(SRB32_ExecSCSICmd)+SENSEDATA_SIZE);
#if USE_SCSIPT
	}
#endif

	return RET_OK;
}

void FreeDrive(DRIVE *drive)
{
	CmdLogFree(drive);
	if(drive->data_buf!=NULL){
		free(drive->data_buf);
		drive->data_buf=NULL;
	}
	if(drive->srb!=NULL){
		free(drive->srb);
		drive->srb=NULL;
	}
#if USE_SCSIPT
	if(drive->hDrive!=NULL){
		DebugLog("CloseHandle : 0x%08lX\n", drive->hDrive);
		CloseHandle(drive->hDrive);
		drive->hDrive=NULL;
	}
#endif
}

static void CheckAtapi(DRIVE *drive)
{
	BYTE cdb[12];
	int ret;

	memset(cdb, 0, sizeof(cdb));
	cdb[0] = CMD_MODE_SENSE10;
	cdb[2] = 0x2a;
	cdb[8] = 12;
	ret = SendCmd(drive, cdb, 12, REQ_DATAIN);
	if(ret!=RET_OK){
		drive->atapi = FALSE;
	}
	else{
		drive->atapi = drive->data_buf[8]==0x2a;
	}
}


int SetAspiSetting(DRIVE *drive, int *ha, int *tg, DWORD *timeout)
{
	BOOL bChanged=FALSE;

	if(ha!=NULL){
		if(drive->hid != *ha){
			drive->hid = *ha;
			bChanged = TRUE;
		}
	}
	if(tg!=NULL){
		if(drive->tid != *tg){
			drive->tid = *tg;
			bChanged = TRUE;
		}
	}
	if(timeout!=NULL){
		drive->timeout = *timeout;
	}

	if(bChanged){
#if USE_SCSIPT
		if(g_bUseScsiPT){
			char cDriveLetter[6+1];
			if(drive->hDrive!=NULL){
				DebugLog("CloseHandle : 0x%08lX\n", drive->hDrive);
				CloseHandle(drive->hDrive);
				drive->hDrive = NULL;
			}
			sprintf(cDriveLetter, "\\\\.\\%c:", 'C'+drive->tid);
			drive->hDrive = CreateFile(cDriveLetter, GENERIC_WRITE|GENERIC_READ,
								FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
			DebugLog("CreateHandle : 0x%08lX\n", drive->hDrive);
			if(drive->hDrive==INVALID_HANDLE_VALUE){
				drive->hDrive=NULL;
				return RET_NG;
			}
		}
#endif	/* USE_SCSIPT */
		CheckAtapi(drive);
	}
	return RET_OK;
}

void GetAspiSetting(DRIVE *drive, int *ha, int *tg, DWORD *timeout)
{
	if(ha!=NULL){
		*ha = drive->hid;
	}
	if(tg!=NULL){
		*tg = drive->tid;
	}
	if(timeout!=NULL){
		*timeout = drive->timeout;
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////
//	R}hR[h(cdb0)ACDBQbg
///////////////////////////////////////////////////////////////////////////////////////////////
static BYTE GetCDBLen( BYTE cdb0 )
{
	BYTE len=0;

	switch(cdb0&0xe0)
	{
	case 0x00:  // 000X XXXX
		len = 6;  break;
	case 0x20:  // 001X XXXX
	case 0x40:  // 010X XXXX
	case 0x60:	// 011X XXXX
		len = 10;  break;
	case 0x80:  // 100X XXXX
	case 0xA0:  // 101X XXXX
	case 0xC0:  // 110X XXXX
	case 0xE0:  // 111X XXXX
		len = 12;  break;
	}

	return len;
}

int SendCmd(DRIVE *drive, BYTE *cdb, DWORD buflen, BYTE reqflag)
{
	long starttime, nowtime;
	int retcode = RET_CMDERR;
	SRB32_ExecSCSICmd *srb;
	DWORD timeout;

#if ASPIBUF_CHECK
	if(buflen > (DWORD)drive->bufsize){
		CmdLog(drive, cdb, buflen, reqflag, RET_MEMERR);
		return RET_MEMERR;
	}
#endif

#if USE_SCSIPT
	if(g_bUseScsiPT){
		SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER swb;
		BOOL bStatus;
		ULONG ulReturned;
		ZeroMemory(&swb, sizeof(swb));
		swb.spt.Length = sizeof(SCSI_PASS_THROUGH_DIRECT);
		swb.spt.CdbLength = GetCDBLen(cdb[0]);
		swb.spt.SenseInfoLength = SENSEDATA_SIZE;
		switch(reqflag){
		case REQ_DATAOUT:	swb.spt.DataIn=SCSI_IOCTL_DATA_OUT;	break;
		case REQ_DATAIN:	swb.spt.DataIn=SCSI_IOCTL_DATA_IN;	break;
		default:			swb.spt.DataIn=SCSI_IOCTL_DATA_UNSPECIFIED;	break;
		}
		swb.spt.DataTransferLength = buflen;
		swb.spt.TimeOutValue = drive->timeout==0 ? 1 : drive->timeout;
		swb.spt.DataBuffer = buflen==0 ? NULL : drive->data_buf;
		swb.spt.SenseInfoOffset = offsetof(SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER, ucSenseBuf);
		CopyMemory(swb.spt.Cdb, cdb, 12);
		bStatus = DeviceIoControl(drive->hDrive,
				IOCTL_SCSI_PASS_THROUGH_DIRECT,
				&swb, sizeof(SCSI_PASS_THROUGH_DIRECT),
				&swb, sizeof(swb),
				&ulReturned, NULL);
		CopyMemory(drive->sense_data, swb.ucSenseBuf, SENSEDATA_SIZE);
		if(bStatus!=FALSE && swb.spt.ScsiStatus==0){
			if(SD_SENSEKEY(drive)==0 &&
				SD_ASC(drive)==0x00 && SD_ASCQ(drive)==0x00){
				retcode = RET_OK;
			}
		}
	}
	else{
#endif /* USE_SCSIPT */
		drive->cmdcode = cdb[0];
		srb = drive->srb;
		time(&starttime);
		srb->SRB_Cmd        = SC_EXEC_SCSI_CMD;
		srb->SRB_HaId       = drive->hid;
		srb->SRB_Hdr_Rsvd   = 0;
		srb->SRB_Target     = drive->tid;
		srb->SRB_Lun        = 0;
		srb->SRB_BufLen     = buflen;
		srb->SRB_SenseLen   = SENSEDATA_SIZE;
		srb->SRB_BufPointer = drive->data_buf;
		srb->SRB_PostProc   = NULL;

		switch(reqflag){
		case REQ_DATAOUT:	srb->SRB_Flags = SRB_DIR_OUT;	break;
		case REQ_DATAIN:	srb->SRB_Flags = SRB_DIR_IN;	break;
		default:	srb->SRB_Flags = SRB_DIR_IN|SRB_DIR_OUT;
		}

		if(srb->SRB_Flags==(SRB_DIR_IN|SRB_DIR_OUT)){
			srb->SRB_BufLen = 0;
			srb->SRB_BufPointer = NULL;
		}

		srb->SRB_CDBLen = GetCDBLen(cdb[0]);

		memcpy( srb->CDBByte, cdb, 12 );
		memset( srb->SenseArea, 0x00, SENSEDATA_SIZE);

		pfnSendASPI32Command(srb);
		SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_LOWEST);	/* D揇ʂ */
		if(drive->timeout==0) timeout=1;
		else timeout = drive->timeout;

		while(srb->SRB_Status==SS_PENDING){
			time(&nowtime);
			if(nowtime>=(starttime+(long)timeout)){
				retcode = RET_TIMEOUT;
				break;
			}
		}
		SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_NORMAL);	/* D揇ʂ߂ */
		if(srb->SRB_Status==SS_COMP && (srb->SRB_HaStat==0 || srb->SRB_HaStat==0x12)){
			retcode = RET_OK;
		}
		memcpy(drive->sense_data, srb->SenseArea, SENSEDATA_SIZE);
#if USE_SCSIPT
	}
#endif /* USE_SCSIPT */

	CmdLog(drive, cdb, buflen, reqflag, retcode);

	return retcode;
}


int GetHostAdapterCount()
{
	return g_NumHa;
}

int GetMaxTarget(void)
{
#if USE_SCSIPT
	return g_bUseScsiPT ? 26-2 : 16;
#else
	return 16;
#endif
}

BOOL CheckDiscDevice(DRIVE *drive)
{
	SRB32_GDEVBlock GDB;

#if USE_SCSIPT
	if(g_bUseScsiPT){
		return TRUE;
	}
#endif
	memset(&GDB, '\0', sizeof(GDB));
	GDB.SRB_Cmd = SC_GET_DEV_TYPE;
	GDB.SRB_HaId = drive->hid;
	GDB.SRB_Target = drive->tid;

	pfnSendASPI32Command(&GDB);

	drive->hid = GDB.SRB_HaId;
	drive->tid = GDB.SRB_Target;

	if(GDB.SRB_Status!=SS_COMP){
		return FALSE;
	}

	return (GDB.SRB_DeviceType==DTYPE_WORM ||
			GDB.SRB_DeviceType==DTYPE_CDROM ||
			GDB.SRB_DeviceType==DTYPE_OPTI);
}

BOOL UsingSPTI()
{
#if USE_SCSIPT
	return g_bUseScsiPT;
#else
	return FALSE;
#endif
}
