/*
# mrcImageShapeSearch : $Revision$  
# $Date$ 
# Created by $Author$
# Usage : mrcImageShapeSearch
# Attention
#   $Loccker$
#  	$State$ 
#
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>                  
#define GLOBAL_DECLARATION
#include "../inc/config.h"

#define DEBUG
#include "genUtil.h"
#include "Memory.h"
#include "Matrix3D.h"
#include "Vector.h"
#include "mrcImage.h"

typedef struct lmrcImageShapeSearchInfo {
	float radius; // Sylinder, half disk, sphere   
	float minRadius;
	float maxRadius;
	float delRadius;

	float length; // Sylinder
	float minLength;
	float maxLength;
	float delLength;

	float minTheta;
	float maxTheta;
	float delTheta;

	float minPhi;
	float maxPhi;
	float delPhi;

	float minPsi;
	float maxPsi;
	float delPsi;

	// temporary
	float x;
	float y;
	float z;

	int interpMode;
	int thresZscore;

	mrcImage shape; // Template Structure 

	// Output
	mrcImage average; // Average for all orientations
	mrcImage SD;      // SD for all 
	mrcImage Max;     // Max for all 
	mrcImage Zscore;  // Z-score;
	mrcImage PCA;     // PCA
	int nShapeInfo;
	mrcImage* shapeInfo; // Shape, Orientation, ... 
} lmrcImageShapeSearchInfo;

typedef enum lmrcImageShapeSearchMode {
	lmrcImageShapeSearchModeSylinder=0,
	lmrcImageShapeSearchModeDisk=1,
	lmrcImageShapeSearchModeSphere=2
} lmrcImageShapeSearchMode;

extern void	lmrcImageShapeSearch(mrcImage* out, mrcImage* in, lmrcImageShapeSearchInfo* linfo, int mode);
extern void	lmrcImageShapeSearchCalc0(mrcImage* out, mrcImage* in, lmrcImageShapeSearchInfo* linfo, int mode);
extern void	lmrcImageShapeSearchSylinder(double* data, mrcImage* in, Matrix3D mat, lmrcImageShapeSearchInfo* linfo, int mode);
extern void	lmrcImageShapeSearchDisk(double* data, mrcImage* in, Matrix3D mat, lmrcImageShapeSearchInfo* linfo, int mode);
extern void	lmrcImageShapeSearchSphere(double* data, mrcImage* in, lmrcImageShapeSearchInfo* linfo, int mode);
extern void lmrcImageShapeSearchModePrint(FILE* fpt);

int
main(int argc, char* argv[]) 
{
	mrcImageShapeSearchInfo info;
	lmrcImageShapeSearchInfo linfo;
	mrcImage in;
	mrcImage out;
	char filename[1024];
	int i;

	init0(&info);
    argCheck(&info, argc, argv);
    init1(&info);

	DEBUGPRINT("Program Start\n");
	linfo.minRadius = info.minR;
	linfo.maxRadius = info.maxR;
	linfo.delRadius = info.delR;
	//
	linfo.minLength = info.minL;
	linfo.maxLength = info.maxL;
	linfo.delLength = info.delL;
	//	
	linfo.minPhi   = info.minPhi*RADIAN;
	linfo.maxPhi   = info.maxPhi*RADIAN;
	linfo.delPhi   = info.delPhi*RADIAN;
	//
	linfo.minTheta = info.minTheta*RADIAN;
	linfo.maxTheta = info.maxTheta*RADIAN;
	linfo.delTheta = info.delTheta*RADIAN;
	//
	linfo.minPsi = info.minPsi*RADIAN;
	linfo.maxPsi = info.maxPsi*RADIAN;
	linfo.delPsi = info.delPsi*RADIAN;
	//
	linfo.thresZscore = info.thresZscore;
	linfo.interpMode = info.interpMode;

	mrcFileRead(&in, info.In, "in main", 0); 

	lmrcImageShapeSearch(&out, &in, &linfo, info.mode);

	mrcFileWrite(&out, info.Out, "in main", 0);
	sprintf(filename, "%s.average", info.Out);	
	mrcFileWrite(&linfo.average, filename, "in main", 0);
	sprintf(filename, "%s.sd", info.Out);	
	mrcFileWrite(&linfo.SD, filename, "in main", 0);
	sprintf(filename, "%s.zscore", info.Out);	
	mrcFileWrite(&linfo.Zscore, filename, "in main", 0);
	sprintf(filename, "%s.max", info.Out);	
	mrcFileWrite(&linfo.Max, filename, "in main", 0);
	sprintf(filename, "%s.PCA", info.Out);	
	mrcFileWrite(&linfo.PCA, filename, "in main", 0);
	for(i=0; i<linfo.nShapeInfo; i++) {
		sprintf(filename, "%s.shapeinfo-%02d", info.Out, i);	
		mrcFileWrite(&linfo.shapeInfo[i], filename, "in main", 0);
	}
	//mrcFileWrite(&linfo.shape, info.Shape, "in main", 0);

	exit(EXIT_SUCCESS);
}

void
additionalUsage()
{
	fprintf(stderr, "----- Additional Usage -----\n");
	lmrcImageShapeSearchModePrint(stderr);
}

void
lmrcImageShapeSearchModePrint(FILE* fpt)
{
	fprintf(fpt, "%d: Cylinder with radius and length\n", lmrcImageShapeSearchModeSylinder);
	fprintf(fpt, "%d: Disk with radius and length(thickness)\n", lmrcImageShapeSearchModeDisk);
	fprintf(fpt, "%d: Sphere with radius \n", lmrcImageShapeSearchModeSphere);
}


void
lmrcImageShapeSearch(mrcImage* out, mrcImage* in, lmrcImageShapeSearchInfo* linfo, int mode)
{
	switch(mode) {
		case lmrcImageShapeSearchModeSylinder: {
			lmrcImageShapeSearchCalc0(out, in, linfo, mode);
			break;
		}
		case lmrcImageShapeSearchModeDisk: {
			lmrcImageShapeSearchCalc0(out, in, linfo, mode);
			break;
		}
		case lmrcImageShapeSearchModeSphere: {
			lmrcImageShapeSearchCalc0(out, in, linfo, mode);
			break;
		}
		default: {
			fprintf(stderr, "Not supported mode: %d\n", mode);
			break;
		}
	}
}

// Sylindar
void
lmrcImageShapeSearchCalc0(mrcImage* out, mrcImage* in, lmrcImageShapeSearchInfo* linfo, int mode)
{
	float x, y, z;
	float srcx, srcy, srcz;
	float srcx0, srcy0, srcz0;
	float dstx, dsty, dstz;
	float theta, phi, psi, radius, length;
	int iPhi, iTheta, iPsi, iRadius, iLength;	
	int nPhi, nTheta, nPsi, nRadius, nLength;	
	float max, ave, sum, sd, zscore, min;
	double* data;
	double  d, cp, sp;
	double score;
	int nData, index;
	float r, l, p;
	Matrix3D mat;
	floatVector v;
	int i, imax;
	int k, k0, nVoxel;
	int PCAMode=0;
	int numPoint;
	Array U;
	Array C;
	Array Lambda;
	double L0, L1, L2;
	Array X;
	Array Ave;
	mrcImageParaTypeInteger nCube; 
	double W;

	nPhi   = (int)((linfo->maxPhi     - linfo->minPhi)  /linfo->delPhi)  +1;
	nTheta = (int)((linfo->maxTheta   - linfo->minTheta)/linfo->delTheta)+1;
	nPsi= (int)((linfo->maxTheta   - linfo->minTheta)/linfo->delTheta)+1;
	nRadius = (int)((linfo->maxRadius - linfo->minRadius)/linfo->delRadius) + 1;
	nLength = (int)((linfo->maxLength - linfo->minLength)/linfo->delLength) + 1;
	
	nData = nPhi*nTheta*nPsi*nRadius*nLength;
	data = (double*)memoryAllocate(sizeof(double)*nData, "in lmrcImageShapeSearchSylinder");

	floatVectorInit(&v, 4);
	v.data[3] = 1;	

	out->Header = in->Header;
	mrcInit(out, NULL);
	linfo->average.Header = in->Header;
	mrcInit(&linfo->average, NULL);
	linfo->SD.Header = in->Header;
	mrcInit(&linfo->SD, NULL);
	linfo->Zscore.Header = in->Header;
	mrcInit(&linfo->Zscore, NULL);
	linfo->Max.Header = in->Header;
	mrcInit(&linfo->Max, NULL);
	linfo->PCA.Header = in->Header;
	mrcInit(&linfo->PCA, NULL);
	linfo->nShapeInfo = 5;
	linfo->shapeInfo = (mrcImage*)memoryAllocate(sizeof(mrcImage)*linfo->nShapeInfo, "in MFPCalc");
	for(i=0; i<linfo->nShapeInfo; i++) {
		linfo->shapeInfo[i].Header = in->Header;
		mrcInit(&linfo->shapeInfo[i], NULL);
	}

	X.dim = 2;
	X.n[0] = 3;
	nCube = ((int)(MAX(linfo->maxLength, 2*linfo->maxRadius+1)+3)/2)*2; 
	X.n[1] = nCube*nCube*nCube; 
	arrayInit(&X, "in");
	
	DEBUGPRINT1("nCube: %d\n", nCube);

	for(z=0; z<in->HeaderN.z; z++) {
		DEBUGPRINT1("z: %f\n", z);
		linfo->z = z;
	for(y=0; y<in->HeaderN.y; y++) {
		//DEBUGPRINT1("y: %f\n", y);
		linfo->y = y;
	for(x=0; x<in->HeaderN.x; x++) { // Each voxel
		linfo->x = x;

		k0 = (int)(x + y*in->HeaderN.x + z*in->HeaderN.x*in->HeaderN.y); 

		//mrcPixelDataGet(in, x, y, z, &d, mrcPixelRePart, mrcPixelHowNearest);
		k = x + y*in->HeaderN.x + z*in->HeaderN.x*in->HeaderN.y;
		d = in->ImageFloatImage[k];
		if(0<d) {
		numPoint = 0;
		for(srcz=-nCube/2; srcz<nCube/2; srcz++) {
		for(srcy=-nCube/2; srcy<nCube/2; srcy++) {
		for(srcx=-nCube/2; srcx<nCube/2; srcx++) {
			//mrcPixelDataGet(in, x+srcx, y+srcy, z+srcz, &d, mrcPixelRePart, mrcPixelHowNearest);	
			srcx0 = x + srcx;
			srcy0 = y + srcy;
			srcz0 = z + srcz;
			k = srcx0 + srcy0*in->HeaderN.x + srcz0*in->HeaderN.x*in->HeaderN.y;
			d = in->ImageFloatImage[k];
			if(0<d) {
				arrayDataSet2(X, 0, numPoint, srcx0);
				arrayDataSet2(X, 1, numPoint, srcy0);
				arrayDataSet2(X, 2, numPoint, srcz0);
				numPoint++;
			}
		}
		}
		}
		//DEBUGPRINT4("%f %f %f numPoint: %d\n", x, y, z, numPoint);
		if(16<numPoint) {
			X.n[1] = numPoint;
			arrayPCA(&U, &C, &Lambda, &X, &Ave, PCAMode);
			if(PCAMode==0) PCAMode=1;
			L0 = arrayDataGet1(Lambda, 0);
			L1 = arrayDataGet1(Lambda, 1);
			L2 = arrayDataGet1(Lambda, 2);

			switch(mode) {
				case lmrcImageShapeSearchModeSylinder: {	
					W = ((L1!=0)?(L0/L1):1)*(L2!=0?(L0/L2):1);		
#ifdef DEBUG
					if(4000<W) {
						DEBUGPRINT3("%f %f %f\n", x, y, z);
						DEBUGPRINT4("W: %f %f %f %f\n", W, L0, L1, L2);
					}
#endif
					break;
				}
				case lmrcImageShapeSearchModeDisk: {	
					W = ((L2!=0)?((L0/L2)*(L1/L2)):1);		
#ifdef DEBUG
					if(1000<W) {
						DEBUGPRINT3("%f %f %f\n", x, y, z);
						DEBUGPRINT4("W: %f %f %f %f\n", W, L0, L1, L2);
					}
#endif
					break;
				}
				case lmrcImageShapeSearchModeSphere: {	
					W = L0*L1+L1*L2+L2*L0/SQR(L0+L1+L2);		
					break;
				}
				default: {
					W = 1;
					break;
				}
			}
		} else {
			W = 1;
		}
		} else {
			W = 1;
		}
		//mrcPixelDataSet(&linfo->PCA, x, y, z, W, mrcPixelRePart);
		linfo->PCA.ImageFloatImage[k] = W;
		if(1<W) {
		for(iPhi=0; iPhi<nPhi; iPhi++) { // Three Rotation	
			phi = linfo->minPhi + iPhi*linfo->delPhi;
		for(iTheta=0; iTheta<nTheta; iTheta++) {	
			theta = linfo->minTheta + iTheta*linfo->delTheta;
		for(iPsi=0; iPsi<nPsi; iPsi++) { 	
			psi= linfo->minPsi+ iPhi*linfo->delPsi;
			
			//DEBUGPRINT3("%f %f %f\n", phi*DEGREE, theta*DEGREE, psi*DEGREE);	
			matrix3DRotationSetFollowingEulerAngle(mat, 
				"ZENS", phi, theta, psi, MATRIX_3D_MODE_INITIALIZE); // Rotation angle set

		for(iRadius=0; iRadius<nRadius; iRadius++) {
			radius = linfo->minRadius + iRadius*linfo->delRadius; 
		for(iLength=0; iLength<nLength; iLength++) {
			length = linfo->minLength + iLength*linfo->delLength; 

			index = iPhi
				   +iTheta *nPhi
				   +iPsi   *nPhi*nTheta
				   +iRadius*nPhi*nTheta*nPsi
				   +iLength*nPhi*nTheta*nPsi*nRadius;

			linfo->radius = radius;	
			linfo->length = length;	
			switch(mode) {
				case lmrcImageShapeSearchModeSylinder: {	
					lmrcImageShapeSearchSylinder(&(data[index]), in, mat, linfo, mode);		
#ifdef DEBUG2
					if(0<data[index]) {
						DEBUGPRINT2("%d %f\n", index, data[index]);
					}
#endif
					break;
				}
				case lmrcImageShapeSearchModeDisk: {	
					lmrcImageShapeSearchDisk(&data[index], in, mat, linfo, mode);		
					break;
				}
				case lmrcImageShapeSearchModeSphere: {	
					lmrcImageShapeSearchSphere(&data[index], in, mat, linfo, mode);		
					break;
				}
				default: {
					fprintf(stderr, "Not supported mode: %d\n", mode);
					exit(EXIT_FAILURE);	
					break;
				}
			}
		}
		}
		}
		}
		}
		//DEBUGPRINT1("nData: %d\n", nData);
		imax = 0;
		max = data[0]; 
		sum = data[0];	
		for(i=1; i<nData; i++) {	
#ifdef DEBUG2
			if(0<data[i]) {
				DEBUGPRINT2("data: %d %f\n", i, data[i]);
			}
#endif
			sum += data[i];
			if(max<data[i]) {
				imax = i;
				max = data[i];
			}
		}
		ave = sum/nData;
		sum = 0;
		for(i=0; i<nData; i++) {	
			sum += SQR(data[i]-ave);
		}
		sd = sqrt(sum/nData);
#ifdef DEBUG2
		if(0<max) {
		DEBUGPRINT3("%f %f %f ", x, y, z); 	
		DEBUGPRINT3("%f %f %f \n", max, ave, sd); 	
		}
#endif
		} else {
			ave = 0;
			sd  = 0;
			max = 0;
			imax = 0;
		}

		//mrcPixelDataSet(&linfo->average, x, y, z, ave, mrcPixelRePart);
		linfo->average.ImageFloatImage[k0] = ave;

		//mrcPixelDataSet(&linfo->SD,      x, y, z, sd, mrcPixelRePart);
		linfo->SD.ImageFloatImage[k0] = sd;

		//mrcPixelDataSet(&linfo->Max,     x, y, z, max, mrcPixelRePart);
		linfo->Max.ImageFloatImage[k0] = max;

		i = imax%nPhi;
		//mrcPixelDataSet(&linfo->shapeInfo[0], x, y, z, (i*linfo->delPhi + linfo->minPhi)*DEGREE, mrcPixelRePart);
		linfo->shapeInfo[0].ImageFloatImage[k0] =  (i*linfo->delPhi + linfo->minPhi)*DEGREE;

		i = (imax%(nPhi*nTheta))/nPhi;
		//mrcPixelDataSet(&linfo->shapeInfo[1], x, y, z, (i*linfo->delTheta + linfo->minTheta)*DEGREE, mrcPixelRePart);
		linfo->shapeInfo[1].ImageFloatImage[k0] =  (i*linfo->delTheta + linfo->minTheta)*DEGREE;

		i = (imax%(nPhi*nTheta*nPsi))/(nPhi*nTheta);
		//mrcPixelDataSet(&linfo->shapeInfo[2], x, y, z, (i*linfo->delPsi+ linfo->minPsi)*DEGREE, mrcPixelRePart);
		linfo->shapeInfo[2].ImageFloatImage[k0] =  (i*linfo->delPsi+ linfo->minPsi)*DEGREE;

		i = (imax%(nPhi*nTheta*nPsi*nRadius))/(nPhi*nTheta*nPsi);
		//mrcPixelDataSet(&linfo->shapeInfo[3], x, y, z, i*linfo->delRadius + linfo->minRadius, mrcPixelRePart);
		linfo->shapeInfo[3].ImageFloatImage[k0] =  i*linfo->delRadius + linfo->minRadius;

		i =  imax/(nPhi*nTheta*nPsi*nRadius);
		//mrcPixelDataSet(&linfo->shapeInfo[4], x, y, z, i*linfo->delLength + linfo->minLength, mrcPixelRePart);
		linfo->shapeInfo[4].ImageFloatImage[k0] =  i*linfo->delLength + linfo->minLength;
	}
	}
	}

	nVoxel = in->HeaderN.x*in->HeaderN.y*in->HeaderN.z;
	max = -1e30;
	min =  1e30;
	ave = 0;
	sd  = 0;
	for(k=0; k<nVoxel; k++) {
		d = linfo->Max.ImageFloatImage[k];
		ave += d;
		if(sd<linfo->SD.ImageFloatImage[k]) {
			sd = linfo->SD.ImageFloatImage[k];
		}
		if(max<d) {
			max = d;
		} 
		if(ave<min) {
			min = d;
		}
	}
	ave /= nVoxel;
	DEBUGPRINT2("aveOfmax: %f maxOfsd: %f\n", ave, sd);	
	DEBUGPRINT2("max: %f min: %f\n", max, min);	

	sum = 100./(max - min);
	for(k=0; k<nVoxel; k++) {
		d = linfo->Max.ImageFloatImage[k]; 
		zscore = (d-linfo->average.ImageFloatImage[k])/linfo->SD.ImageFloatImage[k]; 
		//zscore = (d-ave)/sd; 
		linfo->Zscore.ImageFloatImage[k] = zscore*in->ImageFloatImage[k];
		
		d = (d-min)*sum;
		out->ImageFloatImage[k] = d*in->ImageFloatImage[k];

		//DEBUGPRINT3("%d z: %f out: %f\n", k, zscore, d);
	}
}

void
lmrcImageShapeSearchSylinder(double* data, mrcImage* in, Matrix3D mat, lmrcImageShapeSearchInfo* linfo, int mode)
{
	double p, l, r;
	double x, y, z;
	double srcx, srcy, srcz;
	double dstx, dsty, dstz;
	double scorePos, scoreNeg;
	double cp, sp, d;
	int k, k0;
	int countPos, countNeg;

	//DEBUGPRINT("lmrcImageShapeSearchSylinder start\n");
	scorePos = 0; 
	scoreNeg = 0; 
	countPos = 0;
	countNeg = 0;
	for(p=0; p<2*M_PI;  p+=linfo->delPsi) {
		cp = cos(p);
		sp = sin(p);
	for(l=0; l<=linfo->length; l+=linfo->delLength) {
		z = l-linfo->length/2; 
	for(r=0; r<=linfo->radius; r+=linfo->delRadius) {
		x = r*cp;
		y = r*sp;

		srcx = mat[0][0]*x + mat[1][0]*y + mat[2][0]*z;
		srcy = mat[0][1]*x + mat[1][1]*y + mat[2][1]*z; 
		srcz = mat[0][2]*x + mat[1][2]*y + mat[2][2]*z;

		dstx = linfo->x + srcx; 
		dsty = linfo->y + srcy;
		dstz = linfo->z + srcz; 
		if(-0.5<dstx && dstx < in->HeaderN.x-0.5
		 &&-0.5<dsty && dsty < in->HeaderN.y-0.5
		 &&-0.5<dstz && dstz < in->HeaderN.z-0.5) {
			k = (int)(dstx+0.5) + (int)(dsty+0.5)*in->HeaderN.x + (int)(dstz+0.5)*in->HeaderN.x*in->HeaderN.y; 
			d = in->ImageFloatImage[k];
		} else {
			d = 0;
		}

		countPos++;
		scorePos += d;
	}
	// Edge Check
	x = r*cos(p);
	y = r*sin(p);
	srcx = mat[0][0]*x + mat[1][0]*y + mat[2][0]*z;
	srcy = mat[0][1]*x + mat[1][1]*y + mat[2][1]*z; 
	srcz = mat[0][2]*x + mat[1][2]*y + mat[2][2]*z;

	dstx = linfo->x + srcx;
	dsty = linfo->y + srcy;
	dstz = linfo->z + srcz; 
	if(-0.5<dstx && dstx < in->HeaderN.x-0.5
	 &&-0.5<dsty && dsty < in->HeaderN.y-0.5
	 &&-0.5<dstz && dstz < in->HeaderN.z-0.5) {
		k = (int)(dstx+0.5) + (int)(dsty+0.5)*in->HeaderN.x + (int)(dstz+0.5)*in->HeaderN.x*in->HeaderN.y; 
		d = in->ImageFloatImage[k];
	} else {
		d = 0;
	}
	scoreNeg += d;
	countNeg++;
	}
	}
	k0 =  (int)(linfo->x+0.5) + (int)(linfo->y+0.5)*in->HeaderN.x + (int)(linfo->z+0.5)*in->HeaderN.x*in->HeaderN.y; 
	*data = (0.8*scorePos/countPos - 0.2*scoreNeg/countNeg);
#ifdef DEBUG2
	//DEBUGPRINT1("data: %f\n", score);
	if(0<score) {
		DEBUGPRINT1("data: %f\n", score);
	}
#endif
}

void
lmrcImageShapeSearchDisk(double* data, mrcImage* in, Matrix3D mat, lmrcImageShapeSearchInfo* linfo, int mode)
{

	double p, l, r;
	double x, y, z;
	double zz;
	double srcx, srcy, srcz;
	double dstx, dsty, dstz;
	double scorePos, scoreNeg;
	double cp, sp, d;
	int k, k0;
	int countPos, countNeg;

	//DEBUGPRINT("lmrcImageShapeSearchSylinder start\n");
	scorePos = 0; 
	scoreNeg = 0; 
	countPos = 0;
	countNeg = 0;
	for(p=0; p<2*M_PI;  p+=linfo->delPsi) {
		cp = cos(p);
		sp = sin(p);
	for(r=0; r<=linfo->radius; r+=linfo->delRadius) {
		x = r*cp;
		y = r*sp;
	for(l=0; l<=linfo->length; l+=linfo->delLength) {
		z = l-linfo->length/2; 

		srcx = mat[0][0]*x + mat[1][0]*y + mat[2][0]*z;
		srcy = mat[0][1]*x + mat[1][1]*y + mat[2][1]*z; 
		srcz = mat[0][2]*x + mat[1][2]*y + mat[2][2]*z;

		dstx = linfo->x + srcx; 
		dsty = linfo->y + srcy;
		dstz = linfo->z + srcz; 
		if(-0.5<dstx && dstx < in->HeaderN.x-0.5
		 &&-0.5<dsty && dsty < in->HeaderN.y-0.5
		 &&-0.5<dstz && dstz < in->HeaderN.z-0.5) {
			k = (int)(dstx+0.5) + (int)(dsty+0.5)*in->HeaderN.x + (int)(dstz+0.5)*in->HeaderN.x*in->HeaderN.y; 
			d = in->ImageFloatImage[k];
		} else {
			d = 0;
		}

		countPos++;
		scorePos += d;
	}
	// Edge Check
	for(zz = -1; zz <= 1; zz+=2) {
		x = r*cos(p);
		y = r*sin(p);
		z = l - (linfo->length/2 + linfo->delLength)*zz;     
		srcx = mat[0][0]*x + mat[1][0]*y + mat[2][0]*z;
		srcy = mat[0][1]*x + mat[1][1]*y + mat[2][1]*z; 
		srcz = mat[0][2]*x + mat[1][2]*y + mat[2][2]*z;
		dstx = linfo->x + srcx;
		dsty = linfo->y + srcy;
		dstz = linfo->z + srcz; 

		if(-0.5<dstx && dstx < in->HeaderN.x-0.5
		 &&-0.5<dsty && dsty < in->HeaderN.y-0.5
		 &&-0.5<dstz && dstz < in->HeaderN.z-0.5) {
				k = (int)(dstx+0.5) + (int)(dsty+0.5)*in->HeaderN.x + (int)(dstz+0.5)*in->HeaderN.x*in->HeaderN.y; 
				d = in->ImageFloatImage[k];
		} else {
				d = 0;
		}
		scoreNeg += d;
		countNeg++;
	}

	}
	}
	k0 =  (int)(linfo->x+0.5) + (int)(linfo->y+0.5)*in->HeaderN.x + (int)(linfo->z+0.5)*in->HeaderN.x*in->HeaderN.y; 
	*data = (0.8*scorePos/countPos - 0.2*scoreNeg/countNeg);
#ifdef DEBUG2
	//DEBUGPRINT1("data: %f\n", score);
	if(0<score) {
		DEBUGPRINT1("data: %f\n", score);
	}
#endif
}


void
lmrcImageShapeSearchSphere(double* data, mrcImage* in, Matrix3D mat, lmrcImageShapeSearchInfo* linfo, int mode)
{

	double phi, psi, r;
	double x, y, z;
	double zz;
	double srcx, srcy, srcz;
	double dstx, dsty, dstz;
	double scorePos, scoreNeg;
	double cp, sp, d;
	int k, k0;
	int countPos, countNeg;

	//DEBUGPRINT("lmrcImageShapeSearchSylinder start\n");
	scorePos = 0; 
	scoreNeg = 0; 
	countPos = 0;
	countNeg = 0;
	for(phi=0; phi<2*M_PI;  phi+=linfo->delPhi) {
	for(psi=0; psi<2*M_PI;  psi+=linfo->delPsi) {
	for(r=0; r<=linfo->radius; r+=linfo->delRadius) {
		
		countPos++;
		scorePos += d;
	}
	// Edge Check
		scoreNeg += d;
		countNeg++;
	}

	}
	}
	k0 =  (int)(linfo->x+0.5) + (int)(linfo->y+0.5)*in->HeaderN.x + (int)(linfo->z+0.5)*in->HeaderN.x*in->HeaderN.y; 
	*data = (0.8*scorePos/countPos - 0.2*scoreNeg/countNeg);
}


