
#==============================================================================#
# $Id: pe.rb,v 1.2 2003/10/15 08:34:01 yuya Exp $
#==============================================================================#

module Exerb

  class Stream

    def initialize(data, pos = 0)
      @data = data
      @pos  = pos
    end

    attr_reader :data, :pos

    def read(size, fmt)
      value = @data[@pos, size].unpack(fmt)[0]
      @pos += size
      return value
    end

    def read_char8
      return read(8, 'Z8')
    end

    def read_byte
      return read(1, 'C')
    end

    def read_word
      return read(2, 'S')
    end

    def read_dword
      return read(4, 'L')
    end

    def seek_from_begin(pos)
      @pos = pos
      return self
    end

    def seek_from_current(pos)
      @pos += pos
      return self
    end

    def seek_from_end(pos)
      @pos = @data.size + pos
      return self
    end

  end # Stream

  class Win32Struct

    def self.create_class(arg)
      klass = Class.new(Win32Struct)

      init = arg.collect { |type, name| "@#{name} = nil\n" }.join
      attr = arg.collect { |type, name| ":#{name}" }.join(",")
      read = arg.collect { |type, name| "obj.#{name} = stream.read_#{type}\n" }.join
      dump = arg.collect { |type, name| "dump_#{type}('#{name}', @#{name});" }.join

      klass.module_eval("def initialize; @pos = nil; #{init}; end")
      klass.module_eval("attr_accessor :pos, #{attr}")
      klass.module_eval("def self.new_from_stream(stream); obj = self.new; obj.pos = stream.pos; #{read}; return obj; end")
      klass.module_eval("def dump; #{dump}; end")

      return klass
    end

    def dump_char8(name, value)
      printf("%s : %s\n", name.ljust(40), value)
    end

    def dump_byte(name, value)
      printf("%s : 0x%02X\n", name.ljust(40), value)
    end

    def dump_word(name, value)
      printf("%s : 0x%04X\n", name.ljust(40), value)
    end

    def dump_dword(name, value)
      printf("%s : 0x%08X\n", name.ljust(40), value)
    end

  end # Win32Struct

  class PeFile

    def initialize(data)
      @data         = data
      @dos_header   = nil
      @nt_headers32 = nil
      @sections     = nil
    end

    def self.new_from_file(path)
      return File.open(path) { |file| self.new(file.read) }
    end

    def dos_header
      if @dos_header.nil?
        stream = Stream.new(@data)
        @dos_header = DosHeader.new_from_stream(stream)
      end

      return @dos_header
    end

    def nt_headers32
      if @nt_headers32.nil?
        stream = Stream.new(@data)
        stream.seek_from_begin(self.dos_header.offset_to_new_header)
        @nt_headers32 = NtHeaders32.new_from_stream(stream)
        @sections = (1..(@nt_headers32.file_header.number_of_sections)).collect {
          SectionHeader.new_from_stream(stream)
        }
      end

      return @nt_headers32
    end

    def sections
      return @sections
    end

  end # PeFile

  class NtHeaders32 < Win32Struct

    def initialize
      @pos               = nil
      @signature         = nil
      @file_header       = nil
      @optional_header32 = nil
    end

    attr_accessor :pos, :signature, :file_header, :optional_header32

    def self.new_from_stream(stream)
      obj = self.new

      obj.pos               = stream.pos
      obj.signature         = stream.read_dword
      obj.file_header       = FileHeader.new_from_stream(stream)
      obj.optional_header32 = OptionalHeader32.new_from_stream(stream)

      return obj
    end

    def dump
      dump_dword('signature', @signature)
      @file_header.dump
      @optional_header32.dump
    end

  end # NtHeaders32

  DosHeader = Win32Struct.create_class([
    ["word",  "magic"],
    ["word",  "last_page_size"],
    ["word",  "total_pages_in_file"],
    ["word",  "relocation_items"],
    ["word",  "paragraphs_in_header"],
    ["word",  "minimum_extra_paragraphs"],
    ["word",  "maximum_extra_paragraphs"],
    ["word",  "initial_stack_segment"],
    ["word",  "initial_stack_pointer"],
    ["word",  "complemented_checksum"],
    ["word",  "initial_instraction_pointer"],
    ["word",  "initial_code_segment"],
    ["word",  "relocation_table_offset"],
    ["word",  "overlay_number"],
    ["word",  "reserved1_1"],
    ["word",  "reserved1_2"],
    ["word",  "reserved1_3"],
    ["word",  "reserved1_4"],
    ["word",  "oem_identifier"],
    ["word",  "oem_information"],
    ["word",  "reserved2_1"],
    ["word",  "reserved2_2"],
    ["word",  "reserved2_3"],
    ["word",  "reserved2_4"],
    ["word",  "reserved2_5"],
    ["word",  "reserved2_6"],
    ["word",  "reserved2_7"],
    ["word",  "reserved2_8"],
    ["word",  "reserved2_9"],
    ["word",  "reserved2_10"],
    ["dword", "offset_to_new_header"],
  ])

  FileHeader = Win32Struct.create_class([
    ["word",  "machine"],
    ["word",  "number_of_sections"],
    ["dword", "time_date_stamp"],
    ["dword", "pointer_to_symbol_table"],
    ["dword", "number_of_symbols"],
    ["word",  "size_of_optional_header"],
    ["word",  "characteristics"],
  ])

  OptionalHeader32 = Win32Struct.create_class([
    ["word",  "magic"],
    ["byte",  "major_linker_version"],
    ["byte",  "minor_linker_version"],
    ["dword", "size_of_code"],
    ["dword", "size_of_initialized_data"],
    ["dword", "size_of_uninitialized_data"],
    ["dword", "address_of_entry_point"],
    ["dword", "base_of_code"],
    ["dword", "base_of_data"],
    ["dword", "image_base"],
    ["dword", "section_alignment"],
    ["dword", "file_alignment"],
    ["word",  "major_operating_system_version"],
    ["word",  "minor_operating_system_version"],
    ["word",  "major_image_version"],
    ["word",  "minor_image_version"],
    ["word",  "major_subsystem_version"],
    ["word",  "minor_subsystem_version"],
    ["dword", "win32_version_value"],
    ["dword", "size_of_image"],
    ["dword", "size_of_headers"],
    ["dword", "checksum"],
    ["word",  "subsystem"],
    ["word",  "dll_characteristics"],
    ["dword", "size_of_stack_reserve"],
    ["dword", "size_of_stack_commit"],
    ["dword", "size_of_heap_reserve"],
    ["dword", "size_of_heap_commit"],
    ["dword", "loader_flags"],
    ["dword", "number_of_rva_and_sizes"],
    ["dword", "data_directory0_virtual_address"],
    ["dword", "data_directory0_size"],
    ["dword", "data_directory1_virtual_address"],
    ["dword", "data_directory1_size"],
    ["dword", "data_directory2_virtual_address"],
    ["dword", "data_directory2_size"],
    ["dword", "data_directory3_virtual_address"],
    ["dword", "data_directory3_size"],
    ["dword", "data_directory4_virtual_address"],
    ["dword", "data_directory4_size"],
    ["dword", "data_directory5_virtual_address"],
    ["dword", "data_directory5_size"],
    ["dword", "data_directory6_virtual_address"],
    ["dword", "data_directory6_size"],
    ["dword", "data_directory7_virtual_address"],
    ["dword", "data_directory7_size"],
    ["dword", "data_directory8_virtual_address"],
    ["dword", "data_directory8_size"],
    ["dword", "data_directory9_virtual_address"],
    ["dword", "data_directory9_size"],
    ["dword", "data_directory10_virtual_address"],
    ["dword", "data_directory10_size"],
    ["dword", "data_directory11_virtual_address"],
    ["dword", "data_directory11_size"],
    ["dword", "data_directory12_virtual_address"],
    ["dword", "data_directory12_size"],
    ["dword", "data_directory13_virtual_address"],
    ["dword", "data_directory13_size"],
    ["dword", "data_directory14_virtual_address"],
    ["dword", "data_directory14_size"],
    ["dword", "data_directory15_virtual_address"],
    ["dword", "data_directory15_size"],
  ])

  SectionHeader = Win32Struct.create_class([
    ["char8", "name"],
    ["dword", "misc"],
    ["dword", "virtual_address"],
    ["dword", "size_of_raw_data"],
    ["dword", "pointer_to_raw_data"],
    ["dword", "pointer_to_relocations"],
    ["dword", "pointer_to_linenumbers"],
    ["word",  "number_of_relocations"],
    ["word",  "number_of_linenumbers"],
    ["dword", "characteristics"],
  ])

end # Exerb

#==============================================================================#

if $0 == __FILE__
  path = "/usr/local/bin/ruby.exe"
  pe = Exerb::PeFile.new_from_file(path)
  #pe.dos_header.dump
  #pe.nt_headers32.dump
  pe.nt_headers32
  pe.sections.each { |sec|
    sec.dump
  }
end

#==============================================================================#
#==============================================================================#
