#!/usr/bin/env ruby
#
#

require 'remastering/pluginBaseOs'
require 'remastering/remastertool_utility'
require 'remastering/remastertool_const'

require "gettext"
require "fileutils"
require "tmpdir"


class Knoppix50_ja < PluginBaseOS
  include GetText

	PROG_NAME = "remastertool_plugin_knoppix50_ja"
	bindtextdomain(PROG_NAME, nil, nil, "UTF-8")

	# constant definition
	DISPLAY_NAME	= "KNOPPIX5.0_ja"	# display name
	VERSION			= "V1.0"			# plugin version 
	PACKAGE_TYPE	= "Deb"				# acceptable package type
	OS_NAME			= "KNOPPIX"			# OS name
	TARGET_VERSION	= "5.0"				# KNOPPIX remastering Target version

	MASTER_DIR		= "master"			# CD media Dir
	SOURCE_DIR		= "source"			# Compressed_fs extract Dir
	KNOPPIX_DIR		= "KNOPPIX"			# KNOPPIIX dir
	COMPRESS_FS		= "KNOPPIX"			# coprsssed_fs name
	MODULE_DIR		= "modules"			# insmod/cloop module dir
	INSMOD_CMD		= "insmod"			# insmod command
	CLOOP_MOD		= "cloop.ko"		# cloop module file
	CLOOP_DEV		= "cloop"			# cloop device name

	DIR_PROFILE		= "remastering"		# directory which contains profile setting file
	FILE_PROFILE	= "/tmp/remastering/bootup.read_blocks"

	RESOLV_CONF = "/etc/resolv.conf"
	SYSLINUX_PATH = "boot/isolinux"
	SYSLINUX_CONF = "isolinux.cfg"

	# for init setting
##	CMD_CP_CD = "rm -rf %s && cp -a %s %s"
	CMD_CP_CD = "rm -rf %s && rsync -a --exclude \"/KNOPPIX/KNOPPIX\" %s/ %s"
	CMD_CP_CMPFS = "rm -rf %s && mkdir -p %s && nice --5 cp -a %s/* %s"
	CMD_RM_SRC = "rm -rf %s"

	CMD_MKCOMPRESSFS = '(cd %s; rm -f " + MASTER_DIR + "/" + KNOPPIX_DIR + "/"+ COMPRESS_FS + " && mkisofs -R -U -hide-rr-moved -cache-inodes -no-bak -pad ' + SOURCE_DIR + "/" +  KNOPPIX_DIR + " | nice -5 create_compressed_fs - 65536 > " + MASTER_DIR + "/" + KNOPPIX_DIR + "/" + \
			COMPRESS_FS + " && sync && cd " + MASTER_DIR + " && rm -f " + KNOPPIX_DIR + \
			"/md5sums && find -type f -not -name md5sums -not -name boot.cat -exec md5sum {} \\; >> " + \
			KNOPPIX_DIR + "/md5sums)"

	# for profiling
	CMD_SYSLIN_EDT = "awk 'BEGIN { done = 0 } /^APPEND / && done == 0 { print $0, \" chkblk=10000 nocbr\"; done = 1; next;} {print;}' %s > %s"


	CMD_OPTIMIZE = '(cd %s && mv ' + MASTER_DIR + "/" + KNOPPIX_DIR + "/" + COMPRESS_FS + \
					" ./KNOPPIX.normal && cloopoptimizer ./KNOPPIX.normal " + \
					FILE_PROFILE + " > " + MASTER_DIR + "/" + KNOPPIX_DIR + "/" + COMPRESS_FS + \
					" && rblk2bl " + FILE_PROFILE + " > " + MASTER_DIR + "/" + KNOPPIX_DIR + "/KNOPPIX.boot.lst)"
	CMD_PORTMAP = "ps aux | grep portmap | grep -vq grep || /etc/init.d/portmap start"
	CMD_NFS_CMN = "ps aux | grep rpc.statd | grep -vq grep || /etc/init.d/nfs-common start"
	CMD_NFS_SVR1 = "ps aux | grep rpc.nfsd | grep -vq grep || /usr/sbin/rpc.nfsd"
	CMD_NFS_SVR2 = "ps aux | grep rpc.mountd | grep -vq grep || /usr/sbin/rpc.mountd"
	CMD_EXPORTFS = 'exportfs -i -o sync,rw,insecure "*:/tmp/remastering"'

	# cloop device status
	CLOOP_LOADED		= 0
	CLOOP_NOT_LOADED	= 1
	CLOOP_INVALID_LOADED = 2

	# Message definition
	MSG_QUE_UNLOAD_CLOOP = _("The cloop device that cannot be used by the compressed_fs by KNOPPIX is loaded.\nMay I unload the current cloop dvice?")
	MSG_ERR_COPY_MEDIA = _("Initialize of environment encountered a problem.\n%s")
	MSG_ERR_LOSETUP_CLOOP = _("cloop device is busy.")
	MSG_ERR_MOUNT_CLOOP = _("Unable to mount cloop device(%d).")
	MSG_ERR_LOAD_CLOOP = _("Unable to load cloop device(%d).")
	MSG_ERR_COPY_COMPRESSEDFS = _("Unable to copy compressed_fs(%d).")
	MSG_WAIT_TEXT = _("Initializing environment.\nPlease wait for a while.")
	MSG_INF_MAKE_OS = _("Now creating OS image.\nPlease wait for a while.")
	MSG_ERR_MAKE_OS = _("Failed to create OS image(%d).")
	MSG_INF_PREPARE_PROFILING = _("The profiling is prepared for acceleration.\nPlease wait for a while.")
	MSG_ERR_PREPARE_PROFILE = _("Profiling preparation is encountered a problem.\n\n%s")
	MSG_ERR_PROFILE = _("Profiling is encountered a problem.\n\n%s")
	MSG_ERR_OPTIMIZE = _("Command for optimization is get error(%d).")
	MSG_ERR_NO_PROFILE = _("profiling result is not exist.")

	MSG_INF_PROFILING = _("Now profiling OS image for acceleration.\nPlease wait for a while.")

	#=== コンストラクタ
	#
	#プラグイン初期設定を行う。
	#
	#dir:: 自プラグイン格納ディレクトリ
	#復帰値:: なし
	#
	def initialize(dir)
		super()
		@my_dir = dir
		@display_name = DISPLAY_NAME
		@version = VERSION
		@target_version = TARGET_VERSION
		@package_type = PACKAGE_TYPE
		@os_name = OS_NAME
		@os_version = ""
		@copyright = SSL_COPYRIGHT
		@license = SSL_LICENSE
		@have_about = true				# about dialog is having
		@have_option = false			# Option dialog is not having
		@can_accel = true

		@tool_min_vl = "V1.0"
		@modified = false
		@resolv_conf = RESOLV_CONF

		@path_boot_image = {"isolinux" => "boot/isolinux/isolinux.bin", "syslinux" => "boot/isolinux/*" }
		@path_boot_catalog = {"isolinux" => "boot/isolinux/boot.cat", "syslinux" => "boot/isolinux/boot.cat"}

		init_environ				# initialize environment
	end

	#=== リマスタリングツールとプラグインの整合性チェック
	#
	#本プラグインがリマスタリングツールのバージョン下で動作可能かチェックする。
	#
	#tool_version:: リマスタリングツールのバージョン
	#復帰値:: BOOL true=動作可能、false=動作不可能
	#
	def start_plugin(tool_version)
		return tool_version >= @tool_min_vl
	end

	#===自プラグイン用のOSかチェック
	#
	#指定ディレクトリのメディアが自プラグインでサポートしているOSかチェックする。
	#
	#dir:: メディアマウントディレクトリ or 作業用ディレクトリ
	#復帰値:: BOOL  true=自プラグインのサポートOS, false=未知のOS
	#
	def isYou?(dir)
		version = ""
		begin
			version = File.readlines(File.join(dir, KNOPPIX_DIR, "knoppix-version"))[0].chomp!.split[0]
		rescue
			RTUtility.get_logger.debug("Failed to reading version file(media):" + $!.to_s)
		end
		begin
			version = File.readlines(
				File.join(dir, MASTER_DIR, KNOPPIX_DIR, "knoppix-version"))[0].chomp!.split[0] \
				if version.empty?
		rescue
			RTUtility.get_logger.debug("Failed to reading version file(work):" + $!.to_s)
		end
		if version =~ /^#{@target_version}/
			@os_version = version
			RTUtility.get_logger.debug("knoppix version=" + @os_version)
			true
		else
			false
		end
	end

	#===リマスタリングメディアのコピー
	#
	#指定されたリマスタリングメディア(パラメタ:from)を作業用ディレクトリ(パラメタ:to)
	#にコピーする。
	#
	#from:: メディアマウントディレクトリ
	#to:: 作業用ディレクトリ
	#復帰値:: BOOL  true=コピー成功, false=コピー失敗
	#
	def copy_media(from, to)
		retVal = false
		temp = ""

		begin
			# cloop デバイスのチェック
			case checK_cloop
			when CLOOP_INVALID_LOADED
				return false	# return if cloop is invalid and refuse unload
			when CLOOP_NOT_LOADED
				load_cloop(from)
			end

			RTUtility.show_wait_window(RTUtility.mainWindow, MSG_WAIT_TEXT)

			# cdを作業用ディレクトリにコピー
			copy_cddir(from, File.join(to, MASTER_DIR))
			# cloop デバイスのマウント
##			temp = mount_cloop(File.join(to, MASTER_DIR, KNOPPIX_DIR, COMPRESS_FS))
			temp = mount_cloop(File.join(from, KNOPPIX_DIR, COMPRESS_FS))
			# cloop デバイス内ファイルを作業用ディレクトリにコピー
			copy_compressedfs(File.join(to, SOURCE_DIR), temp)
			retVal = true

			use_directory(to)
			RTUtility.end_wait_window

		rescue
			# exception occurred
			RTUtility.end_wait_window
			dlg = Gtk::MessageDialog.new(RTUtility.mainWindow, Gtk::Dialog::MODAL, Gtk::MessageDialog::ERROR,
										Gtk::MessageDialog::BUTTONS_OK, sprintf(MSG_ERR_COPY_MEDIA, $!.to_s))
			dlg.run
			dlg.destroy
			
			FileUtils.rm_rf([File.join(to, MASTER_DIR), File.join(to, SOURCE_DIR)])
		ensure
			unless temp.empty?
				umount_cloop(temp)
				FileUtils.rm_rf(temp)
			end
		end

		retVal
	end

	#===作業用ディレクトリか初期化済みかチェック
	#
	#パラメタで指定された作業用ディレクトリが初期設定済みか(リマスタリングメディアが
	#作業用ディレクトリにコピーされているか)チェックする。
	#
	#workDir:: 作業用ディレクトリ
	#復帰値:: BOOL  true=初期設定済み, false=初期設定未
	#
	def isInitialized?(workDir)
		FileTest.file?(File.join(workDir, SOURCE_DIR, KNOPPIX_DIR, "etc", "knoppix-version"))
	end

	#=== 作業用ディレクトリの設定
	#
	#指定されたディレクトリを作業用ディレクトリとして扱う。
	#isInitialized? メソッドで true を返却したディレクトリが指定される。
	#
	#work_dir:: 作業用ディレクトリ
	#復帰値:: なし
	#
	def use_directory(work_dir)
		@root_dir = File.join(work_dir, SOURCE_DIR, KNOPPIX_DIR)
		@media_dir = File.join(work_dir, MASTER_DIR)
		@os_version = File.readlines(File.join(@media_dir, KNOPPIX_DIR, "knoppix-version"))[0].chomp!.split[0]
	end

	#===プロファイリング前設定
	#
	#プロファイリングのための前設定を行う。
	#ここではプロファイリングデータの削除、プロファイリングデータ取得用スクリプトのCD
	#へのコピー、NFSサーバの設定を行う。
	#
	#復帰値:: BOOL  true=正常終了、 false=異常終了
	#
	def prepare_profile
		# syslinuxのファイルパスの設定
		syslinuxcfg = File.join(@media_dir, SYSLINUX_PATH, SYSLINUX_CONF)
		syslinuxcfg_bak = syslinuxcfg + ".remasterbak"

		begin
			# ちょっと待ってくださいダイアログを表示
			RTUtility.show_wait_window(RTUtility.mainWindow, MSG_INF_PREPARE_PROFILING)

			# プロファイリング用設定ファイルをメディアディレクトリにコピーする
			FileUtils.mkdir_p File.dirname(FILE_PROFILE)
			FileUtils.chmod(0777, File.dirname(FILE_PROFILE))
			FileUtils.rm_rf(File.join(@media_dir, DIR_PROFILE))
			FileUtils.cp_r(File.join(@my_dir, DIR_PROFILE), @media_dir)
			FileUtils.mv(syslinuxcfg, syslinuxcfg_bak)
			RTUtility.exec_workOS_command(sprintf(CMD_SYSLIN_EDT, syslinuxcfg_bak, syslinuxcfg))

			# プロファイリングした結果ファイル名が存在していたら消す
			FileUtils.rm_f(FILE_PROFILE)
	
			# NFSサーバプロセスの実行
			[CMD_PORTMAP, CMD_NFS_CMN, CMD_NFS_SVR1, CMD_NFS_SVR2, CMD_EXPORTFS].each { |cmd|
				RTUtility.exec_workOS_command(cmd)
			}
			RTUtility.end_wait_window
			true

		rescue
			RTUtility.end_wait_window
			dlg = Gtk::MessageDialog.new(RTUtility.mainWindow, Gtk::Dialog::MODAL, Gtk::MessageDialog::ERROR,
											Gtk::MessageDialog::BUTTONS_OK,
											sprintf(MSG_ERR_PREPARE_PROFILE, $!.to_s))
			dlg.run
			dlg.destroy

			# 最適化に使用したプロファイリングファイルを削除する
			FileUtils.rm_rf([File.join(@media_dir, DIR_PROFILE), "/tmp/remastering"])
			FileUtils.mv(syslinuxcfg_bak, syslinuxcfg)

			false
		end
	end

	#===プロファイリング
	#
	#エミュレータプラグインでOSイメージを起動してプロファイリングを実施する。
	#
	#復帰値:: BOOL  true=正常終了、 false=異常終了
	#
	def do_profile(media_image, emu_plugin)
		begin
			gc_status = GC.disable

			# 別スレッドでエミュレータを起動する
			thr = Thread.start {
				begin
					#エミュレータ実行
					emu_plugin.run(media_image, true)
				rescue
				end
			}

			# エミュレータ配下で実行、かつ、起動後にxxxアイコンをクリックして実行する旨をメッセージ表示
			RTUtility.show_wait_window(RTUtility.mainWindow, MSG_INF_PROFILING)

			# エミュレータからプロファイルリング結果が出力されるまで待つ
			until (File.file?(FILE_PROFILE) || thr.alive? == false)
				Gtk.main_iteration while Gtk.events_pending?
				sleep 1
			end
			thr.raise "end profile" if thr.alive? # エミュレータを終了させる
			raise MSG_ERR_NO_PROFILE unless File.file?(FILE_PROFILE)	

			# OSイメージを最適化する
			cmd = sprintf(CMD_OPTIMIZE, File.join(@media_dir, ".."))
			status = RTUtility.exec_workOS_command(cmd)
			raise sprintf(MSG_ERR_OPTIMIZE, status) unless status == 0

			RTUtility.end_wait_window

			true

		rescue
			RTUtility.end_wait_window
			dlg = Gtk::MessageDialog.new(RTUtility.mainWindow, Gtk::Dialog::MODAL, Gtk::MessageDialog::ERROR,
											Gtk::MessageDialog::BUTTONS_OK,
											sprintf(MSG_ERR_PROFILE, $!.to_s))
			dlg.run
			dlg.destroy
			false

		ensure
			# 最適化に使用したプロファイリングファイルを削除する
			FileUtils.rm_rf([File.join(@media_dir, DIR_PROFILE), "/tmp/remastering"])
			syslinuxcfg = File.join(@media_dir, SYSLINUX_PATH, SYSLINUX_CONF)
			FileUtils.mv(syslinuxcfg + ".remasterbak", syslinuxcfg)

			GC.enable unless gc_status
		end
	end

	#===OSイメージの作成
	#
	#作業用ディレクトリ中のOSファイルツリーからOSイメージを作成する。
	#
	#復帰値:: BOOL : true=成功、false=失敗
	#
	def create_os()
		# 作成に時間がかかるのでwaitウインドウを表示する。
		RTUtility.show_wait_window(RTUtility.mainWindow, MSG_INF_MAKE_OS)
		# OSイメージ作成コマンドの実行
		cmd = sprintf(CMD_MKCOMPRESSFS, File.expand_path(File.join(@root_dir, "../..")))
		status = RTUtility.exec_workOS_command(cmd)

		# waitウインドウを閉じる。
		RTUtility.end_wait_window

		if status != 0
			dlg = Gtk::MessageDialog.new(RTUtility.mainWindow, Gtk::Dialog::MODAL, Gtk::MessageDialog::ERROR,
											Gtk::MessageDialog::BUTTONS_OK,
											sprintf(MSG_ERR_MAKE_OS, status))
			dlg.run
			dlg.destroy
			false
		else
			true
		end
	end

	#===リマスタリング環境の初期化
	#
	#リマスタリング環境を初期状態に設定する。
	#
	#復帰値:: なし
	#
	def init_environ
		@media_dir = ""
		@root_dir = ""
		@os_version = ""
	end

	private

	#===loadされているcloopデバイスのチェック
	#
	#clooデバイスがロードされているかチェックする。ロードされている場合、正しいcloopデバイス(lcat対応デバイス)
	#がロードされているかチェックする。
	#
	#復帰値:: Integer CLOOP_LOADED: lcat対応のcloopデバイス対応がロード済み
	#                CLOOP_NOT_LOADED: cloopデバイスがロードされていない
	#                CLOOP_INVALID_LOADED: lcat未対応のcloopデバイス対応がロード済み
	#
	def checK_cloop()
		# check cloop device is loaded or not loaded
		return CLOOP_NOT_LOADED unless File.readlines("|lsmod | grep cloop").size > 0

		# check cloop device is correct or invalid
		return CLOOP_LOADED if File.readlines("|grep cloop /proc/kallsyms | grep proc_cloop_read_blocks").size > 0

		# invalid cloop device is loaded. ask to user can unloading
		dlg = Gtk::MessageDialog.new(RTUtility.mainWindow, Gtk::Dialog::MODAL, Gtk::MessageDialog::QUESTION,
										Gtk::MessageDialog::BUTTONS_YES_NO, MSG_QUE_UNLOAD_CLOOP)
		response = dlg.run
		dlg.destroy

		return CLOOP_INVALID_LOADED unless response == Gtk::Dialog::RESPONSE_YES

		# unload invalid cloop device
		unload_cloop
		CLOOP_NOT_LOADED
	end

	#===CDメディアのコピー
	#
	#CDメディアをメディアディレクトリ(パラメタ from)から作業用ディレクトリ(パラメタto)
	#にコピーする。
	#
	#from:: メディアマウントディレクトリ
	#to:: 作業用ディレクトリ/master
	#復帰値:: なし
	#
	def copy_cddir(from, to)
		status = RTUtility.exec_workOS_command(sprintf(CMD_CP_CD, to, from, to))
		if status != 0
			raise ""
		end
	end

	#===cloopファイルシステム内ファイルのコピー
	#
	#テンポラリディレクトリ(パラメタtemp)にマウントされたKNOPPIX5.0のcloopファイル
	#システム内のファイルを作業用ディレクトリ(パラメタto)にコピーする。
	#
	#to:: 作業用ディレクトリ/source
	#temp:: KNOPPIX5.0のcloopファイル名システムのマウント先ディレクトリ
	#復帰値:: なし
	#
	def copy_compressedfs(to, temp)
		# copy source
		destDir = File.join(to, KNOPPIX_DIR)
		status = RTUtility.exec_workOS_command(sprintf(CMD_CP_CMPFS, to, destDir, temp, destDir))
		if status != 0
			RTUtility.exec_workOS_command(sprintf(CMD_RM_SRC, destDir))
			raise ""
		end
	end

	#===cloopファイルシステムのマウント
	#
	#cloopファイルをテンポラリディレクトリにマウントする。
	#
	#compress_fs:: cloopファイル
	#復帰値:: KNOPPIX5.0のcloopファイル名システムのマウント先ディレクトリ
	#
	def mount_cloop(compress_fs)
		# create temporary mount point for compressed fs.
		temp = File.join(Dir.tmpdir, ("remaster" + $$.to_s))	# temprary mount dir
		FileUtils.mkdir_p(temp)

		# finding unuse cloop device
		device = ""
		stderr_dup = STDERR.dup
		STDERR.reopen("/dev/null")
		begin
			Dir["/dev/cloop?"].each{ |dev|
				system("losetup", dev, compress_fs)
				if $?.exitstatus == 0
					device = dev
					break
				end
			}
			raise MSG_ERR_LOSETUP_CLOOP if device.empty?	# raise exception because all cloop device is busy.
	
			# mount cloop device
			system("mount", "-o", "loop,ro", device, temp)
			raise sprintf(MSG_ERR_MOUNT_CLOOP, $?.exitstatus) if $?.exitstatus != 0

		rescue
			FileUtils.rm_rf temp
			raise
		ensure
			STDERR.reopen(stderr_dup)
		end

		temp
	end

	#===cloopファイルシステムのアンマウント
	#
	#cloopファイルをアンマウントする。
	#
	#temp:: KNOPPIX5.0のcloopファイル名システムのマウント先ディレクトリ
	#復帰値:: なし
	#
	def umount_cloop(temp)
		system("umount", temp)
	end

	#===cloopデバイスのアンロード
	#
	#cloopデバイスをアンロードする。
	#
	#復帰値:: なし
	#
	def unload_cloop
		RTUtility.get_logger.debug("Unloading cloop device.")
		system("rmmod", "cloop")
	end
	
	#===cloopデバイスのロード
	#
	#lcat対応cloopデバイスをロードする。
	#
	#dir:: メディアマウントディレクトリ
	#復帰値:: なし
	#
	def load_cloop(dir)
		RTUtility.get_logger.debug("Loading cloop device.")
		system("insmod", File.join(dir, KNOPPIX_DIR, MODULE_DIR, CLOOP_MOD))
		raise sprintf(MSG_ERR_LOAD_CLOOP, $?.exitstatus) if $?.exitstatus != 0
	end
end
