/*
Copyright (c) 2003, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


package org.jibx.binding;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.jibx.binding.def.BindingBuilder;
import org.jibx.binding.def.BindingDefinition;
import org.jibx.runtime.JiBXException;
import org.jibx.runtime.impl.UnmarshallingContext;

/**
 * Binding compiler support class. Supplies common methods for use in compiling
 * binding definitions.
 *
 * @author Dennis M. Sosnoski
 * @version 1.0
 */
 
public class Utility
{
    // private constructor to prevent any instance creation
    private Utility() {}
    
    /**
     * Recurse through jar file path component, adding all jars referenced from
     * the original jar to the path collection. Silently ignores problems
     * loading jar files.
     *
     * @param path jar path component
     * @param paths set of paths processed (added to by call)
     */
    
    private static void recursePathJars(String path, ArrayList paths) {
        try {
            
            // check class path information in jar file
            JarFile jfile = new JarFile(path, false);
            Manifest mfst = jfile.getManifest();
            if (mfst != null) {
                
                // look for class path information from manifest
                Attributes attrs = mfst.getMainAttributes();
                String cpath = (String)attrs.get(Attributes.Name.CLASS_PATH);
                if (cpath != null) {
                
                    // set base path for all relative references
                    int split = path.lastIndexOf(File.separatorChar);
                    String base = (split >= 0) ?
                        path.substring(0, split+1) : "";
                
                    // process all references in jar class path
                    while (cpath != null) {
                        split = cpath.indexOf(' ');
                        String item;
                        if (split >= 0) {
                            item = cpath.substring(0, split);
                            cpath = cpath.substring(split+1).trim();
                        } else {
                            item = cpath;
                            cpath = null;
                        }
                        String ipath = base + item;
                        if (!paths.contains(ipath)) {
                            paths.add(ipath);
                            split = ipath.lastIndexOf('.');
                            if (split >= 0 && "jar".equalsIgnoreCase
                                (ipath.substring(split+1))) {
                                recursePathJars(ipath, paths);
                            }
                        }
                    }
                }
            }
        } catch (IOException ex) { /* silently ignore problems in loading */ }
    }
    
    /**
     * Method builds a string array of items in the class path.
     *
     * @return array of classpath components
     */
    
    public static String[] getClassPaths() {
        
        // get all class path components
        String path = System.getProperty("java.class.path");
        ArrayList paths = new ArrayList();
        int start = 0;
        int mark;
        while (path != null) {
            mark = path.indexOf(File.pathSeparatorChar, start);
            String item;
            if (mark >= 0) {
                item = path.substring(start, mark);
            } else {
                item = path.substring(start);
                path = null;
            }
            if (!paths.contains(item)) {
                paths.add(item);
                int split = item.lastIndexOf('.');
                if (split >= 0 &&
                    "jar".equalsIgnoreCase(item.substring(split+1))) {
                    recursePathJars(item, paths);
                }
            }
            start = mark + 1;
        }
        paths.add(".");
        String[] clsspths = new String[paths.size()];
        paths.toArray(clsspths);
        return clsspths;
    }
    
    /**
     * Generate binding name. This takes a base name (such as a file name with
     * extension stripped off) and converts it to legal form by substituting '_'
     * characters for illegal characters in the base name.
     *
     * @param name base binding name
     * @return converted binding name
     */
    
    public static String convertName(String name) {
        
        // convert name to use only legal characters
        StringBuffer buff = new StringBuffer(name);
        if (!Character.isJavaIdentifierStart(buff.charAt(0))) {
            buff.insert(0, 'X');
        }
        for (int i = 1; i < buff.length(); i++) {
            if (!Character.isJavaIdentifierPart(buff.charAt(i))) {
                buff.setCharAt(i, '_');
            }
        }
        return buff.toString();
    }
    
    /**
     * Extract base file name from a full path.
     *
     * @param path full file path
     * @return file name component from path
     */
    
    public static String fileName(String path) {
        int split = path.lastIndexOf(File.separatorChar);
        return path.substring(split+1);
    }
    
    /**
     * Load binding definition from stream. Note that the reader variation of
     * this method is preferred, since the character encoding used in the input
     * document may not be interpreted property when the source is a stream.
     *
     * @param fname binding definition full name
     * @param sname short form of name to use as the default name of the binding
     * @param strm input stream for binding definition document
     * @return constructed binding definition
     * @exception FileNotFoundException if path cannot be accessed
     * @exception JiBXException if error in processing the binding definition
     */
    
    public static BindingDefinition loadBinding(String fname, String sname,
        InputStream strm) throws JiBXException, IOException {
        
        // construct and return the binding definition
        UnmarshallingContext uctx = new UnmarshallingContext(0, 
            new String[0], new String[0], new String[0], new String[0]);
        uctx.setDocument(strm, fname);
        return BindingBuilder.unmarshalBindingDefinition(uctx, sname);
    }
    
    /**
     * Load binding definition from reader.
     *
     * @param fname binding definition full name
     * @param sname short form of name to use as the default name of the binding
     * @param rdr reader for binding definition document
     * @return constructed binding definition
     * @exception FileNotFoundException if path cannot be accessed
     * @exception JiBXException if error in processing the binding definition
     */
    
    public static BindingDefinition loadBinding(String fname, String sname,
        Reader rdr) throws JiBXException, IOException {
        
        // construct and return the binding definition
        UnmarshallingContext uctx = new UnmarshallingContext(0, 
            new String[0], new String[0], new String[0], new String[0]);
        uctx.setDocument(rdr, fname);
        return BindingBuilder.unmarshalBindingDefinition(uctx, sname);
    }
    
    /**
     * Load binding definition from file.
     *
     * @param path file path for binding definition
     * @return constructed binding definition
     * @exception IOException if error accessing file
     * @exception JiBXException if error in processing the binding definition
     */
    
    public static BindingDefinition loadFileBinding(String path)
        throws JiBXException, IOException {
        
        // extract basic name of binding file from path
        String fname = fileName(path);
        String sname = fname;
        int split = sname.indexOf('.');
        if (split > 0) {
            sname = sname.substring(0, split);
        }
        sname = convertName(sname);
        
        // construct and return the binding definition
        Reader rdr = new FileReader(path);
        return loadBinding(fname, sname, rdr);
    }
}