#include "stdafx.h"

#include "FWatchApp.hpp"
#include "resource.h"

#include "Utility.hpp"

#include "ImpersonateUser.hpp"
#include "WatchThread.hpp"

#include <process.h>
#include <assert.h>

#include <string>


#include <strsafe.h>
#pragma comment(lib, "strsafe.lib")

namespace
{

	/*!
	 * Ďnh̃\[XǗNX
	 */
	class WatchHandle
	{
	private:
		const CSettingInfo& settingInfo_;

		FILETIME creation_;

		FILETIME lastChallenged_;

		HANDLE findHandle_;

		Logger* pLogger_;

	public:
		WatchHandle(const CSettingInfo& settingInfo)
			: settingInfo_(settingInfo)
			, creation_(FILETIME())
			, lastChallenged_(FILETIME())
			, findHandle_(INVALID_HANDLE_VALUE)
		{
		}

		~WatchHandle()
		{
			Close();
		}

		void SetLogger(Logger* pLogger)
		{
			pLogger_ = pLogger;
		}

		Logger* GetLogger() const
		{
			return pLogger_;
		}

		void Close()
		{
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				DWORD err = 0;
				if (!::FindCloseChangeNotification(findHandle_)) {
					err = GetLastError();
				}

				findHandle_ = INVALID_HANDLE_VALUE;

				if (pLogger_) {
					pLogger_->LogMessage(4, NULL, err, IDS_LOGMES_WATCH_CLOSE);
				}
			}
		}

		bool checkAndRenew()
		{
			if ( !settingInfo_.isUsingDirNotificationAPI()) {
				// t@CĎnhgpȂꍇ͉Ȃ
				return false;
			}

			FILETIME ftNow;
			::GetSystemTimeAsFileTime(&ftNow);

			DWORD retryInterval = settingInfo_.getDirNotificationAPIRetryInterval();
			DWORD expireSpan = settingInfo_.getDirNotificationAPIExpirySpan();

			// nh쐬ALԂ𒴂Ă΃nhN[YB
			bool renew = false;
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				if (expireSpan > 0 && IsExpired(ftNow, expireSpan * 1000L, creation_)) {
					Close();
					renew = true;
				}
			}

			if (findHandle_ != INVALID_HANDLE_VALUE) {
				// nhLłΉ߂B
				return true;
			}

			if (!renew && retryInterval > 0 && !IsExpired(ftNow, retryInterval * 1000L, lastChallenged_)) {
				// ŌɃgCĂwbo߂ĂȂꍇ̓nh쐬ɖ߂
				return false;
			}

			// 쐬sݎɂ
			lastChallenged_ = ftNow;

			// Ďnh̍쐬
			PropertyStringExpander expander;
			app.setAppConfVariablesTo(expander);
			tstring watchDir = expander.expandString(settingInfo_.getWatchDir());

			findHandle_ = ::FindFirstChangeNotification(
				watchDir.c_str(),
				settingInfo_.getMaxDepth() > 0 ? TRUE : FALSE,
				FILE_NOTIFY_CHANGE_FILE_NAME
					| (settingInfo_.getMaxDepth() > 0 ? FILE_NOTIFY_CHANGE_DIR_NAME : 0)
					| (settingInfo_.isArchiveOnly() ? FILE_NOTIFY_CHANGE_ATTRIBUTES : 0)
					| FILE_NOTIFY_CHANGE_LAST_WRITE
					| FILE_NOTIFY_CHANGE_SIZE
				);
			if (findHandle_ != INVALID_HANDLE_VALUE) {
				// nh쐬ꂽꍇA쐬Zbg
				creation_ = ftNow;

				if (pLogger_) {
					pLogger_->LogMessage(
						4,
						NULL,
						0,
						renew ? IDS_LOGMES_WATCH_RENEW : IDS_LOGMES_WATCH_CREATED
						);
				}

			}
			else {
				// nh쐬łȂꍇ
				if (pLogger_) {
					DWORD errcode = GetLastError();
					pLogger_->LogMessage(0, NULL, errcode, IDS_LOGMES_WATCHERROR);
				}
			}

			return findHandle_ != INVALID_HANDLE_VALUE;
		}

		operator HANDLE() const
		{
			return findHandle_;
		}

	};

}

/*!
 * ~CxgĂΒ~OX[đWvwpB
 */
void CWatchThreadInterruptedException::checkAndThrow(HANDLE hStopEvent)
{
	if (!hStopEvent) {
		throw CWatchThreadInterruptedException();
	}
	DWORD waitResult = WaitForSingleObject(hStopEvent, 0);
	if (waitResult != WAIT_TIMEOUT) {
		// ~Cxgɂً}E
		throw CWatchThreadInterruptedException();
	}
}


////

CWatchThread::CWatchThread(const CSettingInfo& v_settingInfo)
	: settingInfo_(v_settingInfo)
	, fileTimeComparator_(v_settingInfo.getTolerance())
	, nThreadID_(0)
	, hThread_(NULL)
	, nameMatcher_(v_settingInfo.getWatchFile(), app.IsNameMatcherIgnoreCase())
{
	hStopEvent_ = ::CreateEvent(NULL, TRUE, FALSE, NULL);
	assert(hStopEvent_ != NULL && "~Cxgnh擾ł܂B");

	// {ݒ̎擾
	baseCycle_ = app.GetLong(_T("CONFIG"), _T("BASE_CYCLE"), 50);
	baseCycleOnIdle_ = app.GetLong(_T("CONFIG"), _T("BASE_CYCLE_ON_IDLE"), 1000);
	spanSweepDeletedEntries_ = app.GetLong(_T("CONFIG"), _T("SPAN_SWEEP_DELETED_ENTRIES"), 100);
	spanHeapCompaction_ = app.GetLong(_T("CONFIG"), _T("SPAN_HEAP_COMPACTION"), 600 * 1000);
	spanSavePersist_ = app.GetLong(_T("CONFIG"), _T("SPAN_SAVE_PERSIST"), 300 * 1000);
	nResetRunCount_ = app.GetLong(_T("CONFIG"), _T("RESET_RUN_COUNT"), 0);
	bPersistAtForceInterval_ = app.GetLong(_T("CONFIG"), _T("PERSIST_AT_FORCEINTERVAL"), 1) != 0;
	bContinueWhenFailedImpersonate_ = app.GetLong(_T("SECURITY"), _T("CONTINUE_WHEN_FAILED_IMPERSONATING"), 0) != 0;
	bRevertToSelfOnAction_ = app.GetLong(_T("SECURITY"), _T("REVERT_TO_SELF_ON_ACTION"), 0) != 0;

	// ݒt@C̊ϐEϐWJT|[g
	PropertyStringExpander expander;
	app.setAppConfVariablesTo(expander);

	// it@C̕ۑ
	persist_ = expander.expandString(settingInfo_.getPersist());

	// t@CK[̍쐬
	tstring logPath = expander.expandString(settingInfo_.getLog());

	FileLogger* pFileLogger = new FileLogger();
	pLogger_.reset(pFileLogger);
	pFileLogger->SetLogPath(logPath.c_str());
	pFileLogger->SetEnableLogging(settingInfo_.isEnableLogging());
	pFileLogger->SetLogThreshold(settingInfo_.getLogLevel());

	// ANVsIuWFNg̍쐬
	ActionInvokerFactory actionInvokerFactory;
	pActionInvoker_ = actionInvokerFactory.create(settingInfo_);
	pActionInvoker_->setLogger(pFileLogger);
}


CWatchThread::~CWatchThread()
{
	Stop();
	::CloseHandle(hStopEvent_);
}


const CSettingInfo& CWatchThread::getSettingInfo() const
{
	return settingInfo_;
}


bool CWatchThread::isRunning() const
{
	if (hThread_ == NULL) {
		return false;
	}
	return (::WaitForSingleObject(hThread_, 0) == WAIT_TIMEOUT);
}


void CWatchThread::StopNoWait()
{
	if (hThread_ == NULL) {
		return;
	}

	::SetEvent(hStopEvent_);
}


void CWatchThread::Stop()
{
	if (hThread_ == NULL) {
		return;
	}

	// ~v
	StopNoWait();

	// Xbh~̂ҋ@B
	::WaitForSingleObject(hThread_, INFINITE);

	// Xbh~̂ŃnhB
	::CloseHandle(hThread_);
	hThread_ = NULL;
}


void CWatchThread::Start()
{
	if (hThread_ != NULL) {
		if (WaitForSingleObject(hThread_, 0) == WAIT_TIMEOUT) {
			// łɊJnĂ邽߉ȂB
			return;
		}

		// ~ς݂̃XbhȂ̂ŃnhB
		::CloseHandle(hThread_);
		hThread_ = NULL;
	}

	// Xe[^X̐ݒƃXbh̊Jn
	::ResetEvent(hStopEvent_);
	hThread_ = (HANDLE) _beginthreadex(NULL, 0, ThreadProc, this, CREATE_SUSPENDED, (unsigned int*) &nThreadID_);
	if (hThread_ != NULL) {
		::ResumeThread(hThread_);
	}
}

/*!
 * t@C̐΃pXw肵āÃt@C̃t@CTCY擾B
 * ܂AɁAt@Cǂݍ݃[hŃI[vł邩`FbNAłȂꍇfalseԂ܂B
 * falseԂꂽꍇ͊i[͕ύX܂B
 * Win32G[R[h擾ꍇGetLastError()Ăяo܂B
 * \param v_path t@C̐΃pX
 * \param pSize t@CTCY̕ۑANULL̏ꍇ͊i[Ȃ
 * \param pLastModified ŏIXV̕ۑANULL̏ꍇ͊i[Ȃ
 * \param pAttrs ̕ۑANULL̏ꍇ͊i[Ȃ
 * \return t@CTCY̎擾Ɏsꍇ
 */
bool CWatchThread::getFileInfo(const tstring& v_path, unsigned __int64 *pSize, FILETIME *pLastModified, DWORD *pAttrs) throw()
{
	//required:
	assert(!v_path.empty() && "t@Cɋ󕶎͎wł܂B");
	
	//do:
	bool result = false;
	DWORD errcode = NO_ERROR;
	HANDLE fileHandle = ::CreateFile(v_path.c_str(),
			GENERIC_READ, // ȂƂǂݍ݃I[vł邱ƁB
			FILE_SHARE_READ | FILE_SHARE_WRITE,
			NULL,
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL,
			NULL);
	if (fileHandle == INVALID_HANDLE_VALUE) {
		errcode = GetLastError();
	}
	else {
		BY_HANDLE_FILE_INFORMATION fileData = {0};

		if ( !::GetFileInformationByHandle(fileHandle, &fileData)) {
			errcode = GetLastError();
		}
		else {
			ULARGE_INTEGER size = {fileData.nFileSizeLow, fileData.nFileSizeHigh};
			if (pSize) {
				*pSize = size.QuadPart;
			}
			if (pLastModified) {
				*pLastModified = fileData.ftLastWriteTime;
			}
			if (pAttrs) {
				*pAttrs = fileData.dwFileAttributes;
			}
			result = true;
		}

		::CloseHandle(fileHandle);
	}

	::SetLastError(errcode);
	return result;
}


/*!
 * 폜\̃GĝAP\Ԃ߂GgB
 * 폜ꂽGg̐Ԃ܂B
 * \param ftNow ݎ
 * \param bSkipDeletePending 폜\t@CȂꍇtrueɐݒ肳B(falseɂ͐ݒ肳Ȃ)
 * \return 폜ꂽGg̐
 */
int CWatchThread::sweepDeletedEntries(const FILETIME &ftNow, bool &bSkipDeletePending)
{
	const DWORD pendingLimit = (DWORD) (settingInfo_.getDeletePending() * 100);

	int numOfModified = 0; // Gg̕ύX
	int numOfDeletePending = 0; // 폜ҋ@̃t@C

	for (FileNameList::iterator ite_pendingList = pendingList_.begin();
		ite_pendingList != pendingList_.end();) {
		const tstring& absoluteName = *ite_pendingList;
		CFileInfoMap::iterator ite_fileInfo = fileInfoMap_.find(absoluteName);
		
		bool removed = false;
		if (ite_fileInfo == fileInfoMap_.end()) {
			// 폜ҋ@XgɁAt@CGg}bvɑ݂Ȃt@Cw肳ĂꍇA
			// łɍ폜ς݂Ȃ̂ō폜ҋ@Xg폜B
			// (폜\̓Gg̑ɓsxč쐬ꓯĂ͂Ȃ̂Ŗ{肦ȂB)
			assert(false && "폜\ɃGgɂȂڂݒ肳Ă܂B");
			removed = true;
		}
		else {
			CFileInfo& fileInfo = ite_fileInfo->second;
			if (fileInfo.isExist()) {
				// t@Ĉ݂ō폜ҋ@Xg珜
				// (폜\̓Gg̑ɓsxč쐬ꓯĂ͂Ȃ̂Ŗ{肦ȂB)
				assert(false && "폜\ɃGg݂鍀ڂݒ肳Ă܂B");
				removed = true;
			}
			else if (fileInfo.isDeletePending()) {
				if (pendingLimit == 0 || IsExpired(ftNow, pendingLimit, fileInfo.getDeletePendingBegin())) {
					// 폜ҋ@ΏۂłA폜Aw莞Ԃo߂Ă̂
					// t@CGg}bvƁA폜ҋ@Xg̑o폜B
					fileInfoMap_.erase(ite_fileInfo);
					removed = true;
				}
				else {
					// 폜҂̃t@ČJEgB
					numOfDeletePending++;
				}
			}
		}

		if (removed) {
			// ڍ׃O
			if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
				pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEENTRY, absoluteName.c_str());
			}

			// 폜
			numOfModified++;
			ite_pendingList = pendingList_.erase(ite_pendingList);
			continue;
		}
		++ite_pendingList;
	}

	// 폜҂t@C[ł邩肷B
	assert(numOfDeletePending >= 0 && "폜҂t@C̃JEgɌ肪܂B");
	if (numOfDeletePending == 0) {
		// 폜҂t@C[ł΁AɃt@CGgύX܂
		// 폜菈͕svł邱ƂtO
		bSkipDeletePending = true;
		if (pLogger_->GetLogThreshold() >= 10) {
			pLogger_->LogMessage(10, NULL, 0, _T("no undecided file to delete."));
		}
	}

	assert(numOfModified >= 0 && "폜t@CsłB");
	return numOfModified;
}

/*!
 * ĎfBNg𑖍܂B
 * v_enableEventtrueł΁Aǉ܂͍XVɂm҂tOĂ܂B
 * falsȅꍇ̓GgǉEXVE폜邾Ŋm҂͔܂B
 * ǉEύXE폜ꂽGg̐Ԃ܂B
 * \param ftNow ݓ
 * \param v_enableEvent m҂tOĂꍇtrue
 * \return ǉEύXE폜ꂽGg̐
 */
int CWatchThread::EnumFile(const FILETIME &ftNow, const bool v_enableEvent)
{
	PropertyStringExpander expander;
	app.setAppConfVariablesTo(expander);
	tstring watchDir = expander.expandString(settingInfo_.getWatchDir());

	int numOfModified = 0;
	if (EnumFileRecursive(ftNow, watchDir, 0, v_enableEvent, numOfModified)) {
		// ł́AJnfBNg̃t@C񋓂ꍇ̂ݍ폜ҋ@XgZbgĂ܂B
		// āAJnfBNgsꍇ͍폜͎sł܂B
		// (񋓂̂̂sꍇ͈ꎞIȏQR̉\At@C݂ȂǂsȈׁB)

		// U폜L[NAčč쐬B
		pendingList_.clear();

		for (CFileInfoMap::iterator ite_fileInfo = fileInfoMap_.begin();
			ite_fileInfo != fileInfoMap_.end();) {
			const tstring& absolutePath = ite_fileInfo->first;
			CFileInfo& fileInfo = ite_fileInfo->second;

			// ~CxgĂ邩mFB
			CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

			if ( !fileInfo.isExist()) {
				// Ggɑ݂邪̌ʁA݂Ȃ̂̏ꍇ

				if (settingInfo_.getDeletePending() <= 0) {
					// 폜ҋ@Ȃ̏ꍇ͑ɃGgėǂ
					numOfModified++;

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
						pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEENTRY, absolutePath.c_str());
					}

					ite_fileInfo = fileInfoMap_.erase(ite_fileInfo);
					continue;
				}

				if ( !fileInfo.isDeletePending()) {
					// 폜\łȂ̂͐Vɍ폜\ɂB
					// (łɍ폜\̂̂́Â܂܁B)
					fileInfo.setDeletePending(true);
					fileInfo.setDeletePendingBegin(ftNow);
				}

				// 폜\L[ɓB
				// (ȑOɍ폜\ɂȂĂ̂܂ށB)
				pendingList_.push_back(absolutePath);

				// ڍ׃O
				if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 6) {
					pLogger_->LogMessage(6, NULL, 0, IDS_LOGMES_REMOVEPENDING, absolutePath.c_str(), fileInfo.str().c_str());
				}
			}
			++ite_fileInfo;
		}
	}
	// ǉEXVE폜ꂽGg̐Ԃ
	assert(numOfModified >= 0 && "ǉEXVE폜t@CsłB");
	return numOfModified;
}

/*!
 * ĎfBNgċAIɑ܂B
 * \param ftNow ݓ
 * \param v_baseDir fBNg
 * \param v_depth T[݈̌ʒu
 * \param v_enableEvent m҂tOĂꍇtrue
 * \param numOfModified ĎXgɕύXi[Q
 * \return (depth=0)fBNgɐtrueAłȂfalse
 */
bool CWatchThread::EnumFileRecursive(const FILETIME &ftNow,
									 const tstring& v_baseDir,
									 const int v_depth,
									 const bool v_enableEvent,
									 int &numOfModified)
{
	// ΏۃpX𐶐
	const tstring baseDir = EnsureLast(v_baseDir, _T("\\"));

	// ڍ׃O
	if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 8) {
		pLogger_->LogMessage(8, NULL, 0, IDS_LOGMES_SCAN, baseDir.c_str());
	}
	
	// ύXƎvt@C
	WIN32_FIND_DATA finddata = {0};
	HANDLE findHandle = ::FindFirstFile((baseDir + _T("*.*")).c_str(), &finddata);
	if (findHandle == INVALID_HANDLE_VALUE) {
		// ΏۂɃANZXs\ȂΉȂŖ߂B
		DWORD err = GetLastError();
		if (v_depth == 0) {
			// ŏ̃fBNg̃ANZXs̏ꍇAG[ƂB
			pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_FINDERROR, v_baseDir.c_str());
		} else {
			// TufBNg̃ANZXs̓Ox3
			pLogger_->LogMessage(3, NULL, err, IDS_LOGMES_FINDERROR, v_baseDir.c_str());
		}
		return false;
	}

	// ŏ̃tH_([g)ꂩJnꍇA܂A
	// ݂Ȃt@C폜邽߁AׂẴt@C݃tOZbgB
	// (lbg[N̈ꎞIȐؒf̉\̂߁AΏۂɃANZXłꍇ̂ݎsB)
	if (v_depth == 0) {
		for (CFileInfoMap::iterator ite = fileInfoMap_.begin();
			ite != fileInfoMap_.end();
			++ite)
		{
			CFileInfo& fileInfo = ite->second;
			fileInfo.setExist(false);
		}
	}

	for (;;) {
		// ~CxgĂ邩mFB
		CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

		// Ώۃt@C̐΃pX
		const tstring absolutePath = baseDir + finddata.cFileName;
		if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 10) {
			pLogger_->LogMessage(10, NULL, 0, IDS_LOGMES_SCAN_ENTRIES, finddata.dwFileAttributes, absolutePath.c_str());
		}

		if (finddata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
			// fBNg̏ꍇA
			if (_tcscmp(finddata.cFileName, _T(".")) != 0 && _tcscmp(finddata.cFileName, _T("..")) != 0) {
				if (v_depth < settingInfo_.getMaxDepth()) {
					// TufBNg̐[ɒBĂȂ΃TufBNg
					// (TufBNǧs̗L͕s)
					EnumFileRecursive(ftNow, absolutePath, v_depth + 1, v_enableEvent, numOfModified);
				}
			}
		}
		else if (!settingInfo_.isArchiveOnly() || (finddata.dwFileAttributes & FILE_ATTRIBUTE_ARCHIVE) != 0) {
			// t@C̏ꍇA
			// AA[JCû݂̏ꍇ̓A[JCuȂ̂͌oȂB

			if (nameMatcher_(finddata.cFileName)) {
				// Ώۃt@CɃ}b`ꍇA

				CFileInfoMap::iterator ite = fileInfoMap_.find(absolutePath);
				if (ite != fileInfoMap_.end()) {
					// łɃGgɓo^ς݂̃t@C
					CFileInfo& fileInfo = ite->second;

					unsigned __int64 fileSize = CFileInfo::GetFileSize(finddata);
					if (v_enableEvent && (
							!fileTimeComparator_(finddata.ftLastWriteTime, fileInfo.getLastModified()) ||
							fileSize != fileInfo.getSize())) {
						// Cxgʒm胂[hŁAAt@C̍XVA܂̓TCYύXĂꍇ
						// ύXmZbgB(łɊm҂̏ꍇ͍Ċm҂ƂȂB)
						fileInfo.setDetectModified(true);
						fileInfo.setDetectBegin(ftNow);

						// ύXʒmO
						pLogger_->LogMessage(5, NULL, 0, IDS_LOGMES_FILEMODIFIED, absolutePath.c_str());
					}
					// t@C̏XV.
					if (fileInfo.update(finddata)) {
						numOfModified++;
					}

					// Ă邱Ƃ
					fileInfo.setExist(true);

					// 폜\B
					fileInfo.setDeletePending(false);

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 9) {
						pLogger_->LogMessage(9, NULL, 0, IDS_LOGMES_UPDATEENTRY, absolutePath.c_str(), fileInfo.str().c_str());
					}
				}
				else {
					// 炽ɃGgɒǉt@C
					bool detectModified = v_enableEvent;
					CFileInfo fileInfo(finddata, detectModified, true);
					fileInfo.setDeletePending(false);
					fileInfo.setDetectBegin(ftNow);
					fileInfoMap_.insert(CFileInfoMap::value_type(absolutePath, fileInfo));
					numOfModified++;

					// ڍ׃O
					if (pLogger_->IsEnableLogging() && pLogger_->GetLogThreshold() >= 9) {
						pLogger_->LogMessage(9, NULL, 0, IDS_LOGMES_ADDENTRY, absolutePath.c_str(), fileInfo.str().c_str());
					}
					// ύXʒmO
					if (detectModified) {
						pLogger_->LogMessage(5, NULL, 0, IDS_LOGMES_FILEMODIFIED, absolutePath.c_str());
					}
				}
			}
		}
		if ( !::FindNextFile(findHandle, &finddata)) {
			DWORD err = GetLastError();
			if (err != ERROR_NO_MORE_FILES) {
				// ̃t@C̓ǂݎɎsꍇ
				if (v_depth == 0) {
					// ŏ̃fBNg̃ANZXs̏ꍇAG[ƂB
					pLogger_->LogMessage(0, NULL, err,IDS_LOGMES_FINDNEXTERROR, v_baseDir.c_str());
				} else {
					// TufBNg̃ANZXs̓Ox3
					pLogger_->LogMessage(3, NULL, err,IDS_LOGMES_FINDNEXTERROR, v_baseDir.c_str());
				}
			}
			::FindClose(findHandle);
			break;
		}
	}

	return true;
}

/*!
 * m҂̃Ggm肵`FbNAm肵ꍇAANVs܂B
 * ̊m蔻ɂ蕡̃Ggm肷ꍇAs̓t@C̍XV̌ÂA
 * łʂƂȂꍇ͐΃pX̏ōs܂B
 * \param ftNow ݎ
 * \param bSkipCommitPending m҂t@CA܂͊mt@CȂꍇtrueɐݒ肳B(falseւ̐ݒ͂Ȃ)
 * \param bAsyncProcess 񓯊Ɏsꂽꍇtrueɐݒ肳B(falseւ̐ݒ͂Ȃ)
 * \param impersonateUser [ŰUIuWFNg
 * \return m肵Gg
 */
int CWatchThread::CheckDoAction(const FILETIME &ftNow, bool &bSkipCommitPending, bool &bAsyncProcess, CImpersonateUser& impersonateUser)
{
	const DWORD writeWaitLimit = (DWORD) (settingInfo_.getWaitWrite() * 100);

	// m肷ł낤Xg
	typedef std::list<std::pair<tstring, CFileInfo*> > candidates_t;
	candidates_t candidates;

	// Ċm҂A͊mς݂̃t@C
	int numOfDelected = 0;

	// ANV⒆AANVsꂽt@C (t@CGg̕ύXƓB)
	int numOfCommited = 0;

	// m҂Ggm肵肷B
	for (CFileInfoMap::iterator ite = fileInfoMap_.begin();
		ite != fileInfoMap_.end();
		++ite) {

		// ~CxgĂ邩mFB
		CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

		const tstring& absolutePath = ite->first;
		CFileInfo& fileInfo = ite->second;

		if (fileInfo.isDetectModified() && !fileInfo.isDeletePending()) {
			// ύXmꂽ(m҂)ACeA폜\łȂƁB
			numOfDelected++;
			if (IsExpired(ftNow, writeWaitLimit, fileInfo.getDetectBegin())) {
				// t@CTCY܂őҋ@Ԃo߂Ăꍇ
				unsigned __int64 nFileSize = 0;
				FILETIME lastModified = {0};
				DWORD attrs = 0;
				DWORD errcode = NO_ERROR;
				bool bAvailableFileInfo = getFileInfo(absolutePath, &nFileSize, &lastModified, &attrs);
				if ( !bAvailableFileInfo) {
					errcode = GetLastError();
				}
				if ( !bAvailableFileInfo
					|| (fileInfo.getSize() != nFileSize)
					|| !fileTimeComparator_(fileInfo.getLastModified(), lastModified)) {
					// t@C̏ǂݎI[v[hŎ擾łȂA
					// t@C̍XVt܂̓TCYύXĂ΍Ċm҂
					// (̕ύX͖BƂA[JCuƂĂ̍đɔCB)
					if (bAvailableFileInfo) {
						// 擾łŐV̏ōXVB
						fileInfo.setAttributes(attrs);
						fileInfo.setSize(nFileSize);
						fileInfo.setLastModified(lastModified);
					}
					fileInfo.setDetectBegin(ftNow);
					// O
					if (pLogger_->GetLogThreshold() >= 9) {
						pLogger_->LogMessage(9, NULL, errcode, IDS_LOGMES_FILEMODIFY_RETRY, absolutePath.c_str());
					}
				}
				else {
					// t@CTCYɕωȂꍇ́AقڊmB
					// (Ƃ̓ANVsł邩ۂB)
					candidates.push_back(candidates_t::value_type(absolutePath, &fileInfo));
				}
			}
		}
	}

	if ( !candidates.empty()) {
		// t@C̍XV̌Â()At@C̍XVΐ΃pX̏ŔrRp[^
		struct PredicateWriteTime
		{
			bool operator()(const candidates_t::value_type &a, const candidates_t::value_type &b)
			{
				const tstring& absolutePath_a = a.first;
				const tstring& absolutePath_b = b.first;
				const CFileInfo* pFileInfo_a = a.second;
				const CFileInfo* pFileInfo_b = b.second;

				int ret = pFileInfo_a->compare(*pFileInfo_b);
				if (ret == 0) {
					ret = absolutePath_a.compare(absolutePath_b);
				}
				return ret < 0;
			}
		};

		// t@C̍XV̌Â()At@C̍XVΐ΃pX̏Ń\[gB
		candidates.sort(PredicateWriteTime());

		// mɑ΂Ďsł邩mFȂAsłȂΊm肵Ă䂭B
		for (candidates_t::iterator ite = candidates.begin();
			ite != candidates.end();
			++ite) {

			// ~CxgĂ邩mFB
			CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

			if ( !pActionInvoker_->canDoAction()) {
				// vZXEɒB̂ŁAŏI
				break;
			}

			const tstring& absolutePath = ite->first;
			CFileInfo &fileInfo = *ite->second;

			// sς݂Ƃ݂Ȃ߁AύXmtOoffB
			fileInfo.setDetectModified(false);
			numOfCommited++;

			// s񐔂̃JEgAbv
			fileInfo.incrementRunCount();
			
			// O
			pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_FILEMODIFY_COMMITED, absolutePath.c_str());
			
			// ANVsB
			// ANVɋUw肳Ă΁AUԂUB
			bool impersonateOld = bRevertToSelfOnAction_ && impersonateUser.revertToSelf();
			try {
				if (pActionInvoker_->doAction(absolutePath, fileInfo)) {
					// 񓯊ɎsꂽƂtOݒ肷.
					// ([vĂяôtrueɂ邱Ƃ͂Ăfalseɂ邱Ƃ͂ȂB)
					bAsyncProcess = true;
				}
			}
			catch (...) {
				// OꍇAUԂ߂Ă烊X[
				if (impersonateOld) {
					impersonateUser.impersonate();
				}
				throw;
			}
			if (impersonateOld) {
				// ANVsOɋUĂAUĊJB
				try {
					impersonateUser.impersonate();
				}
				catch (const CImpersonateUserException&) {
					if ( !bContinueWhenFailedImpersonate_) {
						// OC[ŰUɎsĂpwȂ
						// OX[ăXbhIB
						// łȂ΁A̗O͖B
						throw;
					}
				}
			}
		}
	}

	// m҂A͊m肵Ăt@C[ł邩?
	assert((numOfDelected - numOfCommited) >= 0 && "m҂A͊mt@C̃JEgɌ肪܂B");
	if ((numOfDelected - numOfCommited) == 0) {
		// Ƀt@CGgXVȂ`FbNsvł邱ƂB
		bSkipCommitPending = true;
		if (pLogger_->GetLogThreshold() >= 10) {
			pLogger_->LogMessage(10, NULL, 0, _T("no undecided file to commit."));
		}
	}

	assert(numOfCommited >= 0 && "mt@CsłB");
	return numOfCommited;
}

void CWatchThread::run()
{
	// t@CɎgpfBNgpX̍쐬
	PropertyStringExpander expander;
	app.setAppConfVariablesTo(expander);
	tstring watchDir = expander.expandString(settingInfo_.getWatchDir());
	const tstring baseDir = EnsureLast(watchDir, _T("\\"));

	// JnbZ[W
	pLogger_->LogMessage(
		1,
		NULL,
		0,
		IDS_LOGMES_INIT,
		baseDir.c_str(),
		settingInfo_.getMaxDepth(),
		settingInfo_.getWatchFile().c_str()
		);

	// iۑvtO
	bool bPersistRequired = true;
	// ivtO
	bool bForcePersist = false;

	try {
		// [UUB
		CImpersonateUser impersonateUser;
		impersonateUser.SetLogger(pLogger_.get());
		try {
			if (settingInfo_.isImpersonate()) {
				impersonateUser.logon(
					settingInfo_.getImpersonateUser(),
					settingInfo_.getImpersonateDomain(),
					settingInfo_.getImpersonatePassword()
					);
				// UJnB
				impersonateUser.impersonate();
			}
		}
		catch (const CImpersonateUserException&) {
			if ( !bContinueWhenFailedImpersonate_) {
				// OC[ŰUɎsĂpwȂ
				// OX[ăXbhIB
				// łȂ΁A̗O͖B
				throw;
			}
		}

		FILETIME ftNow;
		::GetSystemTimeAsFileTime(&ftNow);

		FILETIME lastSweep = FILETIME();
		FILETIME lastHeapCompact = FILETIME();
		FILETIME lastForceTickCount = FILETIME();
		FILETIME lastCheck = FILETIME();
		FILETIME lastSavePersist = FILETIME();

		// Ďnh
		WatchHandle watchHandle(settingInfo_);
		watchHandle.SetLogger(pLogger_.get());

		// ʒmnh̍쐬(KvȂ)
		watchHandle.checkAndRenew();

		// Ďt@CXg[h
		bPersistRequired = false;
		bool loaded = Load();
		if (loaded) {
			lastSavePersist = ftNow;
			// O (Ox1)
			pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_LOADPERSIST, fileInfoMap_.size());
		}

		// ÑXg̏
		// Ďt@CXg̕ɐꍇ͑Oۑ̏ԕω
		// Ď邽߃CxgsB(ƂĂ͏񑶍݂t@C͖`ɂȂB)
		// ́uNɑ݂t@Cʒmvw肳ĂꍇlB
		// ȊO͏Nɑ݂t@CCxgȂŎ擾B
		int numOfModified = EnumFile(ftNow, loaded || settingInfo_.isNotifyOnStart());
		if (numOfModified > 0 || !loaded) {
			// ĎԂĂȂAύXmꂽꍇ͉iKvB
			bPersistRequired = true;
		}
		lastForceTickCount = ftNow;
		pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_INITLIST, numOfModified, fileInfoMap_.size());

		pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_START);

		// tH_vtO
		bool bEnumerationRequired = false;

		// 폜ҋ@݂ȂƂm肵ĂtO
		bool bSkipDeletePending = false;
		// mҋ@݂ȂƂm肵ĂtO
		bool bSkipCommitPending = false;
		// IĎvZX݂ȂƂm肵ĂtO
		bool bSkipProcessPending = false;
		// 1ÕTCNł̊emtÕrbg
		DWORD prevState = 0;

		// t@CύXʒm[v̊Jn
		for (;;) {
			// ݎ̎擾
			::GetSystemTimeAsFileTime(&ftNow);

			// Gg̍폜
			if ( !bSkipDeletePending && IsExpired(ftNow, spanSweepDeletedEntries_, lastSweep)) {
				int numOfBeforeEntries = static_cast<int>(fileInfoMap_.size());
				numOfModified = sweepDeletedEntries(ftNow, bSkipDeletePending);
				if (numOfModified > 0) {
					bPersistRequired = true;
					if (pLogger_->GetLogThreshold() >= 5) {
						pLogger_->LogMessage(5, NULL, 0, IDS_LOGMES_REMOVELIST, numOfModified, numOfBeforeEntries);
					}
				}
				// ݎ̍Ď擾(Gg폜)
				::GetSystemTimeAsFileTime(&ftNow);
				lastSweep = ftNow;
			}

			// q[v̈k(s)
			if (spanHeapCompaction_ > 0 && IsExpired(ftNow, spanHeapCompaction_, lastHeapCompact)) {
				pLogger_->LogMessage(10, NULL, 0, _T("heap compaction"));
				_heapmin();
				lastHeapCompact = ftNow;
			}

			// OC^ɒr[g𑗐MB
			pLogger_->Tick(ftNow);

			// vZX̏I҂ĂGgAIς݂̃vZXN[ibvB
			if (pActionInvoker_->sweepGarbage() == 0) {
				bSkipProcessPending = true;
			}

			// ANV̎s
			bool bAsyncProcess = false;
			if ( !bSkipCommitPending && CheckDoAction(ftNow, bSkipCommitPending, bAsyncProcess, impersonateUser) > 0) {
				bPersistRequired = true;
			}
			if (bAsyncProcess) {
				bSkipProcessPending = false;
			}

			// Xe[g̎Z
			DWORD state = (bSkipDeletePending ? 1 : 0)
						| (bSkipCommitPending ? 2 : 0)
						| (bSkipProcessPending ? 4 : 0);
			DWORD waitTime = (state == 0x07) ? baseCycleOnIdle_ : baseCycle_;
			if (state != prevState) {
				if (pLogger_->GetLogThreshold() >= 10) {
					pLogger_->LogMessage(10, NULL, 0, IDS_STATECHANGE, state, waitTime);
				}
				prevState = state;
			}

			// Cxgҋ@nh
			// 0: ~Cxg
			// 1: tH_Ďnh(ȂNULL)
			HANDLE findHandle = (HANDLE) watchHandle;
			HANDLE handles[] = {
				hStopEvent_,
				NULL
			};
			int handleCount = 1;
			if (findHandle != INVALID_HANDLE_VALUE) {
				handles[1] = findHandle;
				handleCount = 2;
			}

			// Cxgҋ@B
			// bZ[WbaseCycleȑOɖ߂\B
			const DWORD ret = ::WaitForMultipleObjects(
				handleCount,
				handles,
				FALSE,
				waitTime
				);
			if (ret == WAIT_OBJECT_0 || ret == WAIT_ABANDONED_0) {
				// ~CxgB
				pLogger_->LogMessage(10, NULL, 0, _T("stop event"));
				break;
			}
			
			if (ret == WAIT_OBJECT_0 + 1 || ret == WAIT_ABANDONED_0 + 1) {
				// tH_ĎCxgB
				bEnumerationRequired = true;

				// O
				pLogger_->LogMessage(3, NULL, 0, IDS_LOGMES_CHANGENOTIFY);

				// ̃tH_ĎCxg󗝉\ɂB
				if ( !::FindNextChangeNotification(findHandle)) {
					// tH_ĎCxg̎󗝍ĊJɎsꍇ
					DWORD err = GetLastError();
					pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_WATCHNEXTERROR, baseDir.c_str());

					// nhU邪͌pB
					watchHandle.Close();
				}
			}

			// Ďnhč쐬B(KvȂ)
			watchHandle.checkAndRenew();

			// XVw肳ĂAŌɌĂw莞Ԃo߂Ă邽ߋsB
			if (settingInfo_.getForceInterval() > 0 &&
				IsExpired(ftNow, settingInfo_.getForceInterval() * 1000L, lastForceTickCount)) {
				bEnumerationRequired = true;
				lastForceTickCount = ftNow;

				// O
				pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_FORCEINTERVAL);

				// tH_`FbN͖ɉis?
				if (bPersistAtForceInterval_) {
					bForcePersist = true;
				}
			}

			// tH_̑v΁AtH_𑖍B
			if (bEnumerationRequired) {
				DWORD watchDelay = settingInfo_.getWatchDelay();
				if (watchDelay == 0 || IsExpired(ftNow, watchDelay, lastCheck)) {
					// AAʒmЂƂɂ܂Ƃ߂Ԃ0ȏł΁A
					// ŌɑĂwԂo߂ĂȂΑȂB
					bEnumerationRequired = false;
					lastCheck = ftNow;
					numOfModified = EnumFile(ftNow, true);
					if (numOfModified > 0) {
						bPersistRequired = true;
					}
					bSkipDeletePending = false;
					bSkipCommitPending = false;
					pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_UPDATELIST, numOfModified, fileInfoMap_.size());

					// ݎ̍Ď擾()
					::GetSystemTimeAsFileTime(&ftNow);
				}
			}

			// t@CXgɕωAŌɕۑĂw莞Ԍo߂Ă
			// i{B
			// (͋iwĂꍇ)
			if (bForcePersist ||
				(bPersistRequired && spanSavePersist_ > 0 && IsExpired(ftNow, spanSavePersist_, lastSavePersist))) {
				if (Save()) {
					// ꍇ͉itOZbg
					bPersistRequired = false;

					// O (Ox7)
					pLogger_->LogMessage(7, NULL, 0, IDS_LOGMES_SAVEPERSIST, fileInfoMap_.size());
				}
				// (iɎsĂAĎsȂ悤ɂ邽߁A̗L킸̃C^[oJnB)
				bForcePersist = false;
				lastSavePersist = ftNow;
			}
		}
	}
	catch (const CWatchThreadInterruptedException&) {
		// ~Cxgɂ銄荞݂ɂI
		pLogger_->LogMessage(10, NULL, 0, _T("stop interrupted."));
	}
	catch (const CImpersonateUserException&) {
		assert(false && "OC[ŰUŃG[܂B");
		// OC[ŰUɎsꍇ
		// O͏o͍ς݂Ȃ̂ŁAł͏Î݁B
	}
	catch (...) {
		pLogger_->LogMessage(0, NULL, 0, IDS_LOGMES_UNKNOWNEXCEPTION);
		throw; // X[ăAvP[VG[ɂB(ɂVXeɏQOL^B)
	}

	try {
		// Ďt@CXgۑB
		if (bPersistRequired && Save()) {
			// O (Ox1)
			pLogger_->LogMessage(1, NULL, 0, IDS_LOGMES_SAVEPERSIST, fileInfoMap_.size());
		}

		// Ďt@CXgj
		fileInfoMap_.clear();

		// IbZ[W
		pLogger_->LogMessage(
			1,
			NULL,
			0,
			IDS_LOGMES_FINISHED,
			baseDir.c_str(),
			settingInfo_.getMaxDepth(),
			settingInfo_.getWatchFile().c_str()
			);
	}
	catch (...) {
		pLogger_->LogMessage(0, NULL, 0, IDS_LOGMES_UNKNOWNEXCEPTION);
		throw; // X[ăAvP[VG[ɂB(ɂVXeɏQOL^B)
	}
}


unsigned __stdcall CWatchThread::ThreadProc(void* pParam)
{
	SetThreadName((DWORD) -1, "FWatchThread");

	CWatchThread* me = (CWatchThread*) pParam;
	if (FAILED(::CoInitializeEx(NULL, COINIT_APARTMENTTHREADED))) {
		app.showMessageBox(MB_SERVICE_NOTIFICATION | MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_COINITFAILED);
		return 0;
	}
	try {
		me->run();
	}
	catch (...) {
		::CoUninitialize();
		throw;
	}
	::CoUninitialize();
	return 0;
}



// it@C̎ʎq 
#ifdef _UNICODE
// {7AFE7178-9D7D-4d2e-89A9-65EF4D078ABB}
static const GUID SERIALIZE_SIG = { 0x7afe7178, 0x9d7d, 0x4d2e, { 0x89, 0xa9, 0x65, 0xef, 0x4d, 0x7, 0x8a, 0xbb } };
#else
// {9082879D-C96D-4a01-8672-6E65681CFE7B}
static const GUID SERIALIZE_SIG = { 0x9082879d, 0xc96d, 0x4a01, { 0x86, 0x72, 0x6e, 0x65, 0x68, 0x1c, 0xfe, 0x7b } };
#endif

bool CWatchThread::Save()
{
	if (persist_.empty()) {
		return false;
	}
	HANDLE hPersist = ::CreateFile(
		persist_.c_str(),
		GENERIC_READ | GENERIC_WRITE,
		FILE_SHARE_READ,
		NULL,
		CREATE_ALWAYS,
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
		NULL);
	if (hPersist == INVALID_HANDLE_VALUE) {
		DWORD err = GetLastError();
		pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_SAVEPERSIST_ERROR, _T(""), persist_.c_str());
		return false;
	}

	bool succeeded = false;
	try {
		CObjectWriter writer(hPersist);
		
		// ANV̏Ԃ̕ۑ
		pActionInvoker_->Save(writer);

		// ʎq̏
		writer.WriteSig(SERIALIZE_SIG);

		DWORD numOfInfos = static_cast<DWORD>(fileInfoMap_.size());
		writer.WriteDWORD(numOfInfos);

		// ĎACe̕ۑ
		DWORD count = 0;
		for (CFileInfoMap::const_iterator ite = fileInfoMap_.begin();
			ite != fileInfoMap_.end();
			++ite) {
			const tstring& absolutePath = ite->first;
			const CFileInfo& fileInfo = ite->second;

			writer.WriteSig(count++);
			writer.WriteString(absolutePath);
			fileInfo.Save(writer);
		}

		if (!::FlushFileBuffers(hPersist)) {
			DWORD err = GetLastError();
			throw CObjectReaderWriterException(err);
		}

		succeeded = true;
	}
	catch (const CObjectReaderWriterException &v_exception) {
		DWORD err = v_exception.GetErrorCode();
		pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_SAVEPERSIST_ERROR, _T(""), persist_.c_str());
	}
	catch (const std::exception &v_exception) {
		const char *pMes = v_exception.what();
		CA2T mes(pMes ? pMes : "");
		pLogger_->LogMessage(0, NULL, 0, IDS_LOGMES_SAVEPERSIST_ERROR, (LPCTSTR)mes, persist_.c_str());
	}

	::CloseHandle(hPersist);

	return succeeded;
}

bool CWatchThread::Load()
{
	if (persist_.empty()) {
		return false;
	}
	HANDLE hPersist = ::CreateFile(
		persist_.c_str(),
		GENERIC_READ,
		FILE_SHARE_READ,
		NULL,
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
		NULL);
	if (hPersist == INVALID_HANDLE_VALUE) {
		DWORD err = GetLastError();
		if (err == ERROR_FILE_NOT_FOUND || err == ERROR_ACCESS_DENIED) {
			pLogger_->LogMessage(1, NULL, err, IDS_LOGMES_PERSIST_NOT_FOUND, persist_.c_str());
		}
		else {
			pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_LOADPERSIST_ERROR, _T(""), persist_.c_str());
		}
		return false;
	}

	ULARGE_INTEGER siz = {0};
	siz.LowPart = ::GetFileSize(hPersist, &siz.HighPart);
	DWORD err = GetLastError();
	if ((siz.LowPart == -1 && err != NO_ERROR) || siz.QuadPart == 0) {
		// t@CȂAt@C̃TCYǂݎɎsꍇ
		if (err != NO_ERROR) {
			// G[̏ꍇ̓OɏoB
			pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_LOADPERSIST_ERROR, _T(""), persist_.c_str());
		}
		::CloseHandle(hPersist);
		return false;
	}

	bool succeeded = false;
	try {
		CObjectReader reader(hPersist);
		
		// ANV̏Ԃ̕
		pActionInvoker_->Load(reader);

		// ʎq̌
		reader.VerifySig(SERIALIZE_SIG);

		// ݂̏̃NA
		fileInfoMap_.clear();

		// ۑĂACe̎擾
		DWORD numOfInfos = reader.ReadDWORD();

		// ĎACẽ[h
		for (DWORD count = 0; count < numOfInfos; count++) {

			// ~CxgĂ邩mFB
			CWatchThreadInterruptedException::checkAndThrow(hStopEvent_);

			// Gg̊Jn}[N̊mF
			reader.VerifySig(count);

			tstring absolutePath;
			reader.ReadString(absolutePath);

			CFileInfo fileInfo;
			fileInfo.Load(reader);

			// 폜ҋ@łȂΎ݃tOĂB
			// (fBNg͎݃tO͈UZbgԂɂ邽
			// ̊Ԓɒ~ꍇ͎݃tO͞BƂȂB)
			if ( !fileInfo.isDeletePending()) {
				fileInfo.setExist(true);
			}

			fileInfoMap_.insert(CFileInfoMap::value_type(absolutePath, fileInfo));
		}

		// s񐔂ZbgB(KvȂ΁B)
		if (nResetRunCount_ > 0) {
			pActionInvoker_->Reset();
		}
		if (nResetRunCount_ > 1) {
			for (CFileInfoMap::iterator ite = fileInfoMap_.begin();
				ite != fileInfoMap_.end();
				++ite) {
				CFileInfo& fileInfo = ite->second;
				fileInfo.setRunCount(0);
			}
		}

		succeeded = true;
	}
	catch (const CWatchThreadInterruptedException&) {
		// ~Cxgɂ銄荞݂ɂ钆f
		// X[ČĂяoƂŏB
		throw;
	}
	catch (const CObjectReaderWriterException &v_exception) {
		DWORD err = v_exception.GetErrorCode();
		pLogger_->LogMessage(0, NULL, err, IDS_LOGMES_LOADPERSIST_ERROR, _T(""), persist_.c_str());
	}
	catch (const std::exception &v_exception) {
		const char *pMes = v_exception.what();
		CA2T mes(pMes ? pMes : "");
		pLogger_->LogMessage(0, NULL, 0, IDS_LOGMES_LOADPERSIST_ERROR, (LPCTSTR)mes, persist_.c_str());
	}

	::CloseHandle(hPersist);

	if ( !succeeded) {
		// G[ꍇ͕sSȏԂ̉\邽߁A
		// ԂZbgB
		fileInfoMap_.clear();
	}

	return succeeded;
}
