# -*- coding: ascii -*-
#
#  phrases.py - Phrase manipulator for GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: phrases.py,v 1.15 2004/08/18 10:19:13 atzm Exp $
#

import os, sys, re, string
import gtk

from common import *

class PhraseManager:
	DBFILE = "phrases"
	DBWRITEFREQ = 10 # write phrases to file per 10 operations
	def __init__(self):
		self.dbpath = os.path.join(open_bottlecase(), self.DBFILE)
		self.dblife = self.DBWRITEFREQ
		self.load_database()

	def load_database(self):
		self.phrases = []
		try:
			file = open(self.dbpath)
		except IOError:
			return
		while 1:
			line = unicode(file.readline(), "utf-8")
			if not line:
				break
			elif line[0] == "#":
				continue
			elif line[-2:] == "\r\n":
				line = line[:-2]
			elif line[-1] in ["\r", "\n"]:
				line = line[:-1]
			self.phrases.append(line)
		file.close()

	def save_database(self):
		try:
			file = open(self.dbpath, "w")
		except IOError:
			sys.stderr.write("Error: cannot write %s\n" % self.dbpath)
			return
		file.write("# This is an auto-generated file.\n")
		file.write("# Don't edit this file while running %s!\n" % APP)
		for line in self.phrases:
			file.write(line.encode("utf-8") + "\n")
		file.close()

	def close(self):
		self.save_database()

	def sync(self):
		self.dblife = self.dblife - 1
		if self.dblife == 0:
			self.dblife = self.DBWRITEFREQ
			self.save_database()

	def overwrite(self, phrases):
		self.phrases = phrases

	def list(self):
		return self.phrases[:]

	def get(self, index):
		return self.phrases[index]

	def append(self, phrase):
		self.phrases.append(phrase)
		self.sync()

	def insert(self, index, phrase):
		self.phrases.insert(index, phrase)
		self.sync()

	def modify(self, index, phrase):
		self.phrases[index] = phrase
		self.sync()

	def delete(self, index):
		del self.phrases[index]
		self.sync()

	def move_up(self, index):
		if index - 1 < 0:
			return
		self.phrases[index], self.phrases[index - 1] = \
							 self.phrases[index - 1], self.phrases[index]
		self.sync()

	def move_down(self, index):
		if index + 1 >= len(self.phrases):
			return
		self.phrases[index], self.phrases[index + 1] = \
							 self.phrases[index + 1], self.phrases[index]
		self.sync()

class PhraseWindow:
	TARGET_DEFINES = 0
	TARGET_SCRIPT  = 1

	def __init__(self, app, manager):
		self.app = app
		self.manager = manager
		accel_group = gtk.AccelGroup()

# === Create Menubar === #
		self.set_self_msg()
		menu_items = self.get_menu_items()

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.menu_list = {}
		for m in menu_items:
			# keeping ref to each items
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		self.menubar = self.item_factory.get_widget("<main>")
		self.menubar.show()

		# keep edit menu as popup menu
		self.popup_menu = self.item_factory.get_widget("/%s" % self.app.msg["edit"][1])

# === Create Buttons === #
		nb  = get_icon_button(gtk.STOCK_NEW)
		cb  = get_icon_button(gtk.STOCK_COPY)
		db  = get_icon_button(gtk.STOCK_DELETE)
		upb = get_icon_button(gtk.STOCK_GO_UP)
		dnb = get_icon_button(gtk.STOCK_GO_DOWN)

		nb.connect('clicked', self.insert_new)
		cb.connect('clicked', self.edit_duplicate)
		db.connect('clicked', self.delete)
		upb.connect('clicked', self.edit_up)
		dnb.connect('clicked', self.edit_down)

		self.tooltips = gtk.Tooltips()
		self.tooltips.set_tip(nb, self.msg['new'])
		self.tooltips.set_tip(cb, self.msg['duplicate'])
		self.tooltips.set_tip(db, self.app.msg['delete'])
		self.tooltips.set_tip(upb, self.msg['up'])
		self.tooltips.set_tip(dnb, self.msg['down'])

# === Bring Buttons together to HBox === #
		bbox = gtk.HBox(gtk.FALSE, 0)
		bbox.show()

		bbox.pack_start(nb, gtk.FALSE, gtk.TRUE, 0)
		bbox.pack_start(cb, gtk.FALSE, gtk.TRUE, 0)
		bbox.pack_start(db, gtk.FALSE, gtk.TRUE, 0)
		bbox.pack_start(upb, gtk.FALSE, gtk.TRUE, 0)
		bbox.pack_start(dnb, gtk.FALSE, gtk.TRUE, 0)

		bf = gtk.Frame()
		bf.show()
		bf.set_shadow_type(gtk.SHADOW_OUT)
		bf.add(bbox)

# === Create CList for Phrases View === #
		self.entries = gtk.CList(2, (unicode(_('Name'), 'utf-8'), unicode(_('Script'), 'utf-8')))
		self.entries.show()
		self.entries.set_shadow_type(gtk.SHADOW_IN)
		self.entries.set_selection_mode(gtk.SELECTION_BROWSE)
		self.entries.set_reorderable(gtk.TRUE)
		self.entries.set_column_width(0, 100)
		self.entries.set_column_auto_resize(1, gtk.TRUE)

		self.entries.connect("select-row", self.select)
		self.entries.connect("button-press-event", self.popup)
		self.entries.connect("key-press-event", self.keypress)

		# connect self.sort for sort
		self.skey = [gtk.SORT_DESCENDING, gtk.SORT_DESCENDING]
		for i in range(len(self.skey)):
			self.entries.get_column_widget(i).get_parent().connect("clicked", self.sort, i)

		sw = gtk.ScrolledWindow()
		sw.show()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.set_size_request(200, 100)
		sw.add(self.entries)

# === Create Entries, name and script editor === #
		# name
		self.name_text = gtk.Entry()
		self.name_text.show()
		self.name_text.connect("activate", self.keypress)

		nf = gtk.Frame(unicode(_('Name'), 'utf-8'))
		nf.set_shadow_type(gtk.SHADOW_NONE)
		nf.show()
		nf.add(self.name_text)

		# script
		self.script_text = gtk.Entry()
		self.script_text.show()
		self.script_text.connect("activate", self.keypress)

		sf = gtk.Frame(unicode(_('Script'), 'utf-8'))
		sf.set_shadow_type(gtk.SHADOW_NONE)
		sf.show()
		sf.add(self.script_text)

# === Bring name and script editor together to HPaned === #
		hpane = gtk.HPaned()
		hpane.show()
		hpane.pack1(nf)
		hpane.pack2(sf)

# === Bring PhrasesView and EditorPane together to VPaned === #
		pane = gtk.VPaned()
		pane.show()
		pane.pack1(sw, gtk.TRUE)
		pane.pack2(hpane, gtk.FALSE, gtk.FALSE)

# === Create Buttons, OK, APPLY, CANCEL === #
		self.ok  = gtk.Button(unicode(_("OK"), "utf8"), gtk.STOCK_OK)
		self.ok.show()
		self.apl = gtk.Button(unicode(_("Apply"), "utf8"), gtk.STOCK_APPLY)
		self.apl.show()
		self.can = gtk.Button(unicode(_("Cancel"), "utf8"), gtk.STOCK_CANCEL)
		self.can.show()

		self.ok.connect('clicked', self.okey)
		self.apl.connect('clicked', self.apply)
		self.can.connect('clicked', self.cancel)

# === Bring Buttons together to HBox === #
		buttons = gtk.HBox()
		buttons.show()
		buttons.pack_end(self.ok, gtk.FALSE, gtk.TRUE, 0)
		buttons.pack_end(self.can, gtk.FALSE, gtk.TRUE, 0)
		buttons.pack_end(self.apl, gtk.FALSE, gtk.TRUE, 0)

# === Bring all items together to Main Window Finally === #
		self.main_vbox = gtk.VBox(gtk.FALSE, 0)
		self.main_vbox.show()

		self.main_vbox.pack_start(self.menubar, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(bf, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(pane, gtk.TRUE, gtk.TRUE, 0)
		self.main_vbox.pack_end(buttons, gtk.FALSE, gtk.TRUE, 0)

		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.add_accel_group(accel_group)
		self.window.set_size_request(350, 250)
		self.window.connect("delete_event", self.close)
		self.window.set_title(unicode(_(APP), "utf-8") + ":" + \
							  unicode(_("Phrases"), "utf-8"))
		self.window.add(self.main_vbox)

# === Initialization of PhraseWindow === #
		self.initialize()

	def edit_copy_clipboard_defines(self, widget=None, event=None):
		self.edit_copy_clipboard(widget, event, self.TARGET_DEFINES)
	def edit_copy_clipboard_script(self, widget=None, event=None):
		self.edit_copy_clipboard(widget, event, self.TARGET_SCRIPT)
	def edit_copy_clipboard(self, widget, event, target):
		if not self.entries.selection:
			return
		self.app.dispose_selection()
		self.app.copy_clipboard(unicode(self.entries.get_text(self.entries.selection[0], target), 'utf-8'))

	def keypress(self, widget, event=None):
		if type(widget) is gtk.CList:
			if event.keyval == 65535: # delete key
				self.delete(widget, event)
		elif type(widget) is gtk.Entry:
			self.apply(widget, event)

	def popup(self, widget, event):
		sel = self.entries.get_selection_info(int(event.x), int(event.y))
		if sel:
			self.set_focus_row(sel[0], self.entries.rows)
			if event.button == 3:
				self.popup_menu.popup(None, None, None, event.button, event.time)
			if event.type == 5:
				self.edit_copy(widget, event)

	def sort(self, widget, colnum):
		if type(widget) is gtk.Button:
			self.skey[colnum] = not self.skey[colnum]

		self.entries.freeze()
		self.entries.set_sort_type(self.skey[colnum])
		self.entries.set_sort_column(colnum)
		self.entries.sort()
		self.entries.thaw()
		self.entries.grab_focus()

		if self.entries.selection:
			self.entries.moveto(self.entries.selection[0], colnum, 0.5, 0)
			self.set_focus_row(self.entries.selection[0], self.entries.rows)

	def edit_up(self, widget=None, event=None):
		if not self.entries.selection:
			return
		row = self.entries.selection[0]
		if row != 0:
			self.entries.swap_rows(row, row-1)
			self.set_focus_row(row-1, self.entries.rows)
			self.apply(widget, event)

	def edit_down(self, widget=None, event=None):
		if not self.entries.selection:
			return
		row = self.entries.selection[0]
		if row+1 < self.entries.rows:
			self.entries.swap_rows(row+1, row)
			self.set_focus_row(row+1, self.entries.rows)
			self.apply(widget, event)

	def edit_duplicate(self, widget=None, event=None):
		if not self.entries.selection:
			return
		row = self.entries.selection[0]
		name = self.entries.get_text(row, 0)
		script = self.entries.get_text(row, 1)
		self.entries.insert(row, (name, script))

	def edit_copy(self, widget=None, event=None):
		if not self.entries.selection:
			return
		script = self.entries.get_text(self.entries.selection[0], 1)
		if not script:
			return
		self.app.insert(script)

	def select(self, clist, row, col, event):
		name   = unicode(self.entries.get_text(row, 0), "utf-8")
		script = unicode(self.entries.get_text(row, 1), "utf-8")
		self.name_text.set_text(name)
		self.script_text.set_text(script)

	def initialize(self):
		self.entries.freeze()
		self.entries.clear()
		for p in self.manager.list():
			p = string.split(p)
			self.entries.append(p)
		self.entries.thaw()
		self.name_text.set_text('')
		self.script_text.set_text('')
		self.entries.select_row(0, 0)
		self.set_focus_row(0, self.entries.rows)
		self.entries.grab_focus()

	def delete(self, widget=None, event=None):
		row = 0
		if self.entries.selection:
			row = self.entries.selection[0]
		self.entries.freeze()
		self.entries.remove(row)
		self.entries.thaw()
		self.set_focus_row(row-1, self.entries.rows)

	def insert_new(self, widget=None, event=None):
		row = 0
		name = ''
		script = ''
		if self.entries.selection:
			row = self.entries.selection[0] + 1
			name   = self.name_text.get_text()
			script = self.script_text.get_text()
		self.entries.freeze()
		self.entries.insert(row, (name, script))
		self.entries.thaw()
		self.set_focus_row(row, self.entries.rows)

	def apply(self, widget=None, event=None):
		if self.entries.selection:
			name   = self.name_text.get_text()
			script = self.script_text.get_text()
			row = self.entries.selection[0]
			self.entries.freeze()
			self.entries.set_text(row, 0, name)
			self.entries.set_text(row, 1, script)
			self.entries.thaw()

		p = []
		for r in range(self.entries.rows):
			n = unicode(self.entries.get_text(r, 0), "utf-8")
			s = unicode(self.entries.get_text(r, 1), "utf-8")
			p.append("%s\t%s" % (n, s))
		self.manager.overwrite(p)
		self.manager.save_database()

	def okey(self, widget=None, event=None):
		self.apply(widget, event)
		self.initialize()
		self.close()

	def cancel(self, widget=None, event=None):
		self.initialize()
		self.close()

	def set_focus_row(self, row, row_count):
		if row_count == 0:
			return
		if row_count == 1:
			r = 0
		else:
			r = min((row+0.00001)/(row_count-1),1.0)
		self.entries.emit("scroll-vertical", gtk.SCROLL_JUMP, r)

	def open(self, widget=None, data=None):
		self.window.show()
		self.entries.grab_focus()

	def close(self, widget=None, e=None, data=None):
		self.window.hide()
		return gtk.TRUE

	def set_self_msg(self):
		self.msg = {
			"close": unicode(_("Close"), "utf-8"),
			"copy to main editor": unicode(_("Copy to main editor"), "utf-8"),
			"copy name":   unicode(_("Copy name"), "utf-8"),
			"copy script":   unicode(_("Copy script"), "utf-8"),
			"cut":	 unicode(_("Cut"), "utf-8"),
			"new":   unicode(_("New"), "utf-8"),
			"duplicate": unicode(_("Duplicate"), "utf-8"),
			"up":    unicode(_("Up"), "utf-8"),
			"down":  unicode(_("Down"), "utf-8"),
			}

	def get_menu_items(self):
		menu_items = (
			( "/%s" % self.app.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["close"]), "<control>q", self.close, 0, "<StockItem>", gtk.STOCK_CLOSE ),

			( "/%s" % self.app.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy to main editor"]),
			  None, self.edit_copy, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy name"]),
			  None, self.edit_copy_clipboard_defines, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy script"]),
			  "<control>c", self.edit_copy_clipboard_script, 0, "<StockItem>", gtk.STOCK_COPY ),
			( "/%s/sep" % self.app.msg["edit"][1], None, None, 0, '<Separator>' ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["new"]), None, self.insert_new, 0, "<StockItem>", gtk.STOCK_NEW ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["duplicate"]), None, self.edit_duplicate, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.app.msg["delete"]), None, self.delete, 0, "<StockItem>", gtk.STOCK_DELETE ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["up"]), None, self.edit_up, 0, "<StockItem>", gtk.STOCK_GO_UP ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["down"]), None, self.edit_down, 0, "<StockItem>", gtk.STOCK_GO_DOWN ),

			( "/%s" % self.app.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s" % (self.app.msg["help"][1], self.app.msg["about"]), None, self.app.about, 0, "<StockItem>", gtk.STOCK_DIALOG_INFO ),
			)
		return menu_items
