# -*- coding: utf-8 -*-
#
#  loglist.py - Log list view
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: loglist.py,v 1.6 2009/12/04 17:43:31 atzm Exp $
#

import gtk, gobject
import time, re

try:
	import migemo
except ImportError:
	pass

import config
from common import *

class LogList(gtk.TreeView):
	LISTSTORE_PLAYING  = 0
	LISTSTORE_DATETIME = 1
	LISTSTORE_GHOST    = 2
	LISTSTORE_CHANNEL  = 3
	LISTSTORE_VOTES    = 4
	LISTSTORE_AGREES   = 5
	LISTSTORE_URL      = 6
	LISTSTORE_SCRIPT   = 7
	LISTSTORE_ALL      = -1

	def __init__(self):
		liststore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING,
								  gobject.TYPE_STRING, gobject.TYPE_STRING,
								  gobject.TYPE_STRING, gobject.TYPE_STRING,
								  gobject.TYPE_STRING, gobject.TYPE_STRING)
		gtk.TreeView.__init__(self, liststore)

		columns = [
			gtk.TreeViewColumn(unicode(_("Datetime"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Ghost"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Channel"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Votes"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Agrees"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Script"), "utf-8")),
			]

		l = 0
		for i in xrange(len(columns)):
			self.append_column(columns[i])
			if l == self.LISTSTORE_PLAYING or l == self.LISTSTORE_URL:
				cell = gtk.CellRendererPixbuf()
				columns[i].pack_start(cell, False)
				columns[i].add_attribute(cell, 'stock-id', l)
				l += 1
			cell = gtk.CellRendererText()
			columns[i].pack_start(cell, False)
			columns[i].add_attribute(cell, 'text', l)
			columns[i].set_resizable(True)
			columns[i].set_clickable(False)
			l += 1

		self.set_rules_hint(True)
		self.set_search_column(True)
		self.set_reorderable(False)
		self.set_size_request(450, 120)
		self.connect('cursor-changed', self.cursor_changed)
		self.show()

		self.mids = []
		self.playing_mid = ''
		self.url_regex = re.compile('(https?|ftp|file)://')

	def columns(self):
		return len(self.get_columns())

	def rows(self):
		liststore = self.get_model()
		if liststore is None:
			return 0
		return len(liststore)

	def is_selected(self):
		try:
			treeselection = self.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			if not treeiter:
				return False
		except:
			return False
		return True

	def selection(self):
		if self.is_selected():
			treeselection = self.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			pathes = liststore.get_path(treeiter)
			return pathes[0]
		else:
			raise RuntimeError('LogList is not selected.')

	def get_text(self, row, col):
		liststore = self.get_model()
		if liststore is None:
			return ''
		liter = liststore.get_iter(row)
		return liststore.get_value(liter, col)

	def cursor_changed(self, loglist, se=False):
		loglist.emit('select-cursor-row', False)

	def tab_changed(self):
		self.grab_focus()
		if self.is_selected():
			return self

	def set_local_log(self, loglist):
		loglist.reverse()
		liststore = self.get_model()
		if liststore is None:
			return None
		liststore.clear()
		for datetime, mid, votes, agrees, channel, ghost, script in loglist:
			date = ''

			# old format?
			try:
				date = time.strptime(datetime, '%Y%m%d%H%M%S')
			except ValueError:
				pass

			# Bottle Client for Win or new GBotter format?
			try:
				date = time.strptime(datetime, '%y/%m/%d %H:%M:%S')
			except ValueError:
				pass

			if not date:
				continue

			date = time.strftime("%y/%m/%d %H:%M:%S", date)
			votes  = str(votes)
			agrees = str(agrees)
			if self.url_regex.search(script):
				line = (STATE_BOTTLE_NONE, date, ghost, channel, "%2d" % int(votes), "%2d" % int(agrees), gtk.STOCK_JUMP_TO, script)
			else:
				line = (STATE_BOTTLE_NONE, date, ghost, channel, "%2d" % int(votes), "%2d" % int(agrees), '', script)
			liststore.append(line)
			self.mids.append(mid)

	def search(self, arrow, text, widget=None, data=None):
		if not self.is_selected():
			return None

		if globals().has_key('migemo') and config.get('migemo', 'migemo_enabled', 'boolean'):
			m = migemo.Migemo(config.get('migemo', 'migemo_dict'))
			pat = re.compile(m.query(unicode(text, 'utf-8')).encode('utf-8'), re.IGNORECASE)
		else:
			pat = re.compile(text, re.IGNORECASE)

		current = self.selection()
		rng = []

		if arrow == gtk.ARROW_UP:
			rng = range(0, current)
			rng.reverse()
		elif arrow == gtk.ARROW_DOWN:
			rng = range(current+1, self.rows())

		for i in rng:
			for j in [self.LISTSTORE_DATETIME, self.LISTSTORE_GHOST, self.LISTSTORE_CHANNEL,
					  self.LISTSTORE_VOTES, self.LISTSTORE_AGREES, self.LISTSTORE_SCRIPT]:
				m = pat.search(self.get_text(i, j))
				if m:
					self.set_cursor(i)
					return None

	def update(self, mid, channel, ghost, script, datetime):
		if self.url_regex.search(script):
			line = (STATE_BOTTLE_NONE, datetime, ghost, channel, "%2d" % 0, "%2d" % 0, gtk.STOCK_JUMP_TO, script)
		else:
			line = (STATE_BOTTLE_NONE, datetime, ghost, channel, "%2d" % 0, "%2d" % 0, '', script)
		liststore = self.get_model()
		if liststore is None:
			return None
		liststore.prepend(line)
		self.mids.insert(0, mid)

	def cancel_play(self):
		if not self.is_selected():
			return None
		try:
			mid = self.mids[self.selection()]
		except:
			print 'Invalid MID array? This is a bug!'
			return None
		self.set_bottle_state(mid, STATE_BOTTLE_NONE)
		return mid

	def cancel_play_all(self):
		liststore = self.get_model()
		if liststore is None:
			return None
		for i in xrange(len(liststore)):
			liststore[i][self.LISTSTORE_PLAYING] = STATE_BOTTLE_NONE

	def get_bottle_state(self, mid):
		liststore = self.get_model()
		if liststore is None:
			return STATE_BOTTLE_NONE
		try:
			row = self.mids.index(mid)
		except:
			return STATE_BOTTLE_NONE
		return liststore[row][self.LISTSTORE_PLAYING]

	def set_bottle_state(self, mid, stock=STATE_BOTTLE_PLAYING):
		liststore = self.get_model()
		if liststore is None:
			return mid
		try:
			row = self.mids.index(mid)
		except:
			return None
		if liststore is None:
			return None
		iter = liststore.get_iter(row)
		liststore.set_value(iter, self.LISTSTORE_PLAYING, stock)
		return mid

	def set_playing_bottle(self, mid):
		self.clear_playing_bottle()
		self.set_bottle_state(mid)
		self.playing_mid = mid

	def clear_playing_bottle(self, state=STATE_BOTTLE_NONE):
		if self.set_bottle_state(self.playing_mid, state) is not None:
			self.playing_mid = ''

	def has_mid(self, mid):
		if self.mids.count(mid) > 0:
			return True
		else:
			return False

	def log_votes(self, mid, type, num):
		col = -1
		if type == VOTE:
			col = self.LISTSTORE_VOTES
		elif type == AGREE:
			col = self.LISTSTORE_AGREES
		else:
			return

		try:
			row = self.mids.index(mid)
		except:
			return
		liststore = self.get_model()
		if liststore is None:
			return
		iter = liststore.get_iter(row)
		liststore.set_value(iter, col, "%2d" % num)

	def voting(self, widget, data, parent):
		if not self.is_selected():
			return None, None

		try:
			mid = self.mids[self.selection()]
		except:
			print 'Invalid MID array? This is a bug!'
			return None, None

		if mid is None:
			return None, None

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Vote this bottle?"), 'utf-8'),
										parent)
		d.show()
		res = d.run()
		d.destroy()

		if not res or res == gtk.RESPONSE_NO:
			return None, None
		elif res == gtk.RESPONSE_YES:
			return mid, VOTE
		return None, None

	def agreeing(self, widget, data, parent):
		if not self.is_selected():
			return None, None

		try:
			mid = self.mids[self.selection()]
		except:
			print 'Invalid MID array? This is a bug!'
			return None, None

		if mid is None:
			return None, None

		d = get_simple_yes_or_no_dialog(unicode(_("Really?"), 'utf-8'),
										unicode(_("Agree this bottle?"), 'utf-8'),
										parent)
		d.show()
		res = d.run()
		d.destroy()

		if not res or res == gtk.RESPONSE_NO:
			return None, None
		elif res == gtk.RESPONSE_YES:
			return mid, AGREE
		return None, None

	def get_plain_array(self):
		lines = []
		for row in xrange(self.rows()):
			line = []

			for col in [self.LISTSTORE_DATETIME, self.LISTSTORE_GHOST, self.LISTSTORE_CHANNEL,
						self.LISTSTORE_VOTES, self.LISTSTORE_AGREES, self.LISTSTORE_SCRIPT]:
				line.append(self.get_text(row, col))

			line.append(self.mids[row])
			lines.append(line)
		return lines
