# -*- coding: utf-8 -*-
#
#  svgmanager.py - The Manager of SVG
#  Copyright (C) 2004 by Takuya KAWAHARA <num@sann.ne.jp>
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: svgmanager.py,v 1.52 2009/12/14 14:28:56 atzm Exp $

import gtk, gobject
import os, sys, shutil
import string, re

try:
	import migemo
except ImportError:
	pass

from showingsurface      import ShowingSurface
from common              import *
from viewer.viewercommon import *
from viewer.ghostmanager import create_ghost
import config

class SVGManagerWindow(gtk.Window):
	def __init__(self, app, ghostmanager):
		self.app = app
		self.ghostmanager = ghostmanager

		accel_group = gtk.AccelGroup()

## Create status (left) window
		## Create GhostList window

		self.ghost_list_str = gtk.ListStore(gobject.TYPE_BOOLEAN, gobject.TYPE_STRING,
											gobject.TYPE_BOOLEAN,  gobject.TYPE_STRING)

		self.ghost_list = gtk.TreeView(self.ghost_list_str)
		self.ghost_list.set_enable_search(True)
		self.ghost_list.set_rules_hint(True)
		self.ghost_list.set_search_column(True)
		self.ghost_list.set_reorderable(True)
		self.ghost_list.connect("cursor-changed", self.cursor_changed)
		self.ghost_list.connect("select-cursor-row", self.select_row)

		self.get_config()

		# make ghostlist column
		columns = [
			gtk.TreeViewColumn(unicode(_('Use'), 'utf-8')),
			gtk.TreeViewColumn(unicode(_('Ghost'), 'utf-8')),
			gtk.TreeViewColumn(unicode(_('FMO'), 'utf-8')),
			gtk.TreeViewColumn(unicode(_('Path'), 'utf-8')),
		]
		for i in xrange(len(columns)):
			columns[i].set_resizable(True)
			columns[i].set_clickable(True)
			columns[i].connect("clicked", self.sort, i)
			self.ghost_list.append_column(columns[i])

			if i == 0:
				cell = gtk.CellRendererToggle()
				cell.connect("toggled", self.check_use_toggle)
				columns[i].pack_start(cell, True)
				columns[i].add_attribute(cell, 'active', i)
			elif i == 2:
				cell = gtk.CellRendererToggle()
				cell.connect("toggled", self.check_fmo_toggle)
				columns[i].pack_start(cell, True)
				columns[i].add_attribute(cell, 'active', i)
			else:
				cell = gtk.CellRendererText()
				columns[i].pack_start(cell, True)
				columns[i].add_attribute(cell, 'text', i)
				if   i == 1: cell.set_property('width', 80)
				elif i == 2: cell.set_property('width', 40)

		self.sw = gtk.ScrolledWindow()
		self.sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.sw.set_shadow_type(gtk.SHADOW_IN)
		self.sw.add(self.ghost_list)

		# Create GhostInformation main
		self.sakura_box = gtk.Image()
		self.unyuu_box  = gtk.Image()

		sakura_f = gtk.Frame()
		sakura_f.set_border_width(0)
		sakura_f.set_size_request(SIZE_SURFACE_H + 1, SIZE_SURFACE_V + 1)
		sakura_f.set_shadow_type(gtk.SHADOW_NONE)
		sakura_f.add(self.sakura_box)

		unyuu_f = gtk.Frame()
		unyuu_f.set_border_width(0)
		unyuu_f.set_size_request(SIZE_SURFACE_H + 1, SIZE_SURFACE_V + 1)
		unyuu_f.set_shadow_type(gtk.SHADOW_NONE)
		unyuu_f.add(self.unyuu_box)

		#info_ghost is 'showing surface [0] and [10]'
		info_ghost = gtk.HBox(False)
		info_ghost.pack_start(unyuu_f, True, False, 0)
		info_ghost.pack_start(sakura_f, True, False, 0)

		self.button_id = None
		self.url_button = get_icon_button(gtk.STOCK_JUMP_TO, size=gtk.ICON_SIZE_BUTTON,
										  relief=gtk.RELIEF_NORMAL, label=unicode(_('URL'), 'utf-8'))
		self.url_button.set_sensitive(False)
		self.url_button.add_accelerator("clicked",  accel_group, ord('J'),
										gtk.gdk.CONTROL_MASK|gtk.gdk.MOD1_MASK,
										gtk.ACCEL_VISIBLE)

		self.button_id_s = None
		self.sur_button = get_icon_button(gtk.STOCK_INDEX, size=gtk.ICON_SIZE_BUTTON,
										  relief=gtk.RELIEF_NORMAL, label=unicode(_('Surface'), 'utf-8'))
		self.sur_button.set_sensitive(False)
		self.sur_button.add_accelerator("clicked",  accel_group, ord('S'),
										gtk.gdk.CONTROL_MASK|gtk.gdk.MOD1_MASK,
										gtk.ACCEL_VISIBLE)

		buttonbox = gtk.HBox(True)
		buttonbox.pack_start(self.url_button, False, False, 0)
		buttonbox.pack_start(self.sur_button, False, False, 0)

		info_ghost_a = gtk.VBox(False)
		info_ghost_a.pack_start(info_ghost, True, False, 0)
		info_ghost_a.pack_start(buttonbox, False, False, 0)

		## Showing Ghost Status from svg.txt
		self.info_svg_str = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
		info_svg = gtk.TreeView(self.info_svg_str)
		info_svg.set_headers_visible(False)
		info_svg.set_rules_hint(True)

		svg_cols = [
			gtk.TreeViewColumn(),
			gtk.TreeViewColumn(unicode(_('Ghost Status'), 'utf-8'))
		]
		for i in xrange(len(svg_cols)):
			info_svg.append_column(svg_cols[i])
			cell = gtk.CellRendererText()
			svg_cols[i].pack_start(cell, True)
			svg_cols[i].add_attribute(cell, 'text', i)

		isv = gtk.ScrolledWindow()
		isv.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		isv.set_size_request(200, 200)
		isv.set_shadow_type(gtk.SHADOW_IN)
		isv.add(info_svg)

		info_hbox = gtk.HBox()
		info_hbox.pack_start(info_ghost_a, False, False, 0)
		info_hbox.pack_start(isv, True, True, 0)

		v_paned = gtk.VPaned()
		v_paned.pack1(info_hbox)
		v_paned.pack2(self.sw)

		self.ghost_name = gtk.Entry()
		self.ghost_name.set_editable(False)

		# Create Search Entry
		self.msg = unicode(_("Search"), "utf-8")
		self.search_entry = gtk.Entry()
		self.search_entry.set_text(self.msg)
		self.search_entry.connect("activate", self.search)
		self.search_entry.connect("key-press-event", self.entry_keypress)
		self.search_entry.connect("focus-in-event", self.set_entry_guide, 'in')
		self.search_entry.connect("focus-out-event", self.set_entry_guide, 'out')

		self.s_arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_NONE)

		search_allow_button = gtk.ToggleButton()
		search_allow_button.add(self.s_arrow)
		search_allow_button.connect('clicked', 
				lambda x: self.s_arrow.set((not self.s_arrow.get_property('arrow-type')),
				gtk.SHADOW_NONE))
		search_allow_button.connect('focus-in-event', lambda x, y: self.search_entry.grab_focus())

		search_box = gtk.HBox()
		search_box.pack_start(search_allow_button, False, True, 0)
		search_box.pack_start(self.search_entry)

		status_vbox = gtk.VBox(False, 0)
		status_vbox.pack_start(self.ghost_name, False, False, 0)
		status_vbox.pack_start(v_paned, True, True, 5)
		status_vbox.pack_start(search_box, False, False, 0)

## Create Bottons (right) window
		ctrl_vbox = gtk.VBox(False, 0)
		for item in [
			[gtk.STOCK_ADD,     unicode(_('Add'),    'utf-8'), self.add_ghost,    None],
			[gtk.STOCK_YES,     unicode(_('FMO'),    'utf-8'), self.check_fmo,    None],
			[gtk.STOCK_NO,      unicode(_('Use'),    'utf-8'), self.check_use,    None],
			[gtk.STOCK_REMOVE,  unicode(_('Remove'), 'utf-8'), self.delete_ghost, [ord('D'), gtk.gdk.CONTROL_MASK]],
			[gtk.STOCK_REFRESH, unicode(_('Reload'), 'utf-8'), self.reload,       [ord('R'), gtk.gdk.CONTROL_MASK]],
			[gtk.STOCK_SAVE,    unicode(_('Save'),   'utf-8'), self.save,         [ord('S'), gtk.gdk.CONTROL_MASK]],
			[gtk.STOCK_CLOSE,   unicode(_('Close'),  'utf-8'), self.close,        [ord('Q'), gtk.gdk.CONTROL_MASK]],
			]:
			btn = get_icon_button(item[0], size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL, label=item[1])
			btn.connect('clicked', item[2])

			if item[3] is not None:
				btn.add_accelerator('clicked', accel_group, item[3][0], item[3][1], gtk.ACCEL_VISIBLE)

			ctrl_vbox.pack_start(btn, True, True, 5)

		main_hbox = gtk.HBox(False, 0)
		main_hbox.pack_start(status_vbox, True, True, 0)
		main_hbox.pack_start(ctrl_vbox, False, False, 5)

## Create Base Window
		gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
		self.connect("delete-event", self.close)
		self.set_title(unicode(APP, "utf-8") + ": " + unicode(_("SVGManager"), "utf-8"))
		self.set_default_size(config.get('svgmanager', 'width', 'int'),
							  config.get('svgmanager', 'height', 'int'))
		self.add_accel_group(accel_group)
		self.add(main_hbox)
		self.clear_info()

	def open(self, widget=None, data=None):
		self.show_all()

	def close(self, widget=None, event=None, data=None):
		self.hide()
		return True

	def sort(self, treeviewcol, col_num):
		id_ = self.ghost_list_str.get_sort_column_id()
		if id_[0] == col_num:
			self.ghost_list_str.set_sort_column_id(col_num, not id_[1])
		else:
			self.ghost_list_str.set_sort_column_id(col_num, gtk.SORT_DESCENDING)

	def search(self, widget=None, data=None):
		text = self.search_entry.get_text()

		if globals().has_key('migemo') and config.get('migemo', 'migemo_enabled', 'boolean'):
			m = migemo.Migemo(config.get('migemo', 'migemo_dict'))
			pat = re.compile(m.query(unicode(text, 'utf-8')).encode('utf-8'), re.IGNORECASE)
		else:
			pat = re.compile(text, re.IGNORECASE)

		current = self.selection()
		rng = []

		if self.s_arrow.get_property('arrow-type') == gtk.ARROW_UP:
			if current == None: current = len(self.ghost_list_str)
			rng = range(0, current)
			rng.reverse()
		elif self.s_arrow.get_property('arrow-type') == gtk.ARROW_DOWN:
			if current == None: current = 0
			rng = range(current+1, len(self.ghost_list_str))

		for i in rng:
			for j in [1, 3]:
				m = pat.search(self.ghost_list_str.get_value(self.ghost_list_str.get_iter(i), j))
				if m:
					self.ghost_list.set_cursor(i)
					return None

	def is_selected(self):
		try:
			treeselection = self.ghost_list.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			if not treeiter:
				return False
		except:
			return False
		return True

	def selection(self):
		if self.is_selected():
			treeselection = self.ghost_list.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			pathes = liststore.get_path(treeiter)
			return pathes[0]
		else:
			return None

	def entry_keypress(self, widget, event):
		return self.scroll(widget, event)

	def set_entry_guide(self, widget, event, data):
		if data == 'in':
			if widget.get_text() == self.msg:
				widget.set_text('')
		elif data == 'out':
			if not widget.get_text():
				widget.set_text(self.msg)

	def scroll(self, widget, event):
		if event.state == 1:            # shift
			if event.keyval == 65362:   # up
				self.sw.emit('scroll-child', gtk.SCROLL_PAGE_UP, False)
				return True
			elif event.keyval == 65364: # down
				self.sw.emit('scroll-child', gtk.SCROLL_PAGE_DOWN, False)
				return True
			return False
		return False

## 'cursor-changed' signal gtk.TreeView
	def cursor_changed(self, treeview):
		treeview.emit("select-cursor-row", False)

	def set_info(self, ghost_name='', sakura='', unyu='', craftman='', hpname='', hpurl='',
				 supported='', action='', text_path='', surface_path='', version='', info=''):
		status = [
			['sakura',    sakura],
			['unyu',      unyu],
			['craftman',  craftman],
			['HPname',    hpname],
			['URL',       hpurl],
			['supported', supported],
			['action',    action],
			['DEF.',      text_path],
			['SUF.',      surface_path],
			['Ver.',      version],
			['info',      info],
		]

		self.info_svg_str.clear()
		for l in status:
			self.info_svg_str.append(l)
		self.ghost_name.set_text(ghost_name)

	def clear_info(self):
		self.sur_button.set_sensitive(False)
		self.url_button.set_sensitive(False)
		self.sakura_box.set_from_pixbuf(gtk.gdk.pixbuf_new_from_xpm_data(XPM_VANISH))
		self.unyuu_box.set_from_pixbuf(gtk.gdk.pixbuf_new_from_xpm_data(XPM_VANISH))
		self.set_info()
		self.disconnect_buttons()

	def error_info(self, txt_path):
		self.sur_button.set_sensitive(False)
		self.url_button.set_sensitive(False)
		self.sakura_box.set_from_stock(gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_LARGE_TOOLBAR)
		self.unyuu_box.set_from_stock(gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_LARGE_TOOLBAR)
		self.set_info(unicode(txt_path, 'utf-8') + unicode(_(' is NOT existing!!'), 'utf-8'))
		self.disconnect_buttons()

	def disconnect_buttons(self):
		if self.button_id:
			self.url_button.disconnect(self.button_id)
		self.button_id = None

		if self.button_id_s:
			self.sur_button.disconnect(self.button_id_s)
		self.button_id_s = None

	def select_row(self, treeview, se=False):
		if len(self.ghost_list_str) <= 0:
			self.clear_info()
			return True

		txt_path = expand_quote(self.ghost_list_str.get_value(self.ghost_list_str.get_iter(self.ghost_list.get_cursor()[0][0]), 3))
		txt_path = os.path.join(self.ghostmanager.get_basedir(), txt_path)
		ghost    = create_ghost(txt_path)
		if ghost is None:
			self.error_info(txt_path)
			return True

		## get and show sakura, unyuu surfaces
		sur_sakura = ghost.get_surface(0, SIDE_SAKURA)
		sur_unyuu  = ghost.get_surface(10, SIDE_KERO)

		self.sakura_box.set_from_pixbuf(sur_sakura)
		self.unyuu_box.set_from_pixbuf(sur_unyuu)

		hpurl = ghost.get_url()

		self.set_info(ghost.get_ghostname(), ghost.get_sakura(), ghost.get_unyu(), ghost.get_craftman(),
					  ghost.get_hpname(), hpurl, ghost.get_supported(), ghost.get_sakura_action(), txt_path,
					  os.path.join(os.path.dirname(txt_path), ghost.get_surfacefile()),
					  ghost.get_update(), ghost.get_inform())

		self.disconnect_buttons()

		if hpurl:
			self.url_button.set_sensitive(True)
			self.button_id = self.url_button.connect("clicked", self.url_jump, hpurl)
		else:
			self.url_button.set_sensitive(False)

		self.sur_button.set_sensitive(True)
		self.button_id_s = self.sur_button.connect("clicked", self.show_surface, ghost)

	def _copy_svg(self, src_file, dest_file):
		if os.path.exists(src_file) and os.path.exists(dest_file) and os.path.samefile(src_file, dest_file):
			return True

		if os.path.exists(dest_file):
			d = get_simple_yes_or_no_dialog(unicode(_('Really?'), 'utf-8'),
											unicode(_('File already exists.'), 'utf-8') + \
											' ' + unicode(_('overwrite?'), 'utf-8') + \
											'\n%s' % dest_file, self)
			d.show()
			res = d.run()
			d.destroy()
			if not res or res != gtk.RESPONSE_YES:
				return False

		dest_dir = os.path.dirname(dest_file)
		if not os.path.isdir(dest_dir):
			try:
				os.makedirs(dest_dir)
			except OSError:
				return False

		try:
			shutil.copyfile(src_file, dest_file)
		except:
			return False
		return True

	def copy_svg(self, src_textfile, dest_dir):
		src_textfile = os.path.abspath(src_textfile)

		if not os.path.exists(src_textfile):
			open_error_dialog(unicode(_('File not found'), 'utf-8') + ': %s' % src_textfile, self)
			return None

		ghost = create_ghost(src_textfile)
		if not ghost:
			open_error_dialog(unicode(_('Invalid file'), 'utf-8') + ': %s' % src_textfile, self)
			return None

		imagefile     = ghost.get_surfacefile()
		src_imagefile = os.path.join(os.path.dirname(ghost.get_path()), imagefile)
		if not os.path.exists(src_imagefile):
			open_error_dialog(unicode(_('File not found'), 'utf-8') + ': %s' % src_imagefile, self)
			return None

		dest_textfile  = os.path.join(dest_dir, os.path.basename(src_textfile))
		dest_imagefile = os.path.join(dest_dir, os.path.basename(src_imagefile))

		if self._copy_svg(src_textfile, dest_textfile):
			if self._copy_svg(src_imagefile, dest_imagefile):
				return create_ghost(dest_textfile)
			return None
		return None

## if pressed button
	def add_ghost(self, button):
		ffilter = gtk.FileFilter()
		ffilter.add_pattern('*.txt')
		filew = gtk.FileChooserDialog(parent=self, action=gtk.FILE_CHOOSER_ACTION_OPEN,
									  buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
											   gtk.STOCK_OK, gtk.RESPONSE_OK))
		filew.set_filter(ffilter)
		filew.set_select_multiple(False)
		filew.connect("destroy", lambda x: filew.destroy())
		filew.show()
		res = filew.run()
		src_svg_file = filew.get_filename()
		filew.destroy()

		if res != gtk.RESPONSE_OK:
			return True

		dest_dir = os.path.join(self.ghostmanager.get_basedir(), 'Ghost')
		ghost    = self.copy_svg(src_svg_file, dest_dir)
		if not ghost:
			sys.stderr.write("!!! couldn't copy %s\n" % src_svg_file)
			return True

		ghost_name = ghost.get_sakura2()
		if not ghost_name:
			ghost_name = ghost.get_sakura()

		if not ghost_name and not ghost.get_ghostname():
			open_error_dialog(unicode(_("Couldn't open file"), 'utf-8'), self)
			return True

		dest_relpath = os.path.join(os.path.basename(dest_dir), os.path.basename(ghost.get_path()))
		self.ghost_list_str.append([True, ghost_name, False, '"%s"' % dest_relpath])
		self.ghost_list.set_cursor(len(self.ghost_list_str)-1)
		return True

	def check_fmo(self, button):
		try:
			i = self.ghost_list.get_cursor()[0][0]
			now = self.ghost_list.get_selection()
			[model, iter] = now.get_selected()
			if model.get_value(iter, 2):
				self.delete_fmo(model, iter)
			else:
				self.set_fmo(model, iter)
			self.ghost_list.set_cursor(i)
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def check_use(self, button):
		try:
			i = self.ghost_list.get_cursor()[0][0]
			now = self.ghost_list.get_selection()
			[model, iter] = now.get_selected()
			if model.get_value(iter, 0):
				self.unuse_ghost(model, iter)
			else:
				self.use_ghost(model, iter)
			self.ghost_list.set_cursor(i)
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def delete_ghost(self, button):
		try:
			i = self.ghost_list.get_cursor()[0][0]
			self.ghost_list_str.remove(self.ghost_list_str.get_iter(i))
			if i == len(self.ghost_list_str):
				self.ghost_list.set_cursor(i-1)
			else:
				self.ghost_list.set_cursor(i)
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def reload(self, button):
		self.get_config()
		self.app.notify_svg_changed()

	def save(self, button):
		self.set_config()
		self.app.notify_svg_changed()

	def get_fmo_ghostnames(self):
		return self.ghostmanager.get_fmo_ghostnames()

	def check_use_toggle(self, cellrenderer, path):
		model = self.ghost_list_str
		iter = model.get_iter(path)
		try:
			if self.ghost_list_str.get_value(iter, 0):
				self.unuse_ghost(model, iter)
			else:
				self.use_ghost(model, iter)
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def check_fmo_toggle(self, cellrenderer, path):
		model = self.ghost_list_str
		iter = model.get_iter(path)
		try:
			if self.ghost_list_str.get_value(iter, 2):
				self.delete_fmo(model, iter)
			else:
				self.set_fmo(model, iter)
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def url_jump(self, button, hpurl):
		try:
			os.system(config.get('browser', 'browser_command') % hpurl + ' &')
		except:
			open_error_dialog(unicode(_('No selected Ghost!'), 'utf-8'), self)

	def show_surface(self, button, ghost):
		try:
			path   = os.path.join(os.path.dirname(ghost.get_path()), ghost.get_surfacefile())
			pixbuf = gtk.gdk.pixbuf_new_from_file(path)
		except:
			return
		ss = ShowingSurface(self.app, pixbuf, ghost)

	def get_config(self):
		self.ghost_list_str.clear()
		self.ghostmanager.load_ghost_text()
		for line in self.ghostmanager.get_ghosts().values():
			self.ghost_list_str.append(line)

	def set_config(self):
		self.ghostmanager.save_ghost_text(self.ghost_list_str)

	def use_ghost(self, model, iter):
		## delete '#' from  first item '#GHOST'
		self.ghost_list_str.set_value(iter, 0, True)

	def unuse_ghost(self, model, iter):
		## insert '#' in first item 'GHOST'
		self.ghost_list_str.set_value(iter, 0, False)

	def set_fmo(self, model, iter):
		## insert 'FMO' the third item
		self.ghost_list_str.set_value(iter, 2, True)

	def delete_fmo(self, model, iter):
		## delete 'FMO' the third item 'FMO'
		self.ghost_list_str.set_value(iter, 2, False)

if __name__ == "__main__":
	sm = SVGManagerWindow()
	gtk.main()
