#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  ghost.py - The Ghost
#  Copyright (C) 2004 by Takuya KAWAHARA <num@sann.ne.jp>
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: ghost.py,v 1.23 2009/12/15 18:45:55 atzm Exp $
#

__version__ = '$Revision: 1.23 $'
__author__ = '$Author: atzm $'

import gtk
import os
import sys
import string
import re
import random
from types import *

from viewercommon import *
import config

def _debugprint(desc, text):
	if __debug__:
		try:
			text = unicode(text, 'sjis')
		except TypeError:
			pass
		print >> sys.stderr, '%-15s: %s' % (desc, text.encode('euc-jp', 'ignore'))

class Ghost:
	def __init__(self, path):
		_debugprint('Ghost def text', path)

		self.path           = path           # abspath
		self.splite_dic     = {}             # surface_id -> splite_id
		self.surface_dic    = {}             # splite_id  -> surface_id
		self.alias_dic      = {SIDE_SAKURA: {}, SIDE_KERO: {}}
		self.sequential_dic = {}
		self.ghost_pixbuf   = None

		self.svg_dic = {
			'sakura':        None,
			'unyu':          None,
			'user':          None,
			'craftman':      None,
			'url':           None,
			'hpname':        None,
			'surfacefile':   None,
			'update':        None,
			'sakura2':       None,
			'ghostname':     None,
			'supported':     None,
			'sakura-action': None,
			'inform':        None,
			'shellname':     None,           # ???
			'version':       None,           # ???
			# 'alias' and 'sequential' are handled specially
		}

		self.load_deftext()

	def __str__(self):
		sakura    = self.get_sakura()
		ghostname = self.get_ghostname()
		sakura2   = self.get_sakura2()

		if sakura:
			return sakura
		if ghostname:
			return ghostname
		if sakura2:
			return sakura2
		return '<Ghost unnamed>'

	def get_path(self):
		return self.path

	def get_sakura(self):
		return self.svg_dic['sakura']

	def get_unyu(self):
		return self.svg_dic['unyu']

	def get_user(self):
		return self.svg_dic['user']

	def get_craftman(self):
		return self.svg_dic['craftman']

	def get_url(self):
		return self.svg_dic['url']

	def get_hpname(self):
		return self.svg_dic['hpname']

	def get_surfacefile(self):
		return self.svg_dic['surfacefile']

	def get_update(self):
		return self.svg_dic['update']

	def get_sakura2(self):
		return self.svg_dic['sakura2']

	def get_ghostname(self):
		if not self.svg_dic['ghostname']:
			return self.get_sakura()
		return self.svg_dic['ghostname']

	def get_supported(self):
		return self.svg_dic['supported']

	def get_sakura_action(self):
		return self.svg_dic['sakura-action']

	def get_inform(self):
		return self.svg_dic['inform']

	def get_surface_id(self, splite_id):
		return self.surface_dic[splite_id]

	def get_splite_id(self, surface_id):
		return self.splite_dic[surface_id]

	def get_alias_surface_id(self, surface_id, side):
		try:
			return random.choice(self.alias_dic[side][surface_id])
		except KeyError:
			return surface_id

	def get_surface(self, surface_id, side):
		return self._get_surface(surface_id, side, self._get_surface_pixbuf)

	def get_surface_seq(self, surface_id, side):
		return self._get_surface(surface_id, side, self._get_surface_pixbuf_seq)

	def _get_surface(self, surface_id, side, func):
		try:
			surface_id = int(surface_id)
		except ValueError:
			surface_id = -1

		if config.get('gviewer', 'use_alias', 'boolean'):
			surface_id = self.get_alias_surface_id(surface_id, side)

		try:
			splite_id = int(self.get_splite_id(surface_id))
		except KeyError:
			if surface_id == -1:
				splite_id = -1
			elif side == SIDE_KERO:
				splite_id = 10
			else:
				splite_id = 0

		return func(splite_id)

	def _get_surface_pixbuf_seq(self, splite_id):
		try:
			splites = self.sequential_dic[splite_id]
		except KeyError:
			return [self._get_surface_pixbuf(splite_id)]

		surfaces  = []
		last_surf = '*'
		for splt in splites:
			if splt == '':
				surfaces.append(last_surf)
			elif splt == '*':
				surfaces.append('*')
			elif splt == '-2':
				break
			else:
				try:
					last_surf = self._get_surface_pixbuf(int(splt))
					surfaces.append(last_surf)
				except ValueError:
					_debugprint('Invalid sequence contains', str(splite_id))

		return surfaces

	def _get_surface_pixbuf(self, splite_id):
		# try to load ghost image file if not cached
		if self.ghost_pixbuf is None:
			self.load_image()

		# return XPM_VANISH if loading failed
		if self.ghost_pixbuf is None:
			splite_id = -1

		if splite_id < 0:
			area = gtk.gdk.pixbuf_new_from_xpm_data(XPM_VANISH)
		else:
			svg_width  = self.ghost_pixbuf.get_width()
			svg_height = self.ghost_pixbuf.get_height()

			x = svg_width  / SIZE_SURFACE_H
			y = svg_height / SIZE_SURFACE_V
			a = splite_id  % x
			b = splite_id  / x

			area = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, True, 8,
								  SIZE_SURFACE_H, SIZE_SURFACE_V)
			self.ghost_pixbuf.copy_area(SIZE_SURFACE_H * a, SIZE_SURFACE_V * b,
										SIZE_SURFACE_H, SIZE_SURFACE_V, area, 0, 0)

		return area

	def load_image(self):
		surf_file = self.get_surfacefile()
		if surf_file is None:
			return 

		image_path = os.path.join(os.path.dirname(self.path), surf_file)
		try:
			self.ghost_pixbuf = gtk.gdk.pixbuf_new_from_file(image_path)
		except:
			_debugprint('Could not load', image_path)
			self.ghost_pixbuf = None

	def load_deftext(self):
		file_ = None
		try:
			file_ = open(self.path)
		except:
			_debugprint('open error', self.path)
			pass
		if not file_:
			return

		# discard first line. it's version string :P
		file_.next()

		last_splite = 0
		for line in file_:
			line = string.rstrip(line)

			if not line:                     # empty line
				continue
			if line[:4] == '/EOF':           # pseudo eof
				break
			if line[0] == '/':               # comment line
				continue

			# only surface definition has newline (shorthand)
			if line.find('=') > 0 and line[:8] != 'surface=':
				keyval = line.split('=', 1)

				if len(keyval) != 2:
					_debugprint('Invalid line(1)', line)
					continue

				key = keyval[0].strip()
				val = expand_quote(unicode(keyval[1].strip(), 'sjis', 'replace'))

				if key in self.svg_dic:
					self.svg_dic[key] = val

				elif key == 'sequential':
					if not self._parse_sequential(val):
						_debugprint('Invalid sequential', line)

				elif key == 'alias':
					if not self._parse_alias(val):
						_debugprint('Invalid alias', line)

				else:
					_debugprint('Invalid line(2)', line)

				continue

			if line[:8] == 'surface=':
				line = line[8:]              # for longhand (surface=1:1...)
				if not line:
					continue

			for d in line.split(','):
				if not d:
					continue

				keyval = d.split(':', 1)

				if len(keyval) != 2:
					_debugprint('Invalid surface defs(1)', d)
					continue

				if not keyval[0] or not keyval[1]:
					_debugprint('Invalid surface defs(2)', d)
					continue

				surf_id, splt_id = keyval

				if splt_id == '-1':          # do not display like \s[-1]
					continue
				if splt_id == '-2':          # undefined surface
					continue

				if splt_id == '*':
					try:                     # 1:*   --> 1:1
						i = int(surf_id)
						self.splite_dic[i] = last_splite = i

					except ValueError:       # 1-3:* --> 1:1,2:2,3:3
						surf_range = surf_id.split('-', 1)
						start = int(surf_range[0])
						end   = int(surf_range[1])
						for i in xrange(start, end+1):
							self.splite_dic[i] = last_splite = i

				elif splt_id == '+':
					try:                     # 0:10,1:+   --> 1:11
						i = int(surf_id)
						self.splite_dic[i] = last_splite = last_splite + 1

					except ValueError:       # 0:10:1-3:+ --> 1:11,2:12:3:13
						surf_range = surf_id.split('-', 1)
						start = int(surf_range[0])
						end   = int(surf_range[1])
						for i in xrange(start, end+1):
							self.splite_dic[i] = last_splite = last_splite + 1

				else:
					try:                     # 0:10   --> 0:10
						self.splite_dic[int(surf_id)] = last_splite = int(splt_id)

					except ValueError:       # 1-3:10 --> 1:10,2:10,3:10
						surf_range = surf_id.split('-', 1)
						start = int(surf_range[0])
						end   = int(surf_range[1])
						s     = int(splt_id)
						for i in xrange(start, end+1):
							self.splite_dic[i] = last_splite = s

		for i in self.splite_dic.iterkeys():
			if i not in self.surface_dic:
				self.surface_dic[self.splite_dic[i]] = i

	# '[30=31>>32>>*]' --> {30: ['31', '', '32', '', '*']}
	def _parse_sequential(self, data):
		try:
			idx, seq = data[1:-1].split('=', 1)
			splites  = seq.split('>')
			dummy    = int(splites[0])               # check first item
			self.sequential_dic[int(idx)] = splites
		except (ValueError, IndexError):
			return False
		return True

	# '[sakura.22=37|38]' --> {SIDE_SAKURA: {22: [37, 38]}}
	def _parse_alias(self, data):
		try:
			key, val = data[1:-1].split('=', 1)
		except ValueError:
			return False

		try:
			side, idx = key.split('.', 1)
			if side == 'sakura':
				side = SIDE_SAKURA
			elif side == 'kero':
				side = SIDE_KERO
			else:
				raise ValueError
		except ValueError:
			return False

		try:
			self.alias_dic[side][int(idx)] = [int(i) for i in val.split('|')]
		except ValueError:
			return False

		return True
